/***************************************************************************
  qbrewdoc.h
  -------------------
  Recipe document class for QBrew
  -------------------
  begin         September 20th, 1999
  author        David Johnson <david@usermode.org>
  -------------------
  Copyright 1999, 2001, David Johnson
  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this
  list of conditions and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice,
  this list of conditions and the following disclaimer in the documentation
  and/or other materials provided with the distribution.

  3. Neither name of the copyright holders nor the names of its contributors may
  be used to endorse or promote products derived from this software without
  specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE
  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ***************************************************************************/

#ifndef QBREWDOC_H
#define QBREWDOC_H

#include "resource.h"

#include "grain.h"
#include "hops.h"
#include "miscingredient.h"

class OldStore;

// TODO: rename as recipe?

class QBrewDoc : public QObject  {
    Q_OBJECT
public:
    // constructor
    QBrewDoc(QWidget *parent=0);
    // destructor
    ~QBrewDoc();
    
    // create a new document
    void newDoc(const QString &style);
    // save the document into named file
    bool save(const QString &filename);
    // load a document
    bool load(const QString &filename);
    // has the document been modified
    bool isModified();
    // is the document new?
    bool isNew();
    // unset the modified flag
    void unsetModified();

    // add a new grain to the grain list
    GrainIterator addGrain(const QString &name, const double &quantity, const double &extract,
                const double &color, const QString &use);
    // apply a new grain to the grain list
    void applyGrain(GrainIterator it, const QString &name, const double &quantity, const double &extract,
                const double &color, const QString &use);
    // remove a grain from the grain list
    void removeGrain(GrainIterator it);
    //return a copy of the grain list
    GrainList *grainList();

    // add a new hop to the hops list
    HopIterator addHop(const QString &name, const double &quantity, const QString &form,
                const double &alpha, const unsigned &time);
    // apply a new hop to the hops list
    void applyHop(HopIterator it, const QString &name, const double &quantity, const QString &form,
                const double &alpha, const unsigned &time);
    // remove a hop from the hops list
    void removeHop(HopIterator it);
    // return a copy of the hops list
    HopsList *hopsList();

    // add a new misc ingredient to the misc list
    MiscIngredientIterator addMiscIngredient(const QString &name, const double &quantity, const QString &notes);
    // apply a new misc ingredient to the misc list
    void applyMiscIngredient(MiscIngredientIterator it, const QString &name, const double &quantity, const QString &notes);
    // remove a misc ingredient from the misc list
    void removeMiscIngredient(MiscIngredientIterator it);
    // return a copy of the misc list
    MiscIngredientList *miscIngredientList();

    // convert an old style recipe to new format
    bool convert(const QString &filename);

    // get recipe title
    const QString title();
    // get brewer's name
    const QString brewer();
    // get recipe style
    const QString style();
    // get recipe size
    const double size();
    // get whether batch is mashed
    const bool mash();

signals:
    // sends if document has changed (created, loaded)
    void documentChanged();
    // sends if document has been modified
    void documentModified();
    // sends if document needs to be recalculated
    void recalc();
    // sends if the style has changed
    void styleChanged();

public slots:
    // set recipe title
    void setTitle(const QString &title);
    // set brewer
    void setBrewer(const QString &brewer);
    // set recipe style
    void setStyle(const QString &style);
    // set recipe style
    void setSize(double size);
    // set mash flag
    void setMash(bool mash);

private:
    QMap<QString, int> groupmap_;
    QMap<QString, int> fieldmap_;
    QWidget* parent_;
    
    bool modified_;
    bool new_;
    
    QString title_;
    QString brewer_;
    QString style_;
    double size_;
    bool mash_;
    GrainList grainlist_;
    HopsList hoplist_;
    MiscIngredientList misclist_;
};

//////////////////////////////////////////////////////////////////////////////
// Inlined Methods

inline bool QBrewDoc::isModified() { return modified_; }

inline bool QBrewDoc::isNew() { return new_; }

inline void QBrewDoc::unsetModified() { modified_ = false; }

inline const QString QBrewDoc::title() { return title_; }

inline const QString QBrewDoc::brewer() { return brewer_; }

inline const QString QBrewDoc::style() { return style_; }

inline const double QBrewDoc::size() { return size_; }

inline const bool QBrewDoc::mash() { return mash_; }

inline GrainList *QBrewDoc::grainList() { return &grainlist_; }

inline HopsList *QBrewDoc::hopsList() { return &hoplist_; }

inline MiscIngredientList *QBrewDoc::miscIngredientList() { return &misclist_; }

#endif // QBREWDOC_H
