/***************************************************************************
  controller.h
  -------------------
  Control class for QBrew
  -------------------
  Copyright 1999-2004, David Johnson
  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

  1. Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
***************************************************************************/

#ifndef CONTROLLER_H
#define CONTROLLER_H

#include <qguardedptr.h>
#include <qmainwindow.h>

#include "resource.h"
#include "configstate.h"

class QAction;
class QCloseEvent;
class QImage;
class QPopupMenu;
class QTimer;
class QToolBar;

class AlcoholTool;
class Configure;
class HelpWindow;
class HydrometerTool;
class Model;
class View;

/**
 * The Controller class for QBrew. The controller is the "command" interface
 * to the application, and contains the frame window and main menu. This is
 * a singleton class.
 */
class Controller : public QMainWindow {
    Q_OBJECT
public:
    // Return pointer to the controller.
    static Controller *instance();
    // Initialize the controller
    void initialize(const QString &filename = "");

    // return recipe as an ascii string
    QString getRecipeText(); // TODO: put in model?
    
    // return out the base dir for the help documents
    QString docBase();
    // return out the base dir for the data files
    QString dataBase();

    // return image from embedded image data (qembed)
    static const QImage& findImage(const QString& name);

public slots:
    // generate a new document
    void fileNew();
    // open a document
    void fileOpen();
    // open a recent document
    void fileRecent(int item);
    // save a document
    void fileSave();
    // save a document under a different filename
    void fileSaveAs();
    // print the document
    void filePrint();
    
    // toggle the toolbar
    void optionsToolbar();
    // toggle the statusbar
    void optionsStatusbar();
    // display the configuration dialog
    void optionsConfigure();
    // save application settings
    void optionsSaveConfigure();

    // Popup the alcohol calculator tool
    void toolsAlcohol();
    // Popup the hydrometer correction tool
    void toolsHydrometer();

    // displays help contents
    void helpContents();
    // displays primer contents
    void helpPrimer();
    // shows an about dialog for QBrew
    void helpAbout();

    // received if view widgets have changed
    void documentModified();

private:
    // Private constructor.
    Controller();
    // Private destructor
    ~Controller();

    // initialize actions
    void initActions();
    // initialize the menubar
    void initMenuBar();
    // initialize the toolbar
    void initToolBar();
    // initialize the statusbar
    void initStatusBar();
    // initialize autosave
    void initAutoSave();

    // query user to save the current file
    int querySave();
    // query to overwrite existing file
    int queryOverwrite(const QString filename);
    // add a file to recent file menu
    void addRecent(const QString &filename);
    // open a named file
    bool openFile(const QString &filename);
    // backup the file
    bool backupFile();

    // read in configuration
    void readConfig();
    // write out configuration
    void writeConfig();
    // restore application state
    void restoreState();
    // save application state
    void saveState();

    // catch the close event
    void closeEvent(QCloseEvent* e);

    // TODO: temporary conversion function
    void convertPreferences();

private slots:
    // setup recent file menu
    void recentMenuShow();

    // apply states from config dialog
    void applyGeneralState(const GenConfigState &state);
    void applyRecipeState(const RecipeConfigState &state);
    void applyCalcState(const CalcConfigState &state);

    // autosave time
    void autoSave(); 

private:
    friend class nofriendsatall; // to avoid warnings
    static Controller *instance_;

    Model *model_;
    View *view_;

    ConfigState state_;
    QGuardedPtr<Configure> configure_;
    QGuardedPtr<HelpWindow> helpwin_;
    QGuardedPtr<HelpWindow> primerwin_;
    QGuardedPtr<AlcoholTool> alcoholtool_;
    QGuardedPtr<HydrometerTool> hydrometertool_;
    QString filename_;
    bool newflag_;

    // TODO: replace with a QDict, like in pyxis
    QAction *filenew_;
    QAction *fileopen_;
    QAction *filesave_;
    QAction *filesaveas_;
    QAction *fileprint_;
    QAction *filequit_;
    QAction *optionstoolbar_;
    QAction *optionsstatusbar_;
    QAction *optionssetup_;
    QAction *optionssavesetup_;
    QAction *toolhydrometer_;
    QAction *toolalcohol_;
    QAction *helpcontents_;
    QAction *helpprimer_;
    QAction *helpabout_;
    QAction *helpcontext_;

    QPopupMenu* filemenu_;
    QPopupMenu* optionsmenu_;
    QPopupMenu* toolsmenu_;
    QPopupMenu* helpmenu_;
    QToolBar* toolbar_;

    QStringList recent_;
    QPopupMenu* recentmenu_;

    QTimer *autosave_;
    bool backed_;
};

#endif // CONTROLLER_H
