/***************************************************************************
  configure.cpp
  -------------------
  Application configuration dialog
  -------------------
  Copyright 2006 David Johnson
  Please see the header file for copyright and licehnse information
 ***************************************************************************/

#include <QPushButton>
#include <QStyleFactory>

#include "configure.h"
#include "data.h"
#include "quantity.h"
#include "resource.h"

using namespace Resource;

enum {
    GENERAL_PAGE = 0,
    CALC_PAGE,
    RECIPE_PAGE
};

//////////////////////////////////////////////////////////////////////////////
// Construction, Destruction                                                //
//////////////////////////////////////////////////////////////////////////////

Configure::Configure(QWidget* parent)
    : QDialog(parent), calc_(0), general_(0), recipe_(0)
{
    setWindowTitle(TITLE + " - Configure");

    // tab widget
    tabwidget_ = new QTabWidget(this);

    // buttons
    QPushButton *okbutton = new QPushButton("&OK");
    QPushButton *applybutton = new QPushButton("&Apply");
    QPushButton *cancelbutton = new QPushButton("&Cancel");
    QPushButton *defaultsbutton = new QPushButton("&Defaults");
    okbutton->setAutoDefault(true);

    connect(okbutton, SIGNAL(clicked()), this, SLOT(accept()));
    connect(cancelbutton, SIGNAL(clicked()), this, SLOT(reject()));
    connect(applybutton, SIGNAL(clicked()), this, SLOT(slotApply()));
    connect(defaultsbutton, SIGNAL(clicked()), this, SLOT(slotDefault()));

    // pages
    QWidget *page = new QWidget();
    general_ = new Ui::GeneralConfig();
    general_->setupUi(page);
    tabwidget_->insertTab(GENERAL_PAGE, page, "&General");
    page = new QWidget();
    calc_ = new Ui::CalcConfig();
    calc_->setupUi(page);
    tabwidget_->insertTab(CALC_PAGE, page, "&Calculations");
    page = new QWidget();
    recipe_ = new Ui::RecipeConfig();
    recipe_->setupUi(page);
    tabwidget_->insertTab(RECIPE_PAGE, page, "&Recipe");

    // additional setup
    recipe_->batch->
        setSuffix(" " + Data::instance()->defaultSize().unit().symbol());
    connect(calc_->units, SIGNAL(activated(const QString&)),
            this, SLOT(convertBatchSpin(const QString&)));

    // layout
    QHBoxLayout *buttonlayout = new QHBoxLayout();
    buttonlayout->addStretch(1);
    buttonlayout->addWidget(okbutton);
    buttonlayout->addWidget(defaultsbutton);
    buttonlayout->addWidget(applybutton);
    buttonlayout->addWidget(cancelbutton);

    QVBoxLayout *mainlayout = new QVBoxLayout();
    mainlayout->addWidget(tabwidget_);
    mainlayout->addLayout(buttonlayout);
    setLayout(mainlayout);
}

Configure::~Configure() { ; }

//////////////////////////////////////////////////////////////////////////////
// Miscellaneous                                                            //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// setState()
// ----------
// Read in the config to get the states

void Configure::setState(const ConfigState &state)
{
    state_ = state;
    setGeneralState(state_.general);
    setCalcState(state_.calc);
    setRecipeState(state_.recipe);
}

//////////////////////////////////////////////////////////////////////////////
// setGeneralState()
// -----------------
// Set the state for the general page

void Configure::setGeneralState(const GenConfigState &state)
{
    if (general_) {
        general_->lookfeel->clear();
        general_->lookfeel->addItem(QString::null);
        general_->lookfeel->addItems(QStyleFactory::keys());
        general_->showsplash->setChecked(state.showsplash);
        general_->autosave->setChecked(state.autosave);
        general_->saveinterval->setValue(state.saveinterval);
        general_->autobackup->setChecked(state.autobackup);
        general_->loadlast->setChecked(state.loadlast);
        general_->recentnum->setValue(state.recentnum);

        // set combo for lookfeel
        int index;
        if (state.lookfeel.isEmpty()) {
            index = general_->lookfeel->findText(QApplication::style()->objectName(),
                                                 Qt::MatchExactly);
        } else {
            index = general_->lookfeel->findText(state.lookfeel,
                                                 Qt::MatchExactly);
        }
        general_->lookfeel->setCurrentIndex(index);
    }
}

//////////////////////////////////////////////////////////////////////////////
// setCalcState()
// -----------------
// Set the state for the calculation page

void Configure::setCalcState(const CalcConfigState &state)
{
    if (calc_) {
        calc_->efficiency->setValue(state.efficiency);
        calc_->morey->setChecked(state.morey);
        calc_->tinseth->setChecked(state.tinseth);
        QStringList list = (QStringList() << UNIT_METRIC << UNIT_US);
        calc_->units->clear();
        calc_->units->addItems(list);

        // set combo for units string
        int index = calc_->units->findText(state.units, Qt::MatchExactly);
        if (index >= 0) {
            calc_->units->setCurrentIndex(index);
        } else {
            calc_->units->addItem(state.units);
        }
    }
}

//////////////////////////////////////////////////////////////////////////////
// setRecipeState()
// -----------------
// Set the state for the recipe page

void Configure::setRecipeState(const RecipeConfigState &state)
{
    if (recipe_) {
        recipe_->batch->setValue(state.batch);
        recipe_->stylebox->clear();
        recipe_->stylebox->addItems(Data::instance()->stylesList());
        recipe_->hopform->clear();
        recipe_->hopform->addItems(Data::instance()->formsList());
        recipe_->mash->setChecked(state.mash);

        // set combo for style string
        int index = recipe_->stylebox->findText(state.style, Qt::MatchExactly);
        if (index >= 0) {
            recipe_->stylebox->setCurrentIndex(index);
        } else {
            recipe_->stylebox->addItem(state.style);
        }
        // set combo for hopform string
        index =  recipe_->hopform->findText(state.hopform, Qt::MatchExactly);
        if (index >= 0) {
            recipe_->hopform->setCurrentIndex(index);
        } else {
            recipe_->hopform->addItem(state.hopform);
        }
    }
}

//////////////////////////////////////////////////////////////////////////////
// Slots                                                                    //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// accept()
// --------
// OK button was pressed (QTabDialog doesn't have accept signal)

void Configure::accept()
{
    slotApply();
    QDialog::accept();
    emit configureAccept();
}

//////////////////////////////////////////////////////////////////////////////
// slotDefault()
// -------------
// Set dialog to default values

void Configure::slotDefault()
{
    // only default the current page
    switch (tabwidget_->currentIndex()) {
      case GENERAL_PAGE:
          setGeneralState(GenConfigState()); break;
      case CALC_PAGE:
          setCalcState(CalcConfigState()); break;
      case RECIPE_PAGE:
          setRecipeState(RecipeConfigState()); break;
    }
}

//////////////////////////////////////////////////////////////////////////////
// slotApply()
// -----------
// Apply values

void Configure::slotApply()
{
    if (general_) {
        // general page
        state_.general.lookfeel = general_->lookfeel->currentText();
        state_.general.showsplash = general_->showsplash->isChecked();
        state_.general.autosave = general_->autosave->isChecked();
        state_.general.saveinterval = general_->saveinterval->value();
        state_.general.autobackup = general_->autobackup->isChecked();
        state_.general.loadlast = general_->loadlast->isChecked();
        state_.general.recentnum = general_->recentnum->value();
        emit generalApply(state_.general);
    }

    if (calc_) {
        // calc page
        state_.calc.efficiency = calc_->efficiency->value();
        state_.calc.morey = calc_->morey->isChecked();
        state_.calc.tinseth = calc_->tinseth->isChecked();
        state_.calc.units = calc_->units->currentText();
        emit calcApply(state_.calc);
    }

    if (recipe_) {
        // recipe page
        state_.recipe.batch = recipe_->batch->value();
        state_.recipe.style = recipe_->stylebox->currentText();
        state_.recipe.hopform = recipe_->hopform->currentText();
        state_.recipe.mash = recipe_->mash->isChecked();
        emit recipeApply(state_.recipe);
    }
}

//////////////////////////////////////////////////////////////////////////////
// convertBatchSpin()
// ------------------
// Set the appropriate suffice for the batch spinbox

void Configure::convertBatchSpin(const QString &selection)
{
    if (selection == Resource::UNIT_US)
        recipe_->batch->setSuffix(" " + Volume::gallon.symbol());
    else if (selection == Resource::UNIT_METRIC)
        recipe_->batch->setSuffix(" " + Volume::liter.symbol());
    else
        recipe_->batch->setSuffix("");
}
