/***************************************************************************
  databasetool.cpp
  -------------------
  Database editor for QBrew
  -------------------
  Copyright 2005-2006, David Johnson
  Please see the header file for copyright and license information
 ***************************************************************************/

#include <QDir>
#include <QFile>
#include <QHeaderView>
#include <QMessageBox>
#include <QTableView>

#include "data.h"
#include "resource.h"

#include "graindelegate.h"
#include "grainmodel.h"
#include "hopdelegate.h"
#include "hopmodel.h"
#include "miscdelegate.h"
#include "miscmodel.h"
#include "styledelegate.h"
#include "stylemodel.h"
#include "databasetool.h"

//////////////////////////////////////////////////////////////////////////////
// Construction, Destruction                                                //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// DatabaseTool()
// --------------
// Constructor

DatabaseTool::DatabaseTool(QWidget* parent)
    : QMainWindow(parent), grainmodel_(0), hopmodel_(0), miscmodel_(0),
      grainview_(0), hopview_(0), miscview_(0), modified_(false)
{
    ui.setupUi(this);
    statusBar()->hide();
    ui.actionsave->setEnabled(false);

    // setup actions
    connect(ui.actionsave, SIGNAL(triggered()), this, SLOT(fileSave()));
    connect(ui.actionquit, SIGNAL(triggered()), this, SLOT(close()));

    // get current font information, for sizing
    QFont tinyfont(font());
    tinyfont.setPointSize(tinyfont.pointSize()-2);
    QFontMetrics fm(font());
    unsigned mw = fm.width('M');

    // grain page
    grains_ = Data::instance()->grainmap_.values();
    grainmodel_ = new GrainModel(this, &grains_, 0);
    grainview_ = new QTableView();

    grainview_->setModel(grainmodel_);
    QItemDelegate *delegate = new GrainDelegate(this);
    grainview_->setItemDelegate(delegate);

    grainview_->setSelectionBehavior(QAbstractItemView::SelectRows);
    grainview_->setSelectionMode(QAbstractItemView::SingleSelection);
    grainview_->verticalHeader()->hide();
    grainview_->horizontalHeader()->setClickable(true);
    grainview_->verticalHeader()->setFont(tinyfont);

    grainview_->setColumnHidden(Grain::QUANTITY, true);
    grainview_->setColumnWidth(Grain::NAME, 20*mw);
    grainview_->setColumnWidth(Grain::EXTRACT, 8*mw);
    grainview_->setColumnWidth(Grain::COLOR, 8*mw);
    grainview_->setColumnWidth(Grain::USE, 8*mw);

    grainview_->resizeRowsToContents();

    ui.tabs->addTab(grainview_, "&Grains");

    // hop page
    hops_ = Data::instance()->hopmap_.values();
    hopmodel_ = new HopModel(this, &hops_);
    hopview_ = new QTableView();

    hopview_->setModel(hopmodel_);
    delegate = new HopDelegate(this);
    hopview_->setItemDelegate(delegate);

    hopview_->setSelectionBehavior(QAbstractItemView::SelectRows);
    hopview_->setSelectionMode(QAbstractItemView::SingleSelection);
    hopview_->verticalHeader()->hide();
    hopview_->horizontalHeader()->setClickable(true);
    hopview_->verticalHeader()->setFont(tinyfont);

    hopview_->setColumnHidden(Hop::QUANTITY, true);
    hopview_->setColumnHidden(Hop::TIME, true);
    hopview_->setColumnHidden(Hop::FORM, true);
    hopview_->setColumnWidth(Hop::NAME, 20*mw);
    hopview_->setColumnWidth(Hop::ALPHA, 8*mw);

    hopview_->resizeRowsToContents();

    ui.tabs->addTab(hopview_, "&Hops");

    // misc page
    miscs_ = Data::instance()->miscmap_.values();
    miscmodel_ = new MiscModel(this, &miscs_);
    miscview_ = new QTableView();

    miscview_->setModel(miscmodel_);
    delegate = new MiscDelegate(this);
    miscview_->setItemDelegate(delegate);

    miscview_->setSelectionBehavior(QAbstractItemView::SelectRows);
    miscview_->setSelectionMode(QAbstractItemView::SingleSelection);
    miscview_->verticalHeader()->hide();
    miscview_->horizontalHeader()->setClickable(true);
    miscview_->verticalHeader()->setFont(tinyfont);

    miscview_->setColumnHidden(Misc::QUANTITY, true);
    miscview_->setColumnWidth(Misc::NAME, 20*mw);
    miscview_->horizontalHeader()->setStretchLastSection(true);

    miscview_->resizeRowsToContents();

    ui.tabs->addTab(miscview_, "&Miscellaneous");

    // style page
    styles_ = Data::instance()->stylemap_.values();
    stylemodel_ = new StyleModel(this, &styles_);
    styleview_ = new QTableView();

    styleview_->setModel(stylemodel_);
    delegate = new StyleDelegate(this);
    styleview_->setItemDelegate(delegate);

    styleview_->setSelectionBehavior(QAbstractItemView::SelectRows);
    styleview_->setSelectionMode(QAbstractItemView::SingleSelection);
    styleview_->verticalHeader()->hide();
    styleview_->horizontalHeader()->setClickable(true);
    styleview_->verticalHeader()->setFont(tinyfont);

    styleview_->setColumnWidth(Style::NAME, 20*mw);
    styleview_->setColumnWidth(Style::OGLOW, 8*mw);
    styleview_->setColumnWidth(Style::OGHI, 8*mw);
    styleview_->setColumnWidth(Style::FGLOW, 8*mw);
    styleview_->setColumnWidth(Style::FGHI, 8*mw);
    styleview_->setColumnWidth(Style::IBULOW, 8*mw);
    styleview_->setColumnWidth(Style::IBUHI, 8*mw);
    styleview_->setColumnWidth(Style::SRMLOW, 8*mw);
    styleview_->setColumnWidth(Style::SRMHI, 8*mw);

    styleview_->resizeRowsToContents();

    ui.tabs->addTab(styleview_, "&Styles");

    // setup connections
    connect(grainmodel_, SIGNAL(modified()), this, SLOT(dataModified()));
    connect(grainmodel_, SIGNAL(rowsInserted(const QModelIndex&, int, int)),
            grainview_, SLOT(resizeRowsToContents()));
    connect(hopmodel_, SIGNAL(modified()), this, SLOT(dataModified()));
    connect(hopmodel_, SIGNAL(rowsInserted(const QModelIndex&, int, int)),
            hopview_, SLOT(resizeRowsToContents()));
    connect(miscmodel_, SIGNAL(modified()), this, SLOT(dataModified()));
    connect(miscmodel_, SIGNAL(rowsInserted(const QModelIndex&, int, int)),
            miscview_, SLOT(resizeRowsToContents()));
    connect(stylemodel_, SIGNAL(modified()), this, SLOT(dataModified()));
    connect(stylemodel_, SIGNAL(rowsInserted (const QModelIndex&, int, int)),
            styleview_, SLOT(resizeRowsToContents()));
    connect(grainview_->horizontalHeader(), SIGNAL(sectionClicked(int)),
            grainview_, SLOT(sortByColumn(int)));
    connect(hopview_->horizontalHeader(), SIGNAL(sectionClicked(int)),
            hopview_, SLOT(sortByColumn(int)));
    connect(miscview_->horizontalHeader(), SIGNAL(sectionClicked(int)),
            miscview_, SLOT(sortByColumn(int)));
    connect(styleview_->horizontalHeader(), SIGNAL(sectionClicked(int)),
            styleview_, SLOT(sortByColumn(int)));
}

DatabaseTool::~DatabaseTool() {}

void DatabaseTool::fileSave()
{
    QString localbase = QDIR_HOME + "/." + Resource::DATA_FILE;

    QFileInfo finfo(localbase);
    if (finfo.exists() && !finfo.isWritable()) {
        // no write permission
        QMessageBox::warning(this, Resource::TITLE,
                             QString("<p>Unable to save the database."
                                     "You do not have permission "
                                     "to write to %1").arg(localbase));
    } else {
        // sync with Data...
        foreach(Grain grain, grains_) {
            Data::instance()->insertGrain(grain); // replaces existing
        }
        foreach(Hop hop, hops_) {
            Data::instance()->insertHop(hop); // replaces existing
        }
        foreach(Misc misc, miscs_) {
            Data::instance()->insertMisc(misc); // replaces existing
        }
        foreach(Style style, styles_) {
            Data::instance()->insertStyle(style); // replaces existing
        }

        if (!Data::instance()->saveData(localbase)) {
            // error in saving file
            QMessageBox::warning(this, Resource::TITLE,
                                 QString("<p>Unable to save the database."
                                         "Error in saving %1").arg(localbase));
        }
        ui.actionsave->setEnabled(false);
        modified_ = false;
    }
}

void DatabaseTool::dataModified()
{
    ui.actionsave->setEnabled(true);
    modified_ = true;
}
