/*
 * this file is a part of QPxTool project
 * Copyright (C) 2006, Gennady "ShultZ" Kozlov <qpxtool@mail.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; see the file COPYING.  If not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

// const unsigned char op_blacklist[]={ 0xAC, 0xDE, 0xDF , 0xF8 };

const unsigned char op_blacklist[]={ 0xDE, 0xDF, 0xF8 };  // used in TEST mode
const int           op_blacklist_sz=sizeof(op_blacklist);

int op_blacklisted(unsigned char opcode){
    if (op_blacklist_sz) for (int i=0; i<op_blacklist_sz; i++) {
	if (op_blacklist[i] == opcode) return 1;	
    }
    return 0;
}

#include <stdio.h>
#include <stddef.h>
#include <stdlib.h>
#include <string.h>

#include <sys/stat.h>

#include <qpx_mmc.h>
#include <plextor_features.h>

#include <math.h>

#include "pxfw.h"

#include "version.h"

long fsize(FILE* f){
	struct stat st;
	fstat(fileno(f),&st);
	return st.st_size;
}


int custom_command (drive_info* drive, unsigned char opcode) {
	drive->cmd[0]=opcode;
	if ((drive->err=drive->cmd.transport(NONE,NULL,0))) return drive->err;
	return 0;
}

void FW_convert_to_ID (int* dev_ID, char* buf) {
	if (!strncmp(buf,"PLEXTOR CD-R   PREMIUM",22)) *dev_ID=PX_PREMIUM;
	else
	if (!strncmp(buf,"PLEXTOR DVDR   PX-708",21)) *dev_ID=PX_708;
	else
	if (!strncmp(buf,"PLEXTOR DVDR   PX-712",21)) *dev_ID=PX_712 | PX_708A2;
	else
	if (!strncmp(buf,"PLEXTOR DVDR   PX-716",21)) *dev_ID=PX_716 | PX_714;
	else
	if (!strncmp(buf,"PLEXTOR DVDR   PX-760",21)) *dev_ID=PX_760 | PX_755;
	else
		*dev_ID = 0;
}

void PX_convert_to_ID (drive_info* drive, int* FWSZ, int* FWSZ_CRC) {
	*FWSZ=0;
	*FWSZ_CRC=0;
	if (!strncmp(drive->ven,"PLEXTOR ",8)) {
		drive->ven_ID=WR_PLEXTOR;
		if(!strncmp(drive->dev,"CD-R   PREMIUM",14)) {
			drive->dev_ID=PX_PREMIUM;
			*FWSZ=524288;
			*FWSZ_CRC=509952;
		} else
		if(!strncmp(drive->dev,"DVDR   PX-708A2",15)) {
			drive->dev_ID=PX_708A2;
			*FWSZ=1028096;
			*FWSZ_CRC=1028096;
		} else
		if(!strncmp(drive->dev,"DVDR   PX-708",13)) {
			drive->dev_ID=PX_708;
			*FWSZ=983040;
			*FWSZ_CRC=983040;
		} else
		if(!strncmp(drive->dev,"DVDR   PX-712",13)) {
			drive->dev_ID=PX_712;
			*FWSZ=1028096;
			*FWSZ_CRC=1028096;
		} else
		if(!strncmp(drive->dev,"DVDR   PX-714",13)) {
			drive->dev_ID=PX_714;
			*FWSZ=983040;
			*FWSZ_CRC=983040;
		} else
		if(!strncmp(drive->dev,"DVDR   PX-716",13)) {
			drive->dev_ID=PX_716;
			*FWSZ=983040;
			*FWSZ_CRC=983040;
		} else
		if(!strncmp(drive->dev,"DVDR   PX-755",13)) {
			drive->dev_ID=PX_755;
			*FWSZ=2031616;
			*FWSZ_CRC=2031616;
		} else
		if(!strncmp(drive->dev,"DVDR   PX-760",13)) {
			drive->dev_ID=PX_760;
			*FWSZ=2031616;
			*FWSZ_CRC=2031616;
		} else
			drive->dev_ID=PX_OLD;
	}
}

int plextor_read_eeprom(drive_info* drive, unsigned char offs, unsigned int sz) {
//	char*		data;
	unsigned int	i,j;

	drive->cmd[0] = 0xF1;
	drive->cmd[1] = 0x01;
	drive->cmd[7] = offs;
	drive->cmd[8] = (sz >> 8) & 0xFF;
	drive->cmd[9] = sz & 0xFF;
	if ((drive->err=drive->cmd.transport(READ,drive->rd_buf+offs*sz,sz) ))
		{ sperror ("read EEPROM",drive->err); 	return (0); }

	printf("EEPROM block #%d:\n",offs);
	for(i=0;i<(sz/0x10);i++) {
	    printf("| %X0 | ", i);
	    for(j=0;j<0x10;j++) printf("%02X ",drive->rd_buf[i*0x10+j]);
	    printf("|");
	    for(j=0;j<0x10;j++) {
		if (drive->rd_buf[i*0x10+j] > 0x20) printf("%c",drive->rd_buf[i*0x10+j]);
		else printf(" ");
	    }
	    printf("|\n");
	};
	return 1;
}

int plextor_reboot(drive_info* drive) {
	drive->cmd[0] = 0xEE;
	if ((drive->err=drive->cmd.transport(NONE,NULL,0) ))
		{ sperror ("reboot",drive->err); return 1; }
	return 0;
}

int fwblk_send(drive_info* drive, int offs, int blksz, bool last) {
	drive->cmd[0] = 0x3B;
	drive->cmd[1] = last ? 0x05:0x04;
	drive->cmd[2] = 0x00;
	drive->cmd[3] = (offs >> 16) & 0xFF;
	drive->cmd[4] = (offs >> 8) & 0xFF;
	drive->cmd[5] = offs & 0xFF;
	drive->cmd[6] = (blksz >> 16) & 0xFF;
	drive->cmd[7] = (blksz >> 8) & 0xFF;
	drive->cmd[8] = blksz & 0xFF;
	drive->cmd[9] = 0x00;
	drive->cmd[10]= 0x00;
	drive->cmd[11]= 0x00;

	if ((drive->err=drive->cmd.transport(WRITE,drive->rd_buf,blksz) )){
		sperror ("SEND_FWBLK",drive->err);
		if (!last) return 1;
	}
	return 0;
}

int fwblk_read(drive_info* drive, int offs, int blksz, unsigned char flag) {
        printf("data read...\n");

	drive->cmd[0] = 0x3B;
//	cmd[1] = last ? 0x05:0x04;
	drive->cmd[1] = flag;

	drive->cmd[2] = 0x00;
	drive->cmd[3] = (offs >> 16) & 0xFF;
	drive->cmd[4] = (offs >> 8) & 0xFF;
	drive->cmd[5] = offs & 0xFF;
	drive->cmd[6] = (blksz >> 16) & 0xFF;
	drive->cmd[7] = (blksz >> 8) & 0xFF;
	drive->cmd[8] = blksz & 0xFF;
	drive->cmd[9] = 0x00;
	drive->cmd[10]= 0x00;
	drive->cmd[11]= 0x00;

	if ((drive->err=drive->cmd.transport(READ,drive->rd_buf,blksz) )){
		sperror ("READ_FWBLK",drive->err);
		return 1;
	}
	return 0;
}

void csum_init(unsigned short int *csum){
    *csum = 0;
}

void csum_update(unsigned short int *csum, unsigned char* buf, int len) {
	int		i;
	unsigned int	ad;
	for (i=0; i<len; i++)
		{ ad = buf[i]; *csum += ad; }
}

void usage(char* bin) {
	fprintf (stderr,"\nusage: %s [device] [options]\n",bin);
	printf("\t-h\t\t show help\n");
	printf("\t-v\t\t debug\n");
	printf("\t-u\t\t write FW\n");
	printf("\t-if [file]\t read FW from this file\n");
//	printf("\t-of [file]\t save FW to this file\n");
	printf("\t-f\t\t force flashing, even if drive not recognized or checksum error\n");
	printf("\t-e\t\t read EEPROM\n");
	printf("\t-r\t\t reboot drive\n");
	printf("\t-t\t\t test opcodes supported by drive (EXPERIMENTAL),\n");
	printf("\t\t\t use it only if you know what are you doing!\n");
}

int main(int argc,char *argv[]) {
	drive_info  *drive=0;
	char        *devname=0;
	char	    *fwfname=0;
	FILE	    *fwfile;

	int i,j;
//    char 	wrinfo[0x3F];

	int	FW_dev_ID=0;
//	int	cap;

const	int	fwblk = 4096;
const	int	fwblk_crc= fwblk/2;
	int	blen;
	int	fwblocks;
	int	fwblocks_crc;
	int	fwsz;
	int	crc_offs;
	int	fwfsz;
	int	err;

	int 	last;
	int	flags=0;
	unsigned short int fCSUM=0;
	unsigned short int CSUM=0;
	unsigned short int CSUM_diff=0;

	int crca=0;

	if (argc<1) {
		usage(argv[0]);
		exit (2);
	}
	printf(" ________________________________________\n");
	printf("|\n");
	printf("|** Plextor FW updater v%s (c) ShultZ **\n",VERSION);
	printf("|________________________________________\n");

//	flags = FL_UPDATE;
	if (argc>1) {
//		printf("Parsing additional options...\n");
		for (i=1; i<argc; i++) {
//			printf("arg[%d]: %s\n",i,argv[i]);
			if(!strcmp(argv[i],"-if")) {
				if(argc>(i+1)) {
					i++;
					flags |= FL_FWIF;
					fwfname = argv[i];
				} else {
					printf("Option %s needs parameter\n",argv[i]);
					exit (1);
				}
			}
			else if(!strcmp(argv[i],"-h")) flags |= FL_HELP;
			else if(!strcmp(argv[i],"-u")) flags |= FL_UPDATE;
//			else if(!strcmp(argv[i],"-b")) flags |= FL_BACKUP;
			else if(!strcmp(argv[i],"-v")) flags |= FL_DEBUG;
			else if(!strcmp(argv[i],"-t")) flags |= FL_TEST;
			else if(!strcmp(argv[i],"-e")) flags |= FL_EEPROM;
			else if(!strcmp(argv[i],"-r")) flags |= FL_REBOOT;
			else if(!strcmp(argv[i],"-f")) flags |= FL_FORCE;
			else if (!(flags & FL_DEV)) {
				flags |= FL_DEV;
				devname = argv[1];
			} else {
				printf("Illegal option: %s.\nUse %s -h for details\n",argv[i],argv[0]);
				usage(argv[0]);
				exit (1);
			}
		}
	}
	if (flags & FL_DEBUG) {
		printf("Flags: ");
		if (flags & FL_DEV)    printf(" DEV");
		if (flags & FL_FWIF)   printf(" FWIF");
		if (flags & FL_HELP)   printf(" HELP");
		if (flags & FL_UPDATE) printf(" UPDATE");
		if (flags & FL_BACKUP) printf(" BACKUP");
		if (flags & FL_DEBUG)  printf(" DEBUG");
		if (flags & FL_TEST)   printf(" TEST");
		if (flags & FL_EEPROM) printf(" EEPROM");
		if (flags & FL_REBOOT) printf(" REBOOT");
	}
	if (flags & FL_HELP) {
		usage(argv[0]);
		exit (0);
	}

	if (!(flags & FL_DEV)) {
		printf("\n*** No device selected! ***\n\n");
		usage(argv[0]);
		exit (3);
	}

	drive = new drive_info(devname);
	printf("\nDevice : %s\n",devname);
	inquiry(drive);
	PX_convert_to_ID(drive,&fwsz,&crc_offs);
	detect_mm_capabilities(drive);
	printf("Vendor : '%s'\n",drive->ven); 
	printf("Model  : '%s'", drive->dev);
	if ((drive->ven_ID == WR_PLEXTOR)) {
		plextor_get_TLA(drive);
		printf(" (TLA#%s)",drive->TLA);
	}
	printf("\nF/W    : '%s'\n",drive->fw);
	if (get_drive_serial_number(drive))
		printf("Serial#: %s\n",drive->serial);
	if ((drive->ven_ID != WR_PLEXTOR) && (!(flags & FL_FORCE))){
		printf("%s: Only PLEXTOR drives supported!\n",argv[0]);
		return 1;
	} else {
		if ((drive->dev_ID == PX_OLD)){
			printf("%s: Not supported PLEXTOR drive!\n",argv[0]);
			return 1;
		}
	}

	if (flags & FL_REBOOT) {
		plextor_reboot (drive);
		exit (0);
	}

	if (flags & FL_EEPROM) {
//		unsigned char buf[1024];
		for (unsigned char idx=0; idx<4; idx++) plextor_read_eeprom(drive, idx, 256);
		exit (0);
	}
	
	if (flags & FL_TEST) {
		for (unsigned int opcode=0x0000; opcode<0x0100; opcode++) {
			printf("Trying OpCode: ");
		        print_opcode((unsigned char)opcode);			
			if (!op_blacklisted((unsigned char)opcode)) {			
			        int err = custom_command(drive, (opcode & 0x00FF));
				if (!err) printf ("OK");
				else if (err==-1) printf ("command didn't passed to drive");
				else if (err==0x52000) printf ("*** command not supported ***");
				else print_sense(err);
			} else {
				printf("!!! BLACK LIST !!!");
			}
			printf("\n");
		}
		exit(0);
	}

	if (!(flags & FL_FWIF)) exit(3);

	fwfile = fopen(fwfname, "r");
	if (!fwfile) {
		printf("%s: Can't open file: %s\n",argv[0],fwfname);
		return 1;
	}

//	fwblk=4096;
	fwfsz=fsize(fwfile);
	fwblocks=fwfsz/fwblk;
	fread(drive->rd_buf, fwblk, 1, fwfile);
	const int FB=64;
	const int HH=16;

	printf("FW size:      %d\n",fwsz);
	printf("FW file size: %d\n",fwfsz);
	printf("First %d bytes of FW:\n", FB);

	for (i=0; i<(FB/HH); i++){
		printf("| ");
		for (j=0; j<HH; j++) printf("%02X ",drive->rd_buf[i*HH+j] & 0xFF);
		printf(" | ");
		for (j=0; j<HH; j++)
			if ((drive->rd_buf[i*HH+j] & 0xFF) > 0x20)
				printf("%c",drive->rd_buf[i*HH+j] & 0xFF);
			else
				printf(" ");
		printf(" |\n");
	}
	FW_convert_to_ID(&FW_dev_ID,(char*) drive->rd_buf);
	if (!(drive->dev_ID & FW_dev_ID)) {
		printf("FW is not for selected drive!\n");
		exit(4);
	}
	if (flags & FL_BACKUP) {
		printf("BackUp feature not implemented yet...\n");
		for (i=0; i<fwblk; i++) drive->rd_buf[i]=0;
		i=0;
		if (err = fwblk_read(drive, i*fwblk, fwblk, 2)) {
			printf("** error reading data\n");
		};

		printf("First %d bytes of FW:\n", FB);
		for (i=0; i<(FB/HH); i++){
			printf("| ");
			for (j=0; j<HH; j++) printf("%02X ", drive->rd_buf[i*HH+j] & 0xFF);
			printf(" | ");
			for (j=0; j<HH; j++)
				if ((drive->rd_buf[i*HH+j] & 0xFF) > 0x20)
					printf("%c", drive->rd_buf[i*HH+j] & 0xFF);
				else
					printf(" ");
			printf(" |\n");
		}
	}

	fseek(fwfile,0,SEEK_SET);
	last=0;
	csum_init(&CSUM);
//	printf("fwblocks = %d\n",fwblocks);
	fwblocks_crc=crc_offs/fwblk_crc;
	for (i=0;i<fwblocks_crc;i++) {
//		fseek(fwfile,i*fwblk,SEEK_SET);
		fread(drive->rd_buf, fwblk_crc, 1, fwfile);
		if (i == (fwblocks_crc-1)) last=1;
		if (last) {
		    blen = fwblk_crc - 2;
		    crca = i*fwblk_crc+blen;
		    fCSUM = (drive->rd_buf[fwblk_crc-2] << 8) | drive->rd_buf[fwblk_crc-1];
		} else blen = fwblk_crc;
//		printf("Offset %04X, block # %X\n",i*fwblk,i);
//		printf ("%02X:  %4dB CRC=%04X\n",i,blen,CRC16blk);
		csum_update(&CSUM, drive->rd_buf, blen);
	}
//	fCRC = (FWBUF[fwblk-2] << 8) | FWBUF[fwblk-1];
	CSUM_diff = (CSUM  - fCSUM) & 0xFFFF;
	printf("CheckSum:\n");
	printf("Stored [@%06X]: %04X\n",crca,fCSUM);
	printf("Calculated      : %04X\n",CSUM);
	printf("diff            : %04X\n",CSUM_diff);

	if (flags & FL_UPDATE) {
		if (fwfsz!=fwsz) {
			printf("*** File size does not match FW size! ***\n");
			delete drive;
			fclose(fwfile);
	        	exit(3);
		}
		if (CSUM_diff) {
			if (!(flags & FL_FORCE)) {
				printf("*** CheckSum incorrect! ***\n");
				delete drive;
				fclose(fwfile);
				exit(3);
			} else {
				printf("*** CheckSum incorrect, but -f option found. Force flashing...\n");
			}
		}
		determine_disc_type(drive);
//		printf("Disc type: %02X\n",drive->media.disc_type);
		if (drive->media.disc_type > 1) {
			printf("Disc found, doing eject...\n");
			if (load_eject(drive, 0)) {
				printf("Can't eject disc:( remove disc manualy and try again\n");
				delete drive;
				fclose(fwfile);
				exit (1);
			}
	        }
		printf("Waiting for drive to become ready... ");
		if (!wait_unit_ready(drive,2,0)) {
		    printf(" OK!\n");
		} else {
		    printf("\nDrive not ready! Aborting...\n");
		    delete drive;
		    fclose(fwfile);
		    exit(1);
		}
		last=0;
		printf("Sending FirmWare to drive \n");
		fseek(fwfile,0,SEEK_SET);
//		for (i=0;i<(fwblocks);i++) {
		for (i=0;!last;i++) {
			fread(drive->rd_buf, fwblk, 1, fwfile);
//			printf("Block #%d:\n",i);
			if (feof(fwfile) || (i == (fwblocks-1) )) {
				last=1;
				printf("\nData transfer complete: %d bytes (%X blocks). Updating...\n", (i+1)*fwblk, i+1);
			}
			err = fwblk_send(drive, i*fwblk, fwblk, last);
			if (err) {
			    printf("FW UPDATE ERROR!\n");
			    last=1;
			}
		}
		inquiry(drive);
		printf("FW update complete! New INQUIRY data:\n");
		printf("Vendor : '%s'\n",drive->ven); 
		printf("Model  : '%s'\n",drive->dev);
		printf("F/W    : '%s'\n",drive->fw);
	}
	printf("\n");
	delete drive;
	fclose(fwfile);
	return 0;
}
