/* Copyright (C) 2003 Nikos Chantziaras.
 *
 * This file is part of the QTads program.  This program is free
 * software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 */

#ifndef QTADSIO_H
#define QTADSIO_H

#include "config.h"

#include <vector>

#include <qnamespace.h>


class QString;
class QColor;
class QFont;
class QKeyEvent;
class QTadsMainWindow;
class QTadsGameWindow;
class QTadsStatusLine;
class QTadsSettings;


/* This is a namespace for input/output related functions.  It
 * represents the QTads I/O system.  That way, other code can use the
 * wrapper-routines provided here rather than the Qt-widgets directly.
 *
 * The I/O system is unusable until init() gets called.
 */
namespace QTadsIO {

	// The following 'extern' variables are not maintained by the
	// I/O system; the I/O system's callers are responsible for it.
	// They are just global variables that are needed in various
	// places in the sources.
	//
	// This approach is ugly and unsafe; it may change in the
	// future.

	// This flag indicates if a game is currently executing (or,
	// more precisely, if the current game should continue
	// executing or not).  This is useful for the various
	// input-routines.  If this flag is false, the routines should
	// return an "end of file" indicator to the game.  If the game
	// is well implemented, it will recognize this and quit itself.
	// If not, it may enter an infinite loop and QTads will freeze
	// (and there's no way to prevent this; QTads must be killed if
	// this happens).
	extern bool gameRunning;

	// If this is true, the user should be prompted with a
	// file-selection dialog as soon as the VM stops running.
	extern bool openNewGame;

	// If not empty, this variable contains the game to load.  The
	// user should not be prompted with a file selection dialog.
	extern QString newGameToOpen;

	// If this is true, the current game should be restarted.
	extern bool restartGame;

	// If true, the application should quit.
	extern bool quitApp;

	// Initializes the I/O system.  This function *must* be called
	// before any I/O operation can take place, and it should be
	// called only *once*, or else the system will simply crash
	// without warning; for speed reasons, the functions in the
	// QTadsIO namespace don't check is the interface has been
	// initialized.  For reinitialization, call reset().
	void
	init();

	// When the interface is no longer needed, call this function.
	// The most important thing it does is to save the user's
	// preferences to disk.
	void
	done();

	// Flush any pending output, clear the screen, and reset the
	// I/O system to its initial state.  This function should be
	// called after quitting the current game and before starting a
	// new one (it doesn't matter if the new game is the same as
	// the previous one; reset() must always be called when quiting
	// and opening a game).
	void
	reset();

	// These functions provide access to the widgets we maintain.
	//
	// FIXME: The whole point of the QTadsIO namespace is to hide
	// the widgets used for actual I/O operations.  Therefore,
	// these functions should go away at some point.
	QTadsMainWindow&
	mainWindow();

	QTadsStatusLine&
	statusLine();

	QTadsGameWindow&
	gameWindow();

	// Returns the settings.  They are already initialized.
	QTadsSettings&
	settings();

	// Applies changes in the settings (including the current
	// theme).
	void
	applySettings();

	// Switch to Tads 3 mode.  This must be called with 'true'
	// before opening a Tads 3 game, and with 'false' before
	// opening a Tads 2 game.
	void
	t3Mode( bool yes );

	// Returns true if Tads 3 mode is enabled.  Certain routines
	// must adapt their operations according to whether this
	// returns 'true' or 'false'.
	bool
	t3Mode();

	// Enables/disables non-stop mode.  In non-stop mode, we never
	// show a more-promt, no matter if the text grows larger than
	// the output-window.  The VM sometimes requests this mode, but
	// enabling it on our own is also legal (although there's no
	// reason to).
	void
	nonstopMode( bool yes );

	// Print a string on the main window.  Nothing is shown on the
	// screen until flush() is called.  This speeds up
	// output-operations immensely.
	void
	print( const QString& txt );

	// Print a string on the left side of the statusline.  Since
	// the statusline only contains one line of text, the operation
	// is unbuffered (no need to call flush(); the text appears
	// immediately).
	void
	statusPrint( const QString& txt );

	// Print a string on the right side of the statusline.  Since
	// the statusline only contains one line of text, the operation
	// is unbuffered (no need to call flush(); the text appears
	// immediately).
	void
	scorePrint( const QString& txt );

	// Print a string on the system statusline (the small area
	// below the game window).  The system statusline isn't part of
	// the game-output, so there's no need to call flush() or
	// anything like that.
	void
	sysStatusPrint( const QString& txt );

	// Like sysStatusPrint(const QString&), but delete the text again after
	// 'time_ms' milliseconds.
	void
	sysStatusPrint( const QString& txt, int time_ms );

	// Delete the currently displayed text in the system
	// statusline.  This applies only to text printed with one of
	// the sysStatusPrint() functions; text that is displayed by
	// the system statusline itself is not cleared.
	//
	// What this means:  Text shown by Qt itself is not affected.
	void
	clearSysStatus();

	// Clears the screen (but not the statusline) and puts the
	// cursor to the upper-left corner of the main window.  No need
	// to call flush() after this.
	void
	clear();

	// Flushes the output-buffer (displays all text printed with
	// print()).
	void
	flush();

	// Is the highlight-attribute currently set?
	bool
	highlight();

	// Set/clear the highlight attribute.  This attribute doesn't
	// affect text that has already been printed with print().
	void
	highlight( bool yes );

	// Is the italics-attribute currently set?
	bool
	italics();

	// Set/clear the italics attribute.  This attribute doesn't
	// affect text that has already been printed with print().
	void
	italics( bool yes );

	// Display the [More] prompt and wait for the user to respond.
	void
	morePrompt();

	// Get a return-terminated input from the user.
	QString
	getInput();

	// Wait until a character is available from the keyboard.  The
	// keystroke isn't evaluated in any way.  This function behaves
	// like morePrompt() except that no more-prompt is shown.
	void
	waitChar();

	// Get a character from the user.  The returned QKeyEvent
	// contains the user's keystroke.
	QKeyEvent
	getRawChar();

	// Returns true if the application is in fullscreen-mode.
	bool
	fullScreen();

	// Switch to fullscreen-mode when `yes' is true, or to
	// windowed-mode when `yes' if false.  Output operations and
	// behavior are not affected by this; only the user sees a
	// difference.
	void
	fullScreen( bool yes );

	// Set the window's title (caption).  The user will only see a
	// caption when the interpreter is not in fullscreen-mode.
	void
	title( const QString& str );

	// Enters the command `cmd' as if the user typed it in.  Useful
	// for automatically entering things like "quit", "save", etc,
	// but any command is allowed (since it isn't interpreted in
	// any way; it is simply passed on to the game).
	void
	enterCommand( const QString& cmd );

	// Enables/disables any actions (menu items, toolbuttons) that
	// can generate user commands.  For example, the user won't be
	// able to click on the "Save" button when these actions are
	// disabled.
	//
	// This function assumes that the QTadsIO::gameRunning flag has
	// a correct value, since disabling or enabling certain actions
	// is only possible when the a game is/isn't running.
	void
	enableCommandActions( bool yes );

	// Prompts the user for a file to open and returns the name of
	// the file that the user selected.  If the returned string is
	// empty, it means that the user canceled the dialog without
	// selecting a file.
	//
	// The returned string is a valid path (absolute or relative; Qt
	// lacks a precise specification).
	QString
	openFile( const QString& startWith, const QString& filter, const QString& caption );

	// Same as openFile(), but for saving operations.
	QString
	saveFile( const QString& startWith, const QString& filter, const QString& caption );

	// Displays a dialog with several buttons and waits for the
	// user to click one of them.  The dialog's message is `txt'.
	// The `buttons' vector contains the labels for each button.
	// The amount of buttons is determined by the vector's length.
	// The default button (the pre-selected one) is `def', indexed
	// from 1.  The returned value is the button the user clicked.
	// If 0 is returned, it means that the user canceled the dialog
	// without clicking any of the buttons.
	//
	// The labels of the buttons can contain the "&" character.  In
	// this case, the letter after the "&" will be underlined and
	// used as a shortcut.  "&Yes" for example, will underline the
	// "Y" and pressing Alt+Y will be the same as clicking on the
	// button.  If several buttons have the same shortcut,
	// Alt+[Key] will only cycle through these buttons.
	int
	inputDialog( const QString& txt, const std::vector<QString>& buttons,
		     unsigned int def );

	// Set the application window's size.  It's possible to make
	// the window larger than the desktop; no check is being done
	// to prevent this.
	void
	resizeWindow( int width, int height );

	// Set the application window's position.  It's possible to
	// place the window outside of the desktop; no check is being
	// done to prevent this.
	void
	moveWindow( int x, int y );

	// Quit the game.  How this is accomplished depends on the
	// user's preferences.  Returns true if the game will quit,
	// false otherwise (the game will not quit if the user clicks
	// on "No" in the confirmation dialog).
	bool
	quit();

	// Restart the game.  How this is accomplished depends on the
	// user's preferences.
	void
	restart();

}; // namespace QTadsIO


#endif // QTADSIO_H
