#include "qppdview.h"
#include "qppdbaseitem.h"
#include "qppdgroupitem.h"
#include "qppdoptionitem.h"
#include "qppdnumericaloptionitem.h"
#include "ppd-util.h"

#include "printer.xpm"
#include "printer_adjust.xpm"

#include <qheader.h>
#include <qstring.h>
#include <qmessagebox.h>

QPPDView::QPPDView(global_ppd_file_t *ppd, QWidget *parent, const char *name)
	: QListView(parent,name)
{
	setSizePolicy(QSizePolicy(QSizePolicy::Preferred,QSizePolicy::Expanding));

	addColumn( tr("Options") );
	header()->hide();
	setFrameStyle(QFrame::WinPanel|QFrame::Sunken);

	rootitem_ = 0;
	ppd_ = 0;
	options_.setAutoDelete(false);
	conflict_ = false;
	if (ppd) initialize(ppd);
}

QPPDView::~QPPDView()
{
}

void QPPDView::initialize(global_ppd_file_t *ppd, cups_dest_t *dest)
{
	ppd_ = ppd;
	clear();
	if (!ppd_) return;
	if (dest) globalCupsMarkOptions(ppd,dest->num_options,dest->options);
	options_.clear();

	setSorting(-1);

	rootitem_ = new QPPDBaseItem(ppd_,this);
	rootitem_->setText(0,ppd->ppd->nickname);
	rootitem_->setOpen(true);
	rootitem_->setPixmap(0,QPixmap(printer));

	// add numerical options
	if (ppd_->num_numerical_options > 0)
	{
		QPPDBaseItem	*numericalitem;
		numericalitem = new QPPDBaseItem(ppd_,rootitem_);
		numericalitem->setText(0, tr("Adjustments") );
		numericalitem->setOpen(true);
		numericalitem->setPixmap(0,QPixmap(printer_adjust));
		for (int i=ppd_->num_numerical_options-1;i>=0;i--)
		{
			numerical_opt_t	*opt = ppd_->numerical_options+i;
			new QPPDNumericalOptionItem(ppd_,opt,numericalitem);
		}
	}

	// add regular options
	for (int i=ppd_->ppd->num_groups-1;i>=0;i--)
	{
		ppd_group_t	*group = ppd_->ppd->groups+i;
		new QPPDGroupItem(ppd_,group,rootitem_);
	}
	rootitem_->setupDict(options_);

	//checkConstraints();
	checkConflict();
}

void QPPDView::markChoices(global_ppd_file_t *ppd)
{
	rootitem_->markChoices(ppd);
}

void QPPDView::updateChoices(global_ppd_file_t *ppd)
{
	rootitem_->updateChoices(ppd);
	checkConflict();
}

void QPPDView::getCupsOptions(int& num_options, cups_option_t **options, bool incldef)
{
	rootitem_->getCupsOptions(num_options,options,incldef);
}

void QPPDView::checkConstraints()
{
	checkConflict();
	if (conflict_)
	{
		QString	msg = ppdConflictErrorMsg(ppd_->ppd);
		if (!msg.isEmpty()) QMessageBox::warning(this,tr("Driver configuration"),msg.data(),QMessageBox::Ok|QMessageBox::Default,0);
	}
}

void QPPDView::resizeEvent(QResizeEvent *e)
{
	QListView::resizeEvent(e);
	setColumnWidth(0,width());
}

bool QPPDView::hasConflict()
{
        return conflict_;
}

void QPPDView::checkConflict()
{
	if (!rootitem_ || !ppd_) return;

	// reset all options
	rootitem_->setConflict(false);
	conflict_ = false;

	// update conflict
	conflict_ = (ppdConflicts(ppd_->ppd) > 0);

	// updates conflicting options
	if (conflict_)
	{
		QDictIterator<QPPDBaseItem>	oit(options_);
		for (;oit.current();++oit) oit.current()->updateConflict();
	}
}

//--------------------------------------------------------------------------------

int parseOption(ppd_option_t *opt, QString& msg)
{
	if (opt->conflicted)
	{
		ppd_choice_t	*ch(0);
		for (int i=0;i<opt->num_choices;i++)
		{
			ch = opt->choices+i;
			if (ch->marked) break;
			else ch = 0;
		}
		if (ch)
		{
			QString	msg2;
			msg2.sprintf("  %s = %s\n",opt->text,ch->text);
			msg.append(msg2.data());
			return 1;
		}
	}
	return 0;
}

int parseGroup(ppd_group_t *gr, QString& msg)
{
	int	n(0);
	for (int i=0;i<gr->num_subgroups;i++) n += parseGroup(gr->subgroups+i,msg);
	for (int i=0;i<gr->num_options;i++) n += parseOption(gr->options+i,msg);
	return n;
}

QString ppdConflictErrorMsg(ppd_file_t *ppd)
{
	QString	msg(QObject::tr("You selected options that are in conflict. Try to resolve the\nproblem before continuing:\n\n"));
	int	n(0);
	for (int i=0;i<ppd->num_groups;i++) n += parseGroup(ppd->groups+i,msg);
	return (n == 0 ? QString("") : msg);
}
