/***************************************************************************
 *   Copyright (C) 2006-2009 by Robert Keevil                              *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; under version 2 of the License.         *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.         *
 ***************************************************************************/

#include <iostream>
#include <fstream>
#include <sstream>
#include <cstring>
#include <cstdlib> // putenv
#include <cmath> // abs
#include "libscrobble.h"
#include "md5.h"
#include "ConvertUTF.h"

#ifdef _MSC_VER
typedef int int32_t;
typedef unsigned int uint32_t;
// disable "'function': was declared deprecated"
#pragma warning (disable: 4996)
#endif

#if defined(__MINGW32__) || defined(_MSC_VER)
typedef unsigned int uint;
#endif

using std::string;

CURL *curl;
CURLcode curl_res;
static char errorBuffer[CURL_ERROR_SIZE];
static string curl_buffer;

double u_tot = 0;
double u_now = 0;
bool progress_cancel = false;

namespace {

inline int32_t bufToInt32_t(unsigned char* buf) {
    return (buf[3] << 24) | (buf[2] << 16) | (buf[1] << 8) | buf[0];
}
};

string utf16tostr(char *widestring, size_t widesize)
{
    size_t utf8size = 3 * widesize + 1;  // worst case
    std::string resultstring;
    resultstring.resize(utf8size, '\0');
    const UTF16* sourcestart = reinterpret_cast<UTF16*>(widestring);
    const UTF16* sourceend = sourcestart + widesize;
    UTF8* targetstart = reinterpret_cast<UTF8*>(&resultstring[0]);
    UTF8* targetend = targetstart + utf8size;
    ConvertUTF16toUTF8(&sourcestart, sourceend, &targetstart, targetend, strictConversion);

    *targetstart = 0;
    return resultstring;
}

static size_t curl_writer(char *data, size_t size, size_t nmemb,
                            string *writerData)
{
    if (writerData == NULL)
        return 0;

    writerData->append(data, size*nmemb);

    return size * nmemb;
}

int curl_progress(void *clientp,
                        double dltotal, double dlnow,
                        double ultotal, double ulnow)
{
    (void)clientp;
    (void)dltotal;
    (void)dlnow;

    u_tot = ultotal;
    u_now = ulnow;

    if (progress_cancel)
    {
        progress_cancel = false;
        return 1;
    }
    else
        return 0;
}

void Scrobble::clear_method()
{
    scrobble_method = SCROBBLE_NONE;
}

double Scrobble::get_u_tot(void)
{
    return u_tot;
}

double Scrobble::get_u_now(void)
{
    return u_now;
}

void Scrobble::cancel_submission(void)
{
    progress_cancel = true;
}

size_t split(std::vector<string>& v, const char* s, char c)
{
    v.clear();
    while (true) {
        const char* begin = s;

        while (*s != c && *s)
            ++s;

        v.push_back(string(begin, s));

        if (!*s) {
            break;
        }

        if (!*++s) {
            v.push_back("");
            break;
        }
    }
    return v.size();
}

string Scrobble::int2string(int a)
{
    std::stringstream ss;
    string b;
    ss << a;
    ss >> b;
    return b;
}

int Scrobble::string2int(string s)
{
    std::stringstream ss;
    int r;
    ss << s;
    ss >> r;
    return r;
}

time_t get_gmt()
{
    time_t t;
    char gmt_tzset[]="TZ=UTC+0";
    putenv(gmt_tzset);
    tzset();
    return mktime(localtime((time(&t), &t)));
}

Scrobble::Scrobble()
{
    error_str = "";
    scrob_init = false;
    mtp_connected = false;
    log_print_level = LOG_INFO;

    proxy_host = "";
    proxy_port = 0;
    proxy_user = "";
    proxy_pass = "";

    /* initialise TZ variables */
    tzset();

    // our own copy - returned via get_dst
    is_dst = daylight;
    (is_dst)?zonename=tzname[1]:zonename=tzname[0];

    if (is_dst < 0)
        add_log(LOG_ERROR, "is_dst < 0");

    offset = -(int)timezone;

#ifdef altzone // defined in <ctime>, but only recent(ish) POSIX
    offset = -(int)altzone;
#else
    // this WILL be the case with MSVC
    offset += 3600; // assume 1 hour difference
#endif

    add_log(LOG_DEBUG, "Detected Timezone: " + zonename);
    add_log(LOG_DEBUG, "Detected DST: " + int2string(is_dst));
    add_log(LOG_DEBUG, "Detected Offset: " + offset_str());
}

Scrobble::~Scrobble()
{
    curl_easy_cleanup(curl);
#ifdef HAVE_MTP
    mtp_disconnect();
#endif
}

bool Scrobble::init(int timeout, int connect_timeout)
{
    // sanity check supplied values
    timeout = (timeout < 2)?2:timeout;
    timeout = (timeout > 120)?120:timeout;
    connect_timeout = (connect_timeout < 2)?2:connect_timeout;
    connect_timeout = (connect_timeout > 120)?120:connect_timeout;

    curl_res = curl_global_init( CURL_GLOBAL_NOTHING
#ifdef WIN32
        | CURL_GLOBAL_WIN32
#endif
    );
    if (curl_res != CURLE_OK)
    {
        add_log(LOG_ERROR, "CURL: failed global init");
        return false;
    }

    curl = curl_easy_init();
    if (curl == NULL)
    {
        add_log(LOG_ERROR, "Failed to create CURL connection");
        return false;
    }

    curl_res = curl_easy_setopt(curl, CURLOPT_ERRORBUFFER, errorBuffer);
    if (curl_res != CURLE_OK)
    {
        add_log(LOG_ERROR, "CURL: failed to set CURLOPT_ERRORBUFFER");
        return false;
    }

    curl_res = curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, curl_writer);
    if (curl_res != CURLE_OK)
    {
        add_log(LOG_ERROR, "CURL: failed to set CURLOPT_WRITEFUNCTION: " + string(errorBuffer));
        return false;
    }

    curl_res = curl_easy_setopt(curl, CURLOPT_NOPROGRESS, 0);
    if (curl_res != CURLE_OK)
    {
        add_log(LOG_ERROR, "CURL: failed to set CURLOPT_NOPROGRESS: " + string(errorBuffer));
        return false;
    }

    curl_res = curl_easy_setopt(curl, CURLOPT_PROGRESSFUNCTION, curl_progress);
    if (curl_res != CURLE_OK)
    {
        add_log(LOG_ERROR, "CURL: failed to set CURLOPT_PROGRESSFUNCTION: " + string(errorBuffer));
        return false;
    }

    curl_res = curl_easy_setopt(curl, CURLOPT_TCP_NODELAY, 1);
    if (curl_res != CURLE_OK)
    {
        add_log(LOG_ERROR, "CURL: failed to set CURLOPT_TCP_NODELAY: " + string(errorBuffer));
        return false;
    }

    curl_res = curl_easy_setopt(curl, CURLOPT_WRITEDATA, &curl_buffer);
    if (curl_res != CURLE_OK)
    {
        add_log(LOG_ERROR, "CURL: failed to set CURLOPT_WRITEDATA: " + string(errorBuffer));
        return false;
    }

    curl_res = curl_easy_setopt(curl, CURLOPT_NOSIGNAL, 1);
    if (curl_res != CURLE_OK)
    {
        add_log(LOG_ERROR, "CURL: failed to set CURLOPT_NOSIGNAL: " + string(errorBuffer));
        return false;
    }

    curl_res = curl_easy_setopt(curl, CURLOPT_TIMEOUT, timeout);
    if (curl_res != CURLE_OK)
    {
        add_log(LOG_ERROR, "CURL: failed to set CURLOPT_TIMEOUT: " + string(errorBuffer));
        return false;
    }

    curl_res = curl_easy_setopt(curl, CURLOPT_CONNECTTIMEOUT, connect_timeout);
    if (curl_res != CURLE_OK)
    {
        add_log(LOG_ERROR, "CURL: failed to set CURLOPT_CONNECTTIMEOUT: " + string(errorBuffer));
        return false;
    }

    //curl_easy_setopt(curl, CURLOPT_VERBOSE, 1);

    scrob_init = true;

    scrobble_method = SCROBBLE_NONE;

    return true;
}

bool Scrobble::parse_file(const string& file_path, int tz)
{
    add_log(LOG_INFO, "Opening log file: " + file_path);
    add_log(LOG_INFO, "UTC Offset: " + int2string(tz));

    if (!scrob_init)
    {
        add_log (LOG_ERROR, "init function hasn't been run!");
        return false;
    }

    scrob_entry temp_entry;

    std::vector<string> v;

    std::ifstream input_stream;
    string line;

    entries.clear();

    input_stream.open(file_path.c_str(), std::ios::in);

    if(input_stream.fail())
    {
        add_log(LOG_ERROR, "Parsing failed - unable to open file " + file_path);
        return false;
    }

    // check the first header line, and get + check the version.
    getline(input_stream, line);
    if (line.find("#AUDIOSCROBBLER/") == 0 && line.length() > 16)
    {
        string ver = line.substr(16);
        add_log(LOG_DEBUG, "Log version: " + ver);
        if (ver.find("1.0") == 0)
            have_mb = false;
        else if (ver.find("1.1") == 0)
            have_mb = true;
        else
        {
            add_log(LOG_ERROR, "Unknown log version: " + ver);
            input_stream.close();
            return false;
        }
    }
    else
    {
        add_log(LOG_ERROR, "Parsing failed - missing #AUDIOSCROBBLER header");
        input_stream.close();
        return false;
    }

    // check the second header line, and get the TZ info.
    getline(input_stream, line);
    if (line.find("#TZ/") == 0 && line.length() > 4)
    {
        string log_tz = line.substr(4);
        add_log(LOG_DEBUG, "Log TZ: " + log_tz);
        if (log_tz.find("UNKNOWN") == 0)
        {
            // we are fine, continue as-is
        }
        else if (log_tz.find("UTC") == 0)
        {
            tz = 0;
            add_log(LOG_INFO, "Overriding detected timezone");
        }
        else
        {
            add_log(LOG_ERROR, "Unknown log TZ: " + log_tz);
            input_stream.close();
            return false;
        }
    }
    else
    {
        add_log(LOG_ERROR, "Parsing failed - missing #TZ header");
        input_stream.close();
        return false;
    }

    // check the third header line, and get the client version info.
    getline(input_stream, line);
    if (line.find("#CLIENT/") == 0 && line.length() > 8)
    {
        string client = line.substr(8);
        add_log(LOG_DEBUG, "Log CLIENT: " + client);
    }
    else
    {
        add_log(LOG_ERROR, "Parsing failed - missing #CLIENT header");
        input_stream.close();
        return false;
    }

    while( getline( input_stream, line ) )
    {
        if (line[0] != '#' && line.length() > 0)
        {
            if (split(v, line.c_str(), '\t') == ((have_mb)?8:7))
            {
                //right number of tabs in the line

                temp_entry.artist = v[0];
                temp_entry.album = v[1];
                temp_entry.title = v[2];
                temp_entry.tracknum = v[3];
                temp_entry.length = string2int(v[4]);
                temp_entry.played = v[5][0];
                temp_entry.when = string2int(v[6]) - tz;
                if (have_mb && v[7].length() == 36)
                    temp_entry.mb_track_id = v[7];
                else
                    temp_entry.mb_track_id = "";
                entries.push_back(temp_entry);
            }
        }
    }

    input_stream.close();

    add_log(LOG_DEBUG, "Parsed " + int2string(entries.size()) + " entries");

    if (entries.size() > 0)
        scrobble_method = SCROBBLE_LOG;

    return true;
}

bool Scrobble::parse_db(const std::string& db_folder, int tz)
{
    // Unix time - number of seconds elapsed since January 1, 1970
    // Apple stores time information as number of seconds elapsed
    // since January 1, 1904
    const uint32_t APPLE_TIME = 2082844800;

    have_mb = false;

    std::ifstream input_stream;
    entries.clear();

    int32_t songs;
    uint32_t type;
    uint32_t headerlen;
    uint32_t blocktype;
    uint32_t blocklen;
    uint32_t numtracks;
    uint32_t tracklen;
    uint32_t tracknum;
    uint32_t playcount;
    uint32_t bookmark;
    uint32_t dateplayed;
    uint32_t entrylen;
    scrob_entry* iTunesTable;
    string fileName;

    fileName = db_folder + "/iPod_Control/iTunes/iTunesDB";
    input_stream.open(fileName.c_str(), std::ios::in | std::ios::binary);

    if(input_stream.fail())
    {
        add_log(LOG_ERROR, "Parsing failed - unable to find iTunesDB in " + db_folder);
        return false;
    }

    char real_buffer[8];
    unsigned char* buffer = (unsigned char*)real_buffer;

    input_stream.read (real_buffer,4);
    type = bufToInt32_t(buffer);

    if ( type != 0x6462686D ) {
        add_log(LOG_ERROR, "Parsing failed - not iTunesDB file " + db_folder);
        return false;
    }

    input_stream.read (real_buffer,4);
    headerlen = bufToInt32_t(buffer);
    input_stream.seekg(headerlen);

    int position = -1;
    bool finished = false;
    while( !input_stream.eof() && !finished ) {
      input_stream.read (real_buffer,4);
      blocktype = bufToInt32_t(buffer);
      switch(blocktype) {
        case 0x616C686D: {    // mhla
            input_stream.read (real_buffer,4);
            blocklen = bufToInt32_t(buffer);
            input_stream.seekg(blocklen - 4, std::ios::cur);
        }
        case 0x6169686D: {    // mhia
            input_stream.read (real_buffer,4);
            input_stream.read (real_buffer,4);
            blocklen = bufToInt32_t(buffer);
            input_stream.seekg(blocklen - 8, std::ios::cur);
        }
        case 0x6473686D: {    // mhsd
            input_stream.read (real_buffer,4);
            blocklen = bufToInt32_t(buffer);
            input_stream.read (real_buffer,8);     // skip mhsd size
            input_stream.seekg(blocklen - 16, std::ios::cur);
        }
        break;
        case 0x746C686D: {    // mhlt
            input_stream.read (real_buffer,4);
            blocklen = bufToInt32_t(buffer);
            input_stream.read (real_buffer,4);
            numtracks = bufToInt32_t(buffer);

            //std::cout << "Seems like you have " << (uint)numtracks << " tracks" << std::endl;
            iTunesTable = new scrob_entry[numtracks];
            input_stream.seekg(blocklen - 12, std::ios::cur);    // skip the rest
        }
        break;
        case 0x7469686D: {    // mhit
            position++;
            input_stream.read (real_buffer,4);
            blocklen = bufToInt32_t(buffer);

            if(blocklen < 48) {
                add_log(LOG_INFO, "mhit seems very small, we won't find length nor tracknum");
                break;
            }
            input_stream.seekg(32, std::ios::cur);	// we just need track lenght and tracknumber
            input_stream.read (real_buffer,4);
            tracklen = bufToInt32_t(buffer);
            iTunesTable[position].length = tracklen / 1000;
            input_stream.read (real_buffer,4);
            tracknum = bufToInt32_t(buffer);
            iTunesTable[position].tracknum = int2string(tracknum);
            input_stream.seekg( blocklen - 48, std::ios::cur );	// we can skip rest
        }
        break;
        case 0x646F686D: {     // mhod
            input_stream.read (real_buffer,4);
            blocklen = bufToInt32_t(buffer);
            input_stream.read (real_buffer,4);
            blocklen = bufToInt32_t(buffer);
            input_stream.read (real_buffer,4);
            type = bufToInt32_t(buffer);

                switch ( type ) {
                    case MHOD_TITLE:		// we just need title, album and artist
                    case MHOD_ALBUM:
                    case MHOD_ARTIST: {

                    const size_t len = ( blocklen - 40 ) / sizeof( char );
                    char * buf = new char [ len + 1 ];

                    input_stream.seekg( 24, std::ios::cur );    // skip stuff
                    input_stream.read(buf, len);

                    std::string temp = utf16tostr(buf, len/2);
                    //for ( uint i = 0; i*2 < len; ++i ) temp[i] = buf[i*2];
                    delete [] buf;


                    switch (type) {
                        case MHOD_TITLE: iTunesTable[position].title = temp; break;
                        case MHOD_ALBUM: iTunesTable[position].album = temp; break;
                        case MHOD_ARTIST: iTunesTable[position].artist = temp; break;
                    }
                break;
                }

                default:
                    input_stream.seekg( blocklen -16, std::ios::cur );  // skip the whole block
                break;
            }
        }
        break;
        case 0x706C686D: // mhlp
        case 0x7079686D: // mhyp
        case 0x7069686D: { // mhip
             finished = true;		//we don't need more info
        }
        break;

        default: {
            char tmp[32];
            sprintf(tmp, "%p", (void*)blocktype);
            add_log(LOG_ERROR, "Unknown blocktype: " + string(tmp));
            return false;
        }
      }
    }
    input_stream.close();

    fileName = db_folder + "/iPod_Control/iTunes/Play Counts";
    input_stream.open(fileName.c_str(), std::ios::in | std::ios::binary);

    if(input_stream.fail())
    {
        add_log(LOG_ERROR, "Parsing failed - unable to find Play Counts in " + db_folder);
        return false;
    }

    input_stream.read (real_buffer,4);
    type = bufToInt32_t(buffer);
    if ( type != 0x7064686D ) {
        add_log(LOG_ERROR, "Parsing failed - not Play Counts file " + db_folder);
        return false;
    }

    input_stream.read (real_buffer,4);
    headerlen = bufToInt32_t(buffer);
    input_stream.read (real_buffer,4);
    entrylen = bufToInt32_t(buffer);
    input_stream.read (real_buffer,4);
    songs = bufToInt32_t(buffer);
    input_stream.seekg(headerlen - 16, std::ios::cur);

    if (position+1 != songs) {
        add_log(LOG_ERROR, "Different number of songs in Play Counts and iTunesDB");
        return false;
    }

    for (int x=0; x<songs; x++) {
        iTunesTable[x].mb_track_id = "";
        input_stream.read (real_buffer,4);
        playcount = bufToInt32_t(buffer);
        if (playcount > 0 && !iTunesTable[x].artist.empty() && !iTunesTable[x].title.empty()) {
            input_stream.read (real_buffer,4);
            dateplayed = bufToInt32_t(buffer);
            input_stream.read (real_buffer,4);
            bookmark = bufToInt32_t(buffer);

            iTunesTable[x].when = dateplayed - APPLE_TIME - tz;
            iTunesTable[x].played = 'L';
            input_stream.seekg(entrylen - 12, std::ios::cur);
        } else {
            iTunesTable[x].played = 'S';
            input_stream.seekg(entrylen - 4, std::ios::cur);
        }
    }
    input_stream.close();

    int min;
    for (int x=0; x<songs; x++) {
        min=x;
        for (int y=x; y<songs; y++)
            if (iTunesTable[y].when<iTunesTable[min].when) min=y;
            scrob_entry tmp;
            tmp = iTunesTable[x];
            iTunesTable[x] = iTunesTable[min];
            iTunesTable[min] = tmp;
    }


    for ( int x=0; x<songs; x++)
        if (iTunesTable[x].played == 'L')
            entries.push_back(iTunesTable[x]);

    delete [] iTunesTable;

    if (entries.size() > 0)
        scrobble_method = SCROBBLE_IPOD;

    return true;
}

void Scrobble::recalc_dt(int base_dt)
{
    const size_t size = entries.size();

    if (size)
        for (size_t i = size; i > 0; i--) {
            scrob_entry temp = entries.at(i-1);
            base_dt -= temp.length;
            temp.when = base_dt;

            update_track(temp, i-1);
        }
}

void Scrobble::recalc_now()
{
    recalc_dt(get_gmt());
}

size_t Scrobble::get_num_tracks()
{
    return entries.size();
}

scrob_entry Scrobble::get_track(size_t track_index)
{
    return entries.at(track_index);
}

void Scrobble::remove_track(size_t track_index)
{
    entries.erase(entries.begin()+track_index);
}

void Scrobble::update_track(scrob_entry track, size_t track_index)
{
    size_t size = entries.size();

    // should be faster than an insert/erase
    entries.push_back(track);
    std::swap(entries[size], entries[track_index]);
    entries.pop_back();
}

void Scrobble::cleanup_tracks(void)
{
    //delete tracks that were skipped or too short
    for ( size_t x = 0; x < entries.size(); x++) {
        scrob_entry tmp = entries.at(x);
        if ( tmp.played != 'L' || tmp.length < 30 )
        {
            entries.erase(entries.begin() + x);
            x--;
        }
    }
}

bool Scrobble::submit(const string& user, const string& pass)
{
    if (!scrob_init)
    {
        add_log(LOG_ERROR, "init function hasn't been run!");
        return false;
    }

    username = user;
    password_hash = pass;
    need_handshake = true;

    if (entries.empty())
    {
        add_log(LOG_ERROR, "Nothing to submit!");
        return false;
    }

    // May already have been called, but better to be safe.
    cleanup_tracks();

    if (check_age())
    {
        add_log(LOG_ERROR, "One or more tracks are too old.  Correct this and try again.");
        return false;
    }

    while (entries.size() > 0)
    {
        if (need_handshake)
        {
            if (!handshake())
                return false;
        } else {
            if (!real_submit())
                return false;
        }
    }

    return true;
}

bool Scrobble::real_submit()
{
    size_t items;

    // clear the buffer
    curl_buffer = "";

    if (entries.size()<50)
        items = entries.size();
    else
        items = 50;

    add_log(LOG_INFO, "Submitting " + int2string(items) + " entries");

    string data = "s=" + sessionid;

    for ( size_t count=0; count<items; count++ ) {
        scrob_entry tmp = entries.at(count);

        char *esc_artist = NULL;
        char *esc_album = NULL;
        char *esc_title = NULL;
        char *esc_tracknum = NULL;

        string str_count = int2string(count);

        string mbid = "";
        if (have_mb && tmp.mb_track_id.length() == 36)
            mbid = tmp.mb_track_id;

#if (LIBCURL_VERSION_NUM < 0x070F04)
        esc_artist = curl_escape(tmp.artist.c_str(), 0);
        esc_album = curl_escape(tmp.album.c_str(), 0);
        esc_title = curl_escape(tmp.title.c_str(), 0);
        esc_tracknum = curl_escape(tmp.tracknum.c_str(), 0);
#else
        esc_artist = curl_easy_escape(curl, tmp.artist.c_str(), 0);
        esc_album = curl_easy_escape(curl, tmp.album.c_str(), 0);
        esc_title = curl_easy_escape(curl, tmp.title.c_str(), 0);
        esc_tracknum = curl_easy_escape(curl, tmp.tracknum.c_str(), 0);
#endif

        if (esc_artist == NULL ||
            esc_album == NULL ||
            esc_title == NULL ||
            esc_tracknum == NULL)
        {
            add_log(LOG_ERROR, "Error with curl(_easy)_escape?!?");
            return false;
        }

        data += "&a[" + str_count + "]=" + esc_artist + \
                "&t[" + str_count + "]=" + esc_title + \
                "&b[" + str_count + "]=" + esc_album + \
                "&m[" + str_count + "]=" + mbid + \
                "&l[" + str_count + "]=" + int2string(tmp.length) + \
                "&i[" + str_count + "]=" + int2string(tmp.when) + \
                "&o[" + str_count + "]=P" + \
                "&n[" + str_count + "]=" + esc_tracknum + \
                "&r[" + str_count + "]=";

        curl_free(esc_artist);
        curl_free(esc_album);
        curl_free(esc_title);
        curl_free(esc_tracknum);
    }

    add_log(LOG_DEBUG, "Data: " + data);

    curl_res = curl_easy_setopt(curl, CURLOPT_URL, submit_url.c_str());
    if (curl_res != CURLE_OK)
    {
        add_log(LOG_ERROR, "CURL: failed to set CURLOPT_URL: " + string(errorBuffer));
        return false;
    }

    curl_res = curl_easy_setopt(curl, CURLOPT_POSTFIELDS, data.c_str());
    if (curl_res != CURLE_OK)
    {
        add_log(LOG_ERROR, "CURL: failed to set CURLOPT_POSTFIELDS: " + string(errorBuffer));
        return false;
    }

    if (proxy_host.length())
        if (!do_proxy())
            return false;

    curl_res = curl_easy_perform(curl);
    if (curl_res != CURLE_OK)
    {
        add_log(LOG_ERROR, "CURL: failed to POST data to " + submit_url + ": " + string(errorBuffer));
        add_log(LOG_ERROR, "Failed to send data to last.fm - check your connection settings, or try again later");
        return false;
    }

    add_log(LOG_INFO, "Server response: " + curl_buffer);

    if (curl_buffer.length() <= 0)
    {
        add_log(LOG_ERROR, "Empty result from server");
        return false;
    }

    if (curl_buffer.find("no POST parameters") != string::npos)
    {
        // Server problem - http://www.last.fm/forum/21716/_/201367
        // "FAILED Plugin bug: Not all request variables are set - no POST parameters"
        // This if statement can be removed if/when fixed
        need_handshake = true;
        return true;
    }

    if (curl_buffer.find("BADSESSION") == 0)
    {
        need_handshake = true;
        return true;
    }

    if (curl_buffer.find("FAILED") == 0)
    {
        add_log(LOG_ERROR, "Server returned an error after sending data: " +  curl_buffer);
        return false;
    }

    if (curl_buffer.find("OK") == 0)
    {
        entries.erase(entries.begin(), entries.begin() + items);
    }

    return true;
}

bool Scrobble::handshake()
{
    time_t gmt = get_gmt();
    string time_str;
    std::stringstream ss;

    // clear the buffer
    curl_buffer = "";

    ss << gmt;
    ss >> time_str;

    string auth = MD5Digest( (password_hash + time_str).c_str() );

    string handshake_url = ( HANDSHAKE_HOST
                                "/?hs=true&p=" PROTOCOL_VERSION
                                "&c=" CLIENT_ID
                                "&v=" CLIENT_VERSION
                                "&u=" + username +
                                "&t=" + time_str +
                                "&a=" + auth );

    add_log(LOG_DEBUG, "Time: " + time_str);
    add_log(LOG_DEBUG, "auth: " + auth);
    add_log(LOG_DEBUG, "url:  " + handshake_url);

    // always reset to a GET, maybe called straight after a POST
    curl_res = curl_easy_setopt(curl, CURLOPT_HTTPGET, 1);
    if (curl_res != CURLE_OK)
    {
        add_log(LOG_ERROR, "CURL: failed to set CURLOPT_HTTPGET: " + string(errorBuffer));
        return false;
    }

    curl_res = curl_easy_setopt(curl, CURLOPT_URL, handshake_url.c_str());
    if (curl_res != CURLE_OK)
    {
        add_log(LOG_ERROR, "CURL: failed to set CURLOPT_URL: " + string(errorBuffer));
        return false;
    }

    if (proxy_host.length())
        if (!do_proxy())
            return false;

    curl_res = curl_easy_perform(curl);
    if (curl_res != CURLE_OK)
    {
        add_log(LOG_ERROR, "CURL: failed to GET from " + string(HANDSHAKE_HOST));
        add_log(LOG_ERROR, "Failed to handshake with last.fm - check your connection settings, or try again later");
        return false;
    }

    add_log(LOG_DEBUG, "Result: " + curl_buffer);

    if (curl_buffer.length() <= 0)
    {
        add_log(LOG_ERROR, "Empty result from server");
        return false;
    }

    if (curl_buffer.find("BANNED") == 0)
    {
        add_log(LOG_ERROR, "Handshake Failed - client software banned.  Check http://qtscrob.sourceforge.net for updates.");
        return false;
    }

    if (curl_buffer.find("BADAUTH") == 0)
    {
        add_log(LOG_ERROR, "Handshake Failed - authentication problem.  Check your username and/or password.");
        return false;
    }

    if (curl_buffer.find("BADTIME") == 0)
    {
        add_log(LOG_ERROR, "Handshake Failed - clock is incorrect.  Check your computer clock and timezone settings.");
        return false;
    }

    if (curl_buffer.find("FAILED") == 0)
    {
        add_log(LOG_ERROR, "Handshake Failed");
        return false;
    }

    if (curl_buffer.find("OK") == 0)
    {
        std::vector<string> v;

        // this check on the number of lines may be overly precise
        // maybe replace ==5 with >3
        // as long as we check we get enough data back
        if (split(v, curl_buffer.c_str(), '\n') == 5)
        {
            //right number of newlines
            sessionid = v[1];
            nowplay_url = v[2];
            submit_url = v[3];
        } else {
            add_log(LOG_ERROR, "Handshake Failed - unexpected number of lines");
            return false;
        }

    } else {
        add_log(LOG_ERROR, "Handshake Failed - unknown response");
        return false;
    }

    add_log(LOG_DEBUG, "sessionid:   " + sessionid);
    add_log(LOG_DEBUG, "nowplay_url: " + nowplay_url);
    add_log(LOG_DEBUG, "submit_url:  " + submit_url);

    need_handshake = false;
    return true;
}

void Scrobble::set_proxy(const string& host, long port, const string& user, const string& pass, bool win)
{
    proxy_host = host;
    proxy_port = port;
    proxy_user = user;
    proxy_pass = pass;
    proxy_winauth = win;
}

bool Scrobble::do_proxy()
{
    add_log(LOG_ERROR, "Setting Proxy to " + proxy_host + ":" + int2string(proxy_port));

    // if host is empty, this will disable the use of a proxy
    curl_res = curl_easy_setopt(curl, CURLOPT_PROXY, proxy_host.c_str());
    if (curl_res != CURLE_OK)
    {
        add_log(LOG_ERROR, "CURL: failed to set CURLOPT_PROXY: " + string(errorBuffer));
        return false;
    }

    // only set other options if the proxy is to be used.
    if (proxy_host.length())
    {
        curl_res = curl_easy_setopt(curl, CURLOPT_PROXYPORT, proxy_port);
        if (curl_res != CURLE_OK)
        {
            add_log(LOG_ERROR, "CURL: failed to set CURLOPT_PROXYPORT: " + string(errorBuffer));
            return false;
        }

        curl_res = curl_easy_setopt(curl, CURLOPT_PROXYTYPE, CURLPROXY_HTTP);
        if (curl_res != CURLE_OK)
        {
            add_log(LOG_ERROR, "CURL: failed to set CURLOPT_PROXYTYPE: " + string(errorBuffer));
            return false;
        }

        if (proxy_user.length() || proxy_winauth)
        {
            std::string auth;
            if (proxy_winauth)
                auth = ":";
            else
                auth= proxy_user + ":" + proxy_pass;

            curl_res = curl_easy_setopt(curl, CURLOPT_PROXYUSERPWD, auth.c_str());
            if (curl_res != CURLE_OK)
            {
                add_log(LOG_ERROR, "CURL: failed to set CURLOPT_PROXYUSERPWD: " + string(errorBuffer));
                return false;
            }

            curl_res = curl_easy_setopt(curl, CURLOPT_PROXYAUTH, CURLAUTH_ANY);
            if (curl_res != CURLE_OK)
            {
                add_log(LOG_ERROR, "CURL: failed to set CURLOPT_PROXYAUTH: " + string(errorBuffer));
                return false;
            }

        } /*else {
            // don't know if this works...
            curl_res = curl_easy_setopt(curl, CURLOPT_PROXYUSERPWD, "");
            if (curl_res != CURLE_OK)
            {
                add_log(LOG_ERROR, "CURL: failed to set CURLOPT_PROXYUSERPWD: " + string(errorBuffer));
                return false;
            }

            curl_res = curl_easy_setopt(curl, CURLOPT_PROXYAUTH, 0);
            if (curl_res != CURLE_OK)
            {
                add_log(LOG_ERROR, "CURL: failed to set CURLOPT_PROXYAUTH: " + string(errorBuffer));
                return false;
            }

        }*/

    }

    return true;
}

string Scrobble::offset_str(void)
{
    char rtn[11];
    memset(rtn, 0, sizeof(rtn));

    sprintf(rtn,
        "UTC %c%d:%02d",
        (offset < 0)?'-':'+',
        abs(offset) / 3600,
        (abs(offset) % 3600) / 60);

    return rtn;
}

// originally from the last.fm client
string MD5Digest( const char *token )
{
    md5_state_t md5state;
    unsigned char md5pword[16];

    md5_init( &md5state );
    md5_append( &md5state, (unsigned const char *)token, (int)strlen(token) );
    md5_finish( &md5state, md5pword );

    char tmp[33];
    char a[3];
    int j;
    for ( j = 0, tmp[32] = 0; j < 16; j++ )
    {
        sprintf( a, "%02x", md5pword[j] );
        tmp[2*j] = a[0];
        tmp[2*j+1] = a[1];
    }

    return tmp;
}

void Scrobble::add_log(LOG_LEVEL level, string msg)
{
    log_entry tmp;
    tmp.level = level;
    tmp.msg = msg;
    log_messages.push_back(tmp);

    if (level <= log_print_level)
        std::cout << level << ": " << msg << std::endl;

    if (LOG_ERROR == level)
        error_str = msg;
}

size_t Scrobble::get_num_logs()
{
    return log_messages.size();
}

log_entry Scrobble::get_log(size_t index)
{
    return log_messages.at(index);
}

void Scrobble::clear_log()
{
    log_messages.clear();
}

void Scrobble::set_log_level(LOG_LEVEL level)
{
    log_print_level = level;
}

bool Scrobble::check_age()
{
    bool too_old = false;
    // Now - 30 days
    time_t max_age = get_gmt() - 2592000;
    for ( size_t i = 0; i < entries.size(); i++) {
        scrob_entry tmp = entries.at(i);
        if ( tmp.when < max_age )
        {
            add_log(LOG_INFO, "Track too old: " + tmp.artist + " " + tmp.title);
            too_old = true;
        }
    }
    return too_old;
}
