/***************************************************************************
                          diagramdialog.cpp  -  description
                             -------------------
    begin                : Sun Oct 5 2003
    copyright            : (C) 2003 by Michael Margraf
    email                : michael.margraf@alumni.tu-berlin.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "diagramdialog.h"
#include "qucsview.h"
#include "qucs.h"

#include <math.h>

#include <qvbox.h>
#include <qlayout.h>
#include <qhbuttongroup.h>
#include <qvbuttongroup.h>
#include <qhgroupbox.h>
#include <qpushbutton.h>
#include <qtabwidget.h>
#include <qlabel.h>
#include <qstringlist.h>
#include <qmessagebox.h>
#include <qptrlist.h>
#include <qvalidator.h>
#include <qcolordialog.h>
#include <qlineedit.h>
#include <qcheckbox.h>


// standard colors: blue, red, magenta, green, cyan, yellow, black
// (white is only a dummy)
static const QRgb DefaultColors[]
          = {0x0000ff, 0xff0000, 0xff00ff, 0x00ff00, 0x00ffff, 0xffff00,
             0xffffff, 0x000000};


DiagramDialog::DiagramDialog(Diagram *d, const QString& _DataSet,
                             QWidget *parent, const char *name )
                    : QDialog(parent, name, TRUE, Qt::WDestructiveClose)
{
  Diag = d;
  Graphs.setAutoDelete(true);
  copyDiagramGraphs();   // make a copy of all graphs
  defaultDataSet = _DataSet;
  setCaption(tr("Edit Diagram Properties"));
  changed = false;
  toTake = false;   // double-clicked variable be inserted into graph list ?
//  setFixedSize(QSize(400, 400));
//  setMinimumSize(QSize(400, 400));

  ValDouble = new QDoubleValidator(-1e200, 1e200, 6, this);

  all = new QVBoxLayout(this); // to provide neccessary size
  QTabWidget *t = new QTabWidget(this);
  all->addWidget(t);

  // ...........................................................
  QVBox *Tab1 = new QVBox(this);
  Tab1->setSpacing(5);

  Label4 = 0;
  yrLabel = 0;
  yAxisBox = 0;

  QVButtonGroup *InputGroup = new QVButtonGroup(tr("Graph Input"), Tab1);
  GraphInput = new QLineEdit(InputGroup);
  connect(GraphInput, SIGNAL(textChanged(const QString&)),
		      SLOT(slotResetToTake(const QString&)));
  QHBox *Box2 = new QHBox(InputGroup);
  Box2->setSpacing(5);
  Expr.setPattern("[0-9]{1,2}");  // valid expression for property input
  Validator = new QRegExpValidator(Expr, this);

  if(Diag->Name == "Tab") {
    Label1 = new QLabel(tr("Number Notation: "), Box2);
    PropertyBox = new QComboBox(Box2);
    PropertyBox->insertItem(tr("real/imaginary"));
    PropertyBox->insertItem(tr("magnitude/angle (degree)"));
    PropertyBox->insertItem(tr("magnitude/angle (radian)"));
    connect(PropertyBox, SIGNAL(activated(int)), SLOT(slotSetNumMode(int)));
    Box2->setStretchFactor(new QWidget(Box2), 5); // stretchable placeholder

    Label2 = new QLabel(tr("Precision:"), Box2);
    Property2 = new QLineEdit(Box2);
    Property2->setValidator(Validator);
    Property2->setMaximumWidth(25);
    Property2->setText("3");
  }
  else {
    Label1 = new QLabel(tr("Color:"),Box2);
    ColorButt = new QPushButton("   ",Box2);
    ColorButt->setMinimumWidth(50);
    ColorButt->setEnabled(false);
    connect(ColorButt, SIGNAL(clicked()), SLOT(slotSetColor()));
    Box2->setStretchFactor(new QWidget(Box2), 5); // stretchable placeholder

    Label3 = new QLabel(tr("Style:"),Box2);
    Label3->setEnabled(false);
    PropertyBox = new QComboBox(Box2);
    PropertyBox->insertItem(tr("solid line"));
    PropertyBox->insertItem(tr("dash line"));
    PropertyBox->insertItem(tr("dot line"));
    PropertyBox->insertItem(tr("long dash line"));
    PropertyBox->insertItem(tr("stars"));
    PropertyBox->insertItem(tr("circles"));
    PropertyBox->insertItem(tr("arrows"));
    connect(PropertyBox, SIGNAL(activated(int)),
			 SLOT(slotSetGraphStyle(int)));
    Box2->setStretchFactor(new QWidget(Box2), 5); // stretchable placeholder

    Label2 = new QLabel(tr("Thickness:"),Box2);
    Property2 = new QLineEdit(Box2);
    Property2->setValidator(Validator);
    Property2->setMaximumWidth(25);
    Property2->setText("0");

    if((Diag->Name=="Rect") || (Diag->Name=="PS") || (Diag->Name=="SP")){
      QHBox *Box3 = new QHBox(InputGroup);
      Box3->setSpacing(5);

      Label4 = new QLabel(tr("y-Axis:"),Box3);
      Label4->setEnabled(false);
      yAxisBox = new QComboBox(Box3);
      if(Diag->Name == "PS") {
        yAxisBox->insertItem(tr("smith Axis"));
        yAxisBox->insertItem(tr("polar Axis"));
      }
      else if(Diag->Name == "SP") {
        yAxisBox->insertItem(tr("polar Axis"));
        yAxisBox->insertItem(tr("smith Axis"));
      }
      else {
        yAxisBox->insertItem(tr("left Axis"));
        yAxisBox->insertItem(tr("right Axis"));
      }
      yAxisBox->setEnabled(false);
      connect(yAxisBox, SIGNAL(activated(int)), SLOT(slotSetYAxis(int)));
      Box3->setStretchFactor(new QWidget(Box3), 5); // stretchable placeholder
    }
  }
  connect(Property2, SIGNAL(textChanged(const QString&)),
			SLOT(slotSetProp2(const QString&)));

  Label1->setEnabled(false);
  PropertyBox->setEnabled(false);
  Label2->setEnabled(false);
  Property2->setEnabled(false);

  QHBox *Box1 = new QHBox(Tab1);
  Box1->setSpacing(5);

  QVButtonGroup *DataGroup = new QVButtonGroup(tr("Dataset"), Box1);
  ChooseData = new QComboBox(false, DataGroup);
  ChooseData->setMinimumWidth(200);
  connect(ChooseData, SIGNAL(activated(int)), SLOT(slotReadVars(int)));
  ChooseVars = new QListView(DataGroup);
  ChooseVars->addColumn(tr("Name"));
  ChooseVars->addColumn(tr("Status"));
  ChooseVars->addColumn(tr("Size"));
// QT 3.2
//  connect(Content, SIGNAL(doubleClicked(QListViewItem*, constQPoint&,int)),
//                   SLOT(slotTakeVar(QListViewItem*, constQPoint&,int)));
  connect(ChooseVars, SIGNAL(doubleClicked(QListViewItem*)),
		      SLOT(slotTakeVar(QListViewItem*)));


  QVButtonGroup *GraphGroup = new QVButtonGroup(tr("Graph"), Box1);
  GraphList = new QListBox(GraphGroup);
  connect(GraphList, SIGNAL(clicked(QListBoxItem*)),
		     SLOT(slotSelectGraph(QListBoxItem*)));
  QPushButton *NewButt = new QPushButton(tr("New Graph"), GraphGroup);
  connect(NewButt, SIGNAL(clicked()), SLOT(slotNewGraph()));
  QPushButton *DelButt = new QPushButton(tr("Delete Graph"), GraphGroup);
  connect(DelButt, SIGNAL(clicked()), SLOT(slotDeleteGraph()));

  t->addTab(Tab1, tr("Data"));

  // ...........................................................
  if(Diag->Name != "Tab") {
    QWidget *Tab2 = new QWidget(t);
    QGridLayout *gp = new QGridLayout(Tab2,10,2,5,5);

    gp->addWidget(new QLabel(tr("x-Axis Label:"), Tab2), 0,0);
    xLabel = new QLineEdit(Tab2);
    gp->addWidget(xLabel,0,1);

    gp->addWidget(new QLabel(tr("left y-Axis Label:"), Tab2), 1,0);
    ylLabel = new QLineEdit(Tab2);
    gp->addWidget(ylLabel,1,1);

    if(Diag->Name == "Rect") {
      gp->addWidget(new QLabel(tr("right y-Axis Label:"), Tab2), 2,0);
      yrLabel = new QLineEdit(Tab2);
      gp->addWidget(yrLabel,2,1);
    }

    GridOn = new QCheckBox(tr("show Grid"), Tab2);
    connect(GridOn, SIGNAL(stateChanged(int)), SLOT(slotSetGridBox(int)));
    gp->addMultiCellWidget(GridOn,3,3,0,1);

    GridLabel1 = new QLabel(tr("Grid Color:"),Tab2);
    gp->addWidget(GridLabel1, 4,0);
    GridColorButt = new QPushButton("        ",Tab2);
    connect(GridColorButt, SIGNAL(clicked()), SLOT(slotSetGridColor()));
    gp->addWidget(GridColorButt, 4,1);

    GridLabel2 = new QLabel(tr("Grid Style: "), Tab2);
    gp->addWidget(GridLabel2, 5,0);
    GridStyleBox = new QComboBox(Tab2);
    GridStyleBox->insertItem(tr("solid line"));
    GridStyleBox->insertItem(tr("dash line"));
    GridStyleBox->insertItem(tr("dot line"));
    GridStyleBox->insertItem(tr("dash dot line"));
    GridStyleBox->insertItem(tr("dash dot dot line"));
    gp->addWidget(GridStyleBox, 5,1);

    // ...........................................................
    // transfer the diagram properties to the dialog
    xLabel->setText(Diag->xAxis.Label);
    ylLabel->setText(Diag->yAxis.Label);
    if(yrLabel)  yrLabel->setText(Diag->zAxis.Label);
    GridOn->setChecked(Diag->xAxis.GridOn);
    if(!Diag->xAxis.GridOn) slotSetGridBox(QButton::Off);
    GridColorButt->setPaletteBackgroundColor(Diag->GridPen.color());
    GridStyleBox->setCurrentItem(Diag->GridPen.style()-1);

    if(Diag->Name == "Rect") {
      GridLogX = new QCheckBox(tr("logarithmical X Axis Grid"), Tab2);
      gp->addMultiCellWidget(GridLogX,6,6,0,1);

      GridLogY = new QCheckBox(tr("logarithmical left Y Axis Grid"), Tab2);
      gp->addMultiCellWidget(GridLogY,7,7,0,1);

      GridLogZ = new QCheckBox(tr("logarithmical right Y Axis Grid"), Tab2);
      gp->addMultiCellWidget(GridLogZ,8,8,0,1);

      // ...........................................................
      // transfer the diagram properties to the dialog
      GridLogX->setChecked(Diag->xAxis.log);
      GridLogY->setChecked(Diag->yAxis.log);
      GridLogZ->setChecked(Diag->zAxis.log);
    }
    else  GridLogX = GridLogY = GridLogZ = 0;

    t->addTab(Tab2, tr("Properties"));

  // ...........................................................
    QVBox *Tab3 = new QVBox(this);
    Tab1->setSpacing(5);

    QHGroupBox *axisX = new QHGroupBox(tr("x-Axis"), Tab3);

    QVBox *VBox1 = new QVBox(axisX);
    VBox1->setStretchFactor(new QWidget(VBox1),5); // stretchable placeholder
    manualX = new QCheckBox(tr("manual"), VBox1);
    connect(manualX, SIGNAL(stateChanged(int)), SLOT(slotManualX(int)));

    QVBox *VBox2 = new QVBox(axisX);
    new QLabel(tr("start"), VBox2);
    startX = new QLineEdit(VBox2);
    startX->setValidator(ValDouble);

    QVBox *VBox3 = new QVBox(axisX);
    new QLabel(tr("step"), VBox3);
    stepX = new QLineEdit(VBox3);
    stepX->setValidator(ValDouble);

    QVBox *VBox4 = new QVBox(axisX);
    new QLabel(tr("stop"), VBox4);
    stopX = new QLineEdit(VBox4);
    stopX->setValidator(ValDouble);


    QHGroupBox *axisY;
    if(Diag->Name == "PS")
      axisY = new QHGroupBox(tr("Smith Axis"), Tab3);
    else if(Diag->Name == "SP")
      axisY = new QHGroupBox(tr("Polar Axis"), Tab3);
    else
      axisY = new QHGroupBox(tr("left y-Axis"), Tab3);

    QVBox *VBox5 = new QVBox(axisY);
    VBox5->setStretchFactor(new QWidget(VBox5),5); // stretchable placeholder
    manualY = new QCheckBox(tr("manual"), VBox5);
    connect(manualY, SIGNAL(stateChanged(int)), SLOT(slotManualY(int)));

    QVBox *VBox6 = new QVBox(axisY);
    new QLabel(tr("start"), VBox6);
    startY = new QLineEdit(VBox6);
    startY->setValidator(ValDouble);

    QVBox *VBox7 = new QVBox(axisY);
    if((Diag->Name=="Smith") || (Diag->Name=="ySmith") || (Diag->Name=="PS"))
      new QLabel(tr("number"), VBox7);
    else  new QLabel(tr("step"), VBox7);
    stepY = new QLineEdit(VBox7);
    stepY->setValidator(ValDouble);

    QVBox *VBox8 = new QVBox(axisY);
    new QLabel(tr("stop"), VBox8);
    stopY = new QLineEdit(VBox8);
    stopY->setValidator(ValDouble);


    QHGroupBox *axisZ;
    if(Diag->Name == "PS")
      axisZ = new QHGroupBox(tr("Polar Axis"), Tab3);
    else if(Diag->Name == "SP")
      axisZ = new QHGroupBox(tr("Smith Axis"), Tab3);
    else
      axisZ = new QHGroupBox(tr("right y-Axis"), Tab3);

    QVBox *VBox9 = new QVBox(axisZ);
    VBox9->setStretchFactor(new QWidget(VBox9),5); // stretchable placeholder
    manualZ = new QCheckBox(tr("manual"), VBox9);
    connect(manualZ, SIGNAL(stateChanged(int)), SLOT(slotManualZ(int)));

    QVBox *VBox10 = new QVBox(axisZ);
    new QLabel(tr("start"), VBox10);
    startZ = new QLineEdit(VBox10);
    startZ->setValidator(ValDouble);

    QVBox *VBox11 = new QVBox(axisZ);
    if(Diag->Name == "SP")  new QLabel(tr("number"), VBox11);
    else  new QLabel(tr("step"), VBox11);
    stepZ = new QLineEdit(VBox11);
    stepZ->setValidator(ValDouble);

    QVBox *VBox12 = new QVBox(axisZ);
    new QLabel(tr("stop"), VBox12);
    stopZ = new QLineEdit(VBox12);
    stopZ->setValidator(ValDouble);


    Tab3->setStretchFactor(new QWidget(Tab3),5); // stretchable placeholder

    t->addTab(Tab3, tr("Limits"));

    // ...........................................................
    // transfer the diagram properties to the dialog
    if(Diag->xAxis.autoScale)  slotManualX(QButton::Off);
    else  manualX->setChecked(true);
    if(Diag->yAxis.autoScale)  slotManualY(QButton::Off);
    else  manualY->setChecked(true);
    if(Diag->zAxis.autoScale)  slotManualZ(QButton::Off);
    else  manualZ->setChecked(true);

    Diag->calcLimits();    // inserts auto-scale values if not manual

    startX->setText(QString::number(Diag->xAxis.limit_min));
    stepX->setText(QString::number(Diag->xAxis.step));
    stopX->setText(QString::number(Diag->xAxis.limit_max));

    startY->setText(QString::number(Diag->yAxis.limit_min));
    stepY->setText(QString::number(Diag->yAxis.step));
    stopY->setText(QString::number(Diag->yAxis.limit_max));

    startZ->setText(QString::number(Diag->zAxis.limit_min));
    stepZ->setText(QString::number(Diag->zAxis.step));
    stopZ->setText(QString::number(Diag->zAxis.limit_max));

    if((Diag->Name == "Smith") || (Diag->Name == "ySmith") ||
       (Diag->Name == "Polar")) {
       axisZ->setEnabled(false);
    }
    if(Diag->Name != "Rect") {
      axisX->setEnabled(false);
      startY->setEnabled(false);
      startZ->setEnabled(false);
    }
  }
  else  stepX = 0;

  connect(t, SIGNAL(currentChanged(QWidget*)), SLOT(slotChangeTab(QWidget*)));
  // ...........................................................
  QHBox *Butts = new QHBox(this);
  Butts->setSpacing(5);
  Butts->setMargin(5);
  all->addWidget(Butts);

  QPushButton *OkButt = new QPushButton(tr("OK"), Butts);
  connect(OkButt, SIGNAL(clicked()), SLOT(slotOK()));
  QPushButton *ApplyButt = new QPushButton(tr("Apply"), Butts);
  connect(ApplyButt, SIGNAL(clicked()), SLOT(slotApply()));
  QPushButton *CancelButt = new QPushButton(tr("Cancel"), Butts);
  connect(CancelButt, SIGNAL(clicked()), SLOT(slotCancel()));
//  QPushButton *HelpButt = new QPushButton(tr("Function Help"), Butts);
//  connect(HelpButt, SIGNAL(clicked()), SLOT(slotFuncHelp()));

  OkButt->setDefault(true);


  // ...........................................................
  // put all data files into ComboBox
  QDir ProjDir(QucsWorkDir);
  QStringList Elements = ProjDir.entryList("*.dat", QDir::Files, QDir::Name);
  QStringList::iterator it;
  for(it = Elements.begin(); it != Elements.end(); ++it) {
    ChooseData->insertItem((*it).left((*it).length()-4));
    if((*it) == defaultDataSet)
      // default dataset should be the current
      ChooseData->setCurrentItem(ChooseData->count()-1);
  }
  slotReadVars(0);  // put variables into the ListView

  // ...........................................................
  // put all graphs into the ListBox
  for(Graph *pg = Diag->Graphs.first(); pg != 0; pg = Diag->Graphs.next())
    GraphList->insertItem(pg->Var);

  if(Diag->Name != "Tab")
    ColorButt->setPaletteBackgroundColor
	(QColor(DefaultColors[GraphList->count()]));
}

DiagramDialog::~DiagramDialog()
{
  delete all;   // delete all widgets from heap
  delete Validator;
  delete ValDouble;
}

// --------------------------------------------------------------------------
void DiagramDialog::slotReadVars(int)
{
  QString DocName = ChooseData->currentText()+".dat";

  QFile file(QucsWorkDir.filePath(DocName));
  if(!file.open(IO_ReadOnly)) {
    return;
  }

  QString Line, tmp;
  QTextStream ReadWhole(&file);
  QString FileString = ReadWhole.read();   // read whole data file
  file.close();

  ChooseVars->clear();
  int i=0, j=0;
  i = FileString.find('<')+1;
  if(i > 0)
  do {
    j = FileString.find('>', i);
    Line = FileString.mid(i, j-i);
    i = FileString.find('<', j)+1;
    if(Line.left(3) == "dep") {
      tmp = Line.section(' ', 2);
      new QListViewItem(ChooseVars, Line.section(' ', 1, 1).remove('>'),
			"dep", tmp.remove('>'));
    }
    else if(Line.left(5) == "indep") {
      tmp = Line.section(' ', 2, 2);
      new QListViewItem(ChooseVars, Line.section(' ', 1, 1).remove('>'),
			"indep", tmp.remove('>'));
    }
  } while(i > 0);
}

// ------------------------------------------------------------------------
// Inserts the double-clicked variable into the Graph Input Line at the
// cursor position. If the Graph Input is empty, then the variable is
// also inserted as graph.
void DiagramDialog::slotTakeVar(QListViewItem *Item)
{
  GraphInput->blockSignals(true);
  if(toTake) GraphInput->setText("");

  int     i  = GraphInput->cursorPosition();
  QString s  = GraphInput->text();
  QString s1 = Item->text(0);
  if(ChooseData->currentText() != defaultDataSet.section('.',0,0))
    s1 = ChooseData->currentText() + ":" + s1;
  GraphInput->setText(s.left(i) + s1 + s.right(s.length()-i));

  if(s.isEmpty()) {
    GraphList->insertItem(GraphInput->text());
    GraphList->setSelected(GraphList->count()-1,true);

    Graph *g = new Graph(GraphInput->text());   // create a new graph

    if(Diag->Name != "Tab") {
      g->Color = ColorButt->paletteBackgroundColor();
      g->Thick = Property2->text().toInt();
      ColorButt->setPaletteBackgroundColor(
		QColor(DefaultColors[GraphList->count()]));
      g->Style   = PropertyBox->currentItem();
      if(yAxisBox) {
        g->yAxisNo = yAxisBox->currentItem();
        yAxisBox->setEnabled(true);
        Label4->setEnabled(true);
      }

      Label3->setEnabled(true);
      ColorButt->setEnabled(true);
    }
    else {
      g->Precision = Property2->text().toInt();
      g->numMode   = PropertyBox->currentItem();
    }

    Graphs.append(g);
    changed = true;
    toTake  = true;
  }

  GraphInput->blockSignals(false);

  Label1->setEnabled(true);
  PropertyBox->setEnabled(true);
  Label2->setEnabled(true);
  Property2->setEnabled(true);
}

// --------------------------------------------------------------------------
// Puts the text of the selected graph into the line edit.
void DiagramDialog::slotSelectGraph(QListBoxItem *item)
{
  if(item == 0) {
    GraphList->clearSelection();
    return;
  }

  int index = GraphList->index(item);
  GraphInput->blockSignals(true);
  GraphInput->setText(GraphList->text(index));
  GraphInput->blockSignals(false);

  Graph *g = Graphs.at(index);
  if(Diag->Name != "Tab") {
    Property2->setText(QString::number(g->Thick));
    ColorButt->setPaletteBackgroundColor(g->Color);
    PropertyBox->setCurrentItem(g->Style);
    if(yAxisBox) {
      yAxisBox->setCurrentItem(g->yAxisNo);
      yAxisBox->setEnabled(true);
      Label4->setEnabled(true);
    }

    Label3->setEnabled(true);
    ColorButt->setEnabled(true);
  }
  else {
    Property2->setText(QString::number(g->Precision));
    PropertyBox->setCurrentItem(g->numMode);
  }
  toTake = false;

  Label1->setEnabled(true);
  PropertyBox->setEnabled(true);
  Label2->setEnabled(true);
  Property2->setEnabled(true);
}

// --------------------------------------------------------------------------
// Is called when the 'delelte graph' button is pressed.
void DiagramDialog::slotDeleteGraph()
{
  int i = GraphList->index(GraphList->selectedItem());
  if(i < 0) return;   // return, if no item selected

  GraphList->removeItem(i);
  Graphs.remove(i);

  GraphInput->setText("");  // erase input line and back to default values
  if(Diag->Name != "Tab") {
    ColorButt->setPaletteBackgroundColor(
		QColor(DefaultColors[GraphList->count()]));
    Property2->setText("0");
    if(yAxisBox) {
      yAxisBox->setCurrentItem(0);
      yAxisBox->setEnabled(false);
      Label4->setEnabled(false);
    }

    Label3->setEnabled(false);
    ColorButt->setEnabled(false);
  }
  else  Property2->setText("3");
  PropertyBox->setCurrentItem(0);
  changed = true;
  toTake  = false;

  Label1->setEnabled(false);
  PropertyBox->setEnabled(false);
  Label2->setEnabled(false);
  Property2->setEnabled(false);
}

// --------------------------------------------------------------------------
void DiagramDialog::slotNewGraph()
{
  if(GraphInput->text().isEmpty()) return;

  GraphList->insertItem(GraphInput->text());

  Graph *g = new Graph(GraphInput->text());   // create a new graph
  if(Diag->Name != "Tab") {
    g->Color = ColorButt->paletteBackgroundColor();
    g->Thick = Property2->text().toInt();
    g->Style = PropertyBox->currentItem();
    if(yAxisBox)  g->yAxisNo = yAxisBox->currentItem();
  }
  else {
    g->Precision = Property2->text().toInt();
    g->numMode   = PropertyBox->currentItem();
  }
  Graphs.append(g);
  changed = true;
  toTake  = false;
}

// --------------------------------------------------------------------------
// Is called if "Ok" button is pressed.
void DiagramDialog::slotOK()
{
  slotApply();
  if(changed) done(QDialog::Accepted);
  else done(QDialog::Rejected);
}

// --------------------------------------------------------------------------
// Is called if "Apply" button is pressed.
void DiagramDialog::slotApply()
{
  if(Diag->Name != "Tab") {
    if(Diag->xAxis.Label.isEmpty())
      Diag->xAxis.Label = ""; // can be not 0 and empty!
    if(xLabel->text().isEmpty()) xLabel->setText("");
    if(Diag->xAxis.Label != xLabel->text()) {
      Diag->xAxis.Label = xLabel->text();
      changed = true;
    }
    if(Diag->yAxis.Label.isEmpty())
      Diag->yAxis.Label = ""; // can be not 0 and empty!
    if(ylLabel->text().isEmpty()) ylLabel->setText("");
    if(Diag->yAxis.Label != ylLabel->text()) {
      Diag->yAxis.Label = ylLabel->text();
      changed = true;
    }
    if(Diag->xAxis.GridOn != GridOn->isChecked()) {
      Diag->xAxis.GridOn = GridOn->isChecked();
      Diag->yAxis.GridOn = GridOn->isChecked();
      changed = true;
    }
    if(Diag->GridPen.color() != GridColorButt->paletteBackgroundColor()) {
      Diag->GridPen.setColor(GridColorButt->paletteBackgroundColor());
      changed = true;
    }
    if(Diag->GridPen.style()!=(Qt::PenStyle)(GridStyleBox->currentItem()+1)) {
      Diag->GridPen.setStyle((Qt::PenStyle)(GridStyleBox->currentItem()+1));
      changed = true;
    }
    if(Diag->Name == "Rect") {
      if(Diag->zAxis.Label.isEmpty())
        Diag->zAxis.Label = ""; // can be not 0 and empty!
      if(yrLabel->text().isEmpty()) yrLabel->setText("");
      if(Diag->zAxis.Label != yrLabel->text()) {
        Diag->zAxis.Label = yrLabel->text();
        changed = true;
      }

      if(Diag->xAxis.log != GridLogX->isChecked()) {
        Diag->xAxis.log = GridLogX->isChecked();
        changed = true;
      }
      if(Diag->yAxis.log != GridLogY->isChecked()) {
        Diag->yAxis.log = GridLogY->isChecked();
        changed = true;
      }
      if(Diag->zAxis.log != GridLogZ->isChecked()) {
        Diag->zAxis.log = GridLogZ->isChecked();
        changed = true;
      }
    }

    if((Diag->Name == "Smith") || (Diag->Name == "ySmith") ||
				  (Diag->Name == "PS"))
      if(stopY->text().toDouble() < 1.0)
        stopY->setText("1");

    if(Diag->Name == "SP")
      if(stopZ->text().toDouble() < 1.0)
        stopZ->setText("1");

    if(Diag->xAxis.autoScale == manualX->isChecked()) {
      Diag->xAxis.autoScale = !(manualX->isChecked());
      changed = true;
    }
    if(Diag->xAxis.limit_min != startX->text().toDouble()) {
      Diag->xAxis.limit_min = startX->text().toDouble();
      changed = true;
    }
    if(Diag->xAxis.step != stepX->text().toDouble()) {
      Diag->xAxis.step = stepX->text().toDouble();
      changed = true;
    }
    if(Diag->xAxis.limit_max != stopX->text().toDouble()) {
      Diag->xAxis.limit_max = stopX->text().toDouble();
      changed = true;
    }
    if(Diag->yAxis.autoScale == manualY->isChecked()) {
      Diag->yAxis.autoScale = !(manualY->isChecked());
      changed = true;
    }
    if(Diag->yAxis.limit_min != startY->text().toDouble()) {
      Diag->yAxis.limit_min = startY->text().toDouble();
      changed = true;
    }
    if(Diag->yAxis.step != stepY->text().toDouble()) {
      Diag->yAxis.step = stepY->text().toDouble();
      changed = true;
    }
    if(Diag->yAxis.limit_max != stopY->text().toDouble()) {
      Diag->yAxis.limit_max = stopY->text().toDouble();
      changed = true;
    }
    if(Diag->zAxis.autoScale == manualZ->isChecked()) {
      Diag->zAxis.autoScale = !(manualZ->isChecked());
      changed = true;
    }
    if(Diag->zAxis.limit_min != startZ->text().toDouble()) {
      Diag->zAxis.limit_min = startZ->text().toDouble();
      changed = true;
    }
    if(Diag->zAxis.step != stepZ->text().toDouble()) {
      Diag->zAxis.step = stepZ->text().toDouble();
      changed = true;
    }
    if(Diag->zAxis.limit_max != stopZ->text().toDouble()) {
      Diag->zAxis.limit_max = stopZ->text().toDouble();
      changed = true;
    }
  }
  Diag->Graphs.clear();   // delete the graphs
  Graphs.setAutoDelete(false);
  for(Graph *pg = Graphs.first(); pg != 0; pg = Graphs.next())
    Diag->Graphs.append(pg);  // transfer the new graphs to diagram
  Graphs.clear();
  Graphs.setAutoDelete(true);

  Diag->loadGraphData(defaultDataSet);
  ((QucsView*)parent())->viewport()->repaint();
  copyDiagramGraphs();
}


// --------------------------------------------------------------------------
// Is called if "Cancel" button is pressed.
void DiagramDialog::slotCancel()
{
//  Diag->loadGraphData(defaultDataSet);
//  ((QucsView*)parent())->viewport()->repaint();
  reject();
}

// --------------------------------------------------------------------------
void DiagramDialog::slotSetColor()
{
  QColor c = QColorDialog::getColor(ColorButt->paletteBackgroundColor(),this);
  if(!c.isValid()) return;
  ColorButt->setPaletteBackgroundColor(c);

  int i = GraphList->index(GraphList->selectedItem());
  if(i < 0) return;   // return, if no item selected

  Graph *g = Graphs.at(i);
  g->Color = c;
  changed = true;
  toTake  = false;
}

// --------------------------------------------------------------------------
void DiagramDialog::slotSetGridColor()
{
  QColor c = QColorDialog::getColor(
			GridColorButt->paletteBackgroundColor(),this);
  if(!c.isValid()) return;
  GridColorButt->setPaletteBackgroundColor(c);
  changed = true;
}

// --------------------------------------------------------------------------
// Is set if the graph input line changes.
void DiagramDialog::slotResetToTake(const QString& s)
{
  int i = GraphList->index(GraphList->selectedItem());
  if(i < 0) return;   // return, if no item selected

  Graph *g = Graphs.at(i);
  g->Var = s;
  GraphList->changeItem(s, i);   // must done after the graph settings !!!
  changed = true;
  toTake  = false;
}

// --------------------------------------------------------------------------
// Is called if the user changes the graph thickness or the precision.
void DiagramDialog::slotSetProp2(const QString& s)
{
  int i = GraphList->index(GraphList->selectedItem());
  if(i < 0) return;   // return, if no item selected

  Graph *g = Graphs.at(i);
  if(Diag->Name == "Tab") g->Precision = s.toInt();
  else  g->Thick = s.toInt();
  changed = true;
  toTake  = false;
}

// --------------------------------------------------------------------------
// Is called if the user changes the number mode.
void DiagramDialog::slotSetNumMode(int Mode)
{
  int i = GraphList->index(GraphList->selectedItem());
  if(i < 0) return;   // return, if no item selected

  Graph *g = Graphs.at(i);
  g->numMode = Mode;
  changed = true;
  toTake  = false;
}

// --------------------------------------------------------------------------
// Is called when the "show grid" checkbox is changed.
void DiagramDialog::slotSetGridBox(int state)
{
  if(state == QButton::On) {
    GridColorButt->setEnabled(true);
    GridStyleBox->setEnabled(true);
    GridLabel1->setEnabled(true);
    GridLabel2->setEnabled(true);
  }
  else {
    GridColorButt->setEnabled(false);
    GridStyleBox->setEnabled(false);
    GridLabel1->setEnabled(false);
    GridLabel2->setEnabled(false);
  }
}

// --------------------------------------------------------------------------
// Is called if the user changes the graph style (combobox).
void DiagramDialog::slotSetGraphStyle(int style)
{
  int i = GraphList->index(GraphList->selectedItem());
  if(i < 0) return;   // return, if no item selected

  Graph *g = Graphs.at(i);
  g->Style = style;
  changed = true;
  toTake  = false;
}

// --------------------------------------------------------------------------
// Makes a copy of all graphs in the diagram.
void DiagramDialog::copyDiagramGraphs()
{
  for(Graph *pg = Diag->Graphs.first(); pg != 0; pg = Diag->Graphs.next())
    Graphs.append(pg->sameNewOne());
}

// --------------------------------------------------------------------------
// Is called if the combobox changes that defines which y axis uses the graph.
void DiagramDialog::slotSetYAxis(int axis)
{
  int i = GraphList->index(GraphList->selectedItem());
  if(i < 0) return;   // return, if no item selected

  Graph *g = Graphs.at(i);
  g->yAxisNo = axis;
  changed = true;
  toTake  = false;
}

// --------------------------------------------------------------------------
void DiagramDialog::slotManualX(int state)
{
  if(state == QButton::On) {
    if(Diag->Name == "Rect")
      startX->setEnabled(true);
    stopX->setEnabled(true);
    if(GridLogX) if(GridLogX->isChecked())  return;
    stepX->setEnabled(true);
  }
  else {
    startX->setEnabled(false);
    stepX->setEnabled(false);
    stopX->setEnabled(false);
  }
}

// --------------------------------------------------------------------------
void DiagramDialog::slotManualY(int state)
{
  if(state == QButton::On) {
    if(Diag->Name == "Rect")
      startY->setEnabled(true);
    stopY->setEnabled(true);
    if(GridLogY) if(GridLogY->isChecked())  return;
    stepY->setEnabled(true);
  }
  else {
    startY->setEnabled(false);
    stepY->setEnabled(false);
    stopY->setEnabled(false);
  }
}

// --------------------------------------------------------------------------
void DiagramDialog::slotManualZ(int state)
{
  if(state == QButton::On) {
    if(Diag->Name == "Rect")
      startZ->setEnabled(true);
    stopZ->setEnabled(true);
    if(GridLogZ) if(GridLogZ->isChecked())  return;
    stepZ->setEnabled(true);
  }
  else {
    startZ->setEnabled(false);
    stepZ->setEnabled(false);
    stopZ->setEnabled(false);
  }
}

// --------------------------------------------------------------------------
// Is called if the current tab of the QTabWidget changes.
void DiagramDialog::slotChangeTab(QWidget*)
{
  if(stepX == 0) return;   // defined ?
  if(GridLogX) {
    if(GridLogX->isChecked())  stepX->setEnabled(false);
    else  if(manualX->isChecked())  stepX->setEnabled(true);
  }
  if(GridLogY) {
    if(GridLogY->isChecked())  stepY->setEnabled(false);
    else  if(manualY->isChecked())  stepY->setEnabled(true);
  }
  if(GridLogZ) {
    if(GridLogZ->isChecked())  stepZ->setEnabled(false);
    else  if(manualZ->isChecked())  stepZ->setEnabled(true);
  }
}
