/***************************************************************************
 *            filter.c
 *
 *  Sun Sep 24 12:11:27 2006
 *  Copyright  2006-2007  Neil Williams
 *  linux@codehelp.co.uk
 ****************************************************************************/
/** @file filter.c
	@author Copyright 2006-2007  Neil Williams <linux@codehelp.co.uk>
	@author Copyright 1999  Robert Lissner, Jay MacDonald,
	@author Copyright 1999  Sam Phillips, Keith Wesolowski. 
*/
/*
    This package is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "config.h"

#include <string.h>
#include <glib.h>
#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <gtkextra/gtksheet.h>
#include "types.h"
#include "dim_list_menu.h"
#include "filter.h"
#include "main.h"

static gint16 working_idx;
static QlFilterInfo old_filter;
static gboolean in_add = FALSE;

typedef enum
{
	QL_FILTER_APPLY,
	QL_FILTER_EDIT,
	QL_FILTER_DELETE
} QlFilterMode;

enum {
	FILTER_COL,
	FROM_COL,
};

/** \brief whether record record_index matches the criterion
 and rule for field field_index. 
 
 This is a subset of applying a filter,
 which may call this function many times, at least once for each record.
 Note that this function only tests one field per call.
*/

#define RETURN_AND_FREE(x) {\
    gint z = (x);\
    if (use_nocase) g_free (entry);\
    if (use_nocase) g_free (criterion);\
    return z;\
}

void
filter_show_all (GtkAction G_GNUC_UNUSED * w, gpointer data)
{
	gint32 i;
	QlContext * qlc;
	QlTabData * tab;

	qlc = ql_get_context (GTK_WIDGET(data));
	tab = ql_get_tabdata (qlc);
	CHECK_CHANGED(tab);
	big_draw_start (tab);
	for (i = 0; i < tab->file->last_row; i++)
		gtk_sheet_row_set_visibility (tab->view->sheet, i, TRUE);
	big_draw_end (tab);
	tab->file->filter_ptr = NULL;
}

static gboolean
filter_compare_record (QlTabData * tab, gint32 field_index, 
	gint32 record_index, gchar * criterionx, QLFilterRule rule, 
	gboolean use_nocase)
{
	gint32 column;
	gchar *entry;
	gchar *entryx; /**< \todo why the duplicate ? */
	gchar *criterion;
	QlFieldInfo * field;

	field = ql_get_fieldinfo (tab, field_index);
	column = field->sheet_column;
	entryx = g_strdup (gtk_sheet_cell_get_text 
		(tab->view->sheet, record_index, column));
	entry = entryx;
	criterion = criterionx;

	/* First handle the case of blank or nonblank criteria */
	if (!entry && rule != FILTER_ISBLANK)
		return FALSE;
	if (rule == FILTER_ISBLANK && (!entry || strlen (entry) == 0))
		return TRUE;
	switch (field->type)
	{
		/* We are required to handle string/text types ourselves */
		case FIELD_TYPE_TEXT:
		{
			/* We'll allow nocase comparison for strings as we do with match/find.
			 * By default, filters are case-sensitive. That's just the Unix way.
			 * It also happens to be the RIGHT way; if you want things to be
			 * considered the same, type them the same.
			 */
			if (use_nocase)
			{
				entry = g_strdup (entryx);
				g_strdown (entry);
				criterion = g_strdup (criterionx);
				g_strdown (criterion);
			}
			else
			{
				entry = entryx;
				criterion = criterionx;
			}
			switch (rule)
			{
			case FILTER_CONTAINS:
				if (strstr (entry, criterion))
					RETURN_AND_FREE (1);
				RETURN_AND_FREE (0);
				break;
			case FILTER_CONTAINSNO:
				RETURN_AND_FREE (!strstr (entry, criterion));
				break;
			case FILTER_IS:
				RETURN_AND_FREE (!strcmp (entry, criterion));
				break;
			case FILTER_ISNOT:
				RETURN_AND_FREE (!!strcmp (entry, criterion));
				break;
			case FILTER_LESS:
				RETURN_AND_FREE ((strcmp (entry, criterion) > 0));
				break;
			case FILTER_MORE:
				RETURN_AND_FREE ((strcmp (entry, criterion) < 0));
				break;
			case FILTER_ISBLANK:
				RETURN_AND_FREE (FALSE);
				break;
			case FILTER_ISNOTBLANK:
				RETURN_AND_FREE (TRUE);
				break;
			}
		}
		break;

#undef RETURN_AND_FREE

		/* We must still handle all of the following for the other three types:
		 * CONTAINS, CONTAINSNO, ISBLANK, ISNOTBLANK.
		 */
		case FIELD_TYPE_NUMERIC:
		case FIELD_TYPE_DATE:
		case FIELD_TYPE_TIME:
		{
			gdouble d_entry =
				qls2d (entry, field->type, field->formatting);
			gdouble d_criterion = qls2d (criterion,
				field->type, field->formatting);
			switch (rule)
			{
			case FILTER_CONTAINS:
				/* For contains/does not contain, we ignore the conversion and do
				 * text comparison.
				 */
				if (strstr (entry, criterion))
					return 1;
				return 0;
				break;
			case FILTER_CONTAINSNO:
				if (strstr (entry, criterion))
					return 0;
				return 1;
				break;
			case FILTER_IS:
				/** \bug comparison likely to fail */
				return ((glong)d_entry == (glong)d_criterion);
				break;
			case FILTER_ISNOT:
				/** \bug comparison likely to fail */
				return ((glong)d_entry != (glong)d_criterion);
				break;
			case FILTER_LESS:
				return (d_entry < d_criterion);
				break;
			case FILTER_MORE:
				return (d_entry > d_criterion);
				break;
			case FILTER_ISBLANK:
				return FALSE;
				break;
			case FILTER_ISNOTBLANK:
				return TRUE;
				break;
			}
		}
		break;
	default:
		{
			level2_error (tab, _("Internal error in tools_compare_record: "
				"Invalid field type."));
			return 0;
		}
		break;
	}
	return 0;
}

/** This is roughly adapted from the sort selection dialog in sort.c. 

\todo complete migration to Gtk2.
*/
static void
filter_select (QlTabData * tab, QlFilterMode G_GNUC_UNUSED mode)
{
	/* Selects a filter to work with and places its index 
	in working_idx */
	GtkWidget *fsel_dialog, *cancel_button, *scroll_w;
	GtkWidget *tree_hbox, * ok_button;
	gchar *select_entries[2] = { "", NULL };
	gint16 idx;
	guint width;
	GtkTreeView * treeview;
	GtkTreeModel * model;
	GtkTreeStore * treestore;
	GtkTreeIter parent_iter, child_iter;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;
	GtkTreeSortable * sort;
	GtkTreeSelection * fselect;

	fsel_dialog = gtk_dialog_new ();
	gtk_window_set_modal (GTK_WINDOW (fsel_dialog), TRUE);
	gtk_window_set_position (GTK_WINDOW (fsel_dialog), GTK_WIN_POS_CENTER);
	gtk_window_set_resizable (GTK_WINDOW (fsel_dialog), TRUE);
	gtk_container_set_border_width (GTK_CONTAINER (fsel_dialog), 5);

	width = 0;
	gtk_window_set_title (GTK_WINDOW (fsel_dialog), _("Select a filter"));
	g_signal_connect (GTK_OBJECT (fsel_dialog), "delete_event",
		G_CALLBACK(close_dlg), fsel_dialog);

	ok_button = gtk_button_new_from_stock (GTK_STOCK_OK);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (fsel_dialog)->action_area),
		ok_button, TRUE, TRUE, 0);
	gtk_widget_show (ok_button);
/*	g_signal_connect (GTK_OBJECT (ok_button), "clicked",
		G_CALLBACK (add_ok), NULL);*/

	cancel_button = gtk_button_new_from_stock (GTK_STOCK_CANCEL);
	g_signal_connect (GTK_OBJECT (cancel_button), "clicked",
		G_CALLBACK(close_dlg), fsel_dialog);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (fsel_dialog)->action_area),
		cancel_button, FALSE, FALSE, 0);
	gtk_widget_show (cancel_button);

	gtk_box_set_spacing (GTK_BOX (GTK_DIALOG (fsel_dialog)->vbox), 10);

	tree_hbox = gtk_hbox_new (FALSE, 5);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (fsel_dialog)->vbox),
		tree_hbox, TRUE, TRUE, 0);

	treeview = GTK_TREE_VIEW (gtk_tree_view_new ());
	model = gtk_tree_view_get_model (treeview);
	treestore = gtk_tree_store_new (SINGLE_COL, G_TYPE_STRING);
	sort = GTK_TREE_SORTABLE (treestore);
	gtk_tree_view_set_model (GTK_TREE_VIEW (treeview), 
		GTK_TREE_MODEL (sort));
	gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (treeview), TRUE);
	gtk_tree_view_set_reorderable (GTK_TREE_VIEW (treeview), TRUE);
	gtk_tree_store_append (treestore, &parent_iter, NULL);
	gtk_tree_store_set (treestore, &parent_iter, FILTER_COL,
		_("Available columns"), -1);
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes
		(_("Column Name"), renderer, "text", FILTER_COL, NULL);
	gtk_tree_view_column_set_sort_column_id (column, FROM_COL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
	fselect = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));
	gtk_tree_selection_set_mode (GTK_TREE_SELECTION(fselect), GTK_SELECTION_SINGLE);

	scroll_w = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll_w),
		GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
	gtk_container_set_border_width (GTK_CONTAINER (scroll_w), 5);
	gtk_container_add (GTK_CONTAINER (scroll_w), 
		GTK_WIDGET(treeview));

	gtk_box_pack_start (GTK_BOX (tree_hbox), scroll_w, TRUE, TRUE, 0);

	/* now populate the treeview with filter names */
	for (idx = 0; idx < tab->file->filter_ct; idx++)
	{
		select_entries[0] = tab->file->filters[idx].name;
		gtk_tree_store_append (treestore, &child_iter, &parent_iter);
		gtk_tree_store_set (treestore, &child_iter, FILTER_COL, 
			select_entries[0], -1);
		width = (strlen(select_entries[0]) > width) ? 
			strlen(select_entries[0]) : width;
	}
	gtk_widget_set_size_request (GTK_WIDGET(treeview), 
		width + 200, 200);
	gtk_tree_view_expand_all (GTK_TREE_VIEW(treeview));
	gtk_widget_show_all (fsel_dialog);
}

void
filter_apply (GtkAction G_GNUC_UNUSED * w, gpointer data)
{
	QlContext * qlc;
	QlTabData * tab;

	qlc = ql_get_context (GTK_WIDGET(data));
	tab = ql_get_tabdata (qlc);
	/* We must first ask which filter to apply, then call filter_do_apply
	 * with its index */
	CHECK_CHANGED(tab);
	in_add = FALSE;
	filter_select (tab, QL_FILTER_APPLY);
}

void
filter_do_apply (QlTabData * tab, gint16 filter_index)
{
	gint i, j;
	gboolean flag, selected = FALSE;
	GtkSheetRange *x;
	QlFilterInfo *cf = &tab->file->filters[filter_index];
	tab->file->filter_ptr = cf;

	x = g_new0 (GtkSheetRange, 1);
	if (filter_index < 0)
	{
		filter_show_all (NULL, NULL);
		return;
	}

	big_draw_start (tab);
	gtk_sheet_unselect_range (tab->view->sheet);

	tab->view->sel_type = SELECT_NONE;
	for (i = 0; i < tab->file->last_row; i++)
	{
		gtk_sheet_row_set_visibility (tab->view->sheet, i, FALSE);
		if (cf->by_and)
		{
			flag = TRUE;
			for (j = 0; j < cf->line_ct; j++)
			{
				if (cf->line[j].field >= 0)
				{
					if (!filter_compare_record (tab, cf->line[j].field, i,
							cf->line[j].compare,
							cf->line[j].type, cf->use_nocase))
						flag = FALSE;
				}
			}
		}
		else
		{
			flag = FALSE;
			for (j = 0; j < cf->line_ct; j++)
			{
				if (cf->line[j].field >= 0)
				{
					if (filter_compare_record (tab, cf->line[j].field, i,
							cf->line[j].compare,
							cf->line[j].type, cf->use_nocase))
						flag = TRUE;
				}
			}
		}
		gtk_sheet_row_set_visibility (tab->view->sheet, i, flag);
		if (!selected && flag)
		{
			x->row0 = i;
			x->col0 = 0;
			x->rowi = i;
			x->coli = 0;
			selected = TRUE;
		}
	}
	gtk_sheet_select_range (tab->view->sheet, x);
	g_free (x);
	big_draw_end (tab);
}

static void G_GNUC_UNUSED
filter_entry_handler (GtkWidget * w, gpointer data)
{
	QlContext * qlc;
	QlTabData * tab;
	gint filter_idx = GPOINTER_TO_INT (data);

	qlc = ql_get_context (GTK_WIDGET(w));
	tab = ql_get_tabdata (qlc);

	if (filter_idx < 0 || !gtk_widget_get_name (w))
	{
		level2_error (tab, _("** PANIC ** in _filter_entry_handler"));
		return;
	}
	if (!strcmp (gtk_widget_get_name (w), "filter"))
	{
		strncpy (tab->file->filters[filter_idx].name,
			gtk_entry_get_text (GTK_ENTRY (w)),
			sizeof (tab->file->filters[filter_idx].name));
	}
	else
	{
		/** \bug remove use of widget names as flags */
/*		gint line_idx = atoi (gtk_widget_get_name (w));
		strncpy (tab->file->filters[filter_idx].line[line_idx].compare,
			gtk_entry_get_text (GTK_ENTRY (w)), sizeof
			(tab->file->filters[filter_idx].line[line_idx].compare));
*/	}
}

static void G_GNUC_UNUSED
filter_boolean_handler (GtkWidget * w, gpointer data)
{
	gint filter;
	QlContext * qlc;
	QlTabData * tab;

	qlc = ql_get_context (w);
	tab = ql_get_tabdata (qlc);
	filter = GPOINTER_TO_INT (data);
	/** \bug remove these nasty casts */
	if (!strcmp (gtk_widget_get_name (w), "bool"))
		tab->file->filters[filter].by_and
			= gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (w));
	else
		tab->file->filters[filter].use_nocase
			= gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (w));
}

/** Handle the selection of a menu item. Since the only menuitems we work with
 are in the add/edit field select and rule select operations, we just do
 all of them right here.

 \todo these casts stink. replace. */
static void
filter_menuitem_select (GtkWidget * w, gpointer data)
{
	QlContext * qlc;
	QlTabData * tab;
	gchar *name;
	gint16 line;

	qlc = ql_get_context (w);
	tab = ql_get_tabdata (qlc);
	name = g_strdup (gtk_widget_get_name (w));
	line = (*(gint16 *) (data) & (gint16) 0xf000) >> 12;
	if (!strcmp (name, "field"))
	{
		gint16 field = *(gint16 *) (data) & (gint16) 0x0fff;
		tab->file->filters[working_idx].line[line].field = field - 1;
	}
	if (!strcmp (name, "rule"))
	{
		gint16 rule = *(gint16 *) (data) & (gint16) 0x0fff;
		tab->file->filters[working_idx].line[line].type = rule;
	}
}

/* Removes the selected filter. */
static void
filter_do_delete (QlTabData * tab, gint16 filter_index)
{
	/** This is kind of slow, but it's simple and it works. I expect all this
	 * stuff will eventually be replaced by a GList of filter structures and
	 * then this kind of crap will go away.
	 
	\todo make this go away */
	gint reportx;

/*	if (dialog1_win)
		cancel_level1 (dialog1_win, NULL, NULL);*/
	tab->file->filter_ct--;

	dim_list_filter_menu (tab->qlc);
	front_is_changed (tab);

	/* Handle report modifications */
	for (reportx = 0; reportx < tab->file->report_ct; reportx++)
	{
		if (tab->file->reports[reportx].filter == filter_index)
			tab->file->reports[reportx].filter = -1;
		else if (tab->file->reports[reportx].filter > filter_index)
			tab->file->reports[reportx].filter--;
	}
	/** \bug rationalise this mess */
	if (tab->file->filter_ptr == &tab->file->filters[filter_index])
		tab->file->filter_ptr = NULL;
	if (filter_index == tab->file->filter_ct)
		return;
	if (tab->file->filter_ptr == &tab->file->filters[tab->file->filter_ct])
		tab->file->filter_ptr = &tab->file->filters[filter_index];
	tab->file->filters[filter_index] = tab->file->filters[tab->file->filter_ct];
}

/** Handles miscellaneous actions, typically from Edit */
static void
filter_action (GtkWidget G_GNUC_UNUSED * w, gpointer G_GNUC_UNUSED data)
{
	QlContext * qlc;
	QlTabData * tab;

	qlc = ql_get_context (w);
	tab = ql_get_tabdata (qlc);
	if (GTK_IS_DIALOG (w) || !strcmp (gtk_widget_get_name (w), "cancel"))
	{
		/* The user has requested that we cancel this operation. So first
		 * find out what the operation is, then don't do it. :) For now, only
		 * _filter_do_edit uses this.
		 */
		if (in_add)
			filter_do_delete (tab, working_idx);
		else
			tab->file->filters[working_idx] = old_filter;

		in_add = FALSE;
		gtk_widget_destroy (w);
		return;
	}
	/* The user clicked the OK Button. Since we've already done everything,
	 * just return.
	 */
	filter_do_apply (tab, working_idx);
	in_add = FALSE;
	dim_list_filter_menu (tab->qlc);
	front_is_changed (tab);
	gtk_widget_destroy (w);
}

/** \todo move elsewhere. */
typedef enum
{
	IN = 0,
	NOT_IN,
	EXACT,
	NOT,
	LESS,
	MORE,
	BLANK,
	NOT_BLANK
} QlRuleName;
/** \todo move elsewhere and handle init and clean up
separately. */
static GHashTable * rules = NULL;

/** Lets the user modify the selected filter */
static void
filter_do_edit (QlTabData * tab, gint16 filter_index)
{
	GtkWidget *edit_dialog, *ok_button, *cancel_button,
	*field_selector_menu, *field_selector_menu_item,
	*rule_selector_menu, *rule_selector_menu_item,
	*entry[MAX_FILTER_NESTING],
	*line_hbox[MAX_FILTER_NESTING],
	*filter_name, *filter_name_label, *name_hbox, 
	*case_check, *boolean_check, *boolean_hbox;

	gint16 line_index;
	gint16 field_index;
	gint rule_index;

	if (!rules)
	{
		rules = g_hash_table_new (NULL, NULL);
		g_hash_table_insert (rules, GINT_TO_POINTER(IN), 
			_("contains"));
		g_hash_table_insert (rules, GINT_TO_POINTER(NOT_IN), 
			_("does not contain"));
		g_hash_table_insert (rules, GINT_TO_POINTER(EXACT), 
			_("is exactly"));
		g_hash_table_insert (rules, GINT_TO_POINTER(NOT), 
			_("is not"));
		g_hash_table_insert (rules, GINT_TO_POINTER(LESS), 
			_("is less than"));
		g_hash_table_insert (rules, GINT_TO_POINTER(MORE), 
			_("is more than"));
		g_hash_table_insert (rules, GINT_TO_POINTER(BLANK), 
			_("is blank"));
		g_hash_table_insert (rules, GINT_TO_POINTER(NOT_BLANK), 
			_("is not blank"));
	}
	old_filter = tab->file->filters[filter_index];

	edit_dialog = gtk_dialog_new ();
	gtk_window_set_modal (GTK_WINDOW (edit_dialog), TRUE);
	gtk_window_set_position (GTK_WINDOW (edit_dialog), GTK_WIN_POS_CENTER);
	gtk_window_set_resizable (GTK_WINDOW (edit_dialog), TRUE);
	gtk_container_set_border_width (GTK_CONTAINER (edit_dialog), 5);

	gtk_window_set_title (GTK_WINDOW (edit_dialog), _("Filter edit"));

	ok_button = gtk_button_new_from_stock (GTK_STOCK_OK);
	gtk_widget_set_name (ok_button, "ok");
	/** \bug the filter handling is VERY confused. FIXME.*/
	g_signal_connect (GTK_OBJECT (ok_button), "clicked",
		G_CALLBACK (filter_action), tab);
	gtk_widget_show (ok_button);

	cancel_button = gtk_button_new_from_stock (GTK_STOCK_CANCEL);
	gtk_widget_set_name (cancel_button, "cancel");
	/** \bug find a new way to handle the filter index
	- passing as a GINT_TO_POINTER is not portable. */
/*	g_signal_connect (GTK_OBJECT (cancel_button), "clicked",
		G_CALLBACK (filter_action), GINT_TO_POINTER(filter_index));
*/	gtk_widget_show (cancel_button);

/*	g_signal_connect (GTK_OBJECT (edit_dialog), "delete_event",
		G_CALLBACK (filter_action), GINT_TO_POINTER(filter_index));
*/
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (edit_dialog)->action_area),
		ok_button, FALSE, FALSE, 5);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (edit_dialog)->action_area),
		cancel_button, FALSE, FALSE, 5);

	filter_name = gtk_entry_new ();
	gtk_widget_set_name (filter_name, "filter");
	filter_name_label = gtk_label_new (_("Filter name"));
	boolean_check = gtk_check_button_new_with_label
		(_("All of these must be true"));
	case_check = gtk_check_button_new_with_label
		(_("Text is case-insensitive"));
	gtk_widget_set_name (boolean_check, "bool");
	gtk_widget_set_name (case_check, "case");

	if (tab->file->filters[filter_index].name != NULL)
		gtk_entry_set_text (GTK_ENTRY (filter_name),
			tab->file->filters[filter_index].name);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (boolean_check),
		tab->file->filters[filter_index].by_and);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (case_check),
		tab->file->filters[filter_index].use_nocase);
/*	g_signal_connect (GTK_OBJECT (boolean_check), "toggled",
		G_CALLBACK (filter_boolean_handler), 
		GINT_TO_POINTER (filter_index));
	g_signal_connect (GTK_OBJECT (case_check), "toggled",
		G_CALLBACK (filter_boolean_handler), 
		GINT_TO_POINTER (filter_index));
*/
	gtk_widget_show (filter_name);
	gtk_widget_show (filter_name_label);
	gtk_widget_show (boolean_check);
	gtk_widget_show (case_check);

	name_hbox = gtk_hbox_new (TRUE, 5);
	boolean_hbox = gtk_hbox_new (TRUE, 5);

	gtk_box_pack_start (GTK_BOX (name_hbox), filter_name_label, TRUE, TRUE,
		5);
	gtk_box_pack_start (GTK_BOX (name_hbox), filter_name, TRUE, TRUE, 5);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (edit_dialog)->vbox),
		name_hbox, TRUE, TRUE, 5);
	gtk_widget_show (name_hbox);
	gtk_box_pack_start (GTK_BOX (boolean_hbox), boolean_check, TRUE, TRUE,
		5);
	gtk_box_pack_start (GTK_BOX (boolean_hbox), case_check, TRUE, TRUE, 5);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (edit_dialog)->vbox),
		boolean_hbox, TRUE, TRUE, 5);
	gtk_widget_show (boolean_hbox);

	for (line_index = 0; line_index < MAX_FILTER_NESTING; line_index++)
	{
		/** \bug does this need to be [32] ?*/
		gchar nbuf[32];

		line_hbox[line_index] = gtk_hbox_new (TRUE, 5);

		entry[line_index] = gtk_entry_new ();
		g_snprintf (nbuf, 32, "%d", line_index);
		gtk_widget_set_name (entry[line_index], nbuf);
		gtk_entry_set_text (GTK_ENTRY (entry[line_index]),
			tab->file->filters[filter_index].line[line_index].compare);
/*		g_signal_connect (GTK_OBJECT (entry[line_index]), "changed",
			G_CALLBACK (filter_entry_handler), 
			GINT_TO_POINTER (filter_index));
*/
		field_selector_menu = gtk_menu_new ();

		/** \bug The attachment mechanism assumes that a pointer can store 
		16 bits but not necessarily 32, since some inferior platforms 
		only do 16. So I use 4 bits for line and 12 for field. This 
		allows only 16 lines; the current implementation is for only 
		6. When this is replaced with GLists of filter lines, this 
		mechanism will have to be re-thought. Probably just go ahead 
		and assume sizeof (void*) >= 4.
		 */
		/** \todo replace with GList */
		for (field_index = 0; field_index <= tab->file->last_field + 1;
			field_index++)
		{
			if (!field_index)
				field_selector_menu_item = gtk_menu_item_new_with_label
					(_("--- Unused ---"));
			else
			{
				QlFieldInfo * field;
				field = ql_get_fieldinfo (tab, (field_index - 1));
				field_selector_menu_item = gtk_menu_item_new_with_label
					(field->name);
			}

			gtk_widget_set_name (field_selector_menu_item, "field");
			{
				/* workaround */
				gint t;
				t = (line_index << 12) | field_index;
				g_signal_connect (GTK_OBJECT (field_selector_menu_item),
					"activate",	G_CALLBACK (filter_menuitem_select), 
					GINT_TO_POINTER (t));
			}
			gtk_menu_attach (GTK_MENU (field_selector_menu),
				field_selector_menu_item, 0, 1, 0, 1);
			if (tab->file->filters[filter_index].line[line_index].field + 1
				== field_index)
				gtk_menu_set_active (GTK_MENU (field_selector_menu),
					field_index);
			gtk_widget_show (field_selector_menu_item);
		}
//      gtk_option_menu_set_menu ((GtkOptionMenu *) field_selector,
//          field_selector_menu);
//      gtk_widget_show (field_selector);

/*		gtk_box_pack_start (GTK_BOX (line_hbox[line_index]),
			    field_selector, TRUE, TRUE, 5);
*/
//      rule_selector = gtk_option_menu_new ();
		rule_selector_menu = gtk_menu_new ();

		for (rule_index = FILTER_CONTAINS; rule_index < NOT_BLANK;
			rule_index++)
		{
			rule_selector_menu_item = 
				gtk_menu_item_new_with_label 
				(g_hash_table_lookup (rules, GINT_TO_POINTER(rule_index)));
			gtk_widget_set_name (rule_selector_menu_item, "rule");
			{
				gint t;
				t = (line_index << 12) | rule_index;
				g_signal_connect (GTK_OBJECT (rule_selector_menu_item),
					"activate",	G_CALLBACK (filter_menuitem_select), 
					GINT_TO_POINTER(t));
			}
			gtk_menu_attach (GTK_MENU (rule_selector_menu),
				rule_selector_menu_item, 0, 1, 0, 1);
			if (tab->file->filters[filter_index].line[line_index].type
				== rule_index)
				gtk_menu_set_active (GTK_MENU (rule_selector_menu),
					rule_index);
			gtk_widget_show (rule_selector_menu_item);
		}
//      gtk_option_menu_set_menu ((GtkOptionMenu *) rule_selector,
//          rule_selector_menu);
//      gtk_widget_show (rule_selector);

/*		gtk_box_pack_start (GTK_BOX (line_hbox[line_index]),
			    rule_selector, TRUE, TRUE, 5);
	gtk_widget_show (line_hbox[line_index]);
*/
		gtk_widget_show (entry[line_index]);
		gtk_box_pack_start (GTK_BOX (line_hbox[line_index]),
			entry[line_index], TRUE, TRUE, 5);

		gtk_box_pack_start (GTK_BOX (GTK_DIALOG (edit_dialog)->vbox),
			line_hbox[line_index], TRUE, TRUE, 5);
	}
	gtk_widget_show (edit_dialog);
}

void
filter_add (GtkAction G_GNUC_UNUSED * w, gpointer data)
{
	QlContext * qlc;
	QlTabData * tab;
	gint j;

	qlc = ql_get_context (GTK_WIDGET(data));
	tab = ql_get_tabdata (qlc);

	/* Create a new filter and then edit it */
	CHECK_CHANGED(tab);
	if (tab->file->filter_ct >= MAX_FILTERS)
	{
		level2_error (tab, _("Unexpected filter count overflow "
			"in filter_add"));
		return;
	}

	in_add = TRUE;

	working_idx = tab->file->filter_ct++;
	/* Allocate and clear a new entry */
	memset (&tab->file->filters[working_idx], 0, sizeof (QlFilterInfo));
	strncpy (tab->file->filters[working_idx].name, "Untitled filter",
		MAX_FIELD_NAME);
	tab->file->filters[working_idx].line_ct = MAX_FILTER_NESTING;
	tab->file->filters[working_idx].by_and = FALSE;
	for (j = 0; j < MAX_FILTER_NESTING; j++)
		tab->file->filters[working_idx].line[j].field = -1;
	tab->file->filter_ptr = tab->file->filters + working_idx;
	filter_do_edit (tab, working_idx);
}

void
filter_edit (GtkAction G_GNUC_UNUSED * w, gpointer data)
{
	QlContext * qlc;
	QlTabData * tab;

	qlc = ql_get_context (GTK_WIDGET(data));
	tab = ql_get_tabdata (qlc);
	CHECK_CHANGED(tab);
	in_add = FALSE;
	filter_select (tab, QL_FILTER_EDIT);
}

void
filter_delete (GtkAction G_GNUC_UNUSED * w, gpointer data)
{
	QlContext * qlc;
	QlTabData * tab;

	qlc = ql_get_context (GTK_WIDGET(data));
	tab = ql_get_tabdata (qlc);
	/* Simply removes the selected filter from memory */
	CHECK_CHANGED(tab);
	in_add = FALSE;
	filter_select (tab, QL_FILTER_DELETE);
}

/* Handles treeview filter selections */
/** \todo complete transition to treeview handler */
/*static G_GNUC_UNUSED void
select_ok (GtkButton * button, gpointer data)
{
	GtkTreeSelection *selection;
	GtkTreeIter iter;
	GtkTreeModel *model;
	gchar * filter_name;
	QlFilterMode mode;

	selection = GTK_TREE_SELECTION (data);
	if (gtk_tree_selection_get_selected (selection, &model, &iter))
	{
		gtk_tree_model_get (model, &iter, FILTER_COL, 
			&filter_name, -1);
	}
	working_idx = row;
//	mode = GPOINTER_TO_INT (data);
	switch (mode)
	{
	case QL_FILTER_APPLY:
		filter_do_apply (working_idx);
		break;
	case QL_FILTER_EDIT:
		filter_do_edit (working_idx);
		break;
	case QL_FILTER_DELETE:
		filter_do_delete (working_idx);
		break;
	default:
		level2_error (_("Unexpected mode in _filter_selection_made."));
		break;
	}
}*/
