/******************************************************************************
*		       							      *
* main.c (part of rCalc)					       	      *
* Copyright 1999, 2000 Gary Benson <rat@spunge.org>		       	      *
*								       	      *
* This program is free software; you can redistribute it and/or modify 	      *
* it under the terms of the GNU General Public License as published by 	      *
* the Free Software Foundation; either version 2 of the License, or    	      *
* (at your option) any later version.				       	      *
*								       	      *
* This program is distributed in the hope that it will be useful,      	      *
* but WITHOUT ANY WARRANTY; without even the implied warranty of       	      *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the	       	      *
* GNU General Public License for more details.			       	      *
*								       	      *
* You should have received a copy of the GNU General Public License    	      *
* along with this program; if not, write to the Free Software	       	      *
* Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.	       	      *
*								       	      *
******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <signal.h>
#include <gnome.h>
#include <zvt/zvtterm.h>

#include "config.h"
#include "console.h"

static pid_t pidParent, pidChild;


static GtkWidget *winMain;		/* The main window		     */
static GtkWidget *appBar;		/*  Statusbar			     */
static GtkWidget *zvtTerm;		/*  Virtual terminal		     */
static GtkWidget *scrollBar;		/*  Scrollbar			     */

static GtkWidget *winPrefs;		/* Preferences window		     */
static GtkWidget *optScrollPos;		/*  Scrollbar position		     */
static GtkWidget *spnScrollback;	/*  Scrollback lines		     */
static GtkWidget *chkScrollKey;		/*  Scroll on keystroke		     */
static GtkWidget *chkScrollOut;		/*  Scroll on output		     */
static GtkWidget *chkCursBlink;		/*  Blinking cursor		     */
static GtkWidget *chkNoBell;		/*  Silence bell		     */
static GtkWidget *fntFontPicker;	/*  Font			     */
static GtkWidget *clrForeground;	/*  Foreground colour		     */
static GtkWidget *clrBackground;	/*  Background colour		     */
static GtkWidget *clrPrompt;		/*  Prompt colour		     */
static GtkWidget *clrError;		/*  Error colour		     */
static GtkWidget *optAngles;		/*  Angle units			     */

/*****************************************************************************/

/* Index into the help system. Slightly non-standard, as I wish to
** be able to call up help pages from within the calculation engine.
*/
static GnomeHelpMenuEntry helpTopic_Refs[NUM_HELPTOPICS]=
{
	/* helpNoHelp 		*/ { NULL, NULL },
	/* helpIntro 		*/ { PACKAGE, "index.html" },
	/* helpPrefBehaviour 	*/ { PACKAGE, "index.html#prf_behave" },
	/* helpPrefAppearance 	*/ { PACKAGE, "index.html#prf_appear" },
	/* helpPrefCalculator 	*/ { PACKAGE, "index.html#prf_calculate" },
	/* helpExit 		*/ { PACKAGE, "index.html#cmd_exit" },
	/* helpHelp 		*/ { PACKAGE, "index.html#cmd_help" },
	/* helpLs 		*/ { PACKAGE, "index.html#cmd_ls" },
	/* helpRm 		*/ { PACKAGE, "index.html#cmd_rm" },
	/* helpMode 		*/ { PACKAGE, "index.html#cmd_mode" },
	/* helpInt 		*/ { PACKAGE, "index.html#fun_int" },
	/* helpAbs 		*/ { PACKAGE, "index.html#fun_abs" },
	/* helpLogs 		*/ { PACKAGE, "index.html#fun_logs" },
	/* helpSqrt 		*/ { PACKAGE, "index.html#fun_sqrt" },
	/* helpTrig 		*/ { PACKAGE, "index.html#fun_trig" },
	/* helpHype 		*/ { PACKAGE, "index.html#fun_hype" },
	/* helpConstants 	*/ { PACKAGE, "index.html#constants" },
};
	
static void displayHelp( HELPTOPIC topic )
{
	Debug( "%s %d -> %s", __FUNCTION__, topic, helpTopic_Refs[topic].path);
	gnome_help_display( winMain, &helpTopic_Refs[topic] );
}

/*****************************************************************************/

/* Apply the UI preferences 
*/
static void applyUIPreferences( void *ignore )
{
	GdkFont *font;
	GdkGeometry hints;
	int i;

	/* Set the scrollbar position
	*/
	if( cfgUI->scrollbarPos == scrollbarHidden )
	{
		gtk_widget_hide( scrollBar );
	}
	else
	{
		GtkWidget *box = scrollBar->parent;
		gtk_box_set_child_packing( GTK_BOX(box), scrollBar,
			FALSE, TRUE, 0,
			(cfgUI->scrollbarPos == scrollbarLeft) ?
			GTK_PACK_START : GTK_PACK_END );
		
		gtk_widget_show (scrollBar);
	}

	/* Set the terminal behaviour.
	*/
	zvt_term_set_bell( ZVT_TERM(zvtTerm), cfgUI->termBell );
	zvt_term_set_blink( ZVT_TERM(zvtTerm), cfgUI->cursBlink );
	zvt_term_set_scrollback( ZVT_TERM(zvtTerm), cfgUI->scrollbackLines );
	zvt_term_set_scroll_on_keystroke(ZVT_TERM(zvtTerm),cfgUI->scrollOnKey);
	zvt_term_set_scroll_on_output( ZVT_TERM(zvtTerm), cfgUI->scrollOnOut );

	/* Set the font (and the resize hints)
	*/
	if(( font = gdk_font_load( cfgUI->fontName ) ))
	{
		zvt_term_set_fonts( ZVT_TERM(zvtTerm), font, 0 );
	}
	hints.base_width  = ( zvtTerm->style->klass->xthickness * 2 );
	hints.base_height = ( zvtTerm->style->klass->ythickness * 2 );

	hints.width_inc  = ZVT_TERM(zvtTerm)->charwidth;
	hints.height_inc = ZVT_TERM(zvtTerm)->charheight;
	hints.min_width  = hints.base_width  + hints.width_inc;
	hints.min_height = hints.base_height + hints.height_inc;

	gtk_window_set_geometry_hints( GTK_WINDOW(winMain),
				       GTK_WIDGET(zvtTerm),
				       &hints,
				       GDK_HINT_RESIZE_INC |
				       GDK_HINT_MIN_SIZE |
				       GDK_HINT_BASE_SIZE );

	/* Set the colours.
	*/
	for( i=0; i<3; i++ )
	{
		cfgUI->palette[i][16] = cfgUI->palette[i][7];
		cfgUI->palette[i][17] = cfgUI->palette[i][0];
	}
	zvt_term_set_color_scheme(
		ZVT_TERM(zvtTerm), cfgUI->palette[0],
		cfgUI->palette[1], cfgUI->palette[2] );
}

/* Read all preferences from the preferences window, and apply them
*/
static void gatherAndApplyPreferences( void )
{
	GtkWidget *menu,*item;
	gushort r[3];
	int i;

	/* Gather the UI preferences
	*/
	menu = GTK_OPTION_MENU(optScrollPos)->menu;
	item = gtk_menu_get_active( GTK_MENU(menu) );
	cfgUI->scrollbarPos =
		g_list_index( GTK_MENU_SHELL(menu)->children, item );

	cfgUI->termBell		=!GTK_TOGGLE_BUTTON( chkNoBell )->active;
	cfgUI->cursBlink	= GTK_TOGGLE_BUTTON( chkCursBlink )->active;
	cfgUI->scrollbackLines	=
		gtk_spin_button_get_value_as_int(
			GTK_SPIN_BUTTON(spnScrollback));
	cfgUI->scrollOnKey	= GTK_TOGGLE_BUTTON( chkScrollKey )->active;
	cfgUI->scrollOnOut	= GTK_TOGGLE_BUTTON( chkScrollOut )->active;

	g_free( cfgUI->fontName );
	cfgUI->fontName = g_strdup(
		gnome_font_picker_get_font_name(
			GNOME_FONT_PICKER(fntFontPicker) ) );

	gnome_color_picker_get_i16(
		GNOME_COLOR_PICKER(clrForeground),
		&r[0], &r[1], &r[2], NULL);
	for( i=0; i<3; i++ ) cfgUI->palette[i][colForeground] = r[i];

	gnome_color_picker_get_i16(
		GNOME_COLOR_PICKER(clrBackground),
		&r[0], &r[1], &r[2], NULL);
	for( i=0; i<3; i++ ) cfgUI->palette[i][colBackground] = r[i];

	gnome_color_picker_get_i16(
		GNOME_COLOR_PICKER(clrPrompt),
		&r[0], &r[1], &r[2], NULL);
	for( i=0; i<3; i++ ) cfgUI->palette[i][colPrompt] = r[i];

	gnome_color_picker_get_i16(
		GNOME_COLOR_PICKER(clrError),
		&r[0], &r[1], &r[2], NULL);
	for( i=0; i<3; i++ ) cfgUI->palette[i][colError] = r[i];

	/* Apply the UI preferences.
	*/
	applyUIPreferences(NULL);

	/* Gather (no need to apply) the calculator preferences.
	*/
	rCalc_config_Lock();

	menu = GTK_OPTION_MENU(optAngles)->menu;
	item = gtk_menu_get_active( GTK_MENU(menu) );
	cfgCalc->angleMode =
		g_list_index( GTK_MENU_SHELL(menu)->children, item );

	rCalc_config_Unlock();
}	

/*****************************************************************************/

static void cbApplyChanges( GtkWidget *w, gint page )
{
	if( page == -1 )
	{
		gatherAndApplyPreferences();
	}
}
static void cbClosePrefs( GtkWidget *w, GdkEvent *e )
{
	winPrefs = NULL;
}
static void cbPrefsHelp( GtkWidget *w, gint page )
{
	HELPTOPIC pageXref[3]= { helpPrefBehaviour,
				 helpPrefAppearance,
				 helpPrefCalculator };
	
	displayHelp( pageXref[page] );
}
static void cbPrefChanged( GtkWidget *w )
{
	gnome_property_box_changed( GNOME_PROPERTY_BOX(winPrefs) );
}
static void cbFontChanged( GtkWidget *w, guchar *name )
{
	cbPrefChanged( w );
}
static void cbColourChanged( GtkWidget *w, guint r, guint g, guint b, guint a )
{
	cbPrefChanged( w );
}

/*****************************************************************************/

/* Build and run the preferences window.
*/
char *scrollbar_position_list[] =
{
	/* Preferences | Behaviour | Scrollbar on left */
	N_("Left"),
	/* Preferences | Behaviour | Scrollbar on right */
	N_("Right"),
	/* Preferences | Behaviour | Scrollbar hidden */
	N_("Hidden"),
	NULL
};
char *angle_unit_list[] =
{
	/* Preferences | Calculator | Angles in degrees */
	N_("Degrees"),
	/* Preferences | Calculator | Angles in radians */
	N_("Radians"),
	NULL
};

static void buildAndShow_winPrefs( void )
{
	GtkWidget *table,*label;
	GtkWidget *menu;
	GtkObject *adj;
	char **menu_list;
	
	winPrefs = gnome_property_box_new ();
	gtk_window_set_title(
		GTK_WINDOW(winPrefs), PACKAGE );
	gtk_signal_connect(
		GTK_OBJECT(winPrefs), "apply",
		GTK_SIGNAL_FUNC(cbApplyChanges), NULL );
	gtk_signal_connect(
		GTK_OBJECT(winPrefs), "destroy",
		GTK_SIGNAL_FUNC(cbClosePrefs), NULL );
	gtk_signal_connect(
		GTK_OBJECT(winPrefs), "help",
		GTK_SIGNAL_FUNC(cbPrefsHelp), NULL );

	/* BEHAVIOUR PAGE */
	table = gtk_table_new( 6, 2, FALSE );
	gnome_property_box_append_page(
		GNOME_PROPERTY_BOX(winPrefs), table,
		/* Preferences | Behaviour */
		gtk_label_new (_("Behavior")));

	/* Preferences | Behaviour | Scrollbar position selector */
	label = gtk_label_new( _("Scrollbar position") );
	gtk_misc_set_alignment( GTK_MISC(label), 1.0, 0.5 );
	gtk_table_attach(
		GTK_TABLE(table), label, 1, 2, 1, 2,
		GTK_FILL, 0, GNOME_PAD, GNOME_PAD );

	optScrollPos = gtk_option_menu_new ();
	menu = gtk_menu_new ();
	menu_list = scrollbar_position_list;
	while( *menu_list )
	{
		GtkWidget *entry;
		entry = gtk_menu_item_new_with_label( _(*menu_list) );
		gtk_widget_show( entry );
		gtk_menu_append( GTK_MENU(menu), entry );
		menu_list++;
	}
	gtk_option_menu_set_menu( GTK_OPTION_MENU(optScrollPos), menu );
	gtk_option_menu_set_history(
		GTK_OPTION_MENU(optScrollPos), cfgUI->scrollbarPos );
	gtk_signal_connect(
		GTK_OBJECT(menu), "deactivate",
		GTK_SIGNAL_FUNC(cbPrefChanged), NULL );
	gtk_table_attach(
		GTK_TABLE(table), optScrollPos, 2, 3, 1, 2,
		GTK_FILL, 0, GNOME_PAD, GNOME_PAD );
	
	/* Preferences | Behaviour | Number of scrollback lines */
	label = gtk_label_new( _("Scrollback lines") );
	gtk_misc_set_alignment( GTK_MISC(label), 1.0, 0.5 );
	gtk_table_attach(
		GTK_TABLE(table), label, 1, 2, 2, 3,
		GTK_FILL, 0, GNOME_PAD, GNOME_PAD );

	adj = gtk_adjustment_new(
		(gfloat)cfgUI->scrollbackLines,
		1.0, 100000.0, 1.0, 5.0, 0.0 );
	spnScrollback = gtk_spin_button_new( GTK_ADJUSTMENT(adj), 0, 0 );
	gtk_signal_connect(
		GTK_OBJECT(spnScrollback), "changed",
		GTK_SIGNAL_FUNC(cbPrefChanged), NULL );
	gnome_dialog_editable_enters(
		GNOME_DIALOG(winPrefs),
		GTK_EDITABLE(spnScrollback) );
	gtk_table_attach(
		GTK_TABLE(table), spnScrollback, 2, 3, 2, 3,
		GTK_FILL, 0, GNOME_PAD, GNOME_PAD );

	/* Scroll on keystroke? */
	chkScrollKey = gtk_check_button_new_with_label(
		/* Preferences | Behaviour | Scroll on keystroke checkbox */
		_("Scroll on keystroke"));
	gtk_toggle_button_set_active(
		GTK_TOGGLE_BUTTON(chkScrollKey), cfgUI->scrollOnKey);
	gtk_signal_connect(
		GTK_OBJECT(chkScrollKey), "toggled",
		GTK_SIGNAL_FUNC(cbPrefChanged), NULL );
	gtk_table_attach(
		GTK_TABLE(table), chkScrollKey, 2, 3, 3, 4, 
		GTK_FILL, 0, GNOME_PAD, GNOME_PAD );

	/* Scroll on output? */
	chkScrollOut = gtk_check_button_new_with_label(
		/* Preferences | Behaviour | Scroll on output checkbox */
		_("Scroll on output"));
	gtk_toggle_button_set_active(
		GTK_TOGGLE_BUTTON(chkScrollOut), cfgUI->scrollOnOut);
	gtk_signal_connect(
		GTK_OBJECT(chkScrollOut), "toggled",
		GTK_SIGNAL_FUNC(cbPrefChanged), NULL );
	gtk_table_attach(
		GTK_TABLE(table), chkScrollOut, 2, 3, 4, 5, 
		GTK_FILL, 0, GNOME_PAD, GNOME_PAD );

	/* Blinking cursor? */
	chkCursBlink = gtk_check_button_new_with_label(
		/* Preferences | Behaviour | Blinking cursor checkbox */
		_("Blinking cursor"));
	gtk_toggle_button_set_active(
		GTK_TOGGLE_BUTTON(chkCursBlink), cfgUI->cursBlink);
	gtk_signal_connect(
		GTK_OBJECT(chkCursBlink), "toggled",
		GTK_SIGNAL_FUNC(cbPrefChanged), NULL );
	gtk_table_attach(
		GTK_TABLE(table), chkCursBlink, 2, 3, 5, 6, 
		GTK_FILL, 0, GNOME_PAD, GNOME_PAD );

	/* Silence bell? */
	chkNoBell = gtk_check_button_new_with_label(
		/* Preferences | Behaviour | Silence bell checkbox */
		_("Silence bell"));
	gtk_toggle_button_set_active(
		GTK_TOGGLE_BUTTON(chkNoBell), !cfgUI->termBell );
	gtk_signal_connect(
		GTK_OBJECT(chkNoBell), "toggled",
		GTK_SIGNAL_FUNC(cbPrefChanged), NULL );
	gtk_table_attach(
		GTK_TABLE(table), chkNoBell, 2, 3, 6, 7, 
		GTK_FILL, 0, GNOME_PAD, GNOME_PAD );

	/* APPEARANCE PAGE
	*/
	table = gtk_table_new( 5, 2, FALSE );
	gnome_property_box_append_page(
		GNOME_PROPERTY_BOX(winPrefs), table,
		/* Preferences | Appearance */
		gtk_label_new (_("Appearance")));

	/* Preferences | Appearance | Font selector */
	label = gtk_label_new( _("Font") );
	gtk_misc_set_alignment( GTK_MISC(label), 1.0, 0.5 );
	gtk_table_attach(
		GTK_TABLE(table), label, 1, 2, 1, 2,
		GTK_FILL, 0, GNOME_PAD, GNOME_PAD );

	fntFontPicker = gnome_font_picker_new();
	gnome_font_picker_set_mode(
		GNOME_FONT_PICKER(fntFontPicker),
		GNOME_FONT_PICKER_MODE_FONT_INFO );
	gnome_font_picker_fi_set_show_size(
		GNOME_FONT_PICKER(fntFontPicker), TRUE );
	gnome_font_picker_set_font_name(
		GNOME_FONT_PICKER(fntFontPicker), cfgUI->fontName );
	gtk_signal_connect(
		GTK_OBJECT(fntFontPicker), "font_set",
		GTK_SIGNAL_FUNC(cbFontChanged), NULL );
	gtk_table_attach(
		GTK_TABLE(table), fntFontPicker, 2, 3, 1, 2,
		GTK_FILL, 0, GNOME_PAD, GNOME_PAD );

	/* Preferences | Appearance | Foreground colour selector */
	label = gtk_label_new( _("Foreground color") );
	gtk_misc_set_alignment( GTK_MISC(label), 1.0, 0.5 );
	gtk_table_attach(
		GTK_TABLE(table), label, 1, 2, 2, 3,
		GTK_FILL, 0, GNOME_PAD, GNOME_PAD );

	clrForeground = gnome_color_picker_new();
	gnome_color_picker_set_i16(
		GNOME_COLOR_PICKER(clrForeground),
		cfgUI->palette[0][colForeground],
		cfgUI->palette[1][colForeground],
		cfgUI->palette[2][colForeground], 0 );
	gtk_signal_connect(
		GTK_OBJECT(clrForeground), "color_set",
		GTK_SIGNAL_FUNC(cbColourChanged), NULL );
	gtk_table_attach(
		GTK_TABLE(table), clrForeground, 2, 3, 2, 3,
		0, 0, GNOME_PAD, GNOME_PAD );
	
	/* Preferences | Appearance | Background colour selector */
	label = gtk_label_new( _("Background color") );
	gtk_misc_set_alignment( GTK_MISC(label), 1.0, 0.5 );
	gtk_table_attach(
		GTK_TABLE(table), label, 1, 2, 3, 4,
		GTK_FILL, 0, GNOME_PAD, GNOME_PAD );

	clrBackground = gnome_color_picker_new();
	gnome_color_picker_set_i16(
		GNOME_COLOR_PICKER(clrBackground),
		cfgUI->palette[0][colBackground],
		cfgUI->palette[1][colBackground],
		cfgUI->palette[2][colBackground], 0 );
	gtk_signal_connect(
		GTK_OBJECT(clrBackground), "color_set",
		GTK_SIGNAL_FUNC(cbColourChanged), NULL );
	gtk_table_attach(
		GTK_TABLE(table), clrBackground, 2, 3, 3, 4,
		0, 0, GNOME_PAD, GNOME_PAD );
	
	/* Preferences | Appearance | Prompt colour selector */
	label = gtk_label_new( _("Prompt color") );
	gtk_misc_set_alignment( GTK_MISC(label), 1.0, 0.5 );
	gtk_table_attach(
		GTK_TABLE(table), label, 1, 2, 4, 5,
		GTK_FILL, 0, GNOME_PAD, GNOME_PAD );

	clrPrompt = gnome_color_picker_new();
	gnome_color_picker_set_i16(
		GNOME_COLOR_PICKER(clrPrompt),
		cfgUI->palette[0][colPrompt],
		cfgUI->palette[1][colPrompt],
		cfgUI->palette[2][colPrompt], 0 );
	gtk_signal_connect(
		GTK_OBJECT(clrPrompt), "color_set",
		GTK_SIGNAL_FUNC(cbColourChanged), NULL );
	gtk_table_attach(
		GTK_TABLE(table), clrPrompt, 2, 3, 4, 5,
		0, 0, GNOME_PAD, GNOME_PAD );
	
	/* Preferences | Appearance | Error colour selector */
	label = gtk_label_new( _("Error color") );
	gtk_misc_set_alignment( GTK_MISC(label), 1.0, 0.5 );
	gtk_table_attach(
		GTK_TABLE(table), label, 1, 2, 5, 6,
		GTK_FILL, 0, GNOME_PAD, GNOME_PAD );

	clrError = gnome_color_picker_new();
	gnome_color_picker_set_i16(
		GNOME_COLOR_PICKER(clrError),
		cfgUI->palette[0][colError],
		cfgUI->palette[1][colError],
		cfgUI->palette[2][colError], 0 );
	gtk_signal_connect(
		GTK_OBJECT(clrError), "color_set",
		GTK_SIGNAL_FUNC(cbColourChanged), NULL );
	gtk_table_attach(
		GTK_TABLE(table), clrError, 2, 3, 5, 6,
		0, 0, GNOME_PAD, GNOME_PAD );

	/* CALCULATOR PAGE
	*/
	table = gtk_table_new( 1, 2, FALSE );
	gnome_property_box_append_page(
		GNOME_PROPERTY_BOX(winPrefs), table,
		/* Preferences | Calculator */
		gtk_label_new (_("Calculator")));

	rCalc_config_Lock();

	/* Preferences | Calculator | Angle units selector */
	label = gtk_label_new( _("Angle units") );
	gtk_misc_set_alignment( GTK_MISC(label), 1.0, 0.5 );
	gtk_table_attach(
		GTK_TABLE(table), label, 1, 2, 1, 2,
		GTK_FILL, 0, GNOME_PAD, GNOME_PAD );

	optAngles = gtk_option_menu_new();
	menu = gtk_menu_new();
	menu_list = angle_unit_list;
	while( *menu_list )
	{
		GtkWidget *entry;
		entry = gtk_menu_item_new_with_label( _(*menu_list) );
		gtk_widget_show( entry );
		gtk_menu_append( GTK_MENU(menu), entry );
		menu_list++;
	}
	gtk_option_menu_set_menu( GTK_OPTION_MENU(optAngles), menu );
	gtk_option_menu_set_history(
		GTK_OPTION_MENU(optAngles), cfgCalc->angleMode );
	gtk_signal_connect(
		GTK_OBJECT(menu), "deactivate",
		GTK_SIGNAL_FUNC(cbPrefChanged), NULL );
	gtk_table_attach(
		GTK_TABLE(table), optAngles, 2, 3, 1, 2,
		GTK_FILL, 0, GNOME_PAD, GNOME_PAD );

	rCalc_config_Unlock();
	
	gnome_dialog_set_parent( GNOME_DIALOG(winPrefs), GTK_WINDOW(winMain) );
	gtk_widget_show_all( winPrefs );
}

/*****************************************************************************/

static void cbPrefs( GtkWidget *w )
{
	if( winPrefs )
	{
		gtk_widget_show( winPrefs );
	}
	else
	{
		buildAndShow_winPrefs();
	}
}
static void cbAbout( GtkWidget *w )
{
	GtkWidget *winAbout;
	const gchar *authors[] =
	{
	    /* About | Authors */
	    _( "Gary Benson <rat@spunge.org>, main programmer." ),
	    /* About | Authors */
	    _( "Johann Glaser <Johann.Glaser@gmx.at>, German translations." ),
	    /* About | Authors */
	    _( "Mathieu Roy <yeupou@free.fr>, French translations." ),
	    /* About | Authors */
	    _( "Luis Recuerda Santiago <lrec@altavista.net>, Spanish translations." ),
	    /* About | Authors */
	    _( "Stefano Barazza <ebarazza@libero.it>, Italian translations." ),
	    NULL
	};

	winAbout = gnome_about_new(
		PACKAGE, VERSION,
		"(C) Copyright 1999, 2000 Gary Benson",
		authors,
		/* About | Description */
		_( "A fast, simple, symbolic calculator." ),
		NULL );

	gnome_dialog_set_parent( GNOME_DIALOG(winAbout), GTK_WINDOW(winMain) );
	gnome_dialog_run_and_close( GNOME_DIALOG(winAbout) );
}
static void cbExit( GtkWidget *w )
{
	gtk_main_quit();
}
static void cbConfigure( GtkWidget *w, GdkEventConfigure *event )
{
/*  	kill( pidChild, SIGWINCH );	// No */
/*  	zvt_term_reset( ZVT_TERM(zvtTerm), FALSE ); // Nor this */
}

/* Stop CTRL-C from being passed to the child (and interrupting it).
** This will have to go if an Edit->Copy menu item is implemented...
*/
static int cbStripCtrlC( GtkWidget *w, GdkEvent *e )
{
	if( e->type == GDK_KEY_PRESS )
	{
		if( e->key.keyval == GDK_c &&
		    e->key.state & GDK_CONTROL_MASK )
		{
			return TRUE;	/* Event handled... */
		}
	}
	return FALSE;
}

/*****************************************************************************/

/* Build and run the main window.
*/
static GnomeUIInfo file_menu[]=
{
  	GNOMEUIINFO_MENU_PREFERENCES_ITEM( cbPrefs, NULL ),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_MENU_EXIT_ITEM( cbExit, NULL ),
	GNOMEUIINFO_END
};
static GnomeUIInfo help_menu[]=
{
	GNOMEUIINFO_HELP( PACKAGE ),
	GNOMEUIINFO_MENU_ABOUT_ITEM( cbAbout, NULL ),
	GNOMEUIINFO_END	
};
static GnomeUIInfo menu_bar[]=
{
	GNOMEUIINFO_MENU_FILE_TREE( file_menu ),
/*	GNOMEUIINFO_MENU_SETTINGS_TREE( settings_menu ), */
  	GNOMEUIINFO_MENU_HELP_TREE( help_menu ),
	GNOMEUIINFO_END
};

static void buildAndShow_winMain( void )
{
	GtkWidget *hbox;

	/* Set up the main window
	*/
	winMain = gnome_app_new( PACKAGE, PACKAGE VERSION );
	gtk_window_set_title( GTK_WINDOW(winMain), PACKAGE );
	gtk_container_border_width( GTK_CONTAINER(winMain), 0 );
	gtk_signal_connect(
		GTK_OBJECT(winMain), "delete_event",
		GTK_SIGNAL_FUNC(cbExit), NULL );
	gtk_signal_connect(
		GTK_OBJECT(winMain), "configure_event",
		GTK_SIGNAL_FUNC(cbConfigure), NULL );

	/* Statusbar */
	appBar = gnome_appbar_new( FALSE, TRUE, GNOME_PREFERENCES_NEVER );
	gnome_app_set_statusbar( GNOME_APP(winMain), GTK_WIDGET(appBar) );

	/* Menubar */
	gnome_app_create_menus( GNOME_APP(winMain), menu_bar );
	gnome_app_install_menu_hints( GNOME_APP(winMain), menu_bar );

	/* TODO: Toolbar? */
	
	/* Build the window's contents.
	*/
	hbox = gtk_hbox_new( FALSE, 0 );
	gtk_box_set_spacing( GTK_BOX(hbox), 1 );
	gtk_container_border_width( GTK_CONTAINER(hbox), 0 );
	gtk_widget_show( hbox );

	scrollBar = gtk_vscrollbar_new( ZVT_TERM(zvtTerm)->adjustment );

	gtk_box_pack_start( GTK_BOX(hbox), scrollBar, FALSE, TRUE, 0 );
	gtk_box_pack_start( GTK_BOX(hbox), zvtTerm, TRUE, TRUE, 0);
	gtk_signal_connect(
		GTK_OBJECT(zvtTerm), "event",
		GTK_SIGNAL_FUNC( cbStripCtrlC ), NULL );
	gtk_signal_connect_after(
		GTK_OBJECT(zvtTerm), "realize",
		GTK_SIGNAL_FUNC( applyUIPreferences ), NULL );
	gtk_widget_show( zvtTerm );

	/* Show the window and apply the preferences to it.
	*/
	gnome_app_set_contents( GNOME_APP(winMain), hbox );
	gtk_widget_show( winMain );

}

/*****************************************************************************/

/* Reflect changes in the calculator's internal state on the user interface.
*/
static void handleStateChange( void )
{
	HELPTOPIC help;
	
	rCalc_config_Lock();	/* CONFIGURATION LOCKED >>> */

	/* Update the angle units in the preferences box.
	*/
	if( cfgCalc->angleMode_mod )
	{
		if( winPrefs )
		{
			gtk_option_menu_set_history(
				GTK_OPTION_MENU(optAngles),
				cfgCalc->angleMode );
		}
		cfgCalc->angleMode_mod = FALSE;
	}

	/* Read off the help topic
	*/
	help = cfgCalc->helpTopic;
	cfgCalc->helpTopic = helpNoHelp;

	rCalc_config_Unlock();	/* <<< CONFIGURATION LOCKED */

	/* Display the help page if requested...
	*/
	if( help != helpNoHelp ) displayHelp( help );
}


/* Generic signal handler.
*/
static RETSIGTYPE signalHandler( int sig )
{
	Debug( "%s %d", __FUNCTION__, sig );
	switch( sig )
	{
	case SIGINT:
		gtk_main_quit();
		break;

	case SIGUSR2:
		gtk_main_quit();
		break;

	case SIGUSR1:
		handleStateChange();
		break;
		
	default:
		Debug( "unhandled signal." );
	}
}

/*****************************************************************************/

int main(int argc,char *argv[])
{
	/* Wake up NLS
	*/
	bindtextdomain( PACKAGE, GNOMELOCALEDIR );
	textdomain( PACKAGE );

	/* Wake up GNOME
	*/
	gnome_init( PACKAGE, VERSION, argc, argv );

	/* Load the configuration (or set to default if not present)
	*/
	rCalc_config_init();
	rCalc_config_load();
	
	/* Create the terminal and fork:
	*/
	zvtTerm = zvt_term_new_with_size( 50, 15 );
	if(( pidChild = zvt_term_forkpty( ZVT_TERM(zvtTerm), FALSE ) )==-1 )
	{
		perror( "fork" );
		exit(EXIT_FAILURE);
	}
	if( pidChild )
	{
		/* Run the GNOME user interface.
		*/
		buildAndShow_winMain();
		signal( SIGINT,  signalHandler );
		signal( SIGUSR1, signalHandler );
		signal( SIGUSR2, signalHandler );
		gtk_main();

		/* Kill the console (child) process
		** if it isn't already dead.
		*/
		signal( SIGUSR2, SIG_IGN );
		kill( pidChild, SIGTERM );
		waitpid( pidChild, NULL, 0 );
	}
	else
	{
		/* Run the calculator (within the zvt).
		*/
		rCalc_console_Startup();
		rCalc_console_ProcessUserInput();
		rCalc_console_Shutdown();

		/* Signal the parent process that we are done,
		** and then sleep forever. The parent will kill
		** us when ready.
		*/
		pidParent = getppid();
		kill( pidParent, SIGUSR2 );
		while(1) sleep(10);
	}

	/* Save the configuration.
	*/
	rCalc_config_save();
	rCalc_config_free();

	exit(EXIT_SUCCESS);
}

/*** end of main.c ***********************************************************/









