#
# debianbts.py - Routines to deal with the debbugs web pages
#
#   Written by Chris Lawrence <lawrencc@debian.org>
#   (C) 1999-2002 Chris Lawrence
#
# This program is freely distributable per the following license:
#
##  Permission to use, copy, modify, and distribute this software and its
##  documentation for any purpose and without fee is hereby granted,
##  provided that the above copyright notice appears in all copies and that
##  both that copyright notice and this permission notice appear in
##  supporting documentation.
##
##  I DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL
##  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL I
##  BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY
##  DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
##  WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
##  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
##  SOFTWARE.
#
# Version ##VERSION##; see changelog for revision history

import sgmllib, urllib, string, glob, os, re, reportbug, rfc822, time

class Error(Exception):
    pass

# Severity levels
SEVERITIES = {
    'critical' : """makes unrelated software on the system (or the
    whole system) break, or causes serious data loss, or introduces a
    security hole on systems where you install the package.""",
    'grave' : """makes the package in question unusable or mostly so,
    or causes data loss, or introduces a security hole allowing access 
   to the accounts of users who use the package.""",
    'serious' : """is a severe violation of Debian policy (that is,
    the problem is a violation of a 'must' or 'required' directive);
    may or may not affect the usability of the package.""",
    'important' : """a bug which has a major effect on the usability
    of a package, without rendering it completely unusable to
    everyone.""",
    'normal' : """a bug that does not undermine the usability of the
    whole package; for example, a problem with a particular option or
    menu item.""",
    'minor' : """things like spelling mistakes and other minor
    cosmetic errors that do not affect the core functionality of the
    package.""",
    'wishlist' : "suggestions and requests for new features.",
    }

# justifications for critical bugs
JUSTIFICATIONS = {
    'critical' : (
    ('breaks unrelated software', """breaks unrelated software on the system
    (packages that have a dependency relationship are not unrelated)"""),
    ('breaks the whole system', """renders the entire system unusable (e.g.,
    unbootable, unable to reach a multiuser runlevel, etc.)"""),
    ('causes serious data loss', """causes loss of important, irreplaceable
    data"""),
    ('root security hole', """introduces a security hole allowing access to
    root (or another privileged system account), or data normally
    accessible only by such accounts"""),
    ('unknown', """not sure, or none of the above"""),
    ),
    'grave' : (
    ('renders package unusable', """renders the package unusable, or mostly
    so, on all or nearly all possible systems on which it could be installed
    (i.e., not a hardware-specific bug); or renders package uninstallable
    or unremovable without special effort"""),
    ('causes non-serious data loss', """causes the loss of data on the system
    that is unimportant, or restorable without resorting to backup media"""),
    ('user security hole', """introduces a security hole allowing access to
    user accounts or data not normally accessible"""),
    ('unknown', """not sure, or none of the above"""),
    )
    }


# Ordering for justifications
JUSTORDER = {
    'critical' :  ['breaks unrelated software',
                   'breaks the whole system',
                   'causes serious data loss',
                   'root security hole',
                   'unknown'],
    'grave' : ['renders package unusable',
               'causes non-serious data loss',
               'user security hole',
               'unknown']
    }

SEVERITIES_gnats = {
    'critical' : """The product, component or concept is completely
    non- operational or some essential functionality is missing.  No
    workaround is known.""",
    'serious' : """The product, component or concept is not working
    properly or significant functionality is missing.  Problems that
    would otherwise be considered `critical' are rated `serious' when
    a workaround is known.""",
    'non-critical' : """The product, component or concept is working
    in general, but lacks features, has irritating behavior, does
    something wrong, or doesn't match its documentation.""",
    }

# Rank order of severities, for sorting
SEVLIST = ['critical', 'grave', 'serious', 'important', 'normal',
           'non-critical', 'minor', 'wishlist', 'fixed']

def convert_severity(severity, type='debbugs'):
    "Convert severity names if needed."
    if type == 'debbugs':
        return {'non-critical' : 'normal'}.get(severity, severity)
    elif type == 'gnats':
        return {'grave' : 'critical',
                'important' : 'serious',
                'normal' : 'non-critical',
                'minor' : 'non-critical',
                'wishlist' : 'non-critical'}.get(severity, severity)
    else:
        return severity

# These packages are virtual in Debian; we don't look them up...
debother = {
    'base' : 'General bugs in the base system',
# Actually a real package, but most people don't have boot-floppies installed for good reason
    'boot-floppies' : 'Bugs in the installation subsystem',
    'bugs.debian.org' : 'The bug tracking system, @bugs.debian.org',
    'cdimage.debian.org' : 'CD Image issues',
# dpkg-iwj -- The dpkg branch maintained by Ian Jackson 
    'ftp.debian.org' : 'Problems with the FTP site',
    'general' : 'General problems (e.g., that many manpages are mode 755)',
    'kernel' : 'Problems with the Linux kernel, or the kernel shipped with Debian',
    'listarchives' :  'Problems with the WWW mailing list archives',
    'lists.debian.org' : 'The mailing lists, debian-*@lists.debian.org.',
    'nonus.debian.org' : 'Problems with the non-US FTP site',
    'press' : 'Press release issues',
    'project' : 'Problems related to Project administration',
    'qa.debian.org' : 'Problems related to the quality assurance group',
#slink-cd -- Slink CD 
#spam -- Spam (reassign spam to here so we can complain about it)
    'security.debian.org' : 'Problems with the security updates server',
    'wnpp' : 'Work-Needing and Prospective Packages list',
    'www.debian.org' : 'Problems with the WWW site'
    }

progenyother = {
    'debian-general' : 'Any non-package-specific bug',
    }

def handle_wnpp(package, bts, ui):
    desc = body = ''
    headers = []
    
    tag = ui.menu('What sort of request is this?', {
        'O' :
        "The package has been `Orphaned'. It needs a new maintainer as soon as possible.",
        'RFA' :
        "This is a `Request for Adoption'. Due to lack of time, resources, interest or something similar, the current maintainer is asking for someone else to maintain this package. He/she will maintain it in the meantime, but perhaps not in the best possible way. In short: the package needs a new maintainer.",
        'ITP' :
        "This is an `Intent To Package'. Please submit a package description along with copyright and URL in such a report.",
        'RFP' :
        "This is a `Request For Package'. You have found an interesting piece of software and would like someone else to maintain it for Debian. Please submit a package description along with copyright and URL in such a report.",
        }, 'Choose the request type: ')
    if not tag: return

    if tag in ('RFP', 'ITP'):
        prompt = 'Please enter the proposed package name: '
    else:
        prompt = 'Please enter the name of the package: '
    package = ui.get_string(prompt)
    if not package: return

    ui.ewrite('Checking status database...\n')
    info = reportbug.get_package_status(package)
    available = info[1]

    severity = 'normal'
    if tag in ('ITP', 'RFP'):
        if available:
            cont = ui.select_options(
                'A package by this name already exists; continue?',
                'yN', {'y': 'Ignore this problem and continue.',
                       'n': 'Exit without filing a report.' })
            if cont == 'n':
                return
            
        severity = 'wishlist'

        desc = ui.get_string(
            'Please briefly describe this package; this should be an '
            'appropriate short description for the eventual package: ')
        if not desc:
            return

        if tag == 'ITP':
            headers.append('X-Debbugs-CC: debian-devel@lists.debian.org')
            ui.ewrite('Your report will be carbon-copied to debian-devel, '
                      'per Debian policy.\n')

        body = """* Package name    : %s
  Version         : x.y.z
  Upstream Author : Name <somebody@some.org>
* URL             : http://www.some.org/
* License         : (GPL, LGPL, BSD, MIT/X, etc.)
  Description     : %s
""" % (package, desc)
    elif tag in ('O', 'RFA'):
        severity = 'normal'
        if not available:
            info = reportbug.get_source_package(package)
            if info:
                info = reportbug.get_package_status(info[0][0])

        if not info:
            cont = ui.select_options(
                "This package doesn't appear to exist; continue?",
                'yN', {'y': 'Ignore this problem and continue.',
                       'n': 'Exit without filing a report.' })
            if cont == 'n':
                return
            desc = fulldesc = ''
        else:
            desc = info[10] or ''
            package = info[11] or package
            fulldesc = info[12]

        if tag == 'O' and info and info[9] in \
               ('required', 'important', 'standard'):
            severity = 'important'

        if fulldesc:
            orphstr = 'intend to orphan'
            if tag == 'RFA':
                orphstr = 'request an adopter for'
            body = ('I %s the %s package.\n'
                    'The package description is:\n') % (orphstr, package)
            body = body + fulldesc
        
    if desc:
        subject = '%s: %s -- %s' % (tag, package, desc)
    else:
        subject = '%s: %s' % (tag, package)

    return (subject, severity, headers, body)

# Supported servers
# Theoretically support for GNATS and Jitterbug could be added here.

SYSTEMS = { 'debian' :
            { 'name' : 'Debian', 'email': '%s@bugs.debian.org',
              'btsroot' : 'http://www.debian.org/Bugs/',
              'query-dpkg' : 1, 'type' : 'debbugs',
              'otherpkgs' : debother,
              'specials' : { 'wnpp': handle_wnpp },
              'ldap' : ('bugs.debian.org', 35567,
                        'ou=Bugs,o=Debian Project,c=US'),
              'cgiroot' : 'http://bugs.debian.org/cgi-bin/',
              'mirrors' : {} },
##            'tdyc' :
##            { 'name' : 'TDYC [Debian KDE]',
##              'email': '%s@bugs.tdyc.com',
##              'btsroot' : 'http://bugs.tdyc.com/',
##              'otherpkgs' : {}, 'type' : 'debbugs',
##              'namefmt' : '%s-debian', 'query-dpkg' : 1, 'cgiroot' : None,
##              'mirrors' : {} },
            'kde' :
            { 'name' : 'KDE Project', 'email': '%s@bugs.kde.org',
              'btsroot': 'http://bugs.kde.org/', 'type' : 'debbugs',
              'query-dpkg' : 1, 'otherpkgs' : {}, 'cgiroot' : None,
              'mirrors' : {} },
            'mandrake' :
            { 'name' : 'Linux-Mandrake', 'email': '%s@bugs.linux-mandrake.com',
              'btsroot': None,
              'type' : 'debbugs', 'query-dpkg' : 0, 'otherpkgs' : {},
              'cgiroot' : None, 'mirrors' : {} },
            'gnome' :
            { 'name' : 'GNOME Project', 'email': '%s@bugs.gnome.org',
              'type' : 'mailto', 'mirrors' : {}, 'cgiroot' : None,
              'query-dpkg' : 0, 'otherpkgs' : {} },
            'ximian' :
            { 'name' : 'Ximian', 'email': '%s@bugs.ximian.com',
              'type' : 'mailto', 'mirrors' : {}, 'cgiroot' : None,
              'query-dpkg' : 1, 'otherpkgs' : {} },
            'progeny' :
            { 'name' : 'Progeny', 'email' : 'bugs@progeny.com',
              'type' : 'gnats', 'mirrors' : {}, 'cgiroot' : None,
              'query-dpkg' : 1, 'otherpkgs' : progenyother },
            'guug' :
            { 'name' : 'GUUG (German Unix User Group)',
              'email' : '%s@bugs.guug.de',
              'type' : 'debbugs', 'mirrors' : {}, 'cgiroot' : None,
              'query-dpkg' : 0, 'otherpkgs' : {} },
            }

SYSTEMS['helixcode'] = SYSTEMS['ximian']

CLASSES = {
    'sw-bug' : """The problem is a bug in the software or code.  For
    example, a crash would be a sw-bug.""",
    'doc-bug' : """The problem is in the documentation.  For example,
    an error in a man page would be a doc-bug.""",
    'change-request' : """You are requesting a new feature or a change
    in the behavior of software, or are making a suggestion.  For
    example, if you wanted reportbug to be able to get your local
    weather forecast, as well as report bugs, that would be a
    change-request.""",
    }

CLASSLIST = ['sw-bug', 'doc-bug', 'change-request']

TAGS = {
    'patch' : 'You are including a patch to fix this problem.',
    'security' : 'This problem raises a security issue.',
##    'potato' : 'This bug only applies to the potato release (Debian 2.2).',
##    'woody' : 'This bug only applies to the woody release (Debian 3.0).',
##    'sid' : 'This bug only applies to the unstable branch of Debian.',
    'done' : 'No more tags.',
    }

EXTRA_TAGS = ['potato', 'woody', 'sid']

TAGLIST = ['patch', 'security', 'done']

def parse_bts_url(url):
    bits = string.split(url, ':', 1)
    if len(bits) != 2: return None

    type, loc = bits
    if loc[0:2] == '//': loc = loc[2:]
    return type, loc

# Dynamically add any additional systems found
for file in glob.glob('/etc/dpkg/origins/*'):
    try:
        fp = open(file)
        system = os.path.basename(file)
        SYSTEMS[system] = SYSTEMS.get(system, { 'otherpkgs' : {},
                                                'query-dpkg' : 1,
                                                'mirrors' : {},
                                                'cgiroot' : None } )
        for line in fp.readlines():
            try:
                (header, content) = string.split(line, ': ')
                header = string.lower(header)
                content = string.strip(content)
                if header == 'vendor':
                    SYSTEMS[system]['name'] = content
                elif header == 'bugs':
                    (type, root) = parse_bts_url(content)
                    SYSTEMS[system]['type'] = type
                    if type == 'debbugs':
                        SYSTEMS[system]['btsroot'] = 'http://'+root+'/'
                        SYSTEMS[system]['email'] = '%s@'+root
                    elif type == 'mailto':
                        SYSTEMS[system]['btsroot'] = None
                        SYSTEMS[system]['email'] = root
                    else:
                        # We don't know what to do...
                        pass
            except ValueError:
                pass
    except IOError:
        pass

ldap = None
try:
    import ldap

    if not ldap.has_key(LDAPError):
        ldap = None
except:
    pass

# URL opener that raises an exception for unfound pages and
# 401 Authorization Required
class OurURLopener(urllib.FancyURLopener):
    def __init__(self, *args):
        apply(urllib.FancyURLopener.__init__, (self,) + args)

    # Default http error handler: close the connection and raises IOError
    def http_error_default(self, url, fp, errcode, errmsg, headers):
        void = fp.read()
        fp.close()
        raise IOError, ('http error', errcode, errmsg, headers)

    def http_error_407(self, url, fp, errcode, errmsg, headers):
        return self.http_error_401(url, fp, errcode, errmsg, headers)

# Shortcut for basic usage
_urlopener = None
def urlopen(url, data=None, proxies=None):
    global _urlopener
    if not _urlopener:
        _urlopener = OurURLopener(proxies)
    try:
        if data is None:
            return _urlopener.open(url)
        else:
            return _urlopener.open(url, data)
    except TypeError:
        raise Error, "check $http_proxy for proper syntax (see man page)"

# For summary pages, we want to keep:
#
# - Contents of <title>...</title>
# - Contents of <h2>...</h2>
# - Contents of each <li>
#
# For individual bugs, we want to keep:
# - Contents of <title>...</title>
# - Contents of final <pre>...</pre>

class BTSParser(sgmllib.SGMLParser):
    def __init__(self, mode='summary', cgi=0):
        sgmllib.SGMLParser.__init__(self)
        self.hierarchy = []
        self.lidata = None
        self.lidatalist = None
        self.savedata = None
        self.title = None
        self.bugcount = 0
        self.mode = mode
        self.cgi = cgi
        self.preblock = ''

    # --- Formatter interface, taking care of 'savedata' mode;
    # shouldn't need to be overridden

    def handle_data(self, data):
        if self.savedata is not None:
            self.savedata = self.savedata + data

    # --- Hooks to save data; shouldn't need to be overridden

    def save_bgn(self):
        self.savedata = ''

    def save_end(self, mode=0):
        data = self.savedata
        self.savedata = None
        if not mode: data = string.join(string.split(data))
        return data

    def check_li(self):
        if self.mode == 'summary':
            data = self.save_end()
            if data:
                self.lidatalist.append(data)
                self.bugcount = self.bugcount + 1

            self.lidata = 0

    def start_h1(self, attrs):
        self.save_bgn()
        self.oldmode = self.mode
        self.mode = 'title'

    def end_h1(self):
        self.title = self.save_end()
        self.mode = self.oldmode

    def start_h2(self, attrs):
        if self.lidata: self.check_li()

        self.save_bgn()

    def end_h2(self):
        if self.mode == 'summary':
            self.hierarchy.append( (self.save_end(), []) )

    def do_br(self, attrs):
        if self.lidata and self.mode == 'summary': self.check_li()

        if self.mode == 'title':
            self.savedata = ""
        
    def do_li(self, attrs):
        if self.mode == 'summary':
            if self.lidata: self.check_li()

            self.lidata = 1
            self.lidatalist = self.hierarchy[-1][1]
            self.save_bgn()

    def start_pre(self, attrs):
        if self.cgi and self.preblock: return
        
        self.save_bgn()

    def end_pre(self):
        if self.cgi and self.preblock: return

        self.preblock = self.save_end(1)

def cgi_report_url(system, number, archived='no'):
    return SYSTEMS[system]['cgiroot']+'bugreport.cgi?bug='+str(number)+'&archive='+archived

def cgi_package_url(system, package, archived='no', source=0):
    package = urllib.quote_plus(string.lower(package))
    qtype = "pkg"
    if source:
        qtype = "src"

    return SYSTEMS[system]['cgiroot'] + 'pkgreport.cgi?%s=%s&archive=%s' % (
        qtype, package, archived)

def get_cgi_reports(package, system='debian', http_proxy='', archived='no',
                    source=0):
    proxies = urllib.getproxies()
    if http_proxy:
        proxies['http'] = http_proxy
        
    url = cgi_package_url(system, package, archived, source)
    try:
        page = urlopen(url, proxies=proxies)
    except IOError, data:
        if data and data[0] == 'http error' and data[1] == 404:
            return (0, None, None)
        else:
            raise

    content = page.read()
    if string.find(content, 'Maintainer') == -1:
        return (0, None, None)
    
    parser = BTSParser(cgi=1)
    parser.feed(content)
    parser.close()

    return parser.bugcount, parser.title, parser.hierarchy

def get_cgi_report(number, system='debian', http_proxy='', archived='no'):
    number = int(number)
    proxies = urllib.getproxies()
    if http_proxy:
        proxies['http'] = http_proxy
        
    url = cgi_report_url(system, number, archived='no')
    try:
        page = urlopen(url, proxies=proxies)
    except IOError, data:
        if data and data[0] == 'http error' and data[1] == 404:
            return None
        else:
            raise Error, data

    content = page.read()
    parser = BTSParser(cgi=1)
    parser.feed(content)
    parser.close()

    try:
        stuff = parser.preblock
    except AttributeError:
        return None

    parts = string.split(stuff, '\n\n')
    match = re.search('^Date: (.*)$', parts[0], re.M | re.I)
    date_submitted = ''
    if match:
        date_submitted = 'Date: '+match.group(1)+'\n'
        
    stuff = string.join(parts[1:], '\n\n')
    if not stuff:
        return None

    while stuff[-1]=='\n': stuff = stuff[:-1]

    return ("#%d: %s" % (number, parser.title)), date_submitted+stuff+'\n'

def get_btsroot(system, mirrors=None):
    if mirrors:
        alternates = SYSTEMS[system]['mirrors']
        for mirror in mirrors:
            if alternates.has_key(mirror):
                return alternates[mirror]
    return SYSTEMS[system]['btsroot']

def package_url(system, package, mirrors=None, source=0):
    btsroot=get_btsroot(system, mirrors)
    package = urllib.quote_plus(string.lower(package))
    return btsroot+('db/pa/l%s.html' % package) 

def report_url(system, number, mirrors=None):
    number = str(number)
    if len(number) < 2: return None
    btsroot=get_btsroot(system, mirrors)
    return btsroot+('db/%s/%s.html' % (number[:2], number))

def get_reports(package, system='debian', ldap_ok=1, mirrors=None,
                http_proxy='', archived='no', source=0):
    if SYSTEMS[system].has_key('ldap') and ldap and ldap_ok:
        return get_ldap_reports(package, system, source) or \
               get_reports(package, system, 0, mirrors, http_proxy, source)
    elif SYSTEMS[system]['cgiroot']:
        result = get_cgi_reports(package, system, http_proxy, archived, source)
        if result: return result

    proxies = urllib.getproxies()
    if http_proxy:
        proxies['http'] = http_proxy
        
    url = package_url(system, package, mirrors, source)
    try:
        page = urlopen(url, proxies=proxies)
    except IOError, data:
        if data and data[0] == 'http error' and data[1] == 404:
            return (0, None, None)
        else:
            raise

    content = page.read()
    if string.find(content, 'Maintainer') == -1:
        return (0, None, None)
    
    parser = BTSParser()
    parser.feed(content)
    parser.close()

    return parser.bugcount, parser.title, parser.hierarchy

def get_report(number, system='debian', ldap_ok=1, mirrors=None,
               http_proxy='', archived='no'):
    number = int(number)
    if SYSTEMS[system].has_key('ldap') and ldap and ldap_ok:
        return get_ldap_report(number, system) or \
               get_report(number, system, 0, mirrors, http_proxy)
    elif SYSTEMS[system]['cgiroot']:
        result = get_cgi_report(number, system, http_proxy, archived)
        if result: return result
        
    proxies = urllib.getproxies()
    if http_proxy:
        proxies['http'] = http_proxy
        
    url = report_url(system, number, mirrors)
    if not url: return None
    try:
        page = urlopen(url, proxies=proxies)
    except IOError, data:
        if data and data[0] == 'http error' and data[1] == 404:
            return None
        else:
            raise

    content = page.read()

    parser = BTSParser()
    parser.feed(content)
    parser.close()

    try:
        stuff = parser.preblock
    except AttributeError:
        return None
    
    parts = string.split(stuff, '\n\n')
    match = re.search('^Date: (.*)$', parts[0], re.M | re.I)
    date_submitted = ''
    if match:
        date_submitted = 'Date: '+match.group(1)+'\n'
        
    stuff = string.join(parts[1:], '\n\n')

    while stuff[-1]=='\n': stuff = stuff[:-1]
    
    return '#%d: %s' % (number, parser.title), date_submitted+stuff+'\n'

ldap_conn = None

def category_name(report):
    if report['severity'][0] == 'wishlist':
        str = 'Wishlist items - '
    else:
        str = string.capitalize(report['severity'][0]) + ' bugs - '

    if report.has_key('forwarded'):
        return str + 'forwarded to upstream software authors:'
    elif report.has_key('done'):
        return str + 'resolved:'
    else:
        return str + 'outstanding:'

def comparebugs(bug1, bug2):
    try:
        pos1, pos2 = SEVLIST.index(bug1['severity'][0]),\
                     SEVLIST.index(bug2['severity'][0])
    except ValueError:
        pos1, pos2 = 0, 0
    
    return (cmp(bug1.has_key('done'), bug2.has_key('done')) or
            cmp(bug1.has_key('forwarded'), bug2.has_key('forwarded')) or
            cmp(pos1, pos2) or
            cmp(int(bug1['bugid'][0]), int(bug2['bugid'][0])))

def get_ldap_reports(package, system='debian', source=0):
    global ldap_conn
    try:
        (server, port, qinfo) = SYSTEMS[system]['ldap']

        if not ldap_conn:
            ldap_conn = q = ldap.open(server, port)
        else:
            q = ldap_conn

        qtype = "package"
        # XXX Fixme
        if source:
            qtype = "src"

        reports = q.search_s(qinfo, ldap.SCOPE_SUBTREE,
                             ('(%s=%s)' % (qtype, package)))
        if not reports: return 0, None, None

        # strip entry info
        reports = map(lambda x: x[1], reports)
        reports.sort(comparebugs)
        hierarchy = []
        curcat = ''
        for report in reports:
            cat = category_name(report)
            if cat != curcat:
                hierarchy.append([cat, []])
                curcat = cat

            if report.has_key('subject'):
                info = '#%s: %s' % (report['bugid'][0], report['subject'][0])
            else:
                info = '#%s: no subject' % (report['bugid'][0])
            hierarchy[-1][1].append(info)

        return len(reports), 'Reports on '+package, hierarchy
    except (ldap.LDAPError, SystemError):
        return None

def get_ldap_report(number, system='debian'):
    global ldap_conn
    try:
        (server, port, qinfo) = SYSTEMS[system]['ldap']

        if not ldap_conn:
            ldap_conn = q = ldap.open(server, port)
        else:
            q = ldap_conn

        number = int(number)
        info = q.search_s(qinfo, ldap.SCOPE_SUBTREE, ('(bugid=%d)' % number),
                          ['report', 'subject', 'date', 'originater'])
        info = info[0]
        if not info[1]:
            return None

        if info[1].has_key('subject'):
            subject = '#%d: %s' % (number, info[1]['subject'][0])
        else:
            subject = '#%d: no subject' % (number)

        if info[1].has_key('report'):
            report = info[1]['report'][0]
            stuff = string.join(string.split(report, '\n\n')[1:], '\n\n')
        else:
            stuff = ''

        while len(stuff) and stuff[-1]=='\n': stuff = stuff[:-1]

        if info[1].has_key('date'):
            stuff = 'Date: '+info[1]['date'][0]+'\n'+stuff
        if info[1].has_key('originater'):
            stuff = 'From: '+info[1]['originater'][0]+'\n'+stuff
        return subject, stuff+'\n'
    except (ldap.LDAPError, SystemError):
        return None

def close_ldap_conn():
    global ldap_conn
    if ldap_conn:
        ldap_conn.unbind()
        ldap_conn = None

try:
    import webbrowser
except:
    webbrowser = None

def launch_browser(info, system, mirrors=None, archived='no', source=0):
    if re.match('^#?\d+$', str(info)):
        url = cgi_report_url(system, info, archived=archived) or \
              report_url(system, info, mirrors)
    else:
        url = cgi_package_url(system, info, archived=archived,
                              source=source) or \
              package_url(system, info, mirrors, source)

    if webbrowser:
        webbrowser.open(url)
        return

    X11BROWSER = os.environ.get('X11BROWSER', 'netscape')
    CONSOLEBROWSER = os.environ.get('CONSOLEBROWSER', 'lynx')

    if (os.environ.has_key('DISPLAY') and
        not os.system('command -v '+X11BROWSER+' &> /dev/null')):
        cmd = "%s '%s' &" % (X11BROWSER, url)
    else:
        cmd = "%s '%s'" % (CONSOLEBROWSER, url)

    os.system(cmd)
