getRawDefs <-
  #
  # Use the S-extension of the python defs parser to read an input file
  # and the copy the contents back to an S list.
  #
  # The result is a simple collection of strings organized by category.
  # This is the converted into a form more suitable for use in generating
  # code via getDefs().
  
function(fileNames = "gtk.defs")
{
 library(RSPython)
 
 x <- try(importPythonModule("Sgenerate"))

 if(inherits(x, "try-error")) {
  stop("Cannot find the Python module Sgenerate or potentially one of the modules on which it depends.\n Set the PYTHONPATH environment variable appropriately and try again.")
 }

 fileNames <- path.expand(fileNames)

 defs <- NULL
 for(i in 1:length(fileNames)) {
    p <- .PythonNew("SDefsParser", fileNames[i], .module="Sgenerate")
    p$startParsing()
    vals <- list(
                  functions = p[["functions"]],
                  enums     = p[["enums"]],
                  flags     = p[["flags"]],
                  classes   = p[["classes"]],
                  boxes     = p[["boxes"]]
                )
    if(length(fileNames) == 1)
      return(vals)
    
    defs <- mergeDefs(defs, vals)
  }

 invisible(defs)
}


getDefs <-
function(fileNames = c("gtk.defs"))
{
  defs <- getRawDefs(fileNames)

  tmp <- names(defs$classes)  
  defs$classes <- lapply(names(defs$classes),
                                       function(name) {
                                          x <- defs$classes[[name]]
                                          names(x) <- c("parent", "fields")
                                          x[["name"]] <- name

                                          if(length(x[["fields"]]) > 0) {
                                            types <- sapply(x[["fields"]][-1], function(x) x[1])
                                            names(types) <- sapply(x[["fields"]][-1], function(x) x[2])

                                            x[["fields"]] <- types
                                          }
 
                                          class(x) <- "ClassDef"                                          
                                          x
                                       })
  names(defs$classes) <- tmp

  tmp <- names(defs$functions)
  defs$functions <- lapply(names(defs$functions),
                             function(name) {
                                   x <- defs$functions[[name]]
                                   names(x) <- c("return", "parameters")
                                   x[["name"]] <- name

                                   if(length(x$parameters)) {
                                     paramNames <- sapply(x$parameters, function(z) {z[2]})
                                     params <- lapply(x$parameters, function(z) z[-2])
                                     names(params) <- paramNames
                                     x$parameters <- params
                                   }
                                   class(x) <- "FunctionDef"                                   
                                   x
                             })
  names(defs$functions) <- tmp

  enumCvt <- function(name, which) {
                          k <- as.character(sapply(defs[[which]][[name]],
                                                    function(x) { x[[2]]}))
                          local <- as.character(sapply(defs[[which]][[name]],
                                                    function(x) { x[[1]]}))                          
                          vals <- as.integer(sapply(defs[[which]][[name]],
                                                    function(x) { ifelse(length(x) > 2, x[[3]][2], NA)}))


                          if(is.na(vals[1]) && !is.na(vals[-1]))
                            vals[1] <- as.integer(0)
                          else if(is.na(vals[-1]) && !is.na(vals[1]))
                            vals <- seq(vals[1], length = length(vals))
                          else if(all(is.na(vals))) {
                            if(which == "enums")
                              vals <- 0:(length(vals)-1)
                            else
                              vals <- 2^(0:(length(vals)-1))
                          }

                          if(length(vals) != length(local)) {
                            print(local)
                            print(vals)
                          }
                          names(vals) <- local
                          names(k) <- local
                          
                          list(names=k, values = vals)
                         }
  
  tmp <- names(defs$enums)
  defs$enums <- lapply(names(defs$enums), enumCvt, "enums")
  names(defs$enums) <- tmp

  tmp <- names(defs$flags)
  defs$flags <- lapply(names(defs$flags), enumCvt, "flags")
  names(defs$flags) <- tmp

 
  defs
}


mergeDefs <-
function(from,  into)  
{
    # local function to modify the value of into via lexical scoping.
  mergeEl <- function(category) {
    which <- match(names(from[[category]]), names(into[[category]]))
    into[[category]][names(from[[category]])[is.na(which)]] <<- from[[category]][is.na(which)]

    if(any(!is.na(which)))  {
      warning("Discarding ", category,  " definitions for ",   paste(names(from[[category]][!is.na(which)]), sep=", "))
    }
  }

  for(i in names(from))
    mergeEl(i)

  into        
}  

##################################

getParameterTypes <-
function(x)
{
  as.character(c(x$"return",
                 as.character(sapply(x$parameters, function(z) { z[1]}))))
}  

getAllTypes <-
function(defs)
{
  tmp <- names(defs$classes)
  params <- sapply(defs$functions, getParameterTypes)
  tmp <- c(tmp, as.character(unlist(params)))

  unique(as.character(tmp))
}

constructorFilter <-
function(obj)
{
  className <- gsub("_new.*", "", obj$name)
  
}


getConstructors <-
function(defs, filter = NULL)
{
  which <- names(defs$functions)[grep("_new", names(defs$functions))]

  classNames <- gsub("_new.*", "", which)
  classNames <- gsub("_","", classNames)

  classMatch <- match(classNames, tolower(names(defs$classes)))
  if(any(is.na(classMatch)))
    warning(paste("_new functions with no classes for", paste(which[is.na(classMatch)],collapse=", ")))
  classNames <- names(defs$classes)[classMatch]

  names(which) <- classNames

  if(!is.null(filter)) {
    ok <- sapply(defs$functions[match(which, names(defs$functions), 0)], filter)
    which <- which[ok]
  }

  which
}  


###################################

getClassFunctionName <-
  #
  # converts a string of the GtkButtonSetText to gtk_button_set_text
  # to give the Gtk C routine name from the S function name
  # (or actually capitalized version of it!)
  #
function(name)
{
 tolower(substring(gsub("([ABCDEFGKHIJKLMNOPQRSTUVWXYZ])","_\\1", name),2))
}  

  


getClassList <-
function(name, classes)
{
  if(is.na(match("GtkObject", names(classes)))) {
    data(GtkDefs)
    a <- is.na(match(names(classes), names(GtkDefs$classes)))
    if(any(a))
      GtkDefs$classes[names(classes)[a]] <- classes[a]
    classes <- GtkDefs$classes
  }
  
  ans <- name
  tmp <- name
  while(!is.null(p <- classes[[tmp]])) {
     tmp <- p$parent
     if(length(tmp) < 1 || tmp == "nil")
       break
     ans <- c(ans, tmp)
  }

  ans
}  


isWidgetClass <-
  #
  # Determine if the specified class extends the GtkWidget class.
  #
function(name, classes, gtkType = "GtkWidget")
{
  k <- getClassList(name, classes)
  if(length(k) == 0)
    return(FALSE)

  return(!is.na(match(gtkType, k)))
}  

###############################################################################

getSArgs <-
function(params)
{
 if(length(params)) {
     names(params)
#    sapply(params, function(x) {
#                    if(length(x) > 1)
#                       " = NULL"
#                    else
#                       character(0)
#                    })
 } else
    character(0)
}  



PrimitiveTypeCoercion <- c("string" = "as.character",
                           "static_string" = "as.character",
                           "int" = "as.integer",
                           "uint" = "as.integer",
                           "gint" = "as.integer",
                           "guint" = "as.numeric",
                           "guint32" = "as.numeric",                                                      
                           "float" = "as.numeric",
                           "bool" = "as.logical"
                           )

getPrimitiveTypeAs <-
function(x)
{
  if(length(x) > 1)
    x <- x[[1]]
  PrimitiveTypeCoercion[[x]]
}

isPrimitiveType <-
function(x)
{
  !is.na(match(x, names(PrimitiveTypeCoercion)))
}  

isEnum <-
function(type, enums)  
{
  !is.na(match(type, names(enums)))
}

isFlag  <-
function(type, flags)  
{
  !is.na(match(type, names(flags)))
}

coerceRValueCode <-
  #
  # This creates R code to coerce or check that an object is of the 
  # appropriate type.
  #
  # should we just use an as(name, type) here
  # and defer the computation.
  #
function(type, name, defs)
{
  # handle enums also
 name <- nameToS(name)
 
 if(isPrimitiveType(type)) {
   fn <- getPrimitiveTypeAs(type)
   coerce <- paste(name, " <- ", fn, "(", name, ")", sep="")
 } else if(isEnum(type, defs$enums) || isFlag(type, defs$flags)) {
   coerce <- paste(name, "  <- map", type[1], "(", name, ")", sep="")
 } else if(type == "string[]") {
   coerce <- paste(name, " <- as.character(", name, ")", sep="")
 } else {
   if(length(type) > 1 && type[2] == "null-ok")
     nullOk <- ", nullOk = TRUE"
   else
     nullOk <- character(0)
   
   coerce <- paste("gtkCheckInherits(", name, ", \"", type[1], "\"", nullOk, ")", sep="")
 }

 list(name = name, coerce = coerce)
}

nameToS <-
function(name)
{
  gsub("_", ".", name)  
}  

nameToC <-
function(name)
{
 paste("S", name, sep="_")
}


flagStringToValue <-
function(value)
{
  els <- strsplit(value, "\\\|")[[1]]
  paste("c(", paste("\"", els, "\"", sep="", collapse=", "),")", sep="")
}  

getDefaultSArgs <-
  #
  # figure out what the default arguments are for each argument 
  # of the S function being generated from the function with these
  # parameters.
  #
  # This has become very ugly. Fix this up.
  #
function(params, defs)
{
  if(length(params) == 0)
   return(character(0))
  
  x <- sapply(params,
               function(x) {
                 if(length(x) < 2) {
                      tmp <- x[[1]][1]
                   if(isEnum(tmp, defs$enums)) {
                     return(paste(" = ", tmp, "[1]", sep=""))
                   } else if(!is.na(match(tmp, names(defs$classes))))
                     return(" = NULL")
                   else
                     return("") 
                 }

                 if(length(x[[2]]) == 2) {
                     v <- x[[2]][2]
                 } else {
                   if(length(x) == 3)
                     v <- x[[3]][2]
                   else
                     if(length(x) == 2 && x[[2]][1] == "null-ok")
                     v <- "NULL"
                   else
                     return("") 
                 }

                type <- x[[1]][1]
                if(isEnum(type, defs$enums)) {
                   v <- paste("\"",v,"\"", sep="")
                } else if(isFlag(type, defs$flags)) {
                   v <- flagStringToValue(v)
                }
                 
                paste(" =", v)
  })

  names(x) <- nameToS(names(params))

  x
}  

genRCode <-
  #
  #  Generate S code for a function representing the function from the defs
  #  description `fun'.
  #  
function(fun, defs, name, sname, className = NULL, package = "RGtk")
{
  if(missing(name))
    name <- fun$name
  
  if(missing(sname)) {
     sname <- nameToS(name)
  }                 

  if(isConstructor(name)) {
    className <- mapToClassName(name, defs$classes, capitalize = FALSE)
    sname <- mapToClassName(name, defs$classes, capitalize = TRUE)
  }
  
  if(length(fun$parameters) > 0) {
    sargs <- getDefaultSArgs(fun$parameters, defs)
  } else 
     sargs <- character(0)

  
   # Add the show argument if this is a constructor
   # And we check this is a widget class.
  if(isConstructor(name) && isWidgetClass(mapToClassName(name, defs$classes, FALSE), defs$classes)) {
       sargs["show"] <- " = TRUE"
       showWidget <- "\t if(show)\n\t   gtkWidgetShow(w)"
  } else
       showWidget <- character(0)

  sargs[".flush"] <- " = TRUE"

#  if(length(sargs) > 0)
   sargs <- paste(names(sargs), sargs, sep="", collapse=", ")    
  
  txt <- paste(sname, " <-", "\n", "function(",  sargs, ")", sep="")
  txt <- c(txt, "{")
  if(length(fun$parameters)) {
    tmp <- lapply(names(fun$parameters), function(x) coerceRValueCode(fun$parameters[[x]], x, defs))

    args <- lapply(tmp, function(x) x[["name"]])
    args <- paste(args, collapse=", ")
    coerce <- sapply(tmp, function(x) x[["coerce"]])
  } else {
    coerce <- character(0)
    args <- character(0)
  }

  if(length(coerce)) {
    coerce <- paste("\t", coerce, collapse="\n")
    txt <- c(txt, coerce)
  }

  txt <- c(txt, "", paste("\t w <- .GtkCall(\"", nameToC(name), "\"",
                          ifelse(length(args) > 0, ", ", ""),
                          args,
                          ", PACKAGE='", package, "'", ", .flush = .flush)", sep=""))

  if(isConstructor(name)) {
    setClass <- setGtkClassCall(className, defs, variable = "w")
  } else {
    setClass <- character(0)
  }


  if(fun$"return" == "none")
    returnValue <- "invisible(return(w))"
  else
    returnValue <- "return(w)"
  
  txt <- c(txt, setClass, showWidget, paste("\t", returnValue, collapse=""), "\n}\n")
  paste(txt, collapse="\n")
}

setGtkClassCall <-
function(className, defs, variable)
{
#    if(missing(className)) {
#      className <- mapToClassName(name)
#    }

  setClass <- paste("\t class(", variable, ") <- c(",
                      paste("\"", getClassList(className, defs$classes), "\"", sep="", collapse=", "),
                      ")", sep="")

  setClass
}  

isConstructor <-
function(name)
{
  length(grep("_new", name)) > 0
}  

if(FALSE) {
 GtkPrimitiveTypes <- c("string" = "char *",
                        "uint" = "unsigned int"
                       )

mapToCType <-
function(type, defs)
{
 if(!is.na(match(type, names(GtkPrimitiveTypes)))) {
   GtkPrimitiveTypes[[type]]
 }
# else if(match(type, names(defs$enums))) {# }
 else {
   type
 }
}  
}


nameToSArg <-
function(name)
{
 paste("s", name, sep="_")
}  

convertToCType <-
function(name, type, defs)
{
 fullType <- type

 nullOk <- FALSE
 if(length(type) > 1 && type[[2]] == "null-ok")
   nullOk <- TRUE

 if(is.list(type))
   type <- type[[1]]
 
 type <- type[1]
 
 name <- paste("s", name, sep="_")
 if(type == "string") {
    paste("CHAR_DEREF( STRING_ELT(", name, ", 0))")
 } else if(type == "int" || type == "uint" || type == "gint") {
    paste("INTEGER_DATA(",name,")[0]")
 } else if(type == "bool") {
    paste("LOGICAL_DATA(",name,")[0]")
 }else if(type == "float" || type == "double" || type == "guint" || type == "guint32") {
    paste("NUMERIC_DATA(",name,")[0]")
 } else if(isEnum(type, defs$enums) || isFlag(type, defs$flags)) {
    paste( "(", type, ")", "INTEGER_DATA(", name, ")[0]")
 } else if(type == "GdkAtom") {
    paste( "( GdkAtom) NUMERIC_DATA(", name, ")[0]")
 } else if(type == "string[]") {
    paste( "asCStringArray(", name, ")")
 } else if(!is.na( k <- match(type, names(defs$classes))) && length(grep("^Gtk", names(defs$classes)[k]))) {
   if(is.list(type))
     type <- type[[1]]

   if(type == "GtkMenuPositionFunc") {
     cat("**** Matched", type, " to " , names(defs$classes)[k], "\n") 
   }

     # looks like I hard coded the mapping here to catch the HTML as a single word
     # I think I have a function that preserves this now.
    if(type == "GtkHTMLEmbedded") {
      macro <- "GTK_HTML_EMBEDDED"
    } else {
      macro <- toupper(gsub("([ABCDEFGHIJKLMNOPQRSTUVWXYZ]+)", "_\\1", type))
      macro <- gsub("_([ABCDEFGHIJKLMNOPQRSTUVWXYZ])_", "_\\1", macro) 
      macro <- gsub("^_", "", macro)
    }
   if(nullOk)
     val <- paste("Rf_length(", name, ") == 0 ? NULL : ", macro, "((GtkObject*) getPtrValue(", name, "))")
   else
     val <- paste(macro, "((GtkObject*) getPtrValue(", name, "))")

   val
 } else {
   # cat("**** Using default for ", type , "\n")
    paste("(", getCType(type, defs) , ") getPtrValue(", name, ")")
 }
}

convertToR <-
  #
  # generate the appropriate command that takes the result from the call 
  # to the real C routine and converts it to the appropriate R value.
  #
function(var, type, defs)
{
 if(type == "int" || type == "uint") {
   fn <- "asRInt"
 } else if(type == "bool") {
   fn <- "asRLogical"
 } else  if(type == "float" || type == "double" || type == "ulong" || type == "time_t") {
   fn <- "asRNumeric"
 } else if(type == "string" || type == "static_string") {
   fn <- "asRCharacter"
 } else if(isEnum(type, defs$enums) || isFlag(type, defs$flags)) {
    return(paste("   _result = ", getEnumCRoutineName(type[1]), "(asRInt(", var, "));", collapse=""))
 } else if(type == "GdkAtom") {
    fn <- "asRGdkAtom"
 } else if(!is.na(match(type, names(defs$classes))) && any(getClassList(type, defs$classes) == "GtkWidget")) {
   return(paste("   _result = R_gtkWidgetReference((GtkWidget*)", var, ", \"GtkWidget\");", collapse="",sep=""))
 } else {
   return(paste("   _result = toRPointer(", var, ", \"", type, "\"",  ");", collapse="",sep=""))
 }

 paste("  _result =", fn,"(", var, ");") 
}

getCType <-
  #
  # get the C type corresponding to the specified Defs type,
  # potentially looking in the enumerations and flags to match
  # the type.
  # By default, this returns a pointer to the input type!
  # One must handle all types for which this is not appropriate.
  #
  #  Convert to a table!
  #
function(type, defs)
{
 if(is.list(type))
   type = type[[1]]
 
 type <- type[1]
 
 if(type == "GtkWidget") 
   str <- paste(type, "*")
 else if(type == "bool")
   str <- "Rboolean"
 else if (type == "uint")
   str <- "unsigned int"
  else if (type == "static_string")
   str <- "string"
 else if (type == "string" || type == "int" || type == "double" || type == "float")
   str <- type
 else if(isEnum(type, defs$enums) || isFlag(type, defs$flags)) {
   str <- type
 } else if(type == "GdkAtom") {
   str  <- "GdkAtom"
 } else if(type == "ulong") {
   str <- "gulong"
 } else if(type == "time_t") {
   str <- type
 } else if(type == "string[]") {
   str <- "string *"
 } else if(type == "guint" || type == "gint" || type == "guint32") {
   str <- type
 } else if(type == "string[]") {
   str <- "string *"
 } else if(is.character(type) && length(grep("Func$", type)) > 0) {
   str <- type
 } else {
   str <- paste(type, "*")
 }

 str
}

CtypeDecl <-
function(type, var, defs)
{
 str <- getCType(type, defs)
 
 paste(str, var)
}  

genCCode <- 
function(fun, defs, name)
{
 if(missing(name)) {
   name <- fun$name
 }

 if(fun$"return" != "none") {
   retVal <- "ans = "
   ansDecl <- paste("   ", CtypeDecl(fun$"return", "ans", defs)[1], ";")
   cvtResult  <- convertToR("ans", fun$"return", defs)
 } else {
   retVal <- character(0)
   ansDecl <- character(0)
   cvtResult <- character(0)  # "\n\treturn(NULL_USER_OBJECT);"
 }


 if(length(fun$parameters)) {
   argDecls <- paste("USER_OBJECT_", nameToSArg(names(fun$parameters)), collapse=", ")

   coerceCode <- sapply(names(fun$parameters),
                        function(name) {
                          type <- getCType(fun$parameters[[name]], defs=defs)
                          paste("  ", type, name, "=",
                                 convertToCType(name, fun$parameters[[name]], defs),";")
                        })
     
 } else {
   argDecls <- character(0)
   coerceCode <- character(0)
 }

 decl <- paste("USER_OBJECT_\n", nameToC(name), "(", argDecls ,")")   
 txt <- c(decl, "{")
 txt <- c(txt, coerceCode, "\n", ansDecl, "   USER_OBJECT_ _result = NULL_USER_OBJECT;\n")

 argCallNames <- getArgCallValues(fun$parameters)
 txt <- c(txt, paste("\t", retVal, name, "(",  paste(argCallNames, collapse=", "), ");"))
   
 txt <- c(txt, cvtResult, "\n\n  return(_result);", "}\n")
 list(code=paste(txt, collapse="\n"),  decl = paste(decl, ";", sep=""))
}

getArgCallValues <-
  #
  #
  # returns the actual form of the values that are to be passed to the
  # real/underlying C function. This can perform the appropriate casts,
  # additional function calls (e.g. GTK_NOTEBOOK) or take the address of the
  # variable in the case of read-write variables.
  #
  #
  
function(params)
{
  if(length(params) == 0)
    return(character(0))
  
  pointer <- sapply(params, function(x) length(x) > 1 && x[2] == "read-write")
  if(any(pointer)) {
     prefix <- rep("", length(params))
     prefix[pointer] <- "&"
     paste(prefix, names(params))
  } else
    names(params)
}  

getEnumCheckDecl <-
function(name)
{
 paste("USER_OBJECT_ ", name, "(USER_OBJECT_);")
 # (USER_OBJECT_, const char *const [], const char *const [], int [], int len, const char *const);")
}

getEnumCRoutineName <-
  #
  # The name of the C routine that implements the verification/mapping of the S object
  # to a valid value of the specified enumeration of flag and returns an S object.
  #
function(name)
{
 paste("S_check",name,"value", sep="_")
}

genCEnum <-
function(enum, name, defs = NULL, local = T, isEnum = T)
  #
  # Generate the C and R code defining and declaring code
  # to handle an enumeration or flag.
  # This returns the R definition of the enum values which is a
  # named integer vector and an R function to map a value into
  # a valid value for this type.
  # Also, it returns C code which ``computes and verifies'' the
  # a value for this type of enumeration or flag and returns a
  # value with the appropriate type.
  # And finally it returns a declaration of that C routine so
  # it can be called from other C code that is automatically generated.
  # This is necessary when such an enum or flag is returned from a C routine
  # that we are mapping.
  #
{
  # for the moment, we will have a separate R function and C code

  # We create a named vector of the form
  #  c(a = 0, b = 1, c = 2)
  # where the names a, b, c are the local or colloquial names
  # from the defs files.
  # The values are inferred from the defs file.

  # enum$values was (1:length(enum))-1.
 localVals <- paste(paste("\"", names(enum$values), "\"", sep=""), enum$values, sep=" = ")

 rvector <- paste(name, "<-", "c(", paste(localVals, collapse=",\n\t") ,")\nstorage.mode(", name,") <- 'integer'")

 tmp <- paste(".", name, sep="")
 robjectNames = c(name, tmp)
 tmpVals <- paste(paste("\"", enum$names, "\"", sep=""), enum$values, sep=" = ", collapse=",\n\t")
 tmp <- paste(tmp, "<-", "c(", tmpVals ,")\nstorage.mode(", tmp,") <- 'integer'\n")

 rvector <- paste(rvector, tmp, sep="\n")
 
 mapName <- paste("map", name, sep="")
 cmapName <- getEnumCRoutineName(name)

 decl <- getEnumCheckDecl(cmapName)

  # The R code to check 
 txt <- paste(mapName, "<-\n",
              "function(val)\n{\n",
              "  if(inherits(val, \"", name, "\"))\n",
              "    return(val)\n",
              "  .Call(\"", cmapName, "\", val)",
              "\n}\n",
             collapse="", sep="")
 robjectNames = c(robjectNames, mapName)

   # Put in \n between each element to ensure line lengths aren't execessive and kil
   # some compilers.
 localCNames <- paste("\"", names(enum$names), "\"", sep="", collapse=",\n\t\t")
 realCNames <- paste("\"", enum$names, "\"", sep="", collapse=",\n\t\t")  
 cCode <- paste("USER_OBJECT_\n", cmapName,
                 "(USER_OBJECT_ val)\n{\n",
                 "  static const char * const localNames[] = {", localCNames, "};\n",
                 "  static const char * const realNames[]  = {", realCNames, "};\n",
                 "  static const int        cValues[]      = {", paste(enum$names, collapse=",\n\t"), "};\n",
                 "  return(S_check",
                      ifelse(isEnum, "Enum", "Flag"), "(val, localNames, realNames, cValues, ", length(enum$names),",\"", name,"\"));\n",
                 "}\n",
                 sep="")
 
 list(func= paste(txt, collapse="\n"), rvector = rvector, cCode = cCode, declaration = decl, robjectNames = robjectNames)
}


####################################################################

genFieldAccessors <-
function(defs)
{
  code <- list()
  for(i in names(defs$classes)) {
    klass <- defs$classes[[i]]
    if(length(klass$fields)) {
     tmp <- list()
     for(f in names(klass$fields)) {
       #  Check there isn't already an explicitly registered function for doing this.
       #  if(is.na(match(paste( classname to gtk_class format, get, i,sep="_") , names(defs$functions)))

      tmp[[f]] <-  genFieldAccessor(f, klass$fields[[f]], i, defs)
     }
     code[[i]] <- tmp
    }
  }

  return(code)
}  


genFieldAccessor <-
  #
  # produces R code to fetch the value of a slot
  # in an object.
  #
  #  gtkFileSelectionGetOkButton <- function(w) {
  #
  #    gtkCheckInherits(w, "GtkFileSelection")
  #    ans <- .Call("S_gtkFileSelectionGetOkButton", w)
  #    class(ans) <- c(..., "GtkObject")
  #    ans
  #  }
  #
  #  and the C code looks something like
  #  
  #  USER_OBJECT_
  #  S_gtkFileSelectionGetOkButton(USER_OBJECT_ sw)
  #  {
  #    GtkFileSelection *w;
  #    GtkWidget *val;
  #         w = GTK_FILE_SELECTION(getPtrValue(sw));
  #         val = w->ok_button;
  #         toRPointer(w, "GtkWidget")
  #  }
  #
function(name, type, className, defs)
{

 tmpName <- changeCapitalization(className, FALSE) 
 tmp <- paste(sapply(strsplit(name,"_")[[1]], changeCapitalization), collapse="")
 sname <- paste(tmpName, "Get", tmp, collapse="", sep="") 

 croutine <- paste("S_", className,"Get", tmp, collapse="", sep="")

 rcode <- genFieldAccessorRCode(sname, className, croutine, type, defs)

 ccode <- genFieldAccessorCCode(name, className, croutine, type, defs)

 list(rcode = paste(rcode, collapse="\n", sep=""),
      ccode = paste(ccode, collapse="\n", sep=""),
      rfuncName = sname) 
}


genFieldAccessorCCode <-
function(name, className, croutine, type, defs)
{
  c("USER_OBJECT_",
    paste(croutine, "(USER_OBJECT_ s_obj)"),
    "{",
    "   USER_OBJECT_ _result;",
    "",
    paste("  ", getCType(className, defs=defs), "obj;"),
    paste("  ", getCType(type, defs=defs), "val;"),
    "",
    paste("  ", "obj =", convertToCType("obj", className, defs=defs), ";"),
    paste("   val = obj->", name, ";", sep=""),
    convertToR("val", type, defs),
    "",
    "   return(_result);",
    "}"
  )  
}
  
genFieldAccessorRCode <-
function(sname, className, croutine, type, defs)
{
   # Converts the return value to the appropriate R type
   # assigning class information deduced from the `type'
   # argument
 setClassInfo <- character(0)

 if(!is.na(match(type, names(defs$classes))))
     # we don't want to determine the class here at compile time
     # but instead arrange to have it dynamically computed
     # at run-time. So use gtkObjectGetClasses() in the generate code
     # rather than getClassList() here in this code.
     setClassInfo <- " class(v) <- gtkObjectGetClasses(v, check = FALSE)"
 
 rcode <- c(paste(sname , "<-"),
            "function(obj)",
            "{",
            paste("  gtkCheckInherits(obj, '", className, "')", collapse="", sep=""),
            paste("  v <- .Call('", croutine, "', obj)", collapse="", sep=""),
            setClassInfo,
            "  v",
            "}")

 rcode
}  


####################################################################




changeCapitalization <-
function(x, up=TRUE) {
        els <- substring(x, 1:nchar(x), 1:nchar(x))
        if(up)
           els[1] <- chartr("a-z", "A-Z", els[1])
        else
           els[1] <- chartr("A-Z", "a-z", els[1])          

        paste(els, collapse="")
}

mapToClassName <-
  #
  # converts a function name into the name of the associated
  # Gtk/Gdk class by 
  #  a) discarding all the contents after (and including)
  #     the second _ in the name
  #
  #  b) capitalizing each letter following a `_' character.
  #
  #  c) discarding the remaining _
  # 
  #  d) capitalizing the first letter
  #
  #
  # e.g.  mapToClassName("gtk_button_new")
  #       [1] "GtkButton"
  #
  #
function(name, classes, capitalize = TRUE)
{
  parts <- strsplit(name, "_")[[1]]


  upto <- pmatch("new", parts)
  if(is.na(upto))
    stop(paste("no `new' in the name of the name passed to mapToClassName", name))
  upto <- upto - 1
  which <- match(paste(parts[1:upto], collapse=""), tolower(names(classes)))
#  if(is.na(which))
#    stop(paste("No class resembling name", paste(parts[1:upto], collapse="")))

  if(!is.na(which))
    className <- names(classes)[which]
  else
    className <- paste(sapply(parts[1:upto], changeCapitalization), collapse="", sep="")
  
  
  if(capitalize == FALSE)
    return(className)

  other <- paste(sapply(parts[-c(1:upto)], changeCapitalization), collapse="")
  return(paste(changeCapitalization(className, FALSE), other, collapse="", sep=""))
}

toCapPhraseName <-
  # converts a string into collapsed capitalized word format
  # from a _ separated word format.
  # i.e. convert a name in the form word1_word2_word3
  # to word1Word2Word3
  # In other words, this removes the word separator `_'
  # and capitalizes each word, optionally capitalizing
  # the first word also.
  #
  # If classNames is specified, then we try to ensure
  # that the prefix for the name matches a class name
  # from this list and preserves capitalization.
  # This is aimed at fixing the mapping of function names
  # such as gtk_clist_set_column_width which would
  # be mapped to gtkClist... rather than gtkCList...
  #
  # We are doing an extra changeCapitalization() in
  # the case of a matching class name and we want to capitalize
  # No big deal at present. This is already a mess!
function(name, capitalize = FALSE, classNames = NULL)
{
  els <- strsplit(name, "_")[[1]]

  className <- character(0)  
  if(!is.null(classNames)) {
    id <- pmatch(tolower(classNames), paste(els, collapse="", sep=""))
    if(any(!is.na(id))) {
      className <- changeCapitalization(classNames[!is.na(id)][1], FALSE)
    }
  }

  
  els <- sapply(els, changeCapitalization)
  val <- paste(els, collapse="")
  
  if(length(className)) {
     # want to replace the first nchar(className) characters
     # with the values from className.
    val <- paste(className, substring(val, nchar(className)+1), collapse="",sep="")
  }
  
  if(capitalize) {
    val <- changeCapitalization(val, FALSE)
  }

  val
}  


collapseClassName <-
  #
  # converts a class name of the form GtkButton
  # to gtk_button.
  # Also handles GtkCList to gtk_clist.
  #
function(name)
{
  tmp <- gsub("([ABCDEFGHIJKLMNOPQRSTUVWXYZ]+)", "_\\1", name)
  tmp <- tolower(substring(tmp, 2))
  gsub("_([abcdefghijklmnopqrstuvwxyz])_","_\\1", tmp)  
}  


  
computeSubClasses <-
  # Computes the table (named list) of
  # classes  and their sub-classes.
  
  # Not actually used but a utility function
  # for the package that others might want to use.
function(classes)
{
  kids <- vector("list", length(classes))
  names(kids) <- names(classes)

  
  sapply(names(classes), function(x) {
                             parent <- classes[[x]]$parent
                             if(!length(parent))
                               parent <- "Orphan Classes"
                             kids[[parent]] <<- c(kids[[parent]], x)

                             T
                          })
                  
  kids
}
