# ifndef _RHEO_ASR_H
# define _RHEO_ASR_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

#include "rheolef/distributor.h"
#include "rheolef/diststream.h"
#include "rheolef/heap_allocator.h"

#ifdef _RHEOLEF_HAVE_MPI
// -------------------------------------------------------------
// send a pair<pair<size_t,size_t>,T> via mpi & serialization
// -------------------------------------------------------------
// non-intrusive version of serialization
namespace boost {
 namespace serialization {
  template<class T, class Archive>
  void serialize (Archive & ar, std::pair<std::pair<size_t,size_t>, T>& g, const unsigned int version) {
    ar & g.first.first;
    ar & g.first.second;
    ar & g.second;
  }
 } // namespace serialization
} // namespace boost

// Some serializable types, have a fixed amount of data stored at fixed field positions.
// When this is the case, boost::mpi can optimize their serialization and transmission to avoid extraneous copy operations.
// To enable this optimization, we specialize the type trait is_mpi_datatype, e.g.:
namespace boost {
 namespace mpi {
  template <>
  struct is_mpi_datatype<std::pair<std::pair<size_t,size_t>, double> > : mpl::true_ { };
 } // namespace mpi
} // namespace boost
#endif // _RHEOLEF_HAVE_MPI

#ifndef TO_CLEAN
// for debug
namespace rheolef {
template<class T>
inline
std::ostream&
operator<< (std::ostream& o, const std::pair<std::pair<size_t,size_t>, T>& x) {
  return o << "triplet{" << x.first.first << ", " << x.first.second << ", "<< x.second << "}";
}
} // namespace rheolef
#endif // TO_CLEAN

// -------------------------------------------------------------
// the asr class suite...
// -------------------------------------------------------------
namespace rheolef {

template<class T, class M> class csr;
template<class T> class csr_seq_rep;
template<class T> class csr_mpi_rep;

// -------------------------------------------------------------
// the sequential representation
// -------------------------------------------------------------
template<class T>
class asr_seq_rep : public std::vector<std::map<size_t, T,
                     std::less<size_t>, heap_allocator<std::pair<size_t,T> > > > {
public:
    typedef typename std::vector<T>::size_type            size_type;
    typedef T                                             element_type;
    typedef sequential                                    memory_type;
    typedef distributor::communicator_type                communicator_type;
    typedef std::vector<std::map<size_type, T,
            std::less<size_type>,
	    heap_allocator<std::pair<size_type,T> > > >   base;
    typedef std::map <size_type, T, std::less<size_type>,
            heap_allocator<std::pair<size_type,T> > >     map_type;     // (j, value) collection
    typedef std::map<std::pair<size_type, size_type>, T, 
            std::less<std::pair<size_type, size_type> >,
            heap_allocator<std::pair<std::pair<size_type, size_type>, T> > >
							  map_ijv_type; // (i,j,value) collection
    typedef map_type                                      row_type;

    asr_seq_rep (const distributor& row_ownership, const distributor& col_ownership);
    void resize (const distributor& row_ownership, const distributor& col_ownership);
    asr_seq_rep (size_type loc_nrow=0, size_type loc_ncol=0);
    void resize (size_type loc_nrow, size_type loc_ncol);
    asr_seq_rep (const asr_seq_rep<T>& a) ;
    explicit asr_seq_rep(const csr_seq_rep<T>& a);

    const distributor& row_ownership () const { return _row_ownership; }
    const distributor& col_ownership () const { return _col_ownership; }
    size_type nrow() const { return std::vector<row_type>::size(); }
    size_type ncol() const { return col_ownership().size(); }
    size_type nnz() const { return _nnz; }
    size_type dis_nrow() const { return nrow(); }
    size_type dis_ncol() const { return ncol(); }
    size_type dis_nnz() const { return _nnz; }
    size_type row_first_index () const { return 0; }
    size_type row_last_index () const { return nrow(); }
    size_type col_first_index () const { return 0; }
    size_type col_last_index () const { return ncol(); }
    T& dis_entry (size_type i, size_type j) {
	row_type& row_i = base::operator[](i);
        std::pair<typename row_type::iterator,bool> status
	= row_i.insert (typename row_type::value_type(j,T(0)));
        if (status.second) _nnz++;
	return (*(status.first)).second;
      }
    T operator() (size_type i, size_type j) const {
        typename row_type::const_iterator pos_aij  = base::operator[](i).find(j);
        if (pos_aij != base::operator[](i).end())
	    return (*pos_aij).second;
	return T(0);
      }
    void dis_entry_assembly_begin () {}
    void dis_entry_assembly_end () {}
    idiststream& get (idiststream&);
    odiststream& put (odiststream&, size_type istart = 0) const;
    void dump (const std::string& name, size_type istart = 0) const;

protected:
    distributor  _row_ownership;
    distributor  _col_ownership;
    size_type    _nnz;
};
// -------------------------------------------------------------
// the distributed representation
// -------------------------------------------------------------
#ifdef _RHEOLEF_HAVE_MPI
template<class T>
class asr_mpi_rep : public asr_seq_rep<T> {
public:
    typedef typename asr_seq_rep<T>::size_type    size_type;
    typedef typename asr_seq_rep<T>::element_type element_type;
    typedef typename asr_seq_rep<T>::row_type     row_type;
    typedef distributed                           memory_type;
    typedef asr_seq_rep<T>                        base;

    asr_mpi_rep(const distributor& row_ownership, const distributor& col_ownership);
    void resize(const distributor& row_ownership, const distributor& col_ownership);
    asr_mpi_rep (size_type dis_nrow=0, size_type dis_ncol=0);
    void resize (size_type dis_nrow, size_type dis_ncol);
    asr_mpi_rep(const asr_mpi_rep<T>& a);
    explicit asr_mpi_rep(const csr_mpi_rep<T>& a);

    const distributor& row_ownership () const { return asr_seq_rep<T>::_row_ownership; }
    const distributor& col_ownership () const { return asr_seq_rep<T>::_col_ownership; }
    const communicator& comm() const { return row_ownership().comm(); }
    size_type dis_nrow() const { return row_ownership().dis_size(); }
    size_type dis_ncol() const { return col_ownership().dis_size(); }
    size_type dis_nnz() const { return _dis_nnz; }
    size_type row_first_index () const { return row_ownership().first_index(); }
    size_type row_last_index () const  { return row_ownership().last_index(); }
    T& dis_entry (size_type dis_i, size_type dis_j);
    void dis_entry_assembly_begin ();
    void dis_entry_assembly_end ();
    idiststream& get (idiststream&);
    odiststream& put (odiststream&) const;
    void dump (const std::string& name) const;
protected:
    // number of local entries in the diagonal bloc
    size_type        _dis_nnz;

    // asr assembly stuff: e.g. for i/o and finite-element assembly
    // => communication during assembly_begin(), assembly_end()
    // TODO: groups it in a asr_mpi_rep_assembly class ?
    typedef typename asr_seq_rep<T>::map_type     map_type;
    typedef typename asr_seq_rep<T>::map_ijv_type map_ijv_type;
    struct message_type {
        std::list<std::pair<size_t,mpi::request> >                waits;
        std::vector<std::pair<std::pair<size_type,size_type>,T> > data;
    };
    map_ijv_type    _stash;
    message_type    _send;
    message_type    _receive;
    size_type       _receive_max_size;
};
#endif // _RHEOLEF_HAVE_MPI
// -------------------------------------------------------------
// the basic class with a smart pointer to representation
// the user-level class with memory-model parameter
// -------------------------------------------------------------
/*Class:asr
NAME:  @code{asr} - associative sparse matrix (@PACKAGE@-@VERSION@)
SYNOPSYS:       
 Associative sparse matrix container stored row by row
 using the STL map class.
IMPLEMENTATION NOTE:
   Implementation use MPI-1.1
   and is inspired from Mat_MPI in PETSc-2.0.22.
TO DO:
   For efficiency purpose, the assembly phase may
   access directly to the asr representation, without
   crossing the reference counting and pointer handler.
   Something like the iterator for dense vectors.
AUTHORS:
    LMC-IMAG, 38041 Grenoble cedex 9, France
   | Pierre.Saramito@imag.fr
DATE:   6 january 1999
METHODS: @asr
End:
*/
//<asr:
template<class R>
class basic_asr : public smart_pointer<R> {
public:

// typedefs:

    typedef typename R::size_type          size_type;
    typedef typename R::element_type       element_type;
    typedef typename R::memory_type        memory_type;
    typedef distributor::communicator_type communicator_type;

// allocators/deallocators:

    basic_asr (size_type dis_nrow = 0, size_type dis_ncol = 0); 
    basic_asr (const distributor& row_ownership, const distributor& col_ownership);
    explicit basic_asr (const csr<element_type,memory_type>&);

// accessors:

    const communicator_type& comm() const;

    // local sizes
    size_type nrow () const;
    size_type ncol () const;
    size_type nnz () const;
    
    // global sizes
    size_type dis_nrow () const;
    size_type dis_ncol () const;
    size_type dis_nnz () const;
    const distributor& row_ownership() const;
    const distributor& col_ownership() const;

    // range on local memory
    size_type row_first_index () const;
    size_type row_last_index () const;
    size_type col_first_index () const;
    size_type col_last_index () const;

// global modifiers:

    element_type& dis_entry (size_type dis_i, size_type dis_j);
    void dis_entry_assembly();
    void dis_entry_assembly_begin ();
    void dis_entry_assembly_end ();
    void resize (size_type dis_nrow = 0, size_type dis_ncol = 0);

// output:

    void dump (const std::string& name) const;
};
template <class T, class M = rheo_default_memory_model>
class asr 
{
    typedef M memory_type;
};
template <class T>
class asr<T,sequential> : public basic_asr<asr_seq_rep<T> > {
public:
    typedef typename basic_asr<asr_seq_rep<T> >::size_type size_type;
    typedef sequential memory_type;
    asr (size_type dis_nrow = 0, size_type dis_ncol = 0);
    asr (const distributor& row_ownership, const distributor& col_ownertship);
    explicit asr(const csr<T,memory_type>&);
};
#ifdef _RHEOLEF_HAVE_MPI
template <class T>
class asr<T,distributed> : public basic_asr<asr_mpi_rep<T> > {
public:
    typedef distributed memory_type;
    typedef typename basic_asr<asr_mpi_rep<T> >::size_type size_type;
    asr (size_type dis_nrow = 0, size_type dis_ncol = 0);
    asr (const distributor& row_ownership, const distributor& col_ownertship);
    explicit asr(const csr<T,memory_type>&);
};
#endif // _RHEOLEF_HAVE_MPI

// inputs/outputs:
template <class T, class M>
idiststream& operator >> (idiststream& s, asr<T,M>& x);

template <class T, class M>
odiststream& operator << (odiststream& s, const asr<T,M>& x);
//>asr:
// -------------------------------------------------------------
// inline'd
// -------------------------------------------------------------
template <class R>
inline
basic_asr<R>::basic_asr (
        size_type dis_nrow,
	size_type dis_ncol)
 : smart_pointer<R> (new_macro(R(dis_nrow,dis_ncol)))
{
}
template <class R>
inline
basic_asr<R>::basic_asr (
        const distributor& row_ownership,
	const distributor& col_ownership)
 : smart_pointer<R> (new_macro(R(row_ownership,col_ownership)))
{
}
template <class R>
inline
basic_asr<R>::basic_asr (
    const csr<typename R::element_type, typename R::memory_type>& a)
 : smart_pointer<R> (new_macro(R(a.data())))
{
}
template <class R>
inline
const distributor&
basic_asr<R>::row_ownership () const
{
    return smart_pointer<R>::data().row_ownership();
}
template <class R>
inline
const typename basic_asr<R>::communicator_type&
basic_asr<R>::comm () const
{
    return row_ownership().comm();
}
template <class R>
inline
const distributor&
basic_asr<R>::col_ownership () const
{
    return smart_pointer<R>::data().col_ownership();
}
template <class R>
inline
typename basic_asr<R>::size_type
basic_asr<R>::nrow () const
{
    return smart_pointer<R>::data().nrow();
}
template <class R>
inline
typename basic_asr<R>::size_type
basic_asr<R>::ncol () const
{
    return smart_pointer<R>::data().ncol();
}
template <class R>
inline
typename basic_asr<R>::size_type
basic_asr<R>::nnz () const
{
    return smart_pointer<R>::data().nnz();
}
template <class R>
inline
typename basic_asr<R>::size_type
basic_asr<R>::dis_nrow () const
{
    return smart_pointer<R>::data().dis_nrow();
}
template <class R>
inline
typename basic_asr<R>::size_type
basic_asr<R>::dis_ncol () const
{
    return smart_pointer<R>::data().dis_ncol();
}
template <class R>
inline
typename basic_asr<R>::size_type
basic_asr<R>::dis_nnz () const
{
    return smart_pointer<R>::data().dis_nnz();
}
template <class R>
inline
typename basic_asr<R>::size_type
basic_asr<R>::row_first_index () const
{
    return smart_pointer<R>::data().row_first_index();
}
template <class R>
inline
typename basic_asr<R>::size_type
basic_asr<R>::row_last_index () const
{
    return smart_pointer<R>::data().row_last_index();
}
template <class R>
inline
typename basic_asr<R>::size_type
basic_asr<R>::col_first_index () const
{
    return smart_pointer<R>::data().col_first_index();
}
template <class R>
inline
typename basic_asr<R>::size_type
basic_asr<R>::col_last_index () const
{
    return smart_pointer<R>::data().col_last_index();
}
template <class R>
inline
typename basic_asr<R>::element_type&
basic_asr<R>::dis_entry (size_type dis_i, size_type dis_j)
{
    return smart_pointer<R>::data().dis_entry (dis_i, dis_j);
}
template <class R>
inline
void
basic_asr<R>::resize (
    size_type nrow,
    size_type ncol)
{
    smart_pointer<R>::data().resize (dis_nrow,dis_ncol);
}
template <class R>
inline
void
basic_asr<R>::dis_entry_assembly_begin()
{
    smart_pointer<R>::data().dis_entry_assembly_begin();
}
template <class R>
inline
void
basic_asr<R>::dis_entry_assembly_end()
{
    smart_pointer<R>::data().dis_entry_assembly_end();
}
template <class R>
inline
void
basic_asr<R>::dis_entry_assembly()
{
    smart_pointer<R>::data().dis_entry_assembly_begin();
    smart_pointer<R>::data().dis_entry_assembly_end();
}
template <class R>
inline
void
basic_asr<R>::dump(const std::string& name) const
{
    smart_pointer<R>::data().dump(name);
}
template <class T>
inline
asr<T,sequential>::asr (
    size_type dis_nrow,
    size_type dis_ncol)
  : basic_asr<asr_seq_rep<T> >(dis_nrow,dis_ncol)
{
}
template <class T>
inline
asr<T,sequential>::asr (
    const distributor& row_ownership,
    const distributor& col_ownership)
  : basic_asr<asr_seq_rep<T> > (row_ownership, col_ownership)
{
}
template <class T>  
inline
asr<T,sequential>::asr (const csr<T,sequential>& a)
 : basic_asr<asr_seq_rep<T> >(a)
{
}
#ifdef _RHEOLEF_HAVE_MPI
template <class T>
inline
asr<T,distributed>::asr (
    size_type dis_nrow,
    size_type dis_ncol)
  : basic_asr<asr_mpi_rep<T> >(dis_nrow,dis_ncol)
{
}
template <class T>
inline
asr<T,distributed>::asr (
    const distributor& row_ownership,
    const distributor& col_ownership)
  : basic_asr<asr_mpi_rep<T> > (row_ownership, col_ownership)
{
}
template <class T>  
inline
asr<T,distributed>::asr (const csr<T,distributed>& a)
 : basic_asr<asr_mpi_rep<T> >(a)
{
}
# endif // _RHEOLEF_HAVE_MPI
template <class T, class M>
inline
idiststream&
operator >> (idiststream& s,  asr<T,M>& x)
{ 
    return x.data().get(s); 
}
template <class T, class M> 
inline
odiststream&
operator << (odiststream& s, const asr<T,M>& x)
{
    return x.data().put(s); 
}
} // namespace rheolef
# endif // _RHEO_ASR_H
