/*
 * rtmap.h
 * 
 * Copyright (c) 2000-2004 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

/** \file
  * Contains the class template <tt>Map</tt>, which implements a key-value
  * map. 
  * @see Map
  */

#ifndef __LRT_MAP__
#define __LRT_MAP__

#include "rtiterator.h"
#include "rtlist.h"
#include "rtstring.h"

namespace lrt {

// fwd declare
template<class K, class V> class Map;

/** A key-value pair. */
template<class K, class V> class Pair {
public:
	/** Creates a new "standard" key-value pair by calling the
	  * key's and value's default constructors. */
	Pair() : key(), value() {}
	/** Creates a key-value pair and copies the given data into it. */
	Pair(const K& k, const V& v) : key(k), value(v) {}
	/** Returns a constant reference to the key. */
	inline const K& getKey() const		{ return key; }
	/** Returns a constant reference to the value. */
	inline const V& getValue() const	{ return value; }
	/** Returns a modifiable reference to the key. */
	inline K& getKey()					{ return key; }
	/** Returns a modifiable reference to the value. */
	inline V& getValue()				{ return value; }
private:
	friend class Map<K,V>;
	K key;
	V value;
};


/** A map collection type (a.k.a.\ associative array).
  * This implementation of a map uses a linked list to store its data and
  * it therefore quite slow (for large data sets). <br>
  * <b>Note:</b> All types which should be used as keys in a Map must implement
  * the comparison operators <tt>&lt;</tt> and <tt>==</tt>!
  */
template<class K, class V> class Map {
public:
	
	/** The map iterator type: You can alternatively use 
	  * Iterator<Map<K,V>::Pair>
	  * or Map<K,V>::Iterator (which is somewhat shorter). */
	typedef Iterator<Pair<K,V> > Iterator;

	/** Creates a new empty map. The map entries are sorted using the default 
	  * <tt>stdCompare()</tt> function. */
	Map();
	/** Creates a new empty map, whose entries are sorted with a given comparison 
	  * function. */
	Map(int (*compareFun)(const K&, const K&));
	/** Creates a new map, copying all the contents over from the given map. */
	Map(const Map &);
	virtual ~Map() {}
	/** Clears this map, then copies all the contents of the given map into it. */
	const Map& operator=(const Map &);

	/** Returns <tt>True</tt> if there is a mapping for the given key in this map. */
	bool isSet(const K&) const;
	/** Returns a constant reference to the value the given key is mapped to, or 
	  * <tt>noValue.getValue()</tt> if it is not mapped to anything. */
	const V& get(const K&) const;
	/** Returns a modifiable reference to the value the given key is mapped to, or
	  * if it was not mapped to anything, inserts the key into the map, so that 
	  * the reference can be assigned to something. */
	V& get(const K&);
	/** Put a new mapping of the given key and value into the map. If the given key
	  * was already mapped to something, the mapping is replaced. 
	  * @return A reference to the inserted key-value pair. */
	Pair<K,V>& put(const K&, const V&);
	/** Copies all mappings from the given map into this one. Existing mappings are
	  * replaced (if they have the same key). */
	void putAll(const Map<K,V>&);
	/** Removes the mapping of the given key from this map. If the key was not mapped
	  * to anything, just does nothing. 
	  * @return <tt>True</tt> if there was actually a mapping removed. */
	bool remove(const K&);
	/** Removes the mapping the given iterator points to.
	  * Moves the iterator to the next mapping of the list. */
	void remove(Iterator& iter);
	/** Clears this map by removing and deleting all entries. */
	void clear();
	//Pair& insert(const Pair&);

	/** Returns a constant reference to the value the given key is mapped to, or 
	  * <tt>noValue.getValue()</tt> if it is not mapped to anything. */
	inline const V& operator[](const K&) const;
	/** Returns a modifiable reference to the value the given key is mapped to, or
	  * if it was not mapped to anything, inserts the key into the map, so that 
	  * the reference can be assigned to something. */
	inline V& operator[](const K&);
	
	/** Returns the element count of this map. */
	int length() const;
	/** Returns an unmodifyable iterator to the first element of the map. 
	  * If the map is empty, the iterator points to nothing; you can use 
	  * Iterator::hasElement() to check this. */
	Iterator begin() const;
	/** Returns an unmodifyable iterator to the last element of the map. 
	  * If the map is empty, the iterator points to nothing; you can use 
	  * Iterator::hasElement() to check this. */
	Iterator end() const;
	/** Returns a modifyable iterator to the first element of the map. 
	  * If the map is empty, the iterator points to nothing; you can use 
	  * Iterator::hasElement() to check this. */
	Iterator begin();
	/** Returns a modifyable iterator to the last element of the map. 
	  * If the map is empty, the iterator points to nothing; you can use 
	  * Iterator::hasElement() to check this. */
	Iterator end();

	/** The pair which is defined to be no value. */
	static Pair<K,V> noValue;

protected:
	/** The function used to compare the keys. */
	int (*compareFun)(const K&, const K&);

private:
	List< Pair<K,V> > data;

};

/** A map using <tt>String</tt>s as keys.
  * The only difference to the basic Map template is, that StringMaps can be case
  * insensitive. 
  */
template<class V> class StringMap : public Map<String,V>
{
public:
	/** Creates an empty StringMap which is case sensitive. */
	StringMap() : Map<String,V>(&String::compare) {}
	/** Creates an empty StringMap, which is optionally case insensitive. */
	StringMap(bool caseSensitive) : 
		Map<String,V>(&String::compare)
	{
		if(!caseSensitive) compareFun = &String::compareIgnoreCase;		
	}
	/** Creates a new StringMap, copying all the contents over from the given map. */
	StringMap(const StringMap& sm) : Map<String,V>(sm) {}
	virtual ~StringMap() {}
	/** Clears this StringMap, then copies all the contents of the given map into it. */
	const StringMap& operator=(const StringMap& sm)
	{ Map<String,V>::operator=(sm); return *this; }

};

} // namespace

#include "rtmap.templ.cpp"

#endif
