
/*
 * Copyright (C) 1999-2001, Ian Main <imain@stemwinder.org>.
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject
 * to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * 
 */

#include <roy.h>

#define RHASH_MIN_SIZE 4

/* Define the growth rate of the table.  A growth rate of
 * 1 will mean we will have at least as many slots available
 * as we do table entries (a 1 to 1 ratio).  Setting this
 * value to 2, would cause 2 entries per hash table slot
 * etc. */
#define RHASH_GROW_RATIO 4

/* Define the shrink ratio.  This sets how far down you have to
 * go before the table will be shrunken.  Setting this to 2 for
 * example, will make the table shrink when less than 1/2 of the
 * hash table slots are used */
#define RHASH_SHRINK_RATIO 2


#define RHASH_INSERT_LIST(head,entry) \
entry->rhash_header.next = head; \
head = entry; \
entry->rhash_header.prev_next_ptr = &head; \
if (entry->rhash_header.next) \
    entry->rhash_header.next->rhash_header.prev_next_ptr = &entry->rhash_header.next


static unsigned int rhash_primes[] =
{
    3,
    7,
    11,
    19,
    37,
    73,
    109,
    163,
    251,
    367,
    557,
    823,
    1237,
    1861,
    2777,
    4177,
    6247,
    9371,
    14057,
    21089,
    31627,
    47431,
    71143,
    106721,
    160073,
    240101,
    360163,
    540217,
    810343,
    1215497,
    1823231,
    2734867,
    4102283,
};

static unsigned int rhash_num_primes = sizeof (rhash_primes) / sizeof (unsigned int);



RHash *
rhash_new (RHashHashFunction hash_func, RHashCompareFunction compare_func)
{
    RHash *hash;

    hash = rchunk_alloc (sizeof (RHash));
    RHASH_INIT (hash, hash_func, compare_func);

    return (hash);
}

void
rhash_free (RHash *hash)
{
    RHASH_DESTRUCT (hash);
    rchunk_free (hash, sizeof (RHash));
}


static void
rhash_table_resize (RHash *hash)
{
    unsigned int new_size = RHASH_MIN_SIZE;
    unsigned int old_size;
    unsigned long index;
    RHashEntry **old_entries;
    unsigned int i = 0;

    /* Calculate new prime size for table */
    while ((rhash_primes[i] < hash->num_entries) && (i < rhash_num_primes)) {
        i++;
    }
    new_size = rhash_primes[i];

    old_entries = hash->entries;
    old_size = hash->table_size;

    hash->entries = (RHashEntry **) rmem_alloc0 (sizeof (RHashEntry *) * new_size);

    hash->table_size = new_size;

    /* walk through the old hash table entries, and just insert them into
     * the new table */
    for (i = 0; i < old_size; i++) {
        RHashEntry *next;
        RHashEntry *ent = old_entries[i];

        while (ent) {
            index = ent->rhash_header.hashval % hash->table_size;
            /* Save 'next' pointer because we have to change it below
             * for its new home */
            next = ent->rhash_header.next;
            RHASH_INSERT_LIST (hash->entries[index], ent);
            ent = next;
        }
    }

    /* free the old table */
    if (old_entries)
        rmem_free (old_entries);
}

void *
rhash_remove (RHash *hash, void *key)
{
    RHashEntry *entry;
    unsigned long index;
    unsigned long hashval;

    if (hash->num_entries < (hash->table_size / RHASH_SHRINK_RATIO))
        rhash_table_resize (hash);

    if (hash->hash_func)
        hashval = hash->hash_func (key);
    else 
        hashval = (unsigned long) key;

    index = hashval % hash->table_size;

    for (entry = hash->entries[index]; entry != NULL; entry = entry->rhash_header.next) {
        if (hashval == entry->rhash_header.hashval) {
            if (hash->compare_func) {
                if (hash->compare_func (rhash_entry_getkey (entry), key)) {
                    rhash_remove_entry (hash, entry);
                    return (entry);
                }
            } else {
                rhash_remove_entry (hash, entry);
                return (entry);
            }
        }
    }

    return (NULL);
}

void *
rhash_lookup (RHash *hash, void *key)
{
    RHashEntry *entry;
    unsigned long index;
    unsigned long hashval;

    if (hash->table_size == 0)
        return (NULL);

    if (hash->hash_func)
        hashval = hash->hash_func (key);
    else 
        hashval = (unsigned long) key;

    index = hashval % hash->table_size;
    for (entry = hash->entries[index]; entry != NULL; entry = entry->rhash_header.next) {
        if (hashval == entry->rhash_header.hashval) {
            if (hash->compare_func) {
                if (hash->compare_func (rhash_entry_getkey (entry), key))
                    return (entry);
            } else {
                return (entry);
            }
        }
    }

    return (NULL);
}

void
rhash_insert (RHash *hash, void *inentry, void *key)
{
    RHashEntry *entry = inentry;
    unsigned long hashval;
    unsigned long index;

    /* Store the key in the entry */
    rhash_entry_setkey ((RHashEntry *) inentry, key);

    if (hash->num_entries >= (hash->table_size * RHASH_GROW_RATIO))
        rhash_table_resize (hash);

    /* Get your index into the hash table */
    if (hash->hash_func)
        hashval = hash->hash_func (key);
    else 
        hashval = (unsigned long) key;

    entry->rhash_header.hashval = hashval;
    index = hashval % hash->table_size;

    /* stick it in the front of the list */
    RHASH_INSERT_LIST (hash->entries[index], entry);

    hash->num_entries++;
}

