
/*
 * Copyright (C) 1999-2001, Ian Main <imain@stemwinder.org>.
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject
 * to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * 
 */

#include <roy.h>


#define RXP_ENCODER_MAYBE_EXPAND(encoder) \
    if (encoder->length > encoder->alloced) { \
        unsigned int new_size; \
        new_size = encoder->alloced * 2; \
        encoder->data = rmem_realloc (encoder->data, new_size); \
        encoder->alloced = new_size; \
    }


RXpEncoder *
rxp_encoder_new (void)
{
    RXpEncoder *encoder;

    encoder = rchunk_alloc (sizeof (RXpEncoder));
    encoder->data = rmem_alloc (4096);
    encoder->cur_pos = encoder->data;
    encoder->length = 4;
    encoder->alloced = 4096;

    /* Encode the length of the block */
    RXP_ENCODE_RINT32 (encoder->data, 4);
    encoder->cur_pos += 4;
    
    return (encoder);
}

void
rxp_encode_reset (RXpEncoder *encoder)
{
    encoder->cur_pos = encoder->data;
    encoder->length = 4;
    
    RXP_ENCODE_RINT32 (encoder->data, 4);
    encoder->cur_pos += 4;
}

void
rxp_encode_rint32 (RXpEncoder *encoder, rint32 value)
{
    /* Update length */
    encoder->length += 8;
    
    /* Insure we have enough space */
    RXP_ENCODER_MAYBE_EXPAND (encoder);
    
    /* Update total length at start of block */
    RXP_ENCODE_RINT32 (encoder->data, encoder->length);

    /* Encode the data type */
    RXP_ENCODE_RINT32 (encoder->cur_pos, RXP_TYPE_INTEGER);
    encoder->cur_pos += 4;
    
    /* Encode integer */
    RXP_ENCODE_RINT32 (encoder->cur_pos, value);
    encoder->cur_pos += 4;
}

void
rxp_encode_rbuf (RXpEncoder *encoder, RBuf *value)
{
    /* NULL is special cased */
    if (value == NULL) {
        encoder->length += 4;
        /* Insure we have enough space */
        RXP_ENCODER_MAYBE_EXPAND (encoder);
        /* Update total length at start of block */
        RXP_ENCODE_RINT32 (encoder->data, encoder->length);
        /* Set data type to NULL_RBUF */
        RXP_ENCODE_RINT32 (encoder->cur_pos, RXP_TYPE_NULL_RBUF);
        encoder->cur_pos += 4;
        
        return;
    }
    
    /* Non-NULL rbuf - Update length */
    encoder->length += rbuf_len (value);
    encoder->length += 8;
    
    /* Insure we have enough space */
    RXP_ENCODER_MAYBE_EXPAND (encoder);
    /* Update total length at start of block */
    RXP_ENCODE_RINT32 (encoder->data, encoder->length);
    /* Set data type to RBUF */
    RXP_ENCODE_RINT32 (encoder->cur_pos, RXP_TYPE_RBUF);
    encoder->cur_pos += 4;
    /* Save length of rbuf */
    RXP_ENCODE_RINT32 (encoder->cur_pos, rbuf_len (value));
    encoder->cur_pos += 4;
    /* Save rbuf itself */
    memcpy (encoder->cur_pos, rbuf_str (value), rbuf_len (value));
    encoder->cur_pos += rbuf_len (value);
}


void
rxp_encoder_free (RXpEncoder *encoder)
{
    rmem_free (encoder->data);
    rchunk_free (encoder, sizeof (RXpEncoder));
}


unsigned int
rxp_decode_msg_length (char *data)
{
    unsigned int length;
    
    RXP_DECODE_RINT32 (data, length);
    return (length);
}

RArray *
rxp_decode_data (char *data)
{
    RArray *items;
    char *position;
    char *end;
    unsigned int length;
    unsigned int type;
    RXpArrayEntry *entry;

    
    items = rarray_new (sizeof (RXpArrayEntry), 16);

    position = data;
   
    /* Get total length */
    RXP_DECODE_RINT32 (position, length);
    position += 4;
    
    end = data + length;

    while (position < end) {
        RXP_DECODE_RINT32 (position, type);
        position += 4;

        if (type == RXP_TYPE_INTEGER) {
           
            if (position + 4 > end) {
                printf ("Error decoding message in rxp: Reading rint32 value will go past end of message.\n");
                return (items);
            }

            entry = rarray_push (items);
            RXP_DECODE_RINT32 (position, entry->data.intval);
            entry->type = RXP_TYPE_INTEGER;
            position += 4;
        
        } else if (type == RXP_TYPE_RBUF) {
            
            RXP_DECODE_RINT32 (position, length);
            position += 4;
            if (position + length > end) {
                printf ("Error decoding message in rxp: Length of rbuf greater than length of message.\n");
                return (items);
            }
            
            entry = rarray_push (items);
            entry->data.buf = rbuf_new_with_data (position, length);
            entry->type = RXP_TYPE_RBUF;
            position += length;
        
        } else if (type == RXP_TYPE_NULL_RBUF) {
            
            entry = rarray_push (items);
            entry->data.buf = NULL;
            entry->type = RXP_TYPE_NULL_RBUF;
        
        } else {
            printf ("Error decoding message in rxp: Unknown item type %d.\n", type);
            return (items);
        }
    }

    return (items);
}




