
/*
 * Copyright (C) 1999-2001, Ian Main <imain@stemwinder.org>.
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject
 * to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * 
 */

#include <roy.h>


#define FILE_BUFFER_SIZE 2048
char TESTCHUNK[]=  "<a a=\"a\" b=\"b\">this<b x=\"y\"/></a>";
char *speed_error = NULL;


void
fast_error_handler (ThermlParser *p, int severity, 
                    int linenum, int colnum, const char *description);

void
error_handler (ThermlParser *p, int severity, int linenum, int colnum, const char *description);

void
pcommand_handler (ThermlParser *p, RBuf *data);


void
tag_start_handler (ThermlParser *parser, RBuf *type, RBHash *attribs);

void
tag_end_handler (ThermlParser *p, RBuf *name, RBuf *data_attr);

void
parse_file (ThermlParser *parser, char *name);

void
#if DEBUG
loop_test (char *filename, int do_debug);
#else
loop_test (char *filename);
#endif





void
fast_error_handler (ThermlParser *p, int severity, 
                    int linenum, int colnum, const char *description)
{
    return;
}

void
error_handler (ThermlParser *p, int severity, int linenum, int colnum, const char *description)
{
    printf("Error line %d, col %d: %s\n", linenum, colnum, description);
}

void
pcommand_handler (ThermlParser *p, RBuf *data)
{
    printf ("Preprocessing command:\n");
    printf ("---------------------------------------\n");
    printf ("%s\n", rbuf_str(data));
    printf ("---------------------------------------\n");
}

void
tag_start_handler (ThermlParser *parser, RBuf *type, RBHash *attribs)
{
    ThermlAttribute *attrib;

    printf ("New tag type '%s'\n", rbuf_str (type));

    RBHASH_FOREACH (attribs, attrib) {
        
        printf ("  '%s'='%s'\n",
                rbuf_str (rbhash_entry_getkey (attrib)),
                rbuf_str (attrib->value));
        
    } RFOREACH_CLOSE;
                
    printf ("\n");
}

void
tag_end_handler (ThermlParser *p, RBuf *name, RBuf *data_attr)
{
    printf ("** End tag: '%s'\n", rbuf_str(name));
    if (rbuf_not_empty (data_attr)) {
        printf ("Associated Data for '%s': '%s'\n", rbuf_str(name), rbuf_str(data_attr));
        rbuf_set_rdonly (data_attr);
    }
}



void
parse_file (ThermlParser *parser, char *name)
{
    FILE *p;
    char buf[FILE_BUFFER_SIZE], buf2[20];
    int read;


    p = fopen (name,"r");
    if (!p) {
        printf ("can't open file '%s', failing.\n", name);
        exit (-1);
    }


    while (!feof (p)) {
        read = fread (buf, 1, FILE_BUFFER_SIZE, p);
        sprintf (buf2, "%%%ds\n", read);
        
        if (therml_parse_chunk (parser, buf, read, read != FILE_BUFFER_SIZE) < 0) {
            fclose (p);
            printf ("Aborting file read due to errors\n");
            return;
        }
    }
    fclose(p);
}

/* When you do the loop test, no handlers etc. are hooked up, so it's the raw
 * speed of the parser, with no output to stdio etc. */

void
#if DEBUG
loop_test (char *filename, int do_debug)
#else
loop_test (char *filename)
#endif
{  
    int i;

    printf ("looping 10000 times\n");

    for (i=0; i < 10000; i++)
    {
        ThermlParser *parser;

        parser = therml_parser_new ();
        parse_file (parser, filename);
        therml_parser_free (parser);  
    }
}


int
main (int argc, char **argv) 
{
    ThermlParser *parser;
    int do_speedtest = FALSE;
    int do_loop = FALSE;
    char *filename = NULL;
#if DEBUG
    int do_debug = FALSE;
    char *optarg;
#endif


    RARGV_SWITCH 
    {
        case 'l':
            do_loop = TRUE;
            break;
        case 's':
            do_speedtest = TRUE;
            break;
#if DEBUG
        case 'd':
            printf ("Turning parser debugging on.\n");
            do_debug = TRUE;
            break;
        case 'm':
            optarg = RARGV_NEXT;
            printf ("Setting rmdbg output to %s\n", optarg);
            rmdbg_set_outfile (optarg);
            break;
#endif
        case '*':
            if (filename)
                printf ("Multiple files specified, ignoring previous\n");

            filename = RARGV_CURRENT;
            break;

        default:
            printf ("ignoring invalid option: %s\n", RARGV_CURRENT);
            break;
    }
    RARGV_CLOSE;

    rinit ();

    parser = therml_parser_new ();

    if (do_speedtest)
    {
        therml_set_error_handler (parser, fast_error_handler);
    }
    else
    {
        therml_set_error_handler (parser, error_handler);
        therml_set_pcommand_handler (parser, pcommand_handler);
        therml_set_node_start_handler (parser, tag_start_handler);
        therml_set_node_end_handler (parser, tag_end_handler);
    }


    if (filename == NULL)
    {
        printf("No file specified, will parse default:\n");
        printf(TESTCHUNK);
        printf("\n\n\n");

        therml_parse_chunk (parser, TESTCHUNK, strlen (TESTCHUNK), 1);

        therml_parser_free (parser);
        return 0;
    }


    if (TRUE == do_loop)
    {
#if DEBUG
        loop_test (filename, do_debug);
#else
        loop_test (filename);
#endif
    }
    else
        parse_file (parser, filename);

    therml_parser_free (parser);

    /*  if (do_speedtest)
        printf ("Errors were: %s\n", speed_error);
     */

    rcleanup ();
    
    return (0);
}



