
/*
 * Copyright (C) 1999-2001, Ian Main <imain@stemwinder.org>.
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject
 * to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * 
 */

#include <roy.h>

#define RBHASH_MIN_SIZE 4

/* Define the growth rate of the table.  A growth rate of
 * 1 will mean we will have at least as many slots available
 * as we do table entries (a 1 to 1 ratio).  Setting this
 * value to 2, would cause 2 entries per hash table slot
 * etc. */
#define RBHASH_GROW_RATIO 4

/* Define the shrink ratio.  This sets how far down you have to
 * go before the table will be shrunken.  Setting this to 2 for
 * example, will make the table shrink when less than 1/2 of the
 * hash table slots are used */
#define RBHASH_SHRINK_RATIO 2


#define RBHASH_INSERT_LIST(head,entry) \
entry->rbhash_header.next = head; \
head = entry; \
entry->rbhash_header.prev_next_ptr = &head; \
if (entry->rbhash_header.next) \
    entry->rbhash_header.next->rbhash_header.prev_next_ptr = &entry->rbhash_header.next


static unsigned int rbhash_primes[] =
{
    3,
    7,
    11,
    19,
    37,
    73,
    109,
    163,
    251,
    367,
    557,
    823,
    1237,
    1861,
    2777,
    4177,
    6247,
    9371,
    14057,
    21089,
    31627,
    47431,
    71143,
    106721,
    160073,
    240101,
    360163,
    540217,
    810343,
    1215497,
    1823231,
    2734867,
    4102283,
};

static unsigned int rbhash_num_primes = sizeof (rbhash_primes) / sizeof (unsigned int);


/* Truly frightning stuff.. :) */

#define RBHASH_SET_HASHVAL(hashval, buf) \
    hashval = 0; \
    if (rbuf_len (buf) >= sizeof (int)) { \
	const unsigned int *i; \
        unsigned int num_ints = rbuf_len (buf) / sizeof (int); \
        for (i = ((unsigned int *) rbuf_str (buf)); num_ints--; i++) { \
            hashval += (hashval << 3) + *i; \
        } \
    } else { \
        const char *p; \
        for (p = rbuf_str (buf); *p != '\0'; p++)  \
            hashval = (hashval << 3) + *p; \
    }



RBHash *
rbhash_new (void)
{
    RBHash *hash;
    /* hash = rchunk_alloc (sizeof (RBHash)); */
    hash = rmem_new0 (RBHash, 1);
    RBHASH_INIT (hash);
    return (hash);
}

void
rbhash_free (RBHash *hash)
{
    RBHASH_DESTRUCT (hash);
    /* rchunk_free (hash, sizeof (RBHash)); */
    rmem_free (hash);
}

static void
rbhash_table_resize (RBHash *hash)
{
    unsigned int new_size = RBHASH_MIN_SIZE;
    unsigned int old_size;
    unsigned long index;
    RBHashEntry **old_entries;
    unsigned int i = 0;

    /* Calculate new prime size for table */
    while ((rbhash_primes[i] < hash->num_entries) && (i < rbhash_num_primes)) {
        i++;
    }
    new_size = rbhash_primes[i];

    old_entries = hash->entries;
    old_size = hash->table_size;

    hash->entries = (RBHashEntry **) rmem_alloc0 (sizeof (RBHashEntry *) * new_size);

    hash->table_size = new_size;

    /* walk through the old hash table entries, and just insert them into
     * the new table */
    for (i = 0; i < old_size; i++) {
        RBHashEntry *next;
        RBHashEntry *ent = old_entries[i];

        while (ent) {
            index = ent->rbhash_header.hashval % hash->table_size;
            /* Save 'next' pointer because we have to change it below
             * for its new home */
            next = ent->rbhash_header.next;
            RBHASH_INSERT_LIST (hash->entries[index], ent);
            ent = next;
        }
    }

    /* free the old table */
    if (old_entries)
        rmem_free (old_entries);
}

void *
rbhash_remove (RBHash *hash, RBuf *key)
{
    RBHashEntry *entry;
    unsigned long index;
    unsigned long hashval;

    if (hash->num_entries < (hash->table_size / RBHASH_SHRINK_RATIO))
        rbhash_table_resize (hash);

    RBHASH_SET_HASHVAL (hashval, key);

    /* Don't cause a SIGFPE if not necessary. */
    if (0 == hash->table_size)
        return NULL;

    index = hashval % hash->table_size;

    for (entry = hash->entries[index]; entry != NULL; entry = entry->rbhash_header.next) {
        if (hashval == entry->rbhash_header.hashval) {
            const RBuf *buf1 = rbhash_entry_getkey (entry);
            const RBuf *buf2 = key;
            unsigned int *b1;
            unsigned int *b2;
            unsigned char *p1;
            unsigned char *p2;
            unsigned int i;
            unsigned int num_ints;

            /* quick and easy check that might catch a lot of them without
             * having to walk the strings */
            if (buf1->len != buf2->len)
                goto nogood;

            b1 = ((unsigned int *) rbuf_str (buf1));
            b2 = ((unsigned int *) rbuf_str (buf2));

            num_ints = rbuf_len (buf1) / sizeof (int);
            for (i = num_ints; i--; b1++, b2++) {
                if (*b1 != *b2)
                    goto nogood;
            }

            /* b1 and b2 now point to the last bit of the string, 
             * so we can just assign our pointers to them, and compare
             * from there */
            i = rbuf_len (buf1) - (num_ints * sizeof (int));

            p1 = (unsigned char *) b1;
            p2 = (unsigned char *) b2;

            for (; i--; p1++, p2++) {
                if (*p1 != *p2)
                    goto nogood;
            }

            rbhash_remove_entry (hash, entry);
            return (entry);
        }
nogood:
	/* Suns broken compiler won't allow a tag right before a }, so we
	 * add this as a fix */
	;
    }

    return (NULL);
}

void *
rbhash_lookup (RBHash *hash, RBuf *key)
{
    RBHashEntry *entry;
    unsigned long index;
    unsigned long hashval;

    if (hash->table_size == 0)
        return (NULL);

    RBHASH_SET_HASHVAL (hashval, key);

    index = hashval % hash->table_size;
    for (entry = hash->entries[index]; entry != NULL; entry = entry->rbhash_header.next) {
        if (hashval == entry->rbhash_header.hashval) {
            const RBuf *buf1 = rbhash_entry_getkey (entry);
            const RBuf *buf2 = key;
            unsigned int *b1;
            unsigned int *b2;
            unsigned char *p1;
            unsigned char *p2;
            unsigned int i;
            unsigned int num_ints;

            /* quick and easy check that might catch a lot of them without
             * having to walk the strings */
            if (buf1->len != buf2->len)
                goto nogood;

            b1 = ((unsigned int *) rbuf_str (buf1));
            b2 = ((unsigned int *) rbuf_str (buf2));

            num_ints = rbuf_len (buf1) / sizeof (int);
            for (i = num_ints; i--; b1++, b2++) {
                if (*b1 != *b2)
                    goto nogood;
            }

            /* b1 and b2 now point to the last bit of the string, 
             * so we can just assign our pointers to them, and compare
             * from there */
            i = rbuf_len (buf1) - (num_ints * sizeof (int));

            p1 = (unsigned char *) b1;
            p2 = (unsigned char *) b2;

            for (; i--; p1++, p2++) {
                if (*p1 != *p2)
                    goto nogood;
            }

            return (entry);
        }
nogood:
	/* Suns broken compiler won't allow a tag right before a }, so we
	 * add this as a fix */
	;
    }

    return (NULL);
}

void *
rbhash_lookup_next (RBHash *hash, void *prev_entry, RBuf *key) 
{
    RBHashEntry *entry = prev_entry;
    unsigned long hashval = entry->rbhash_header.hashval;

    for (entry = entry->rbhash_header.next; entry != NULL; entry = entry->rbhash_header.next) {
        if (hashval == entry->rbhash_header.hashval) {
            const RBuf *buf1 = rbhash_entry_getkey (entry);
            const RBuf *buf2 = key;
            unsigned int *b1;
            unsigned int *b2;
            unsigned char *p1;
            unsigned char *p2;
            unsigned int i;
            unsigned int num_ints;

            /* quick and easy check that might catch a lot of them without
             * having to walk the strings */
            if (buf1->len != buf2->len)
                goto nogood;

            b1 = ((unsigned int *) rbuf_str (buf1));
            b2 = ((unsigned int *) rbuf_str (buf2));

            num_ints = rbuf_len (buf1) / sizeof (int);
            for (i = num_ints; i--; b1++, b2++) {
                if (*b1 != *b2)
                    goto nogood;
            }

            /* b1 and b2 now point to the last bit of the string, 
             * so we can just assign our pointers to them, and compare
             * from there */
            i = rbuf_len (buf1) - (num_ints * sizeof (int));

            p1 = (unsigned char *) b1;
            p2 = (unsigned char *) b2;

            for (; i--; p1++, p2++) {
                if (*p1 != *p2)
                    goto nogood;
            }

            return (entry);
        }
nogood:
	/* Suns broken compiler won't allow a tag right before a }, so we
	 * add this as a fix.. umm, aparently gcc has depricated this now. */
	;
    }

    return (NULL);
}

void
rbhash_insert (RBHash *hash, void *inentry, RBuf *key)
{
    RBHashEntry *entry = inentry;
    unsigned long hashval;
    unsigned long index;

    /* Save key in entry */
    rbhash_entry_setkey ((RBHashEntry *) inentry, key);

    if (hash->num_entries >= (hash->table_size * RBHASH_GROW_RATIO))
        rbhash_table_resize (hash);

    RBHASH_SET_HASHVAL (hashval, key);

    entry->rbhash_header.hashval = hashval;
    index = hashval % hash->table_size;

    /* stick it in the front of the list */
    RBHASH_INSERT_LIST (hash->entries[index], entry);

    hash->num_entries++;
}



