#include <rumba/manifoldFile.h>
#include <rumba/point.hpp>
#include <rumba/exception.h>
#include <rumba/dl.h>

#include <rumba/iohandler_base.h>

#include <config.h>

using RUMBA::ManifoldFile;

const int RUMBA::CACHE_VOL = 1;
const int RUMBA::CACHE_TIME = 2;
const int RUMBA::CACHE_NONE = 3;


std::list<ManifoldFile*>& RUMBA::ManifoldFile::TypeList()
{
	static std::list<ManifoldFile*> _typelist;
	return _typelist;
}

std::list<void*>& handle_list()
{
	static std::list<void*> L;
	return L;
}

ManifoldFile RUMBA::ManifoldFileInstance(Exemplar("ManifoldFile"), "ManifoldFile" );
ManifoldFile* RUMBA::ManifoldFileExemplar=&RUMBA::ManifoldFileInstance;

ManifoldFile*
ManifoldFile::construct(std::string filename, std::ios_base::openmode mode, const BaseManifold* manifoldPtr )
{
	ManifoldFile* retval = 0;
//	extern ManifoldFile* ManifoldFileExemplar;
	if ( TypeList().empty() )
	{
		throw RUMBA::Exception("Fatal -- no filetypes available");
	}	
	std::list<ManifoldFile*>::const_iterator it;
	for ( it = TypeList().begin(); it != TypeList().end(); ++it )  
	{
		if ( *it != ManifoldFileExemplar ) 
		{
			if ( (*it)->isMine(filename)) 
			{
				retval = (*it)->make(filename,mode,manifoldPtr);	
				break;
			}
		}
	}
	return retval;
// }}}
}


ManifoldFile*
ManifoldFile::construct(std::string filename, std::string datatype_request, const intPoint& e)
{
	ManifoldFile* retval = 0;
//	extern ManifoldFile* ManifoldFileExemplar;
	if ( TypeList().empty() )
	{
		throw RUMBA::Exception("Fatal -- no filetypes available");
	}	
	std::list<ManifoldFile*>::const_iterator it;
	for ( it = TypeList().begin(); it != TypeList().end(); ++it )  
	{
		if ( *it != ManifoldFileExemplar ) 
		{
			if ( (*it)->isMine(filename)) 
			{
				retval = (*it)->make(filename,datatype_request,e);	
				break;
			}
		}
	}
	return retval;
// }}}
}

// Provide a sane default in the base class.

bool
RUMBA::ManifoldFile
::isMine(std::string )
{//{{{
	return false;
}//}}}

RUMBA::ManifoldFile::
ManifoldFile(std::string classname) 
: Data(0), classname(classname), Cache(0), CacheStrategy(CACHE_NONE)
{}

RUMBA::ManifoldFile* 
RUMBA::ManifoldFile::
getNew()
{
	return new ManifoldFile("ManifoldFile");
}


RUMBA::ManifoldFile::
ManifoldFile(Exemplar, std::string classname) 
: Data(0), classname(classname), Cache(0), CacheStrategy(CACHE_NONE)

// {{{
{
	static bool called = false;
	TypeList().push_front(this);
	if (!called)
	{
		called = true;
		handle_list() = dl_init();	// if it's the first time we're called, load dl modules.
	}
}
// }}}

RUMBA::ManifoldFile* 
RUMBA::ManifoldFile
::make(  std::string filename, std::ios_base::openmode mode, const BaseManifold* manifoldPtr )
//{{{
{
	log.logName() << "Entering " << classname << "::make( " << filename << " " << mode << ") " << "\n";
	log.logName() << "header extension: " << header_extension << "\n";
	ManifoldFile* res = getNew();
	// get header file
	std::string basename;

	basename.assign (filename, 0, filename.find_last_of("."));
	res->HeaderFile = basename + header_extension;
	res->DataFile = basename + data_extension;

	if ( mode & std::ios::in )
		res->loadHeader(res->HeaderFile.c_str());
	else if (manifoldPtr)
	{
		res->copyHeader ( *manifoldPtr );
		res->saveHeader(res->HeaderFile.c_str());
	}
	else
	{
		// throw an appropriate exception.
		;	
	}


	// Should move this elsewhere.
	res->initFile (mode);
	log.logName() << "Leaving " << classname << "::make() " << "\n";
	return res;

}
//}}}

RUMBA::ManifoldFile::~ManifoldFile()
{
	if (Data && Data->writeable())
	{
		writeCache();
	}

	if (Cache) delete Cache;
	delete Data;
}

RUMBA::ManifoldFile* 
RUMBA::ManifoldFile
::make(  std::string filename, std::string datatype_request, const intPoint& e )
//{{{
{
	log.logName() << "Entering " << classname << "::make( " << filename << " " << datatype_request << "\n";
	log.logName() << "header extension: " << header_extension << "\n";
	ManifoldFile* res = getNew();
	// get header file
	std::string basename;

	basename.assign (filename, 0, filename.find_last_of("."));
	res->HeaderFile = basename + header_extension;
	res->DataFile = basename + data_extension;

	res->setExtent(e);
	res->headerData()["normalized_datatype"] = datatype_request;
	res->saveHeader(res->HeaderFile.c_str());

	// Should move this elsewhere.
	try 
	{
		res->initFile (std::ios::out);
	}
	catch ( RUMBA::Exception& e )
	{
		delete res;
		res = 0;	
	}
	log.logName() << "Leaving " << classname << "::make() " << "\n";
	return res;

}
//}}}




//{{{
/* 
   This method can be used by file formats that lay out their data in a
   flat x,y,z,t file. This works for most formats -- eg Analyze, Stimulate, 
   Brik
*/
/*
template<class TYPE>
void 
RUMBA::ManifoldFile
::loadData( Manifold<TYPE> )
{
	cerr << "Calling ManifoldFile::loadData()\n";
		
}
*/
//}}}

	// Forwarded to IOHandler
void RUMBA::ManifoldFile::seekp (int pos) 
{ 
	Data->seekp(pos); 
}

void RUMBA::ManifoldFile::seekg (int pos) 
{ 
	Data->seekg(pos); 
}

		

void RUMBA::ManifoldFile::cGet (char*   dest, int nelts)
{ 
	Data->cGet(dest,nelts,LittleEndian);
}

void RUMBA::ManifoldFile::cGet (short*  dest, int nelts)
{ 
	Data->cGet(dest,nelts,LittleEndian);
}

void RUMBA::ManifoldFile::cGet (int*    dest, int nelts)
{ 
	Data->cGet(dest,nelts,LittleEndian);
}

void RUMBA::ManifoldFile::cGet (float*  dest, int nelts)
{ 
	Data->cGet(dest,nelts,LittleEndian);
}

void RUMBA::ManifoldFile::cGet (double* dest, int nelts)
{ 
	Data->cGet(dest,nelts,LittleEndian);
}

void RUMBA::ManifoldFile::cPut (char*   src, int nelts)
{ 
	Data->cPut(src,nelts,LittleEndian);
}

void RUMBA::ManifoldFile::cPut (short*  src, int nelts)
{ 
	Data->cPut(src,nelts,LittleEndian);
}

void RUMBA::ManifoldFile::cPut (int*    src, int nelts)
{ 
	Data->cPut(src,nelts,LittleEndian);
}
void RUMBA::ManifoldFile::cPut (float*  src, int nelts)
{ 
	Data->cPut(src,nelts,LittleEndian);
}
void RUMBA::ManifoldFile::cPut (double* src, int nelts)
{ 
	Data->cPut(src,nelts,LittleEndian);
}


RUMBA::BaseManifold*
RUMBA::ManifoldFile
::get(int x, int y, int z, int t, int dx, int dy, int dz, int dt) const
{
//{{{
	return get (
			intPoint (x,y,z,t), 
			intPoint (dx,dy,dz,dt) 
		   );
// }}}
};

RUMBA::BaseManifold*
RUMBA::ManifoldFile
::get(const intPoint& o, const intPoint& ex) const
{
#ifdef DEBUG
	log.logName() << "get() : Skip is :" << Skip.x() << "," << Skip.y() << "," << Skip.z() << "," << Skip.t() << "\n";
#endif
	BaseManifold* result = Data->get(o, ex, Skip, LittleEndian);
	result->copyHeader  (*this);
	result->setExtent(ex);
	return result;
}

void RUMBA::ManifoldFile::get ( 
		const intPoint& o, 
		const intPoint& ex, 
		BaseManifold* res
) const
{
	Data->get(o, ex, Skip, LittleEndian, res);
}

void
RUMBA::ManifoldFile
::get(
		int x, int y, int z, int t, 
		int dx, int dy, int dz, int dt, 
		BaseManifold* res
		) const
{
//{{{
	get (
			intPoint (x,y,z,t), 
			intPoint (dx,dy,dz,dt),
			res
		   );
// }}}
};



void
RUMBA::ManifoldFile
::put(BaseManifold* M, const intPoint& o )
{
	//{{{
	// define this !
	/**
	Data->seekp(x*SkipX+y*SkipY+z*SkipZ+t*SkipT);
	int dx=M->width();
	int dy=M->height();
	int dz=M->depth();
	int dt=M->timepoints();

	for ( int i=0; i<dt; ++i)
		for (int j=0; j<dz; ++j)
			for (int k=0; k<dy; ++k)
			{
				Data->seekp(x*SkipX+(y+k)*SkipY+(z+j)*SkipZ+(t+i)*SkipT);
//				Data->cPut(M,x,dx);
			}
			*/

	Data->put(M,o,M->extent(),Skip,LittleEndian);
	//}}}
}

void
RUMBA::ManifoldFile
::put(BaseManifold* M, int x,int y,int z,int t)
{
	intPoint p(x,y,z,t);
	put(M,p);
}

void 
RUMBA::ManifoldFile
::print(std::ostream& out) const
{
	//{{{
	out << "Dimensions: " 
		<< Extent.x() << "x" 
		<< Extent.y() << "x" 
		<< Extent.z() << "x" 
		<< Extent.t() << "\n";

	out << "Spatial voxel size: " 
		<< VoxelSize.x() << "x" 
		<< VoxelSize.y() << "x" 
		<< VoxelSize.z() <<  "\n";
	
	std::map<std::string, RUMBA::Splodge>::const_iterator it;

	for ( it = HeaderData.begin(); it != HeaderData.end(); ++it )
		out << it->first << ": " << it->second << "\n";
	//}}}
}

double 
RUMBA::ManifoldFile::getElementDouble(int pos) const 
//{{{
{ 
	if (!cached(pos)) 
		setCachePoint(pos);
	switch ( CacheStrategy )
	{
		case CACHE_NONE:
			double res; 
			Data->seekg(pos);
			Data->cGet(&res,1,LittleEndian);
			return res;
		case CACHE_VOL:
			if (!cached(pos)) 
				setCachePoint(pos);
			return Cache->getElementDouble(pos%pixels());
		case CACHE_TIME:
			if (!cached(pos)) 
				setCachePoint(pos);
			return Cache->getElementDouble(pos%Skip.z()+Skip.z()*(pos/Skip.t()));
		default:
			throw RUMBA::Exception("Default fall-through in getElementDouble()");
	}
	return Cache->getElementDouble(pos);
}
///}}}

double 
RUMBA::ManifoldFile::getElementDouble(int x, int y, int z, int t) const
{//{{{ 
	intPoint p(x,y,z,t);
	if (!cached(p)) 
		setCachePoint(p);
	switch ( CacheStrategy )
	{
		case CACHE_NONE:
			double res; 
			Data->seekg(inner_product(p,Skip));
			Data->cGet(&res,1,LittleEndian);
			return res;
		case CACHE_VOL:
			return Cache->getElementDouble(x,y,z,0);
		case CACHE_TIME:
			return Cache->getElementDouble(x,y,0,t);
		default:
			throw RUMBA::Exception("Default fall-through in getElementDouble()");
	}
}//}}}

double 
RUMBA::ManifoldFile
::getElementDouble(const intPoint& p) const
{//{{{ 
	if (!cached(p)) 
		setCachePoint(p);
	switch ( CacheStrategy )
	{
		case CACHE_NONE:
			double res; 
			Data->seekg(inner_product(p,Skip));
			Data->cGet(&res,1,LittleEndian);
			return res;
		case CACHE_VOL:
			return Cache->getElementDouble(p.x(),p.y(),p.z(),0);
		case CACHE_TIME:
			return Cache->getElementDouble(p.x(),p.y(),0,p.t());
		default:
			throw RUMBA::Exception("Default fall-through in getElementDouble()");
	}	
}//}}}


int 
RUMBA::ManifoldFile
::getElementInt(int pos) const
{ //{{{
	int res; 
	if (!cached(pos))
		setCachePoint(pos);
	Data->seekg(pos);
	Data->cGet(&res,1,LittleEndian);
	return res;
} //}}}

int 
RUMBA::ManifoldFile
::getElementInt(int x, int y, int z, int t) const
{ //{{{
	int res; 
	intPoint p(x,y,z,t);
	if (!cached(p))
		setCachePoint(p);
	Data->seekg(inner_product(p,Skip));
	Data->cGet(&res,1,LittleEndian);
	return res;
}//}}}

int 
RUMBA::ManifoldFile
::getElementInt(const intPoint& p) const
{ //{{{
	int res; 
	if (!cached(p))
		setCachePoint(p);
	Data->seekg(inner_product(p,Skip));
	Data->cGet(&res,1,LittleEndian);
	return res;
}//}}}


void 
RUMBA::ManifoldFile
::setElementDouble(int pos, double val)
{ //{{{

	if (!cached(pos)) 
		setCachePoint(pos);
	switch ( CacheStrategy )
	{
		case CACHE_NONE:
			Data->seekp(pos);
			Data->cPut(&val,1,LittleEndian);
			break;
		case CACHE_VOL:
			if (!cached(pos)) 
				setCachePoint(pos);
			Cache->setElementDouble(pos%pixels(),val);
			break;
		case CACHE_TIME:
			if (!cached(pos)) 
				setCachePoint(pos);
			Cache->setElementDouble(pos%Skip.z()+Skip.z()*(pos/Skip.t()),val);
			break;
		default:
			throw RUMBA::Exception("Default fall-through in getElementDouble()");
	}

}//}}}

void 
RUMBA::ManifoldFile
::setElementDouble(int x, int y, int z, int t, double val)
{ //{{{
	Data->seekp(inner_product(intPoint(x,y,z,t),Skip));
	Data->cPut(&val,1,LittleEndian);
}//}}}

void 
RUMBA::ManifoldFile
::setElementDouble(const intPoint& p, double val)
{ //{{{
	Data->seekp(inner_product(p,Skip));
	Data->cPut(&val,1,LittleEndian);
}//}}}



void 
RUMBA::ManifoldFile
::setElementInt(int pos, int val)
{ //{{{
	Data->seekp(pos);
	Data->cPut(&val,1,LittleEndian);
} //}}}

void 
RUMBA::ManifoldFile
::setElementInt(int x, int y, int z, int t, int val)
{ //{{{
	Data->seekp(inner_product(intPoint(x,y,z,t),Skip));
	Data->cPut(&val,1,LittleEndian);
} //}}}

void 
RUMBA::ManifoldFile
::setElementInt(const intPoint& p, int val)
{ //{{{
	Data->seekp(inner_product(p,Skip));
	Data->cPut(&val,1,LittleEndian);
} //}}}


void RUMBA::ManifoldFile::loadData(ManifoldFile* )
{

}

void ManifoldFile::saveData(ManifoldFile* f) const
{
#ifdef DEBUG
	log.logName() << "ManifoldFile::Calling saveData\n";
	log.logName() << "Extents: " << Extent.x() << "," << Extent.y() << "," 
		<< Extent.z() << "," << Extent.t() << "\n";
#endif
	BaseManifold* M;
	intPoint counter(0,0,0,0);
	for ( counter.t()=0; counter.t() < Extent.t(); ++counter.t() )
		for ( counter.z() =0; counter.z() < Extent.z(); ++ counter.z() )
			for ( counter.y() = 0; counter.y() < Extent.y(); ++counter.y() )
			{
				M = get (0,counter.y(),counter.z(),counter.t(),Extent.x(),1,1,1);
				f->put ( M,0,counter.y(),counter.z(),counter.t());
				delete M;
			}
}

// initialise Data
void ManifoldFile::initFile(std::ios_base::openmode mode)
{
	log.logName() << "Calling initFile()" << "\n";
	Factory* f = 0;	

	if ( ! HeaderData.count("normalized_datatype") )
	{
		log.logName() << "Warning: no normalized_datatype, using datatype." << "\n";
		HeaderData["normalized_datatype"] = normalizeDataType ( HeaderData["datatype"].asString() );
	}
	log.logName() <<  "normalize data type: " 
		<< HeaderData["normalized_datatype"].asString() << "\n";

	f = Factory::lookup ( HeaderData["normalized_datatype"].asString() );

	if (f) {
		Data = f->makeIOHandler ( DataFile.c_str(), size(), mode);
	}
	else 
		throw RUMBA::Exception ( "Null factory exception" );

	// else throw an exception !
	log.logName() << "Leaving initFile()" << "\n";
}

std::string ManifoldFile::normalizeDataType(std::string t)
{
	log.logName() << "ManifoldFile::normalizeDataType : t is " << t << "\n";
	for ( std::map<std::string, std::string>::const_iterator it = types.begin(); it != types.end();
			++it )
	{
		log.logName() << "it->first: " << it->first << "\n";
		log.logName() << "it->second: " << it->second << "\n";
		if ( t == it->second )
			return it->first;
	}

	// if type is already normalize, return it.
	for ( std::map<std::string, std::string>::const_iterator it = types.begin(); it != types.end();
			++it )
	{
		if ( t == it->first )
			return it->first;
	}
	log.logName() << "ManifoldFile::normalizeDataType failed" << t << "\n";
	return "";

}

bool RUMBA::ManifoldFile::cached(const intPoint& p) const
{
	switch(CacheStrategy)
	{
		case CACHE_VOL:
			return p.t()==CachePoint;
		case CACHE_TIME:
			return p.z()==CachePoint;
		case CACHE_NONE:
			return false;
		default:
			throw RUMBA::Exception ("Default fall-through in cached()");
	}
}

bool RUMBA::ManifoldFile::cached (int p) const
{
	switch(CacheStrategy)
	{
		case CACHE_VOL:
			return p/Skip.t()==CachePoint;
		case CACHE_TIME:
			return (p/Skip.z())%Extent.z() == CachePoint;
		case CACHE_NONE:
			return false;
		default:
			throw RUMBA::Exception ("Default fall-through in cached()");
	}
}

void RUMBA::ManifoldFile::setCachePoint(int p) const
{
	switch(CacheStrategy)
	{
		case CACHE_VOL:
			if (Cache&&Data->writeable())
				writeCache();
			CachePoint=p/pixels();
			Data->get(
					intPoint(0,0,0,CachePoint), 
					intPoint(Extent.x(),Extent.y(),Extent.z(),1),
					Skip,
					LittleEndian,
					Cache);
			break;
		case CACHE_TIME:
			if (Cache && Data->writeable())
				writeCache();
			CachePoint = (p/Skip.z())%Extent.z();
			Data->get(
					intPoint(0,0,CachePoint,0),
					intPoint(Extent.x(),Extent.y(),1,Extent.t()),
					Skip,
					LittleEndian,
					Cache);
			break;
		case CACHE_NONE:
			break;
		default:
			throw RUMBA::Exception ("Default fall-through in SetCachedPoint(int)");
	}
}

void RUMBA::ManifoldFile::setCachePoint(const intPoint& p) const
{
	switch(CacheStrategy)
	{
		case CACHE_VOL:
			CachePoint=p.t();
			if (Data->writeable())
				writeCache();
			Data->get(
					intPoint(0,0,0,CachePoint), 
					intPoint(Extent.x(),Extent.y(),Extent.z(),1),
					Skip,
					LittleEndian,
					Cache);
			break;
		case CACHE_TIME:
			CachePoint = p.z();
			if (Data->writeable())
				writeCache();
			Data->get( 
					intPoint(0,0,CachePoint,0), 
					intPoint(Extent.x(),Extent.y(),1,Extent.t()),
					Skip,
					LittleEndian,
					Cache);
			break;
		case CACHE_NONE:
			break;
		default:
			throw RUMBA::Exception ("Default fall-through in SetCachedPoint(int)");
	}
}

void RUMBA::ManifoldFile::writeCache() const
{
	switch (CacheStrategy)
	{
		case CACHE_VOL:
			const_cast<ManifoldFile*>(this)->put 
				(Cache, intPoint(0,0,0,CachePoint));
			break;
		case CACHE_TIME:
			const_cast<ManifoldFile*>(this)->put 
				(Cache, intPoint(0,0,CachePoint,0));
			break;
		case CACHE_NONE:
			break;
		default:
			throw RUMBA::Exception("Default fall-through in writeCache()");
	}
}

void RUMBA::ManifoldFile::setCacheStrategy(int x)
{
	CacheStrategy = x;
	if (Cache) delete Cache;
	intPoint p = Extent;
	switch (CacheStrategy)
	{
		case CACHE_VOL: 
			p.t() = 1;
			Cache = Data->get(intPoint(0,0,0,0), p, Skip, LittleEndian );
			CachePoint = 0;
			break;
		case CACHE_TIME:
			p.z() = 1;
			Cache = Data->get(intPoint(0,0,0,0),p, Skip, LittleEndian );
			CachePoint = 0;
			break;
		case CACHE_NONE:
			Cache = 0;
			break;
		default:
			throw 
				RUMBA::Exception("Default fall-through in setCacheStrategy()");
	}
}


