#include <rumba/manifold.h>
#include <rumba/manifoldFile.h>

/**
  * Interface class for mask. the function mask uses this as a callback.
  * A mask is simply a manifold consisting of true/false values, represented
  * as type Manifold<char>. A mask can be used to perform a computation
  * over a subset that meets a certain criteria. Note that we use the term
  * "mask" to describe a spatial mask. So the criteria used to compute a
  * mask typically uses some aggregate of each time series.
  */

class MaskFunctor 
{
public:
	/**
	  * For a given pixel px, which should satisfy 0 <= px <= M->pixels(), 
	  * compute value of the mask at px.
	  */
	virtual bool operator()( int px, const RUMBA::BaseManifold* M ) =0;
};

/**
  * Given a BaseManifold and a MaskFunctor, compute an appropriate mask.
  */
RUMBA::Manifold<char> mask ( MaskFunctor& f, const RUMBA::BaseManifold* M );


/**
  * Threshold functor. This acts as a callback used by anything that needs
  * to threshold. A threshold T is a function that does the following:
  * 
  \code T(x) \endcode
  * would return either x, or some other value if x lay outside a certain 
  * range. In particular, consider the range [min,max]. Then T(x) will either
  * be x or a fixed constant, minreturn, for any x < min, depending on the
  * type of threshold T(x) is. Likewise, 
  * if min <= x <= max, then T(x) = x or T(x) = midreturn, and for x > max,
  * T(x) = maxreturn or T(x) = x. 
  */
class Threshold 
{
public:
	Threshold 
	( 
	 /** 
	   * Pointer to min value. Use a NULL pointer to avoid specifying a min.
	   */
	 const double* min, 
	 /** 
	   * Pointer to max value. Use a NULL pointer to avoid specifying a max.
	   */
	 const double* max, 
	 /**
	   * Pointer to fixed constant that operator()(x) returns if x<(*min).
	   * NULL if operator()(x) should return x for x<(*min)
	   */
	 const double* minreturn, 
	 /**
	   * Pointer to fixed constant that operator()(x) returns if 
	   *  (*min) <= x <= (*max).
	   * NULL if operator()(x) should return x for (*min)<=x<=(*max)
	   */	 
	 const double* midreturn, 
	 /**
	   * Pointer to fixed constant that operator()(x) returns if x>(*max).
	   * NULL if operator()(x) should return x for x>(*max)
	   */	 const double* maxreturn
	 );
	 /**
	   * Callback. Given a double, x, apply the threshold to x.
	   */
	double operator() (double x);
	~Threshold();

private:
	Threshold(const Threshold&);
	Threshold& operator=(const Threshold&);

	double *min; 
	double *max; 
	double *minreturn; 
	double *midreturn; 
	double *maxreturn;
};


/**
  * The threshold mask uses a threshold to decide whether or not a given
  * pixel should be masked. Use pMin and pMax to specify the range that
  * should be used. Call mask() on a ThresholdMask object to create a mask.
  */
class ThresholdMask : public MaskFunctor
{
public:
	/**
	  *  [*pMin,*pMax] is the range that is not masked. Use null pointers
	  * to specify an open range, eg if pMin is NULL, the range is 
	  * (-infinity, *pMax]
	  */
	ThresholdMask (double * pMin, double * pMax) : 
	T(pMin, pMax, &value_false, &value_true, &value_false)
	{}

	virtual bool operator() ( int px, const RUMBA::BaseManifold* M )
	{	
		return T(M->getElementDouble(px))!=value_false;
	}
	virtual ~ThresholdMask() {} 
private: 
	Threshold T;
	ThresholdMask(const ThresholdMask&);
	ThresholdMask& operator=(const ThresholdMask&);
	static const double value_false ;
	static const double value_true ;
};

/**
  * Apply a mask to a data set, and produce a squashed data set that 
  * excludes the masked pixels. The resulting data set can be used 
  * for pixel-wise computations and has the distinct advantage of being
  * smaller. There is sufficient information in the mask and squashed 
  * image to recover an image the same size as the original, the only 
  * information lost is taht contained in the masked pixels.
  *
  * \param mask a mask to be applied to the input 
  * \param in the input data
  * \param out the output data. Needs to have the same number of pixels 
  * as the mask.
  */
void squash 
( 
 const RUMBA::Manifold<char>& mask, 
 const RUMBA::BaseManifold* in, 
 RUMBA::BaseManifold* out
);


/**
  * Recover an unsquashed image from a squashed one. Note that the mask 
  * used in the squashing is needed to do this.
  * \param mask The mask used to generate the squashed manifold
  * \param in A squashed manifold
  * \param out where to put the unsquashed manifold.
  */
void
unsquash
( 
 const RUMBA::Manifold<char>& mask, 
 const RUMBA::BaseManifold* in,
 RUMBA::BaseManifold* out
 );



/**
  * Returns appropriate dimensions for a squashed manifold. In particular,
  * width and height are unchanged, depth is adjusted.
  */
RUMBA::intPoint squashSize 
(const RUMBA::Manifold<char>& M, RUMBA::BaseManifold* N);

/**
  * Returns appropriate dimensions for an unsquashed manifold
  */
inline RUMBA::intPoint 
unSquashSize(const RUMBA::Manifold<char>& M, RUMBA::BaseManifold* N)
{
	RUMBA::intPoint r(M.extent()); r.t()=N->timepoints(); return r;
}
