// rumbaview brainview.cpp (authors: Giorgio Grasso, Ben Bly, Donovan Rebbechi)

#include <rumba/util.hpp>

#include <qimage.h>
#include <qkeycode.h>
#include <qfiledialog.h>
#include <qmessagebox.h>

#include <math.h>
#include <colormaplist.h>
#include <colormap.h>

#include "damnwindowhack.h"

#include "brainview.h"

/*
#include "grey.xpm"
#include "hotmetal.xpm"
#include "jet.xpm"
*/

#include "rumbaview.xpm"

#undef DEBUG

using namespace RUMBA;

/**
  *@author Giorgio Grasso, Ben Bly, Donovan Rebbechi
  */

BrainView::BrainView(QWidget *parent, View v, const char *name, float zoom,bool s, bool param_mainview, dPoint scale) : 
QFrame(parent),
	n_cmap(0),
	thresh_min(0), thresh_max(255),
	view(v), underlay(false), crosshair(true), 
	mainview(param_mainview), 
	coordinates(param_mainview?true:false), threshold(false),
	shown(false), animate(false),stretch(s), 	
	mag_x(zoom), mag_y(zoom),magnification(zoom), 
	gamma(1), contrast(1), Scale(scale)
{
/*
  view = v;
  underlay = false;	
  crosshair = true;	
  threshold = false;	
  stretch = s;
  if (mainview)
    coordinates = true;	
  else
    coordinates = false;	
  shown = false;
  animate = false;
  magnification=zoom;
  mag_x = mag_y = magnification;
  gamma = contrast = 1;
  thresh_min = 0;
  thresh_max = 255;
  n_cmap = 0;
*/
  // set caption for the window
  switch (view) {
  case AXIAL: sprintf(caption,"Axial"); break;
  case SAGITAL: sprintf(caption,"Sagital"); break;
  case CORONAL: sprintf(caption,"Coronal"); break;
  default: break;
  }
  sprintf(caption,"%s - %s",caption,name);
  setCaption(caption);

  // avoid segfault on the first refresh
  image = NULL;

  painter = new QPainter(this);
  animate_timer = new QTimer(this);
  connect(animate_timer,SIGNAL(timeout()),SLOT(updateAnimate()));
  setCursor(crossCursor);

  // set up colormaps
  initColormap();

  // set up colormap sub-menu
  cmap_menu = new QPopupMenu;

/*  
  id_cmap[n_cmap++] = id_grey = cmap_menu->insertItem("Grey",this,SLOT(slotGreyCmap()));
  id_cmap[n_cmap++] = id_hotmetal = cmap_menu->insertItem("Hotmetal",this,SLOT(slotHotmetalCmap()));
  id_cmap[n_cmap++] = id_jet = cmap_menu->insertItem("Jet",this,SLOT(slotJetCmap()));
*/
	id_cmap = new int[ colorMapList().size() ];
	for ( 
		vector<Colormap>::const_iterator it = colorMapList().begin();
		it != colorMapList().end(); 
		++it 
	)
	{
		id_cmap[n_cmap++] = cmap_menu->insertItem(it->name().c_str());
	}
	QObject::connect(cmap_menu, SIGNAL(activated(int)), this, SLOT(slotSetCmap(int)));


  // set up reset sub-menu
  reset_menu = new QPopupMenu;
  reset_menu->insertItem("All",this,SLOT(resetAll()));
  reset_menu->insertItem("Aspect",this,SLOT(resetAspect()));
  reset_menu->insertItem("Size",this,SLOT(resetSize()));
  reset_menu->insertItem("Brightness",this,SLOT(resetBrightness()));
  reset_menu->insertItem("Contrast",this,SLOT(resetContrast()));

  // set up popup menu
  menu = new QPopupMenu;
  menu->insertItem("&Save As...",this,SLOT(save()));
  if (mainview)
    menu->insertItem("&Save All...",this,SLOT(saveAll()));
  menu->insertSeparator();
  if (mainview) {
    id_threshold=menu->insertItem("&Threshold",this,SLOT(changeThreshold()));
    menu->setItemChecked(id_threshold,threshold);
    menu->setItemChecked(id_crosshair,crosshair);
    menu->insertItem("&Recompose",this,SLOT(recompose()));
    menu->insertItem("Colormap",cmap_menu);
    menu->insertSeparator();
  } else menu->insertItem("Colormap",cmap_menu);
  menu->insertItem("Reset",reset_menu);
  id_crosshair=menu->insertItem("&Crosshair",this,SLOT(changeCrossHair()));
  menu->setItemChecked(id_crosshair,crosshair);
  id_coordinates=menu->insertItem("Coordinates",this,SLOT(changeCoordinates()));
  menu->setItemChecked(id_coordinates,coordinates);
  id_animate=menu->insertItem("Animate",this,SLOT(changeAnimate()));
  if (mainview) {
    menu->insertSeparator();
    menu->insertItem("&Quit",this,SLOT(quit()));
  }

  // set up threshold slider
  if (mainview) {
    tslider = new ColorBar(this,Scale);
    tslider->hide();
    connect(tslider,SIGNAL(changeThreshold(int,int)),SLOT(slotChangeThreshold(int,int)));
  }

  setIcon(QPixmap((const char **) rumbaview));

}

BrainView::~BrainView(){
  if (painter) delete painter;
  if (menu) delete menu;
  if (tslider) delete tslider;
  if (mainview) exit(0);
}

void BrainView::setManifold(RUMBA::Manifold<short> *vol){
  volume = vol;

  // initialize cursor at the center of the manifold
  cx = volume->width()/2;
  cy = volume->height()/2;
  cz = volume->depth()/2;
  ct = 0;

  // assigning colormap
//  colormap = grey_cmap;
	colormap = &colorMapList()[0];
  cmap_menu->setItemChecked(id_cmap[0],true);
  if (mainview)
    tslider->setColormap(colormap);

  // set the initial slice view, resize the window and display
  updateImage();
  resize(image->width(),image->height());
  showImage();
}

void BrainView::setUnderlay(RUMBA::Manifold<short> *vol){
  underlay = true;
  underlay_volume = vol;

  // assigning colormaps
  // FIX ME !!!!
  colormap = &colorMapList()[0];
  if ( colorMapList().size() > 1 )
	  underlay_colormap = &colorMapList()[0];
  else
	  underlay_colormap = &colorMapList()[0];

  clearCmapMenu();
  cmap_menu->setItemChecked(id_hotmetal,true);
  if (mainview)
    tslider->setColormap(colormap);

  // set the initial slice view, resize the window and display
  updateImage();
  resize(image->width(),image->height());
  showImage();
}

void BrainView::initColormap(){
	/*
  QImage *cmap = new QImage((const char **)grey);	
  for (int i=0; i<256; i++)
    grey_cmap[i] = cmap->pixel(i,0);

  delete cmap;
  cmap = new QImage((const char **)hotmetal);
  for (int i=0; i<256; i++)
    hotmetal_cmap[i] = cmap->pixel(i,0);

  delete cmap;
  cmap = new QImage((const char **)jet);
  for (int i=0; i<256; i++)
    jet_cmap[i] = cmap->pixel(i,0);

  delete cmap;
  */
}

void BrainView::updateImage(){
  // update slice position and corresponding image
  int slice;
  switch (view) {
  case AXIAL: slice = cz; break;
  case SAGITAL: slice = cx; break;
//  case CORONAL: slice = cy; break;
  case CORONAL: slice = volume->height()-cy-1; break;
  default: slice = 0; break;
  }

  // get a fresh image, create an empty one if not there already
  if (!image) image = new QImage(width(), height(), 32, 1<<24);
  if (underlay && !underlay_image)
    underlay_image = new QImage(width(), height(), 32, 1<<24);
  getSlice(slice);
}

void BrainView::getSlice(int slice ){
  int vw, vh, vd;
  int width, height, depth;
  int skip_x, skip_y, skip_z;
  int skip;
  float scale_x, scale_y, scale_z;

  int uvw, uvh, uvd;
  int uwidth, uheight, udepth;
  int uskip_x, uskip_y, uskip_z;
  int uskip, uslice;
  float uscale_x, uscale_y, uscale_z;

  long color, pos;
  long intensity;

  QImage simg;

  // initialize variables to make compiler happy
  width = height = 0;
  skip_x = skip_y = skip_z = 0;
  scale_x = scale_y = scale_z = 1;
  uvw = uvh = uvd = 0;
  uwidth = uheight = 0;
  uskip_x = uskip_y = uskip_z = 0;
  uscale_x = uscale_y = uscale_z = 1;
  uskip = 0;

  // grab the manifold sizes
  vw = volume->width();
  vh = volume->height();
  vd = volume->depth();

  // if overlay do the appropriate thing
  if (underlay) {
    uvw = underlay_volume->width();
    uvh = underlay_volume->height();
    uvd = underlay_volume->depth();
  }

  // Do the appropriate setup for different views
  switch (view) {
  case AXIAL:
#ifdef DEBUG
    cerr<<"AX. ";
#endif
    width = vw; height = vh; depth = vd;
    skip_x = 1; skip_y = vw; skip_z = vw*vh;
    scale_x = volume->voxelSize().x();
    scale_y = volume->voxelSize().y();
    scale_z = volume->voxelSize().z();
    if (underlay) {
      uwidth = uvw; uheight = uvh; udepth = uvd;
      uskip_x = 1; uskip_y = uvw; uskip_z = uvw*uvh;
      uscale_x = underlay_volume->voxelSize().x();
      uscale_y = underlay_volume->voxelSize().y();
      uscale_z = underlay_volume->voxelSize().z();
    }
    break;
  case SAGITAL:
#ifdef DEBUG
    cerr<<"SAG. ";
#endif
    width = vd; height = vh;depth = vw;
    skip_x = vw*vh; skip_y = vw; skip_z = 1;
    scale_x = volume->voxelSize().z();
    scale_y = volume->voxelSize().y();
    scale_z = volume->voxelSize().x();
    if (underlay) {
      uwidth = uvd; uheight = uvh; udepth = uvw;
      uskip_x = uvw*uvh; uskip_y = uvw; uskip_z = 1;
      uscale_x = underlay_volume->voxelSize().z();
      uscale_y = underlay_volume->voxelSize().y();
      uscale_z = underlay_volume->voxelSize().x();
    }
    break;
  case CORONAL:
#ifdef DEBUG
	cerr<<"COR. ";
#endif
	width = vw; height = vd; depth = vh;
    skip_x = 1; skip_y = vw*vh; skip_z = vw;
    scale_x = volume->voxelSize().x();
    scale_y = volume->voxelSize().z();
    scale_z = volume->voxelSize().y();
    if (underlay) {
      uwidth = uvw; uheight = uvd; udepth = uvh;
      uskip_x = 1; uskip_y = uvw*uvh; uskip_z = uvw;
      uscale_x = underlay_volume->voxelSize().x();
      uscale_y = underlay_volume->voxelSize().z();
      uscale_z = underlay_volume->voxelSize().y();
    }
    break;
  default: break;
  }

  // set the fixed skip
  // todo: add modulo arithmetic for ct in over and underlay...
  skip = (ct)*vw*vh*vd + slice*skip_z;
#ifdef DEBUG
  cerr<<"INFO:"<<endl<<"overlay width, height, depth = "<<width<<" "<<height<<" "<<depth<<endl;
  cerr<<"overlay xyz scale = "<<scale_x<<" "<<scale_y<<" "<<scale_z<<endl;
  cerr<<"skip_x, skip_y, skip_z = "<<skip_x<<" "<<skip_y<<" "<<skip_z<<" "<<endl;
  cerr<<"overlay skip, slice = "<<skip<<" "<<slice<<endl<<endl;
  cerr <<"size of overlay = " << volume->size() << endl;
#endif
  if (underlay) {
    
    if (stretch) 
		uslice = (int) ((float) slice * udepth / depth + .5 ) % udepth;
	else {
		uslice = (int) (udepth/2 + (slice - depth/2) * scale_z / uscale_z );
		if ( uslice < 0 ) uslice = 0;
		if ( uslice >= udepth ) uslice = udepth - 1;
	}
    uskip = uslice * uskip_z;
#ifdef DEBUG
    cerr<<"underlay width, height, depth = "<<uwidth<<" "<<uheight<<" "<<udepth<<endl;
    cerr<<"underlay xyz scale = "<<uscale_x<<" "<<uscale_y<<" "<<uscale_z<<endl;
    cerr<<"underlay skip, slice = "<<uskip<<" "<<uslice<<endl<<endl;
#endif
	assert ( uskip >= 0 );
	assert ( uskip < underlay_volume->size() );
  }

  // Build an image of the slice from the manifold data
  QImage img(width, height, 32, 1<<24);

  for (int i=0; i<width; i++){
    pos = skip + i*skip_x;
	//cerr << pos << endl;
	assert ( pos < volume->size() );
    for (int j=0; j<height; j++){
      intensity = volume->getElement(pos);
      if (thresh_min < thresh_max &&
	  (intensity < thresh_min || intensity > thresh_max)) intensity = 0;
      else if (thresh_min >= thresh_max &&
	       (intensity < thresh_min && intensity > thresh_max)) intensity = 0;
      color = (*colormap)[BrainView::adjust(intensity)];


//	  img.setPixel(i,j,color);
      img.setPixel(i,height - j -1,color);
	  assert ( pos < volume->size() );
      pos += skip_y;


    }
  }
  if (underlay) {
    // build an image of the underlay slice from the manifold data
    QImage uimg(uwidth, uheight, 32, 1<<24);
    for (int i=0; i<uwidth; i++){
      pos = uskip + i*uskip_x;
	  assert ( pos < underlay_volume->size() );
      for (int j=0; j<uheight; j++){
//	color =  grey_cmap[BrainView::adjust(underlay_volume->getElement(pos))] ;
	color =  (*underlay_colormap)[BrainView::adjust(underlay_volume->getElement(pos))] ;


	uimg.setPixel(i,uheight - j - 1,color);
	assert ( pos < underlay_volume->size() );
	pos += uskip_y;
      }
    }

    // stretch the functional image to fit the underlay
    if (stretch) {
		simg = img.smoothScale(uwidth,uheight);
    	for (int i=0; i<uwidth ; i++)
      		for (int j=0; j<uheight ; j++)
				if (simg.pixel(i,j)== (*colormap)[BrainView::adjust(0)])  
					simg.setPixel(i,j,uimg.pixel(i,j));
	}
    else {
		int owidth = (int) ( (float) scale_x / uscale_x * width + 0.5 );
		int oheight = (int) ( (float) scale_y / uscale_y * height + 0.5 );
		simg = img.smoothScale ( owidth, oheight );
    	for ( int i=0; i < owidth; i++ )
      		for ( int j=0; j < oheight; j++ )
				if ( simg.pixel(i,j) == (*colormap)[BrainView::adjust(0)] )  
					if( uimg.valid( i+uwidth/2-owidth/2, j+uheight/2-oheight/2 ))
						simg.setPixel(i,j, uimg.pixel(i+uwidth/2-owidth/2, j+uheight/2-oheight/2 ));
						else simg.setPixel(i,j, 0 );
	}
    // stretch image to fit window
    *image = simg.smoothScale 
		(
		 static_cast<int>(width*scale_x*mag_x),
		 static_cast<int>(height*scale_y*mag_y)
		 );
  } 
  else // no underlay, just stretch the image to fit the window.
    *image = img.smoothScale
		(
		 static_cast<int>(width*scale_x*mag_x),
		 static_cast<int>(height*scale_y*mag_y)
		 );
}

void BrainView::mousePressEvent(QMouseEvent *qme){
  if (qme->button() == LeftButton) {
    if (crosshair) setCursor(blankCursor);
    updateCursor(qme->x(), qme->y());
  } else if (qme->button() == MidButton) {
    mx = qme->x();
    my = qme->y();
    mc = contrast;
    mg = gamma;
  } else if (qme->button() == RightButton) {
    menu->popup(mapToGlobal(qme->pos()));
  }
}

void BrainView::mouseMoveEvent(QMouseEvent *qme){
  if (qme->state() == LeftButton) {
    updateCursor(qme->x(), qme->y());
  } else if (qme->state() == MidButton) {
    setContrast((float)(qme->x()-mx)/image->width());
    setGamma((float)(qme->y()-my)/image->height());
    updateImage();
    showImage();
  }
}

void BrainView::mouseReleaseEvent(QMouseEvent *){
  setCursor(crossCursor);
}

void BrainView::resizeEvent(QResizeEvent *){
  switch (view) {
  case AXIAL:
    mag_x = (float) width()/volume->width()/volume->voxelSize().x();
    mag_y = (float) height()/volume->height()/volume->voxelSize().y();
    break;
  case SAGITAL:
    mag_x = (float) width()/volume->depth()/volume->voxelSize().z();
    mag_y = (float) height()/volume->height()/volume->voxelSize().y();
    break;
  case CORONAL:
    mag_x = (float) width()/volume->width()/volume->voxelSize().x();
    mag_y = (float) height()/volume->depth()/volume->voxelSize().z();
    break;
  default: mag_x = mag_y = magnification; break;
  }
	if (mainview)
		tslider->resize(width(),height()/20);
  updateImage();
}

void BrainView::moveEvent(QMoveEvent *qme){
  if (mainview) {
    int dx = x() - qme->oldPos().x();
    int dy = y() - qme->oldPos().y();
    emit(windowMoved(dx, dy));
  }
}

void BrainView::paintEvent(QPaintEvent *){
  if (mainview) {
    if (!shown) {
      recompose();
      shown = true;
    }
    tslider->showBar();
  }
  showImage();
}

void BrainView::keyPressEvent(QKeyEvent *qke){
  switch(qke->key()) {
  case Key_C: changeCrossHair(); break;
  case Key_Q: quit(); break;			
  case Key_R: recompose(); break;			
  case Key_S: save(); break;			
  case Key_T: changeThreshold(); break;			
  case Key_V: saveAll(); break;			
  default: break;
  }
}

void BrainView::showImage(){
  painter->setRasterOp(CopyROP);
  painter->drawImage(0,0,*image);
  showCursor();
}

void BrainView::save(){
  QFileDialog save_dialog(this);
  QString filestr = save_dialog.getSaveFileName(NULL,"*.bmp");
  char *filename = (char *) filestr.data();
  if (filename != NULL) {
    QFileInfo file(filename);
    if (!file.exists()) image->save(filename,"BMP");
    else {
      QMessageBox file_exists(this);
      int answer = file_exists.warning(NULL,"Save As",
				       "The file already exists\nDo you want to overwrite it?",
				       QMessageBox::Yes,QMessageBox::No | QMessageBox::Default);
      if (answer == QMessageBox::Yes) image->save(filename,"BMP");
    }
  }
}

void BrainView::saveAll(){
}

void BrainView::recompose(){
  if (mainview) {
    resetAspect(1);
    emit(recompose(frameGeometry(),geometry()));
  }
}

void BrainView::resetAspect() {
	resetAspect(true);
}
void BrainView::resetAspect( bool fixwidth ){
//  fixwidth = false;
  if ( fixwidth ) mag_y = mag_x;
  else mag_x = mag_y;

  updateImage();
  resize(image->width(),image->height());
}

void BrainView::resetAll(){
  contrast = gamma = 1;
  mag_y = mag_x = magnification;
  updateImage();
  resize(image->width(),image->height());
}

void BrainView::resetSize(){
  mag_y = mag_x = magnification;
  updateImage();
  resize(image->width(),image->height());
}

void BrainView::resetBrightness(){
  gamma = 1;
  updateImage();
  showImage();
}

void BrainView::resetContrast(){
  contrast = 1;
  updateImage();
  showImage();
}

void BrainView::changeAnimate(){
  animate = !animate;
  if (animate) {
    menu->changeItem("Freeze",id_animate);
    animate_timer->start(500);
  } else {
    menu->changeItem("Animate",id_animate);
    animate_timer->stop();
  }
}

void BrainView::updateAnimate(){
  int slice;
  switch (view) {
  case AXIAL:
    if (++cz >= volume->depth()) cz = 0;
    slice = cz;
    break;
  case SAGITAL:
    if (++cx >= volume->width()) cx = 0;
    slice = cx;
    break;
  case CORONAL:
    if (++cy >= volume->height()) cy = 0;
    slice = cy;
    break;
  default: break;
  }
  updateImage();
  showImage();
  emit(cursorMoved(cx, cy, cz));
}

void BrainView::quit(){
  if (mainview) exit(0);
}

void BrainView::clearCmapMenu(){
  // clear selections
  for (int i=0; i<n_cmap; i++)
    cmap_menu->setItemChecked(id_cmap[i],false);
}


void BrainView::slotSetCmap(int id)
{
	int index = -1;
	for (int i =0; i< n_cmap; ++i)
		if ( id_cmap[i] == id )
			index = i;
	if ( index == -1 )
	{
		throw RUMBA::Exception("Fatal: index lookup failed in slotSetCmap(int)");
	}

	clearCmapMenu();
  	cmap_menu->setItemChecked(id,true);
  	colormap = &(colorMapList()[index]);
  	updateImage();
  	showImage();

  	if (mainview) 
	{
		tslider->setColormap(colormap);
		emit(Cmap(index));
	}
}

void BrainView::slotSetCmapExternal(int index)
{
	int id = id_cmap[index];
	clearCmapMenu();
  	cmap_menu->setItemChecked(id,true);
  	colormap = &(colorMapList()[index]);
  	updateImage();
  	showImage();

  	if (mainview) 
	{
		tslider->setColormap(colormap);
		emit(Cmap(index));
	}
}



/**
void BrainView::slotGreyCmap(){
  clearCmapMenu();

  cmap_menu->setItemChecked(id_grey,true);
  colormap = grey_cmap;
  updateImage();
  showImage();

  if (view == AXIAL) {
    tslider->setColormap(colormap);
    emit(greyCmap());
  }
}

void BrainView::slotHotmetalCmap(){
  clearCmapMenu();

  cmap_menu->setItemChecked(id_hotmetal,true);
  colormap = hotmetal_cmap;
  updateImage();
  showImage();

  if (view == AXIAL) {
    tslider->setColormap(colormap);
    emit(hotmetalCmap());
  }
}

void BrainView::slotJetCmap(){
  clearCmapMenu();

  cmap_menu->setItemChecked(id_jet,true);
  colormap = jet_cmap;
  updateImage();
  showImage();

  if (view == AXIAL){
    tslider->setColormap(colormap);
    emit(jetCmap());
  }
}
*/

void BrainView::changeThreshold(){
  threshold = !threshold;	
  menu->setItemChecked(id_threshold,threshold);
  if (threshold) tslider->show();
  else tslider->hide();
}

void BrainView::changeCrossHair(){
  showCursor();	// if exists disappears
  crosshair = !crosshair;
  showCursor();	// if doesn't exist appears
  menu->setItemChecked(id_crosshair,crosshair);
}

void BrainView::changeCoordinates(){
  coordinates = !coordinates;
  menu->setItemChecked(id_coordinates,coordinates);
  showImage();
}

void BrainView::updateCursor(int X, int Y){
  showCursor();	
  int old_cx = cx;
  int old_cy = cy;
  int old_cz = cz;
  switch (view) {
  case AXIAL:
    cx = (int) (X/volume->voxelSize().x()/mag_x);
    cy = (int) (Y/volume->voxelSize().y()/mag_y);
    break;
  case SAGITAL:
    cz = (int) (X/volume->voxelSize().z()/mag_x);
    cy = (int) (Y/volume->voxelSize().y()/mag_y);
    //    cz = (int) ((height()-Y)/volume->voxelSize().z()/mag_y);
    break;
  case CORONAL:
    cx = (int) (X/volume->voxelSize().x()/mag_x);
    //cz = (int) (Y/volume->voxelSize().z()/mag_y);
    cz = (int) ((height()-Y)/volume->voxelSize().z()/mag_y);
    break;
  default: X = Y = 0; break;
  }
  if (cx < 0) cx = 0;	if (cx >= volume->width()) cx = volume->width()-1;
  if (cy < 0) cy = 0;	if (cy >= volume->height()) cy = volume->height()-1;
  if (cz < 0) cz = 0;	if (cz >= volume->depth()) cz = volume->depth()-1;
  showCursor();	
  if ((old_cx-cx) || (old_cy-cy) || (old_cz-cz))
    emit(cursorMoved(cx, cy, cz));
}

void BrainView::showCursor(){
  if (crosshair) {
    QPen pen(red);
    pen.setStyle(DashLine);

    int X, Y;


	if (coordinates) 
	{
		char coord[50];
		if (volume->timepoints() > 1)
			sprintf(coord,"%d,%d,%d,%d",cx,cy,cz,ct);
		else
			sprintf(coord,"%d,%d,%d",cx,cy,cz);
  
		painter->setRasterOp(CopyROP);
		painter->drawImage(0,0,*image);

		QPen pen(white);
		painter->setPen(pen);
		painter->drawText(2,height()-5,(const char*) coord);
	}

    switch (view) 
	{
    case AXIAL:
      X = (int) (((float)cx+.5)*volume->voxelSize().x()*mag_x);
      Y = (int) (((float)cy+.5)*volume->voxelSize().y()*mag_y);
      break;
    case SAGITAL:
      X = (int) (((float)cz+.5)*volume->voxelSize().z()*mag_x);
      Y = (int) (((float)cy+.5)*volume->voxelSize().y()*mag_y);
//      Y = height() - (int) (((float)cz+.5)*volume->voxelSize().z()*mag_y);
      break;
    case CORONAL:
      X = (int) (((float)cx+.5)*volume->voxelSize().x()*mag_x);
      //Y = (int) (((float)cz+.5)*volume->voxelSize().z()*mag_y);
      Y = height() - (int) (((float)cz+.5)*volume->voxelSize().z()*mag_y);
      break;
    default: X = Y = 0; break;
    }
    painter->setRasterOp(XorROP);
    //painter->setRasterOp(CopyROP);
    painter->setPen(pen);
    painter->drawLine(0,Y,image->width(),Y);
    painter->drawLine(X,0,X,image->height());
  }


}

void BrainView::slotCursorMoved(int x, int y, int z){
  cx = x;
  cy = y;
  cz = z;
  updateImage();
  showImage();
}

void BrainView::slotCursorMoved(int t){
  ct = t;
  updateImage();
  showImage();
}

void BrainView::slotWindowMoved(int dx, int dy){
  move(x()+dx + WIDTHHACK, y()+dy+HEIGHTHACK);
}

void BrainView::slotRecompose(QRect out,QRect in){

  switch (view) {
  case SAGITAL:
	if (mainview) break;
    resize(width(),in.height());
    resetAspect( 0 ); // hold height fixed, not width...
    move(out.right() + WIDTHHACK,out.top()+HEIGHTHACK);
    show();
    break;
  case CORONAL:
	if (mainview) break;
    resize(in.width(),height());
    resetAspect(1);
    move(out.left()+WIDTHHACK,out.bottom()+HEIGHTHACK);
    show();
    emit(recompose(frameGeometry(),geometry()));
    break;
  case AXIAL: 
	if (mainview) break;
  default: throw 0;
  }

}

void BrainView::slotChangeThreshold(int t, int T){
  if (mainview) emit(changeThreshold(t,T));
  thresh_min = t;
  thresh_max = T;
  updateImage();
  showImage();
}

void BrainView::setGamma(float g){
  gamma = mg + g;
  if (gamma<.1) gamma = .1;
  if (gamma>5) gamma = 5;
}

void BrainView::setContrast(float c){
  contrast = mc - c;
  if (contrast<.1) contrast = .1;
  if (contrast>5) contrast = 5;
}

int BrainView::adjust(int color){
  // contrast
  double v = .5 + ((double) color/255 - .5)/contrast;
  if (v < 0) v = 0;
  else if (v > 1) v = 1;
  // brightness
  v = 255*pow(v, gamma);
  return ((int) v);
}
