//rumbaview colorbar.cpp

#include "colorbar.h"

#include <qtooltip.h>

#include <sstream>
#include <string>

/**
  *@author Giorgio Grasso, Ben Bly, Donovan Rebbechi
  */

class ColorBarToolTip : public QToolTip
{
public:
	typedef ColorBar::dPoint dPoint;
	ColorBarToolTip(QWidget* parent, dPoint Scale, int* pmin, int* pmax) 
		: QToolTip(parent),Scale(Scale),pmin(pmin),pmax(pmax) {
	}
	virtual void maybeTip(const QPoint& p)
	{
		std::ostringstream out;
		int max = ( *pmax > *pmin ) ? *pmax : *pmin;
		int min = ( *pmax > *pmin ) ? *pmin : *pmax;

		if ( *pmax < *pmin ) 
			out << "x not in "; 
		else 
			out << "x in ";
		out <<  '(' 
			<< (min*(Scale.second-Scale.first))/255.0 +Scale.first<<','
			<< (max*(Scale.second-Scale.first))/255.0 +Scale.first<<')';
		tip ( parentWidget()->rect(), QString(out.str().c_str()) );
	}
private:
	dPoint Scale;
	int *pmin, *pmax;

};

ColorBar::ColorBar(QWidget *parent, dPoint scale) 
: QFrame(parent), 
p(parent), 
cmap_image(NULL), 
move_min(FALSE), move_max(FALSE),
min(0), max(255),
image(new QImage(256,1,32,1<<24)), painter(new QPainter(this)),
Scale(scale)
{
	ColorBarToolTip* x = new ColorBarToolTip(this,Scale,&min,&max);
}

ColorBar::~ColorBar(){
	if (painter) delete painter;
	if (image) delete image;
	if (cmap_image) delete cmap_image;
}

void ColorBar::setColormap(const RUMBA::Colormap* cmap){
	colormap = cmap;
	if (cmap_image) delete cmap_image;
	cmap_image = new QImage(256, 1, 32, 1<<24);
	for (int i=0; i<256; i++)
		cmap_image->setPixel(i,0,(*colormap)[i]);

	*image = cmap_image->smoothScale(width(),height());
	repaint();
}

void ColorBar::enterEvent(QEvent *){
	setCursor(arrowCursor);
}

void ColorBar::mousePressEvent(QMouseEvent *qme){
	int dmin, dmax;

	dmin = min*(width()-1)/255;
	dmax = (min+height()/2)*(width()-1)/255;
	if (qme->x() > dmin && qme->x() < dmax) move_min = TRUE;

	dmin = (max-height()/2)*(width()-1)/255;
	dmax = max*(width()-1)/255;

	if (qme->x() > dmin && qme->x() < dmax) move_max = TRUE;

	if (move_min || move_max) setCursor(sizeHorCursor);
}

void ColorBar::mouseReleaseEvent(QMouseEvent *){
	setCursor(arrowCursor);
	move_min = move_max = FALSE;
}

void ColorBar::mouseMoveEvent(QMouseEvent *qme){
	if (qme->state() == LeftButton) {
		if (move_min) min = qme->x()*255/(width()-1);
		if (move_max) max = qme->x()*255/(width()-1);
		if (min < 0) min = 0; if (min >= 250) min = 250;
		if (max < 5) max = 5; if (max >= 255) max = 255;
		showBar();
		emit(changeThreshold(min,max));
	}
}

void ColorBar::resizeEvent(QResizeEvent *){
	*image = cmap_image->smoothScale(width(),height());
	repaint();
}

void ColorBar::paintEvent(QPaintEvent *){
	showBar();
}

void ColorBar::showBar(){
	int dmin = min*(width()-1)/255;
	int dmax = max*(width()-1)/255;

	painter->setRasterOp(CopyROP);
	QBrush brush(black);
	// draw only the visible portions of the map
	if (min < max) {
		QBrush brush(black);
		painter->fillRect(0,0,dmin,height(),brush);
		painter->fillRect(dmax,0,width(),height(),brush);
		painter->drawImage(dmin,0,*image,dmin,0,dmax-dmin,height());
	} else {
		painter->fillRect(dmin,0,dmax-dmin,height(),brush);
		painter->drawImage(0,0,*image,0,0,dmax,height());
		painter->drawImage(dmin,0,*image,dmin,0,width()-dmin,height());
	}

	painter->setRasterOp(XorROP);
	painter->setPen(white);
	// draw miminum threshold cursor
	painter->moveTo(dmin,0);
	painter->lineTo(dmin,height());
	painter->lineTo(dmin+height()/2,height()/2);
	painter->lineTo(dmin,0);
	
	// draw maximum threshold cursor
	painter->moveTo(dmax,0);
	painter->lineTo(dmax,height());
	painter->lineTo(dmax-height()/2,height()/2);
	painter->lineTo(dmax,0);
}

void ColorBar::keyPressEvent(QKeyEvent *){
}
