//rumbaview main.cpp (authors: Giorgio Grasso, Ben Bly, Donovan Rebbechi)

#include <unistd.h>
#include <sys/stat.h>



#include <qapplication.h>
#include <qpixmap.h>
#include <qfont.h>
#include <qgrid.h>

#include <rumba/arghandler.h>
#include <rumba/manifold.h>
#include <rumba/manifoldFile.h>
#include <rumba/util.hpp>

#include "brainview.h"
#include "timeview.h"
#include  "colormaplist.h"
#include "header_view.h"

#include <iostream>

/**
  *@author Giorgio Grasso, Ben Bly, Donovan Rebbechi
  */

typedef std::pair<double,double> dPoint;

// custom command line arguments
using RUMBA::Argument;
Argument myArgs[]= {
  Argument("overlay", RUMBA::ALPHA, 'o', ""),	// overlay volume
  Argument("stretch", RUMBA::FLAG, 's', 0),	// stretch volumes to fit
  Argument("colormap", RUMBA::ALPHA, 'c', ""),	// extra colormap file or dir
  Argument("zoom", RUMBA::NUMERIC, 'z', "1.0"),	// zoom factor
  Argument("normalize-vols", RUMBA::FLAG, 'n' ),
  Argument()
};

class ScaleFunctor
{
	double Max,Min,Scale,Offset;
public:
	ScaleFunctor(double Min, double Max, double Scale=256, double Offset=0)
		:Max(Max),Min(Min),Scale(Scale),Offset(Offset) {}
	ScaleFunctor(const ScaleFunctor& x)
		:Max(x.Max),Min(x.Min),Scale(x.Scale),Offset(x.Offset)
		{  }
	double operator()(double x)
	{
		if (Max>Min)
			return (x - Min) * Scale / (Max-Min) + Offset;
		else
			return Offset;
	}

	double max(){ return Max; }
	double min(){ return Min; }


};

ScaleFunctor maxmin(
		RUMBA::ManifoldFile* in, 
		int start, int ext
		)
{
	double tmp;
	double Min,Max;
	Min = Max = in->getElementDouble(start);
	for (int i = 0; i<ext; ++i )
	{
		if ( (tmp=(*in)[i+start])>Max ) 
			Max=tmp;
		if ( (tmp=(*in)[i+start])<Min ) 
			Min=tmp;
	}
	return ScaleFunctor(Min,Max);
}



dPoint volNormalize(RUMBA::ManifoldFile* in, RUMBA::Manifold<short>& out)
{
	assert(in->timepoints()==out.timepoints());
	ScaleFunctor scale(0,0);
	RUMBA::ManifoldFile::iterator it = in->begin();
	RUMBA::Manifold<short>::iterator ot = out.begin();
	for ( int i = 0; i < in->timepoints(); ++i )
	{
		scale = maxmin(in,in->pixels()*i,in->pixels());
		for ( int j = 0; j < in->pixels(); ++j )
			*ot++ = static_cast<short>(scale(*it++));
	}
	out.copyHeader (*in);
	return dPoint(scale.min(), scale.max());
}


dPoint glNormalize(RUMBA::ManifoldFile* in, RUMBA::Manifold<short>& out)
{
	assert(in->timepoints()==out.timepoints());
	ScaleFunctor scale(0,0);
	RUMBA::ManifoldFile::iterator it = in->begin();
	RUMBA::Manifold<short>::iterator ot = out.begin();
	scale = maxmin(in,0,in->size());
	for ( int i = 0; i < in->size(); ++i )
		*ot++ = static_cast<short>(scale(*it++));
	out.copyHeader (*in);
	return dPoint(scale.min(), scale.max());
}

RUMBA::Manifold<short>* fix_orientation( const RUMBA::Manifold<short>& M  )
{
	RUMBA::PointTransform t;
	RUMBA::intPoint ex;
	RUMBA::Manifold<short>* res = 0;
	RUMBA::Orientation x
		( 
		 RUMBA::patient_right(), 
		 RUMBA::patient_front(), 
		 RUMBA::patient_foot(), 
		 RUMBA::intPoint(), true 
		);

	t = RUMBA::transform( M.orientation(),x );
	ex = t.map_extent(M.extent());
	std::cout << ex.x() << " " << ex.y() << " " << ex.z() << " " 
		<< ex.t() << std::endl;


	res = new RUMBA::Manifold<short>(
			RUMBA::Orientation(
				RUMBA::patient_right(), 
				RUMBA::patient_front(),
				RUMBA::patient_foot(),
				ex, true)
			);



	if (!res)
		return 0;

	RUMBA::volume_transform (M, *res);

	res->headerData() = (M.headerData());
	return res;
}

void time_brainview_hookup(BrainView* B, TimeView* T)
{
	if (T&&B)
	{
		B->connect
			(
			 T,SIGNAL(cursorMoved(int)),
			 SLOT(slotCursorMoved(int))
			 );
		T->connect
			(
			 B,SIGNAL(cursorMoved(int,int,int)),
			 SLOT(slotCursorMoved(int,int,int))
			 );
	}

}

/**
 Hookup main windows ops to subordinate windows
*/ 
void asymmetric_hookup ( BrainView * M, BrainView* V )
{
	if ( (M!=V) && M && V)
	{
		V->connect
			(
			 M,SIGNAL(windowMoved(int,int)),
			 SLOT(slotWindowMoved(int,int))
			 );

  		V->connect
			(
			 M,SIGNAL(recompose(QRect,QRect)),
			 SLOT(slotRecompose(QRect,QRect))
			 );

		V->connect
			(
			 M,SIGNAL(changeThreshold(int,int)),
			 SLOT(slotChangeThreshold(int,int))
			 );

		V->connect
			( 
			 M,SIGNAL(Cmap(int)), 
			 SLOT(slotSetCmapExternal(int))
			 );


	}
}



void cursor_hookup(BrainView* x, BrainView* y)
{
	if (x&&y)
	{
		x->connect
			(
			 y,SIGNAL(cursorMoved(int,int,int)),
			 SLOT(slotCursorMoved(int,int,int))
			 );
		y->connect
			(
			 x,SIGNAL(cursorMoved(int,int,int)),
			 SLOT(slotCursorMoved(int,int,int))
			 );

	}
}

int main(int argc, char *argv[])
{
  bool verbose = false;
  std::string infile, overlayfile, addColormap;
  bool overlay = false;
  bool stretch = false;
  float zoom;
  bool normalize_vols = false;
  dPoint scale;

  // Handle command line arguments
  try {
    RUMBA::ArgHandler argh(argc, argv, myArgs);

    // Grab standard arguments from the handler
    argh.arg_or_loose("infile",infile);

    // Loose arguments
    std::vector<string> v = argh.loose();
    std::vector<string>::const_iterator it;
    it=v.begin();
  
    if ( infile == "" ) // If no explicit infile, try to find one.
      {
	if ( !v.empty() ) { // First try to use a loose arg.
	  infile = *it ;
	  it++; // Increment counter to avoid re-using arg.
	}
	else {
	  cerr << "You must supply an input file." << endl;
	  exit (1);
	}
      }

    verbose = argh.arg("verbose");

    // Grab custom arguments from the handler

    overlay = true;
    argh.arg("overlay",overlayfile);
    if ( overlayfile.empty() && it != v.end () ) // If no explicit overlayfile
      {
	overlayfile = *it ;
	it++; // Increment counter to avoid re-using arg.
      }
    if ( overlayfile.empty() ) overlay = false;

	normalize_vols = argh.arg("normalize-vols");

    argh.arg("colormap",addColormap);
    stretch = argh.arg("stretch");
    argh.arg("zoom",zoom);
    if ( zoom <= 0.0 ) zoom = 1.0;
  }

  catch (RUMBA::Exception& e) {
   	std::cerr 
		<< e.error() << std::endl 
		<< "usage: rumbaview -i|--infile <file> [-o|--overlay <file>]\n"
		<< "[-s|--stretch param] [-c|--colormap <file] [-z|--zoom <factor>]\n" 
		<< std::endl ;
    exit(1);
  }

  try {
	  RUMBA::colorMapList();
  }
  catch (RUMBA::Exception& e )
  {
	  cerr << e.error() << endl;
	  exit(1);
  }

  // Read in the manifolds from files
  RUMBA::Manifold<short>* volume = 0;
  RUMBA::Manifold<short>* underlay_volume =  0;

  RUMBA::ManifoldFile* in = 0;

  try {

    struct stat buf ;
    //    struct stat buf2 ;

    if (overlay) {
      stat ( overlayfile.c_str(), &buf ) ;
//      if ( S_ISDIR ( buf.st_mode ) ) {
//	volume->loadVols ( overlayfile.c_str() );
//      }
//      else 
	  in = RUMBA::ManifoldFile::construct(overlayfile.c_str());
	  in->setCacheStrategy(RUMBA::CACHE_VOL);
	  volume = new RUMBA::Manifold<short>(in->extent());
	  if (normalize_vols)
		  scale=volNormalize(in,*volume);
	  else
		  scale=glNormalize(in,*volume);
	  delete in;


	  RUMBA::BaseManifold* tmpvol = volume;
	  volume = fix_orientation(*volume);
	  delete tmpvol;


      stat ( infile.c_str(), &buf ) ;
//      if ( S_ISDIR ( buf.st_mode ) ) {
//	underlay_volume->loadVols ( infile.c_str() );
//      }
//      else 
	  in = RUMBA::ManifoldFile::construct(infile.c_str());
	  in->setCacheStrategy(RUMBA::CACHE_VOL);
	  underlay_volume = new RUMBA::Manifold<short>(in->extent());
	  if (normalize_vols)
		  volNormalize(in,*underlay_volume);
	  else
		  glNormalize(in,*underlay_volume);
	  delete in;

	  tmpvol = underlay_volume;
	  underlay_volume = fix_orientation(*underlay_volume);
	  delete tmpvol;

    }
    else {
      stat ( infile.c_str(), &buf ) ;
//      if ( S_ISDIR ( buf.st_mode ) )
//	volume->loadVols ( infile.c_str() );
//      else 
	  in = RUMBA::ManifoldFile::construct(infile.c_str());
	  if (!in)
		  throw RUMBA::BadFile(infile);
	  in->setCacheStrategy(RUMBA::CACHE_VOL);
//	  volume = new RUMBA::Manifold<short>(in->extent());
	  volume = new RUMBA::Manifold<short>(in->orientation());

	  if (normalize_vols)
		  scale=volNormalize(in,*volume);
	  else
		  scale=glNormalize(in,*volume);
	  delete in;

	  RUMBA::BaseManifold* tmpvol = volume;
	  volume = fix_orientation(*volume);

	  delete tmpvol;


    }

    if ( zoom * volume->height() > 1000) zoom = 1000/volume->height();
    if ( zoom * volume->width() > 1000) zoom = 1000/volume->width();
    if ( zoom * volume->depth() > 1000) zoom = 1000/volume->depth();
  // renormalize voxel range
  RUMBA::normalize((short)0,(short)255,*volume);
  if (overlay) RUMBA::normalize((short)0,(short)255,*underlay_volume);
  if (verbose) {
    volume->print();
    if (overlay) underlay_volume->print();
  }

  // Qt application, set font and style
  QApplication a(argc, argv);
  a.setFont(QFont("helvetica", 12));

  // which one gets to be the main view ? 
	BrainView *axial_view=0, *sagital_view=0, *coronal_view=0;
	BrainView** main_view;
	if ( volume->width() > 1 && volume->height() > 1 )
	{
		main_view = &axial_view;
	}
	else if (volume->width() > 1 && volume->depth() > 1 )
	{
		main_view = &coronal_view;
	}
	else if (volume->height() >1 && volume->depth() > 1 )
	{
		main_view = &sagital_view;
	}
	else
	{
		main_view = &coronal_view;
	}

  HeaderView* header_view = new HeaderView(*volume);

	if ( volume->width() > 1 && volume->height() > 1 ) 
	{
		axial_view = new BrainView
	  (
	   NULL,AXIAL,infile.c_str(),zoom,stretch,main_view == &axial_view,scale
	   );
		axial_view->setManifold(volume);
	}

	if (volume->height() > 1 && volume->depth() > 1 )
	{
		sagital_view = new BrainView
	  	(
	   	NULL,SAGITAL,infile.c_str(),zoom,stretch,
		main_view == &sagital_view,scale
	   	);
		sagital_view->setManifold(volume);
	}


	if (volume->width() > 1 && volume->depth() > 1 )
	{
		coronal_view = new BrainView
	  	(
	   	NULL,CORONAL,infile.c_str(),zoom,stretch, 
		main_view == &coronal_view, scale
	   	);
		coronal_view->setManifold(volume);	
	}


  TimeView *time_view = NULL;
  if (volume->timepoints() > 1) {
    time_view = new TimeView();
    time_view->setManifold(volume);
  }

	if (overlay) {
		if (axial_view)
			axial_view->setUnderlay(underlay_volume);
		if (sagital_view)
			sagital_view->setUnderlay(underlay_volume);
		if (coronal_view)
			coronal_view->setUnderlay(underlay_volume);
  }

  // connect signals and slots for cursor movements


	cursor_hookup(sagital_view, axial_view);
	cursor_hookup(coronal_view, axial_view);
	cursor_hookup(coronal_view, sagital_view);


	asymmetric_hookup(*main_view, axial_view);
	asymmetric_hookup(*main_view, coronal_view);
	asymmetric_hookup(*main_view, sagital_view);

	time_brainview_hookup(axial_view,time_view);
	time_brainview_hookup(sagital_view,time_view);
	time_brainview_hookup(coronal_view,time_view);

  // connect signals and slots for AXIAL window movements
  if (volume->timepoints() > 1)
    time_view->connect(axial_view,SIGNAL(windowMoved(int,int)),SLOT(slotWindowMoved(int,int)));


  if (volume->timepoints() > 1)
    time_view->connect(coronal_view,SIGNAL(recompose(QRect,QRect)),SLOT(slotRecompose(QRect,QRect)));




	// show all views
	if (volume->timepoints() > 1) time_view->show();
	if (coronal_view) coronal_view->show();
	if (sagital_view) sagital_view->show();
	if (axial_view) axial_view->show();
	header_view->show();

  a.setMainWidget(*main_view);
  return  a.exec();

  }
  catch ( ... )
    {
      cerr << "Bad input file: " << infile << endl;
    }


}
