//rumbaview timeview.cpp

#include <qpixmap.h>
#include <qimage.h>

#include <rumba/util.hpp>
#include <rumba/fft.h>

// #include <newmath/newmat.h>
// #include <newmath/newmatap.h>

#include <vector>
#include <complex>
#include <cmath>

#include "damnwindowhack.h"

#include "timeview.h"
#include "hotmetal.xpm"
#include "rumbaview.xpm"
//
/**
  *@author Giorgio Grasso, Ben Bly, Donovan Rebbechi
  */

using namespace RUMBA;
using namespace std;

QImage *cmap = new QImage((const char **)hotmetal);	

TimeView::TimeView(QWidget *parent) : QFrame(parent){
  mode = RAW;
  cursor = TRUE;
  animate = FALSE;
  text_height = 10;
  n_mode = 0;

  painter = new QPainter(this);
  animate_timer = new QTimer(this);
  connect(animate_timer,SIGNAL(timeout()),SLOT(updateAnimate()));

  // set initial size
  resize(200,100);

  // set up popup menu
  menu = new QPopupMenu;

  id_mode[n_mode++] = menu->insertItem("Raw",this,SLOT(setRAW()));
  id_mode[n_mode++] = menu->insertItem("Scaled",this,SLOT(setSCALED()));
  id_mode[n_mode++] = menu->insertItem("Intensity",this,SLOT(setINTENSITY()));
  id_mode[n_mode++] = menu->insertItem("FFT power",this,SLOT(setFFT_POWER()));
  menu->insertSeparator();
  id_animate=menu->insertItem("Animate",this,SLOT(changeAnimate()));

  setIcon(QPixmap((const char **) rumbaview));
}

TimeView::~TimeView(){
  if (painter) delete painter;
  if (points) delete points;

  delete cmap;
}

void TimeView::setManifold(RUMBA::Manifold<short> *vol){
  volume = vol;

  // initialize cursor at the center of the manifold
  cx = volume->width()/2;
  cy = volume->height()/2;
  cz = volume->depth()/2;
  ct = 0;

  // initialize array of points in graph
  points = new int[volume->timepoints()];

  // set initial display mode
  setRAW();
}

void TimeView::slotCursorMoved(int x, int y, int z){
  cx = x;
  cy = y;
  cz = z;
  updateGraph();
  repaint();
}

void TimeView::slotWindowMoved(int dx, int dy){
  move(x()+dx+WIDTHHACK,y()+dy+HEIGHTHACK);
}

void TimeView::slotRecompose(QRect out, QRect in){
  resize(in.width(),in.height());
  move(out.right(),out.top());
  show();
}

void TimeView::showGraph(){
  painter->setPen(black);

  float scale = 1 ;
  if (mode == RAW || mode == SCALED) {
    scale = (float) width()/volume->timepoints();
    painter->moveTo(0,points[0]);
    for (int i=1; i<volume->timepoints(); i++)
      painter->lineTo(static_cast<int>(i*scale),points[i]);
  } 
  else if (mode == INTENSITY) {
    scale = (float) width()/volume->timepoints();
    for (int i = 0; i < volume->timepoints (); i++) {
      int col = points [ i ];
      painter->setPen( QPen( QColor( cmap->pixel(col,0) ), (uint) ( scale + 1 ) ));
      painter->drawLine
		  ( 
		   static_cast<int>(i*scale + scale/2), 0, 
		   static_cast<int>(i*scale + scale/2), height()
		   );
    }
  }
  else { //mode is FFT_POWER
    scale = (float) width()/volume->timepoints()*2;
    QBrush brush(blue);
    for (int i=0; i<=volume->timepoints()/2; i++) {
      painter->fillRect
		  (
		   static_cast<int>(i*scale),
		   points[i],
		   static_cast<int>(scale),
		   static_cast<int>(height()-points[i]),
		   brush
		   );
      painter->drawRect
		  (
		   static_cast<int>(i*scale),points[i],
		   static_cast<int>(scale),height()-points[i]
		   );
    }
  }

  int bottom = height() - text_height;
  if (mode == RAW || mode == SCALED ) {
    int tick_step = volume->timepoints()/10;
    if (!tick_step) tick_step = 1;
    for (int i=0; i<volume->timepoints(); i+= tick_step)
      painter->drawLine
		  ( 
		   static_cast<int>(i*scale),height(), 
		   static_cast<int>(i*scale),bottom
		   );		
  }

  if (mode == SCALED) {
    QPen pen(blue);
    pen.setStyle(DotLine);
    painter->setPen(pen);
    painter->drawLine(0,static_cast<int>(avg),width(),static_cast<int>(avg));		
  }

  if (mode == INTENSITY) {
    QPen pen(red);
    pen.setStyle(DashLine);
    painter->setPen(pen);
    painter->drawLine
		(
		 static_cast<int>(ct*scale + scale/2), 0, 
		 static_cast<int>(ct*scale + scale/2), 
		 height() 
		 );
  }

  if ((mode == RAW || mode == SCALED) && cursor) {
    QPen pen(red);
    pen.setStyle(DashLine);
    painter->setPen(pen);
    painter->drawLine
		(
		 static_cast<int>(ct*scale),0,
		 static_cast<int>(ct*scale),bottom
		 );
  }
}

void TimeView::updateGraph(){
  if (mode == RAW) {
    int bottom = height() - text_height;
    float scale = (float) bottom/255;
    for (int i=0; i<volume->timepoints(); i++)
      points[i] = (int) (bottom - volume->getElement(cx,cy,cz,i)*scale);
  } 
  else if (mode == SCALED) {
    int bottom = height() - text_height;
    int min = volume->getElement(cx,cy,cz,0);
    int max = volume->getElement(cx,cy,cz,0);
    avg = volume->getElement(cx,cy,cz,0);
    for (int i=1; i<volume->timepoints(); i++) {
      avg += volume->getElement(cx,cy,cz,i);
      if (volume->getElement(cx,cy,cz,i) > max)
	max = volume->getElement(cx,cy,cz,i);
      if (volume->getElement(cx,cy,cz,i) < min)
	min = volume->getElement(cx,cy,cz,i);
    }
    avg /= volume->timepoints();
    float scale = (float) bottom/(max-min);
    for (int i=0; i<volume->timepoints(); i++)
      points[i] = (int) (bottom - (volume->getElement(cx,cy,cz,i)-min)*scale);
    avg = (int) (bottom - (avg-min)*scale);
  }
  else if (mode == INTENSITY) {
    //note that the volume is normalized 0-255 in 'main'...
    for (int i=0; i<volume->timepoints(); i++)
      points[i] = (int) (volume->getElement(cx,cy,cz,i));
  }
  else if (mode == FFT_POWER) {
/*
    ColumnVector X ( volume->timepoints() );
    ColumnVector F ( volume->timepoints()/2 );
    ColumnVector G ( volume->timepoints()/2 );
*/
	  std::vector<complex<double> > X;
	  X.resize(volume->timepoints());

    for ( int t = 0;t < volume->timepoints(); t++ ) {
//      X(t+1) = volume->getElement( cx, cy, cz, t );
		X[t] = complex<double> ( volume->getElement(cx,cy,cz,t) );
    }
//    RealFFT ( X, F, G );
	RUMBA::fft(X);
    for ( int t = 0; t <= volume->timepoints()/2; t++ ) {
//      points[t-1] = (int) ( log( 1000 * (F(t)*F(t) + G(t)*G(t)) + 1 ));
		points[t] = (int) ( log(  10 + 1000 *
				X[t].real()*X[t].real() + X[t].imag() * X[t].imag()   
				)//log
				);
      //points[t-1] = (int) ( F(t) * F(t) + G(t) * G(t) );
    }

    int bottom = height()-1;
    int min = points[0]; int max = points[0];
    for (int i=0; i<=volume->timepoints()/2; i++) {
      if (points[i] > max) max = points[i];
      if (points[i] < min) min = points[i];
    }

    float scale = (float) bottom/(max-min);
    for (int i=0; i<=volume->timepoints()/2; i++)
      points[i] = (int) (bottom - (points[i]-min)*scale);
  }
}

void TimeView::mousePressEvent(QMouseEvent *qme){
  if (mode != FFT_POWER && qme->button() == LeftButton) {
    ct = (int) ((float) qme->x()*volume->timepoints()/width() + .5);
    if (ct<0) ct = 0;
    if (ct>=volume->timepoints()) ct = volume->timepoints()-1;
    repaint();
    emit(cursorMoved(ct));
  } else if (qme->button() == RightButton) {
    menu->popup(mapToGlobal(qme->pos()));
  }
}

void TimeView::mouseMoveEvent(QMouseEvent *qme){
  if (mode != FFT_POWER && qme->state() == LeftButton) {
    int new_t = (int) ((float) qme->x()*volume->timepoints()/width() + .5);
    if (new_t<0) new_t = 0;
    if (new_t>=volume->timepoints()) new_t = volume->timepoints()-1;
    if (new_t != ct) {
      ct = new_t;
      repaint();
      emit(cursorMoved(ct));
    }
  }
}

void TimeView::resizeEvent(QResizeEvent *){
  updateGraph();
  repaint();
}

void TimeView::paintEvent(QPaintEvent *){
  showGraph();
}

void TimeView::keyPressEvent(QKeyEvent *){
}

void TimeView::changeAnimate(){
  animate = !animate;
  if (animate) {
    menu->changeItem("Freeze",id_animate);
    animate_timer->start(100);
  } else {
    menu->changeItem("Animate",id_animate);
    animate_timer->stop();
  }
}

void TimeView::updateAnimate(){
  if (++ct >= volume->timepoints()) ct = 0;
  repaint();
  emit(cursorMoved(ct));
}

void TimeView::setRAW(){
  clearModes();
  menu->setItemEnabled(id_animate,TRUE);

  setCaption("Raw");
  menu->setItemChecked(id_mode[RAW],TRUE);
  mode = RAW;
  updateGraph();
  repaint();
}

void TimeView::setSCALED(){
  clearModes();
  menu->setItemEnabled(id_animate,TRUE);

  setCaption("Scaled");
  menu->setItemChecked(id_mode[SCALED],TRUE);
  mode = SCALED;
  updateGraph();
  repaint();
}

void TimeView::setINTENSITY(){
  clearModes();
  menu->setItemEnabled(id_animate,TRUE);

  setCaption("Intensity");
  menu->setItemChecked(id_mode[INTENSITY],TRUE);
  mode = INTENSITY;
  updateGraph();
  repaint();
}

void TimeView::setFFT_POWER(){
  clearModes();
  if (animate) changeAnimate();
  menu->setItemEnabled(id_animate,FALSE);

  setCaption("FFT power");
  menu->setItemChecked(id_mode[FFT_POWER],TRUE);
  mode = FFT_POWER;
  updateGraph();
  repaint();
}

void TimeView::clearModes(){
  // clear selections
  for (int i=0; i<n_mode; i++)
    menu->setItemChecked(id_mode[i],FALSE);
}
