#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <stdio.h>

#ifndef INHIBIT_STRING_HEADER
#if defined (HAVE_STRING_H) || defined (STDC_HEADERS) || defined (_LIBC)
#include <string.h>
#else
#include <strings.h>
#endif
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif

#ifdef HAVE_ERRNO_H
#include <errno.h>
#endif

#if TIME_WITH_SYS_TIME
#include <sys/time.h>
#include <time.h>
#else
#if HAVE_SYS_TIME_H
#include <sys/time.h>
#else
#include <time.h>
#endif
#endif

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_TYPES_H
#include <types.h>
#endif

#ifdef HAVE_SYS_SELECT_H
#include <sys/select.h>
#endif
#ifdef HAVE_SELECT_H
#include <select.h>
#endif

#ifdef HAVE_SYS_WAIT_H
#include <sys/wait.h>
#endif

#ifdef HAVE_WAIT_H
#include <wait.h>
#endif

#ifdef HAVE_SIGNAL_H
#include <signal.h>
#endif

#ifdef HAVE_PWD_H
#include <pwd.h>
#endif


/* IPC stuff */
#include <sys/ipc.h>
#include <sys/msg.h>

#define LOCK_DIR "/tmp/run/"

static void usage (void)
{
    fprintf (stderr, "\
Usage:\n\
\trun [-h|--help] [-r|--restart] [--daemon] [--robust] [--null]\n\
\t    [-q, --quiet] [-i|--interval <n.nn>] <app_name> <arg1> <arg2> ...\n\
\n\
\tisrunning [-h|--help] [-q|--quiet] <app_name>\n\
\n\
\tstop [-h|--help] [-f|--force] [-q|--quiet] [-<signal>] <app_name>\n\
\n\
run  runs a program, but not if  run <app_name>  is already running.\n\
\n\
Returns the exit status of <app_name>.\n\
\n\
    -h, --help             print this message\n\
    -r, --restart          restart <app_name> if it dies\n\
    --max-restarts         maximum restarts, default infinity\n\
    --daemon               fork twice into the background (must preceed)\n\
    --robust               ignore INT, QUIT, PIPE signals\n\
    --null                 redirect input/output to /dev/null\n\
    -f, --force            do not wait for <app_name> to exit\n\
    -q, --quiet            suppress certain messages\n\
    -d                     same as --daemon --robust --null --restart\n\
    -i, --interval n.nn    min interval between restarts in\n\
                           seconds: default 0.1\n\
    -<signal>              signal name or number\n\
\n\
These programs must be installed setuid to work.\n\
");
    exit (1);
}

static int write_lock (int f, off_t offset, int size)
{
    struct flock l;
    l.l_type = F_WRLCK;
    l.l_whence = SEEK_SET;
    l.l_start = offset;
    l.l_len = size;
    l.l_pid = getpid ();
    return fcntl (f, F_SETLK, &l);
}

static int get_lock (int f, off_t offset, int size)
{
    struct flock l;
    l.l_type = F_RDLCK;
    l.l_whence = SEEK_SET;
    l.l_start = offset;
    l.l_len = size;
    l.l_pid = getpid ();
    fcntl (f, F_GETLK, &l);
    if (l.l_type == F_UNLCK)
	return 0;
    return -1;
}

static int send_signal = SIGTERM;
static double per_second = 0.1;
static pid_t child_pid = 0;
static int verbose = 1;
static int option_restarts = 1;
static int msg_queue = 0;
static int option_force = 0;
static int option_null = 0;
static int option_daemon = 0;
static int option_robust = 0;
static int last_status = 0;

static double get_time (void)
{
    struct timeval t;
#ifndef HAVE_GETTIMEOFDAY
# error You need the gettimeofday function for this to work
#endif
    gettimeofday (&t, 0);
    return (double) t.tv_usec / (double) 1000000.0 + (double) t.tv_sec;
}

static void user_priv (void)
{
    if (seteuid ((uid_t) getuid ())) {
	perror ("run: setuid");
	exit (127);
    }
}

static void my_execvp (char **a)
{
    char *e;
    execvp (a[0], a);
    e = (char *) malloc ((size_t) strlen (a[0]) + 10);
    strcpy (e, "run: ");
    strcat (e, a[0]);
    perror (e);
    free (e);
    exit (127);
}

#ifndef HAVE_STRSPN
/* from the Linux kernel src Copyright (C) 1991, 1992  Linus Torvalds */
static size_t strspn (const char *s, const char *accept)
{
    const char *p;
    const char *a;
    size_t count = 0;

    for (p = s; *p != '\0'; ++p) {
	for (a = accept; *a != '\0'; ++a) {
	    if (*p == *a)
		break;
	}
	if (*a == '\0')
	    return count;
	++count;
    }

    return count;
}
#endif

#ifndef HAVE_USLEEP
#error
static void usleep (unsigned long t)
{
    struct timeval tv;
    tv.tv_sec = t / 1000000L;
    tv.tv_usec = t;
    select ((int) 0, NULL, NULL, NULL, &tv);
}

#endif

#if (RETSIGTYPE==void)
#define handler_return return
#else
#define handler_return return 1
#endif

struct _all_signals {
    int n;
    char *name;
} all_signals[] = {

#ifdef SIGHUP
    {
    SIGHUP, "SIGHUP"},
#endif
#ifdef SIGINT
    {
    SIGINT, "SIGINT"},
#endif
#ifdef SIGQUIT
    {
    SIGQUIT, "SIGQUIT"},
#endif
#ifdef SIGILL
    {
    SIGILL, "SIGILL"},
#endif
#ifdef SIGABRT
    {
    SIGABRT, "SIGABRT"},
#endif
#ifdef SIGFPE
    {
    SIGFPE, "SIGFPE"},
#endif
#ifdef SIGKILL
    {
    SIGKILL, "SIGKILL"},
#endif
#ifdef SIGSEGV
    {
    SIGSEGV, "SIGSEGV"},
#endif
#ifdef SIGPIPE
    {
    SIGPIPE, "SIGPIPE"},
#endif
#ifdef SIGALRM
    {
    SIGALRM, "SIGALRM"},
#endif
#ifdef SIGTERM
    {
    SIGTERM, "SIGTERM"},
#endif
#ifdef SIGUSR1
    {
    SIGUSR1, "SIGUSR1"},
#endif
#ifdef SIGUSR2
    {
    SIGUSR2, "SIGUSR2"},
#endif
#ifdef SIGCHLD
    {
    SIGCHLD, "SIGCHLD"},
#endif
#ifdef SIGCONT
    {
    SIGCONT, "SIGCONT"},
#endif
#ifdef SIGSTOP
    {
    SIGSTOP, "SIGSTOP"},
#endif
#ifdef SIGTSTP
    {
    SIGTSTP, "SIGTSTP"},
#endif
#ifdef SIGTTIN
    {
    SIGTTIN, "SIGTTIN"},
#endif
#ifdef SIGTTOU
    {
    SIGTTOU, "SIGTTOU"},
#endif
#ifdef SIGTRAP
    {
    SIGTRAP, "SIGTRAP"},
#endif
#ifdef SIGIOT
    {
    SIGIOT, "SIGIOT"},
#endif
#ifdef SIGEMT
    {
    SIGEMT, "SIGEMT"},
#endif
#ifdef SIGBUS
    {
    SIGBUS, "SIGBUS"},
#endif
#ifdef SIGSYS
    {
    SIGSYS, "SIGSYS"},
#endif
#ifdef SIGSTKFLT
    {
    SIGSTKFLT, "SIGSTKFLT"},
#endif
#ifdef SIGURG
    {
    SIGURG, "SIGURG"},
#endif
#ifdef SIGIO
    {
    SIGIO, "SIGIO"},
#endif
#ifdef SIGPOLL
    {
    SIGPOLL, "SIGPOLL"},
#endif
#ifdef SIGCLD
    {
    SIGCLD, "SIGCLD"},
#endif
#ifdef SIGXCPU
    {
    SIGXCPU, "SIGXCPU"},
#endif
#ifdef SIGXFSZ
    {
    SIGXFSZ, "SIGXFSZ"},
#endif
#ifdef SIGVTALRM
    {
    SIGVTALRM, "SIGVTALRM"},
#endif
#ifdef SIGPROF
    {
    SIGPROF, "SIGPROF"},
#endif
#ifdef SIGPWR
    {
    SIGPWR, "SIGPWR"},
#endif
#ifdef SIGINFO
    {
    SIGINFO, "SIGINFO"},
#endif
#ifdef SIGLOST
    {
    SIGLOST, "SIGLOST"},
#endif
#ifdef SIGWINCH
    {
    SIGWINCH, "SIGWINCH"},
#endif
#ifdef SIGUNUSED
    {
    SIGUNUSED, "SIGUNUSED"},
#endif
};

#define RUNMSG_INTERUPT         1
#define RUNMSG_SIGNAL           2
#define RUNMSG_EXIT             3
#define RUNMSG_REQUEST          4
#define RUNMSG_STATUS_OR_PID    5

static void destroy_queue (void)
{
    msgctl (msg_queue, IPC_RMID, 0);
    msg_queue = 0;
}

struct my_msgbuf {
    struct msgbuf m;
    int status;
    pid_t pid;
};

static void status_exit (int status)
{
    if (WIFSTOPPED (status) && verbose) {
	fprintf (stderr, "run: child stopped\n");
	exit (127);
    }
    if (!WIFEXITED (status)) {
	if (verbose)
	    fprintf (stderr, "run: child did not exit normally\n");
	exit (127);
    }
    exit (WEXITSTATUS (status));
}

static RETSIGTYPE handle_child (int sig_num)
{
    struct my_msgbuf msgp;
    while (wait (&last_status) <= 0);
    child_pid = 0;
    if (option_restarts) {
	if (!--option_restarts) {
	    msgp.m.mtype = RUNMSG_STATUS_OR_PID;
	    msgp.status = last_status;
	    if (msg_queue) {
		if (msgsnd (msg_queue, (struct msgbuf *) &msgp,
			    sizeof (msgp), 0))
		    perror ("run: msgsnd");
	    }
	    status_exit (last_status);
	}
    }
    msgp.m.mtype = RUNMSG_INTERUPT;
    msgsnd (msg_queue, (struct msgbuf *) &msgp, sizeof (msgp), IPC_NOWAIT);
    signal (sig_num, handle_child);
    handler_return;
}

static RETSIGTYPE handle_other (int sig_num)
{
    signal (SIGCHLD, SIG_IGN);
    destroy_queue ();
    exit (0);
    handler_return;
}

static RETSIGTYPE handle_hup (int sig_num)
{
    struct my_msgbuf msgp;
    msgp.m.mtype = RUNMSG_INTERUPT;
    msgsnd (msg_queue, (struct msgbuf *) &msgp, sizeof (msgp), IPC_NOWAIT);
    signal (SIGHUP, handle_hup);
    handler_return;
}

int main (int argc, char **argv)
{
    struct my_msgbuf msgp;
    double t = 0.0;
    char lockname[350];
    char lockcontent[21];
    char **a;
    char *progname, *program;
    int f;
    key_t msg_key;
    if (argc == 1)
	usage ();
    progname = argv[0];
    if ((char *) strchr (progname, '/'))
	progname = (char *) strrchr (progname, '/') + 1;

    if (strcmp (progname, "stop") && strcmp (progname, "isrunning")) {
	signal (SIGCHLD, handle_child);
#ifdef SIGHUP
	signal (SIGHUP, handle_hup);
#endif
#ifdef SIGINT
	signal (SIGINT, handle_other);
#endif
#ifdef SIGQUIT
	signal (SIGQUIT, handle_other);
#endif
#ifdef SIGPIPE
	signal (SIGPIPE, handle_other);
#endif
#ifdef SIGTERM
	signal (SIGTERM, handle_other);
#endif
    }

    for (; argc > 1 && argv[1][0] == '-'; argv++, argc--) {
	int found = 0;
	if (!strcmp (argv[1], "-h") || !strcmp (argv[1], "--help")) {
	    usage ();
	    found = 1;
	}
	if (!strcmp (argv[1], "-r")
	    || !strcmp (argv[1], "--restart")
	    || !strcmp (argv[1], "-d")) {
	    option_restarts = 0;
	    found = 1;
	}
	if (!strcmp (argv[1], "--daemon")
	    || !strcmp (argv[1], "-d")) {
	    option_daemon = 1;
	    found = 1;
	}
	if (!strcmp (argv[1], "--null") || !strcmp (argv[1], "-d")) {
	    option_null = 1;
	    found = 1;
	}
	if (!strcmp (argv[1], "--quiet") || !strcmp (argv[1], "-q")) {
	    verbose = 0;
	    found = 1;
	}
	if (!strcmp (argv[1], "--force") || !strcmp (argv[1], "-f")) {
	    option_force = 1;
	    found = 1;
	}
	if (!strcmp (argv[1], "--verbose") || !strcmp (argv[1], "-v")) {
	    verbose++;
	    found = 1;
	}
	if (!strcmp (argv[1], "--version") || !strcmp (argv[1], "-V")) {
	    printf ("run version " VERSION "\n");
	    exit (0);
	}
	if (!strcmp (argv[1], "--max-restarts")) {
	    if (strspn (argv[2], "0123456789") != strlen (argv[2]))
		usage ();
	    option_restarts = (int) atoi (argv[2]);
	    argv++;
	    found = 1;
	}
	if (!strcmp (argv[1], "--robust")
	    || !strcmp (argv[1], "-d")) {
	    option_robust = 1;
	    found = 1;
	}
	if (!strcmp (argv[1], "-i")
	    || !strcmp (argv[1], "--interval")) {
	    if (strspn (argv[2], "0123456789.") != strlen (argv[2]))
		usage ();
	    per_second = (double) atof (argv[2]);
	    argv++;
	    found = 1;
	}
	if (argv[1][0] == '-') {
	    int i;
	    if (strspn (argv[1] + 1, "0123456789") ==
		strlen (argv[1] + 1) && strlen (argv[1] + 1) > 0) {
		send_signal = (int) atoi (argv[1] + 1);
		found = 1;
	    } else {
		for (i = 0;
		     i <
		     sizeof (all_signals) / sizeof (struct _all_signals);
		     i++) {
		    if (!strcmp (all_signals[i].name + 3, argv[1] + 1)
			|| !strcmp (all_signals[i].name, argv[1] + 1)) {
			send_signal = all_signals[i].n;
			found = 1;
			break;
		    }
		}
	    }
	}
	if (!found)
	    usage ();
    }
    if (argc <= 1)
	usage ();
    if (strspn
	(argv[1],
	 "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789-_./")
	!= strlen (argv[1])) {
	fprintf (stderr,
		 "run: <app_name> must contain no weird characters\n");
	exit (127);
    }
    program = strrchr (argv[1], '/');
    program = program ? program + 1 : argv[1];
    if (strlen (program) < 2) {
	fprintf (stderr, "run: <app_name> is short - huh?\n");
	exit (127);
    }
    mkdir (LOCK_DIR, 0700);
    strcpy (lockname, LOCK_DIR);
    strncat (lockname, program, 256);
    strcat (lockname, ".run");
    sprintf (lockname + strlen (lockname), "-%ld",
	     (long) ((uid_t) getuid ()));
    f = open (lockname, O_CREAT | O_RDWR, 0600);
    if (f == -1) {
	perror ("run: cannot create or open lock file");
	exit (127);
    }

    msg_key = ftok (lockname, '\000');
    if (msg_key == -1) {
	perror ("run: ftok failed");
	exit (127);
    }

/* turn off root privileges */
    user_priv ();
    if (!strcmp (progname, "isrunning")) {
	if (get_lock (f, 0, 20) == -1) {
	    if (verbose) {
		int r;
		pid_t parent;
		read (f, lockcontent, 20);
		parent = atol (lockcontent);
		if (parent > 0) {
		    struct my_msgbuf msgp;
		    msg_queue = msgget (msg_key, S_IWUSR | S_IRUSR);
		    if (msg_queue == -1) {
			perror ("isrunning: msgget");
			exit (127);
		    }
		    msgp.m.mtype = RUNMSG_REQUEST;
		    r =
			msgsnd (msg_queue, (struct msgbuf *) &msgp,
				sizeof (msgp), 0);
/* */
		    if (!r) {
			kill (parent, SIGHUP);
			r = msgrcv (msg_queue, (struct msgbuf *) &msgp,
				    sizeof (msgp), RUNMSG_STATUS_OR_PID,
				    0);
			if (r == -1) {
			    perror ("stop: msgrcv");
			    msgp.pid = -1;
			} else if (msgp.m.mtype != RUNMSG_STATUS_OR_PID) {
			    fprintf (stderr, "bad message???\n");
			    exit (127);
			}
		    } else {
			perror ("stop: msgsnd");
			exit (127);
		    }
		    if (msgp.pid > 0)
			printf
			("%s is running as pid %ld, parent as pid %ld\n",
			 program, (long) msgp.pid, (long) parent);
		    else
			printf
			("%s is between restarts, parent as pid %ld\n",
			 program, (long) parent);
		} else {
		    fprintf (stderr, "stop: lock file corrupted\n");
		    exit (127);
		}
	    }
	    exit (0);
	} else {
	    if (verbose) {
		fprintf (stderr, "%s is not running\n", program);
	    }
	    exit (1);
	}
    }
    if (!strcmp (progname, "stop")) {
	if (get_lock (f, 0, 20) == -1) {
	    pid_t parent;
	    read (f, lockcontent, 20);
	    parent = atol (lockcontent);
	    if (parent > 0) {
		pid_t r;
		struct my_msgbuf msgp;
		msg_queue = msgget (msg_key, S_IWUSR | S_IRUSR);
		if (msg_queue == -1) {
		    perror ("stop: msgget");
		    exit (127);
		}
		msgp.m.mtype = option_force ? RUNMSG_EXIT : RUNMSG_SIGNAL;
		msgp.m.mtext[0] = send_signal;
		r =
		    msgsnd (msg_queue, (struct msgbuf *) &msgp,
			    sizeof (msgp), IPC_NOWAIT);
/* we have to interrupt run if it is usleep()ing - we use SIGHUP */
		if (!r) {
		    if (msgp.m.mtype == RUNMSG_SIGNAL) {
			kill (parent, SIGHUP);
			r = msgrcv (msg_queue, (struct msgbuf *) &msgp,
				    sizeof (msgp), RUNMSG_STATUS_OR_PID,
				    0);
/* should only happen if run is in a usleep when stop interrupts: */
			if (r == -1)
			    perror ("stop: msgrcv");
			else
			    destroy_queue ();
			if (r != -1)
			    if (msgp.m.mtype != RUNMSG_STATUS_OR_PID)
				fprintf (stderr, "bad message???\n");
			while (get_lock (f, 0, 20) == -1)
			    usleep (20000L);
			if (r == -1)
			    exit (127);
			status_exit (msgp.status);
		    } else {
			while (get_lock (f, 0, 20) == -1)
			    usleep (20000L);
			exit (0);
		    }
		} else {
		    perror ("stop: msgsnd");
		}
	    } else {
		fprintf (stderr, "stop: lock file corrupted\n");
	    }
	} else {
	    if (verbose)
		fprintf (stderr, "stop: program is not running\n");
	}
	exit (127);
    }

    if (option_daemon) {
	pid_t r;
	r = fork ();
	if (r == -1) {
	    perror ("run: fork() failed");
	    exit (127);
	}
	if (r)
	    exit (0);
	r = fork ();
	if (r == -1) {
	    perror ("run: fork() failed");
	    exit (127);
	}
	if (r)
	    exit (0);
    }

    if (option_robust) {
#ifdef SIGINT
	signal (SIGINT, SIG_IGN);
#endif
#ifdef SIGQUIT
	signal (SIGQUIT, SIG_IGN);
#endif
#ifdef SIGPIPE
	signal (SIGPIPE, SIG_IGN);
#endif
    }

    if (write_lock (f, 0, 20) == -1) {
	perror ("run: lock failed");
	exit (127);
    }

    sprintf (lockcontent, "%20d", getpid ());
    write (f, lockcontent, 20);
    a = (char **) malloc ((argc + 1) * sizeof (char *));
    memset (a, 0, (argc + 1) * sizeof (char *));
    memcpy (a, argv + 1, (argc - 1) * sizeof (char *));
    msg_queue = msgget (msg_key, IPC_CREAT | S_IWUSR | S_IRUSR);
    if (msg_queue == -1) {
	perror ("run: msgget failed");
	exit (127);
    }
/* clear the queue of any previous messages */
    while (msgrcv
	   (msg_queue, (struct msgbuf *) &msgp, sizeof (msgp), 0,
	    IPC_NOWAIT) > 0);
    if (option_null) {
	close (0);
	close (1);
	close (2);
	dup (open ("/dev/null", O_RDWR));
	dup (open ("/dev/null", O_RDWR));
	dup (open ("/dev/null", O_RDWR));
    }

    for (;;) {
	if (!child_pid) {
	    double s;
	    s = t + per_second - get_time ();
	    if (s > 0.0)
		usleep ((unsigned long) ((double) s * 1000000.0));
	    s = t + per_second - get_time ();
	    if (s <= 0.0) {
		t = get_time ();
		child_pid = fork ();
		if (child_pid == -1) {
		    perror ("run: fork() failed");
		    destroy_queue ();
		    exit (127);
		}
		if (!child_pid)
		    my_execvp (a);
	    }
	}
	while (msgrcv
	       (msg_queue, (struct msgbuf *) &msgp, sizeof (msgp),
		RUNMSG_STATUS_OR_PID, MSG_EXCEPT) <= 0);
read_again:
	if (msgp.m.mtype == RUNMSG_REQUEST) {
	    msgp.pid = child_pid;
	    msgp.m.mtype = RUNMSG_STATUS_OR_PID;
	    if (msgsnd
		(msg_queue, (struct msgbuf *) &msgp, sizeof (msgp),
		 0)) perror ("run: msgsnd");
	} else if (msgp.m.mtype == RUNMSG_SIGNAL) {
	    if (child_pid <= 0) {	/* child already dead */
		msgp.status = last_status;
		msgp.m.mtype = RUNMSG_STATUS_OR_PID;
		if (msgsnd
		    (msg_queue, (struct msgbuf *) &msgp, sizeof (msgp),
		     0)) perror ("run: msgsnd");
		status_exit (last_status);
	    }
	    option_restarts = 1;	/* exit immediately when child next dies */
	    if (msgp.m.mtext[0])
		kill (child_pid, msgp.m.mtext[0]);
	} else if (msgp.m.mtype == RUNMSG_EXIT) {
	    if (msgp.m.mtext[0] && child_pid > 0)
		kill (child_pid, msgp.m.mtext[0]);
	    break;
	}
	if (msgrcv (msg_queue, (struct msgbuf *) &msgp, sizeof (msgp),
		RUNMSG_STATUS_OR_PID, MSG_EXCEPT | IPC_NOWAIT) > 0)
	    goto read_again;
    }
    destroy_queue ();
    exit (127);
    return 0;			/* prevents warning */
}
