#  Copyright (C) 1999-2005
#  Smithsonian Astrophysical Observatory, Cambridge, MA, USA
#  For conditions of distribution and use, see copyright notice in "copyright"

package provide DS9 1.0

# Public Functions

proc CreateColorbar {} {
    global ds9
    global colorbar

    switch -- $ds9(visual) {
	pseudocolor {
	    $ds9(canvas) create colorbar$ds9(visual)$ds9(depth) \
		-private $colorbar(private) \
		-width $colorbar(width) -height $colorbar(height) \
		-tag colorbar

	    # And set the two windows that where created before the 
	    # colorbar was created, to the new colormap

	    colorbar set colormap window "$ds9(main)"

	    # This is a kluge, but it works. We want to reassign the bg color 
	    # for $ds9(main) from the old colormap to the new colormap
	    
	    $ds9(main) configure -background [$ds9(main) cget -background]
	}
	truecolor {
	    $ds9(canvas) create colorbar$ds9(visual)$ds9(depth) \
		-width $colorbar(width) -height $colorbar(height) \
		-colors 2048 -tag colorbar
	    $ds9(canvas) create colorbarrgb$ds9(visual)$ds9(depth) \
		-width $colorbar(width) -height $colorbar(height) \
		-colors 2048 -tag colorbarrgb
	}
    }
}

proc InitColorbar {} {
    global current
    global colorbar

    set current(colorbar) colorbar

    colorbar map "{$colorbar(map)}"
    colorbar invert $colorbar(invert)
}

proc ResetColormap {} {
    global current
    global rgb

    $current(colorbar) reset
    if {$current(frame) != "" } {
	RGBEvalLock rgb(lock,colorbar) "$current(frame) colormap [$current(colorbar) get colormap]"
	set colorbar(invert) [$current(colorbar) get invert]
    }

    UpdateColormapDialog
}

proc LoadColormap {} {
    LoadColormapFile [OpenFileDialog colorbarfbox]
}

proc LoadColormapFile {filename} {
    global current
    global colorbar
    global ds9

    if {$filename != {}} {
	colorbar load "\{$filename\}"
	set id [colorbar get id]
	set colorbar(map) [colorbar get name]

	$ds9(mb).color insert $colorbar(count) radiobutton \
	    -label "$colorbar(map)" \
	    -variable colorbar(map) -command "ChangeColormapID $id"

	if {[winfo exist $colorbar(top)]} {
	    $colorbar(mb).color insert [expr $colorbar(count)-1] \
		radiobutton \
		-label "$colorbar(map)" \
		-variable colorbar(map) -command "ChangeColormapID $id"
	}
	incr colorbar(count)

	ChangeColormapID $id
    }
}

proc SaveColormap {} {
    global ds9
    global current

    FileLast colorbarfbox [colorbar get file name]
    set filename [SaveFileDialog colorbarfbox]
    if {$filename != {}} {
	colorbar save "\{$filename\}"
    }
}

proc LoadContrastBias {} {
    global colorbar
    global colorbardlg

    set filename [OpenFileDialog contrastbiasfbox]
    if {$filename != {}} {
	if {![catch {set ch [open $filename r]}]} {
	    set l [gets $ch]
	    close $ch
	    set colorbardlg(constrast) [lindex $l 0]
	    set colorbardlg(bias) [lindex $l 1]

	    BeginAdjustColormap
	    AdjustColormap 1
	    EndAdjustColormap
	}
    }
}

proc SaveContrastBias {} {
    global colorbar
    global colorbardlg

    set filename [SaveFileDialog contrastbiasfbox]
    if {$filename != {}} {
	if {![catch {set ch [open $filename w]}]} {
	    puts $ch "$colorbardlg(contrast) $colorbardlg(bias)"
	    close $ch
	}
    }
}

proc ButtonColormap {x y} {
    global colorbar
    global current
    global rgb
    global cursor

    # turn off blinking cursor
    if {$cursor(timer)} {
	catch {after cancel $cursor(id)}
	set cursor(id) 0
    }

    if {$current(frame) != ""} {
	# we need to hold the current frame, since we may be blinking
	set colorbar(frame) $current(frame)
	$colorbar(frame) colormap begin $x $y
    }
}

proc MotionColormap {x y} {
    global colorbar
    global current
    global canvas
    global rgb

    # X sets bias
    set cursorX \
	[expr double($x-$canvas(width)/2+$colorbar(speed)/2)/$colorbar(speed)]
    # Y set contrast
    set cursorY [expr double($y)/$colorbar(speed) * $colorbar(mult)]

    if {$colorbar(frame) != {}} {
	RGBEvalLockColorbar "$current(colorbar) adjust $cursorY $cursorX"
	# only update the current colorbar frame
	$colorbar(frame) colormap motion [$current(colorbar) get colormap]
    } else {
	RGBEvalLockColorbar "$current(colorbar) adjust $cursorY $cursorX"
    }

    UpdateColormapDialog
}

proc ReleaseColormap {x y} {
    global colorbar
    global current
    global rgb
    global cursor

    # and turn on blinking cursor if needed
    if {$cursor(timer)} {
	CursorTimer
    }

    # only update the current colorbar frame
    if {$colorbar(frame) != ""} {
	$colorbar(frame) colormap end [$current(colorbar) get colormap]
	set colorbar(frame) {}
    }

    UpdateColormapDialog
}

proc CreateColorMenus {} {
    global current
    global colorbar
    global ds9

    set id [colorbar list id]

    foreach i $id {
	set name [colorbar get name $i]
	$ds9(mb).color insert $colorbar(count) radiobutton \
	    -label "$name" -variable colorbar(map) \
	    -command "ChangeColormapID $i"
	$ds9(mb).prefs.color insert [expr $colorbar(count)-1] radiobutton \
	    -label "$name" -variable prefs(colorbar,map)
	incr colorbar(count)
    }
}

proc ChangeColormapID {id} {
    global current
    global colorbar
    
    SetWatchCursor
    colorbar map $id
    if {$current(frame) != "" } {
	$current(frame) colormap [colorbar get colormap]
	set colorbar(map) [colorbar get name]
	set colorbar(invert) [colorbar get invert]
    }
    UnsetWatchCursor

    UpdateColormapDialog
}

proc ChangeColormap {} {
    global current
    global colorbar
    
    SetWatchCursor
    colorbar map "{$colorbar(map)}"
    if {$current(frame) != "" } {
	$current(frame) colormap [colorbar get colormap]
	set colorbar(invert) [colorbar get invert]
    }
    UnsetWatchCursor

    UpdateColormapDialog
}

proc MatchColorbars {} {
    global ds9
    global current
    global colorbar

    SetWatchCursor
    set id [$current(colorbar) get colormap]
    foreach f $ds9(frames) {
	if {$f!=$current(frame) && [$f get type]==[$current(frame) get type]} {
	    $f colormap $id
	}
    }
    UnsetWatchCursor
}

proc InvertColorbar {} {
    global current
    global colorbar

    SetWatchCursor
    colorbar invert $colorbar(invert)
    if {$current(frame) != "" } {
	$current(frame) colormap [colorbar get colormap]
    }
    UnsetWatchCursor

    UpdateColormapDialog
}

proc ColorbarArea {} {
    global ds9
    global colorbar

    foreach f $ds9(frames) {
	$f colormap area $colorbar(area) $colorbar(mode)
    }
}

proc ColormapDialog {} {
    global colorbar
    global colorbardlg
    global ds9
    global menu

    # see if we already have a window visible

    if [winfo exist $colorbar(top)] {
	raise $colorbar(top)
	return
    }

    set w $colorbar(top)
    set title "Colormap Parameters"
    set length 300

    # create the window

    toplevel $w -colormap $ds9(main)
    wm title $w $title
    wm iconname $w $title
    wm group $w $ds9(top)
    wm protocol $w WM_DELETE_WINDOW ColormapDestroyDialog

    $w configure -menu $colorbar(mb)

    menu $colorbar(mb) -tearoff 0
    $colorbar(mb) add cascade -label File -menu $colorbar(mb).file
    $colorbar(mb) add cascade -label Color -menu $colorbar(mb).color

    menu $colorbar(mb).file -tearoff 0 -selectcolor $menu(selectcolor)
    $colorbar(mb).file add command -label "Apply" -command ColormapApplyDialog
    $colorbar(mb).file add separator
    $colorbar(mb).file add command -label "Load Colormap..." \
	-command LoadColormap
    $colorbar(mb).file add command -label "Save Colormap..." \
	-command SaveColormap
    $colorbar(mb).file add separator
    $colorbar(mb).file add command -label "Load Contrast/Bias..." \
	-command LoadContrastBias
    $colorbar(mb).file add command -label "Save Contrast/Bias..." \
	-command SaveContrastBias
    $colorbar(mb).file add separator
    $colorbar(mb).file add command -label "Close" \
	-command ColormapDestroyDialog

    menu $colorbar(mb).color -tearoff 0 -selectcolor $menu(selectcolor)

    set id [colorbar list id]
    set count 0
    foreach i $id {
	set name [colorbar get name $i]
	$colorbar(mb).color insert $count radiobutton \
	    -label "$name" -variable colorbar(map) \
	    -command "ChangeColormapID $i"
	incr count
    }
    $colorbar(mb).color add separator
    $colorbar(mb).color add checkbutton -label "Invert Colormap" \
	-variable colorbar(invert) -command InvertColorbar
    $colorbar(mb).color add command -label "Reset Colormap" \
	-command ResetColormap

    UpdateColormapDialog

    frame $w.cb  -relief groove -borderwidth 2
    frame $w.buttons -relief groove -borderwidth 2
    pack $w.cb $w.buttons -ipadx 4 -ipady 4 -fill x -expand true

    frame $w.cb.contrast
    scale $w.cb.contrast.slider -from 0 -to 10 -length $length \
	-variable colorbardlg(contrast) -orient horizontal -label "Contrast" \
	-tickinterval 2 -showvalue false -resolution .1 \
	-command AdjustColormap
    entry $w.cb.contrast.value -textvariable colorbardlg(contrast) -width 7
    pack $w.cb.contrast.slider $w.cb.contrast.value -side left -padx 4
    pack $w.cb.contrast.value

    frame $w.cb.bias
    scale $w.cb.bias.slider -from 0 -to 1 -length $length \
	-variable colorbardlg(bias) -orient horizontal -label "Bias" \
	-tickinterval .2 -showvalue false -resolution .01 \
	-command AdjustColormap
    entry $w.cb.bias.value -textvariable colorbardlg(bias) -width 7
    pack $w.cb.bias.slider $w.cb.bias.value -side left -padx 4
    pack $w.cb.bias.value

    pack $w.cb.contrast $w.cb.bias -side top

    button $w.buttons.apply -text "Apply" -command ColormapApplyDialog
    button $w.buttons.close -text "Close" -command ColormapDestroyDialog
    pack $w.buttons.apply $w.buttons.close -side left -padx 10 -expand true

    bind $w.cb.contrast.slider <Button-1> BeginAdjustColormap
    bind $w.cb.contrast.slider <ButtonRelease-1> EndAdjustColormap
    bind $w.cb.bias.slider <Button-1> BeginAdjustColormap
    bind $w.cb.bias.slider <ButtonRelease-1> EndAdjustColormap
}

proc ColormapDestroyDialog {} {
    global colorbar
    global colorbardlg

    if {[winfo exist $colorbar(top)]} {
	destroy $colorbar(top)
	destroy $colorbar(mb)
    }

    if {[info exist colorbardlg]} {
	unset colorbardlg
    }
}

proc ColormapApplyDialog {} {
    BeginAdjustColormap
    AdjustColormap 1
    EndAdjustColormap
}

proc BeginAdjustColormap {} {
    global current
    global colorbar
    global rgb

    set colorbar(adjustok) 1
    if {$current(frame) != ""} {
	RGBEvalLock rgb(lock,colorbar) "$current(frame) colormap begin"
    }
}

proc AdjustColormap {value} {
    # value is ignored
    global current
    global colorbar
    global colorbardlg
    global rgb

    if {[info exists colorbar(adjustok)]} {
	if {$current(frame) != ""} {
	    RGBEvalLockColorbar "$current(colorbar) adjust $colorbardlg(contrast) $colorbardlg(bias)"
	    RGBEvalLock rgb(lock,colorbar) "$current(frame) colormap motion [$current(colorbar) get colormap]"
	}
    }
}

proc EndAdjustColormap {} {
    global current
    global colorbar
    global rgb

    if {[info exists colorbar(adjustok)]} {
	unset colorbar(adjustok)
	if {$current(frame) != ""} {
	    RGBEvalLock rgb(lock,colorbar) "$current(frame) colormap end [$current(colorbar) get colormap]"
	}
    }
}

proc UpdateColormapMenu {} {
    global ds9
    global current
    global menu
    global colorbar

    global debug
    if {$debug(tcl,update)} {
	puts "UpdateColormapMenu"
    }

    if {$current(frame) != ""} {
	if {[$current(frame) get type] == "rgb"} {
	    $ds9(mb).color entryconfig "Invert Colormap" -state disabled
	    for {set i 1} {$i<$colorbar(count)} {incr i} {
		$ds9(mb).color entryconfig $i -state disabled
	    }

	    for {set i 0} {$i<$colorbar(button1)} {incr i} {
		$ds9(buttons).color1.$i configure -state disabled
	    }
	    for {set i $colorbar(button1)} {$i<$colorbar(button2)} {incr i} {
		$ds9(buttons).color2.$i configure -state disabled
	    }
	    $ds9(buttons).color3.invert configure -state disabled

	} else {
	    $ds9(mb).color entryconfig "Invert Colormap" -state normal
	    for {set i 1} {$i<$colorbar(count)} {incr i} {
		$ds9(mb).color entryconfig $i -state normal
	    }

	    for {set i 0} {$i<$colorbar(button1)} {incr i} {
		$ds9(buttons).color1.$i configure -state normal
	    }
	    for {set i $colorbar(button1)} {$i<$colorbar(button2)} {incr i} {
		$ds9(buttons).color2.$i configure -state normal
	    }
	    $ds9(buttons).color3.invert configure -state normal
	}
    } else {
	$ds9(mb).color entryconfig "Invert Colormap" -state normal
	for {set i 1} {$i<$colorbar(count)} {incr i} {
	    $ds9(mb).color entryconfig $i -state normal
	}

	for {set i 0} {$i<$colorbar(button1)} {incr i} {
	    $ds9(buttons).color1.$i configure -state normal
	}
	for {set i $colorbar(button1)} {$i<$colorbar(button2)} {incr i} {
	    $ds9(buttons).color2.$i configure -state normal
	}
	$ds9(buttons).color3.invert configure -state normal
    }
}

proc UpdateColormapDialog {} {
    global current
    global colorbar
    global colorbardlg

    global debug
    if {$debug(tcl,update)} {
	puts "UpdateColormapDialog"
    }

    if {[winfo exist $colorbar(top)]} {
	set colorbardlg(contrast) [$current(colorbar) get contrast]
	set colorbardlg(bias) [$current(colorbar) get bias]

	if {$current(frame) != ""} {
	    if {[$current(frame) get type] == "rgb"} {
		$colorbar(mb).file entryconfig "Load Colormap..." \
		    -state disabled
		$colorbar(mb).file entryconfig "Save Colormap..." \
		    -state disabled
		for {set i 0} {$i<[expr $colorbar(count)-1]} {incr i} {
		    $colorbar(mb).color entryconfig $i -state disabled
		}
		$colorbar(mb).color entryconfig "Invert Colormap" \
		    -state disabled
	    } else {
		$colorbar(mb).file entryconfig "Load Colormap..." -state normal
		$colorbar(mb).file entryconfig "Save Colormap..." -state normal
		for {set i 0} {$i<[expr $colorbar(count)-1]} {incr i} {
		    $colorbar(mb).color entryconfig $i -state normal
		}
		$colorbar(mb).color entryconfig "Invert Colormap" -state normal
	    }
	} else {
	    $colorbar(mb).file entryconfig "Load Colormap..." -state normal
	    $colorbar(mb).file entryconfig "Save Colormap..." -state normal
	    for {set i 0} {$i<[expr $colorbar(count)-1]} {incr i} {
		$colorbar(mb).color entryconfig $i -state normal
	    }
	    $colorbar(mb).color entryconfig "Invert Colormap" -state normal
	}
    }
}

proc ProcessCmapCmd {varname iname} {
    upvar $varname var
    upvar $iname i

    global ds9
    global current
    global colorbar
    global rgb

    switch -- [string tolower [lindex $var $i]] {
	file {
	    incr i
	    set fn [file normalize [lindex $var $i]]
	    LoadColormapFile $fn
	    FileLastFull colormapfbox $fn
	}
	invert {
	    incr i
	    set colorbar(invert) [FromYesNo [lindex $var $i]]
	    InvertColorbar
	}
	value {
	    # we need to be realized
	    RealizeDS9

	    incr i
	    set c [lindex $var $i]
	    incr i
	    set b [lindex $var $i]
	    if {$current(frame) != ""} {
		RGBEvalLockColorbar "$current(colorbar) adjust $c $b"
		RGBEvalLock rgb(lock,colorbar) "$current(frame) colormap begin"
		RGBEvalLock rgb(lock,colorbar) "$current(frame) colormap motion [$current(colorbar) get colormap]"
		RGBEvalLock rgb(lock,colorbar) "$current(frame) colormap end [$current(colorbar) get colormap]"
	    }
	}
	default {
	    set colorbar(map) [lindex $var $i]
	    ChangeColormap
	}
    }
    UpdateColormapDialog
}
