// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include <stdlib.h>
#include <string.h>

#include "tcl.h"
#include <Xlib.h>
#include <Xutil.h>

#include "colorbarpseudo8.h"
#include "util.h"
#include "ps.h"

// Tk Canvas Widget Function Declarations

int ColorbarPseudoColor8CreateProc(Tcl_Interp*, Tk_Canvas, Tk_Item*, int,
				   Tcl_Obj *const []);

// Colorbar Specs

static Tk_CustomOption tagsOption = {
  Tk_CanvasTagsParseProc, Tk_CanvasTagsPrintProc, NULL
};

struct ColorbarPseudoColor8Options {
  Tk_Item item;              // required by tk
  int x, y;                  // Coordinates of positioning point on canvas
  int width;                 // widget width
  int height;                // widget height
  Tk_Anchor anchor;          // Where to anchor widget relative to x,y
  char* cmdName;             // Suggested Tcl command name

  Widget* widget;            // pointer to widget class

  int minColors;
  int maxColors;
  int privateCmap;
  int privateColors;
};

static Tk_ConfigSpec colorbarPseudoColor8Specs[] = {

  {TK_CONFIG_STRING, "-command", NULL, NULL, "colorbar",
   Tk_Offset(ColorbarPseudoColor8Options, cmdName), 
   TK_CONFIG_OPTION_SPECIFIED, NULL},
  {TK_CONFIG_INT, "-x", NULL, NULL, "1",
   Tk_Offset(ColorbarPseudoColor8Options, x), TK_CONFIG_OPTION_SPECIFIED, 
   NULL},
  {TK_CONFIG_INT, "-y", NULL, NULL, "1",
   Tk_Offset(ColorbarPseudoColor8Options, y), TK_CONFIG_OPTION_SPECIFIED, 
   NULL},
  {TK_CONFIG_INT, "-width", NULL, NULL, "512",
   Tk_Offset(ColorbarPseudoColor8Options, width), TK_CONFIG_OPTION_SPECIFIED, 
   NULL},
  {TK_CONFIG_INT, "-height", NULL, NULL, "22",
   Tk_Offset(ColorbarPseudoColor8Options, height), TK_CONFIG_OPTION_SPECIFIED,
   NULL},
  {TK_CONFIG_ANCHOR, "-anchor", NULL, NULL, "nw",
   Tk_Offset(ColorbarPseudoColor8Options, anchor), 0, NULL},
  {TK_CONFIG_CUSTOM, "-tags", NULL, NULL, NULL,
   0, TK_CONFIG_NULL_OK, &tagsOption},

  {TK_CONFIG_INT, "-min", NULL, NULL, "80",
   Tk_Offset(ColorbarPseudoColor8Options, minColors), 0, NULL},
  {TK_CONFIG_INT, "-max", NULL, NULL, "200",
   Tk_Offset(ColorbarPseudoColor8Options, maxColors), 0, NULL},
  {TK_CONFIG_BOOLEAN, "-private", NULL, NULL, "false",
   Tk_Offset(ColorbarPseudoColor8Options, privateCmap), 0, NULL},
  {TK_CONFIG_INT, "-privatecolors", NULL, NULL, "128",
   Tk_Offset(ColorbarPseudoColor8Options, privateColors), 0, NULL},

  {TK_CONFIG_END, NULL, NULL, NULL, NULL, 0, 0, NULL},
};

// Tk Static Structure

static Tk_ItemType colorbarPseudoColor8Type = {
  "colorbarpseudocolor8",       // name
  sizeof(ColorbarPseudoColor8Options), // size
  ColorbarPseudoColor8CreateProc,  // configProc
  colorbarPseudoColor8Specs,       // configSpecs
  WidgetConfigProc,             // configProc
  WidgetCoordProc,              // coordProc
  WidgetDeleteProc,             // deleteProc
  WidgetDisplayProc,            // displayProc
  0,                            // alwaysRedraw
  WidgetPointProc,              // pointProc
  WidgetAreaProc,               // areaProc
  WidgetPostscriptProc,         // postscriptProc
  WidgetScaleProc,              // scaleProc
  WidgetTranslateProc,          // translateProc
  (Tk_ItemIndexProc*)NULL,      // indexProc
  (Tk_ItemCursorProc*)NULL,     // icursorProc
  (Tk_ItemSelectionProc*)NULL,  // selectionProc
  (Tk_ItemInsertProc*)NULL,     // insertProc
  (Tk_ItemDCharsProc*)NULL,     // dCharsProc
  (Tk_ItemType*)NULL            // nextPtr
};

// Non-Member Functions

int ColorbarPseudoColor8_Init(Tcl_Interp* interp)
{
  Tk_CreateItemType(&colorbarPseudoColor8Type);
  return TCL_OK;
}

int ColorbarPseudoColor8CreateProc(Tcl_Interp* interp, Tk_Canvas canvas, 
				   Tk_Item* item, int argc, 
				   Tcl_Obj *const argv[])
{
  ColorbarPseudoColor8* colorbar = 
    new ColorbarPseudoColor8(interp, canvas, item);

  // and set default configuration

  if (colorbar->configure(argc, (const char**)argv, 0) != TCL_OK) {
    delete colorbar;
    Tcl_AppendResult(interp, " error occured while creating colorbar.", NULL);
    return TCL_ERROR;
  }

  return TCL_OK;
}

// ColorbarPseudoColor8

ColorbarPseudoColor8::ColorbarPseudoColor8(Tcl_Interp* i, Tk_Canvas c, 
					   Tk_Item* item) 
  : Colorbar(i, c, item)
{
  configSpecs = colorbarPseudoColor8Specs;  // colorbar configure options

  colormap = 0;
  privateUsed = 0;

  loadDefaultCMaps();
}

ColorbarPseudoColor8::~ColorbarPseudoColor8()
{
  // Tk will free the private map because we used Tk_SetWindowColormap
  //  if (privateUsed)
  //    Tk_FreeColormap(display, colormap);
}

void ColorbarPseudoColor8::setColormapWindowCmd(char* str)
{
  Tk_Window win = Tk_NameToWindow(interp, str, tkwin);

  // Check to see if we have the same visual (and depth)

  if (Tk_Visual(tkwin) == Tk_Visual(win) && Tk_Depth(tkwin) == Tk_Depth(win)) {
    if (win)
      Tk_SetWindowColormap(win, colormap);
    else
      result = TCL_ERROR;
  }
  else
    cerr << "Colorbar Internal Error: Visual mismatch" << endl;
}

// UpdatePixmap. This function is responsable for creating a valid 
// pixmap the size of the current Colorbar

int ColorbarPseudoColor8::updatePixmap(const BBox& bb)
{
  // create a valid pixmap if needed
  // bb is in canvas coords

  if (!pixmap) {
    int& width = options->width;
    int& height = options->height;

    if (!(pixmap = Tk_GetPixmap(display, Tk_WindowId(tkwin), width, height, 
				depth))) {
      cerr << "ColorbarPseudoColor8 Internal Error: Unable to Create Pixmap" 
	   << endl;
      exit(1);
    }

    XImage* xmap = XGETIMAGE(display, pixmap, 0, 0, width, height, AllPlanes, 
			     ZPixmap);
    if (!xmap) {
      cerr << "ColorbarPseudoColor8 Internal Error: Unable to Create XImage" 
	   << endl;
      exit(1);
    }
    char* data = xmap->data;

    // Fill in colorbar data
    // --Calculate first row

    for (int i=0; i<width; i++)
      data[i] = (char)colorIndex[i*colorCount/width];

    // --and duplicate for remaining rows

    for (int j=1; j<height; j++)
      memcpy(data+(j*xmap->bytes_per_line), data, xmap->bytes_per_line);

    // update the pixmap from map image and clean up

    XPutImage(display, pixmap, gc, xmap, 0, 0, 0, 0, width, height);
    XDestroyImage(xmap);

    // border
    renderBorder();
  }

  return TCL_OK;
}

// initColormap allocates the requested number of color colorCells

int ColorbarPseudoColor8::initColormap()
{
  // initialize default or private colormap

  if (((ColorbarPseudoColor8Options*)options)->privateCmap) {
    if (initPrivateMap() == TCL_ERROR)
      return TCL_ERROR;
  }
  else
    if (initDefaultMap() == TCL_ERROR)
      if (initPrivateMap() == TCL_ERROR)
	return TCL_ERROR;

  // by now, we should have a map with at least min colors

  updateColors();

  return TCL_OK;
}

int ColorbarPseudoColor8::initDefaultMap()
{
  // grap default colormap

  colormap = Tk_Colormap(tkwin);

  // see if we can allocate at least min number of colors requested

  int minColors = ((ColorbarPseudoColor8Options*)options)->minColors;
  int maxColors = ((ColorbarPseudoColor8Options*)options)->maxColors;

  unsigned long* cells = 
    new unsigned long[((ColorbarPseudoColor8Options*)options)->maxColors];

  for (int k=maxColors; k>=minColors-1; k--)
    if (XAllocColorCells(display, colormap, False, 0, 0, cells, k)) {
      colorCount = k;

      // copy from long to unsign short

      colorIndex = new unsigned short[colorCount];
      for (int i=0; i<colorCount; i++)
	colorIndex[i] = cells[i];
      colorCells = new unsigned char[colorCount*3];

      delete cells;
      return TCL_OK;
    }

  delete cells;
  return TCL_ERROR;
}

int ColorbarPseudoColor8::initPrivateMap()
{
  int privateColors = ((ColorbarPseudoColor8Options*)options)->privateColors;
#ifndef _WIN32
  int defCount = 254 - 8 - 8 - privateColors;
#else
  // windows locks up 20 colors at the start
  int defCount = 254 - 8 - 8 - privateColors -20;
#endif

  // first, grap most of the default colors from the current color map

  Colormap defColormap = Tk_Colormap(tkwin);

  XColor* defColors= new XColor[defCount];
  for (int i=0; i<defCount; i++)
    defColors[i].pixel = i;
  XQueryColors(display, defColormap, defColors, defCount);

  // now, allocate a new private map

  colormap = Tk_GetColormap(interp, tkwin, "new");

  // allocate the default colors

  unsigned long* defCells = new unsigned long[defCount];
  if (!XAllocColorCells(display, colormap, False, 0, 0, defCells, defCount)) {
    Tcl_AppendResult(interp, " unable to allocate default colors",
		     " in private colormap.", NULL);
    return TCL_ERROR;
  }

  // store default colors

  XStoreColors(display, colormap, defColors, defCount);

  // and finally, allocate the dynamic colors

  colorCount = privateColors;
  unsigned long* cells = new unsigned long[colorCount];
  if (!XAllocColorCells(display, colormap, False, 0, 0, cells, colorCount)) {
    Tcl_AppendResult(interp, " unable to allocate maxium number of colors",
		     " in private colormap.", NULL);
    return TCL_ERROR;
  }

  // copy from long to unsign char

  colorIndex = new unsigned short[colorCount];
  for (int j=0; j<colorCount; j++)
    colorIndex[j] = cells[j];
  colorCells = new unsigned char[colorCount*3];

  // and init the first DEFCMAP colors

  privateUsed = 1;
  // assign private map to this window
  Tk_SetWindowColormap(tkwin, colormap); 

  // force the private map to be loaded now
  XInstallColormap(display, colormap); 

  delete cells;
  delete defCells;
  delete defColors;
  return TCL_OK;
}

//  Initailizes the current Colormap from the current ColorMapInfo

void ColorbarPseudoColor8::updateColors()
{
  XColor* colors = new XColor[colorCount];

  if (currentcmap)
    for(int i=0, j=colorCount-1; i<colorCount; i++, j--) {
      int index = invert ? calcContrastBias(j) : calcContrastBias(i);

      // fill in colors

      colors[i].pixel = colorIndex[i];
      colors[i].red = currentcmap->getRedShrt(index, colorCount);
      colors[i].green = currentcmap->getGreenShrt(index, colorCount);
      colors[i].blue = currentcmap->getBlueShrt(index, colorCount);
      colors[i].flags = DoRed | DoGreen | DoBlue;

      // fill in rest of colorCells
      // Note: we fill the array bgr

      colorCells[i*3] = currentcmap->getBlueChar(index, colorCount);
      colorCells[i*3+1] = currentcmap->getGreenChar(index, colorCount);
      colorCells[i*3+2] = currentcmap->getRedChar(index, colorCount);
    }
  
  XStoreColors(display, colormap, colors, colorCount);

  delete colors;
}

void ColorbarPseudoColor8::psLevel2Head(PSColorSpace mode, int width, 
					int height)
{
  ostringstream str;

  switch (mode) {
  case BW:
  case GRAY:
    {
      str << "[/Indexed /DeviceGray 255" << endl 
	  << '<' << endl;

      int count = 0;
      for (int i=0; i<256; i++) {
	unsigned short value = 0;

	unsigned char red;
	unsigned char green;
	unsigned char blue;
	if (psBaseColor(i, &red, &green, &blue))
	  value = RGB2Gray(red, green, blue);
	else
	  for (int j=0; j<colorCount; j++)
	    if (colorIndex[j]==i) {
	      value = RGB2Gray(colorCells[j*3+2], colorCells[j*3+1], 
			       colorCells[j*3]);
	      break;
	    }

	str << hex << setfill('0') << setw(2) << value << ' ';

	if (++count == 16) {
	  str << endl;
	  count = 0;
	}
      }
    }
    break;
  case RGB:
    {
      str << "[/Indexed /DeviceRGB 255" << endl 
	  << '<' << endl;

      int count = 0;
      for (int i=0; i<256; i++) {
	unsigned short red = 0;
	unsigned short green = 0;
	unsigned short blue = 0;

	unsigned char r;
	unsigned char g;
	unsigned char b;
	if (psBaseColor(i, &r, &g, &b)) {
	  red = r;
	  green = g;
	  blue = b;
	}
	else
	  for (int j=0; j<colorCount; j++)
	    if (colorIndex[j]==i) {
	      red = colorCells[j*3+2];
	      green = colorCells[j*3+1];
	      blue = colorCells[j*3];
	      break;
	    }

	str << hex << setfill('0') << setw(2) << red 
	    << hex << setfill('0') << setw(2) << green
	    << hex << setfill('0') << setw(2) << blue
	    << ' ';

	if (++count == 8) {
	  str << endl;
	  count = 0;
	}
      }
    }
    break;
  case CMYK:
    {
      str << "[/Indexed /DeviceCMYK 255" << endl 
	  << '<' << endl;

      int count = 0;
      for (int i=0; i<256; i++) {
	unsigned char c = 0;
	unsigned char m = 0;
	unsigned char y = 0;
	unsigned char b = 255;

	unsigned char red;
	unsigned char green;
	unsigned char blue;
	if (psBaseColor(i, &red, &green, &blue))
	  RGB2CMYK(red, green, blue, &c, &m, &y, &b);
	else
	  for (int j=0; j<colorCount; j++)
	    if (colorIndex[j]==i) {
	      RGB2CMYK(colorCells[j*3+2], colorCells[j*3+1], colorCells[j*3],
		       &c, &m, &y, &b);
	      break;
	    }

	unsigned short cyan = c;
	unsigned short magenta = m;
	unsigned short yellow = y;
	unsigned short black = b;

	str << hex << setfill('0') << setw(2) << cyan 
	    << hex << setfill('0') << setw(2) << magenta
	    << hex << setfill('0') << setw(2) << yellow
	    << hex << setfill('0') << setw(2) << black
	    << ' ';

	if (++count == 8) {
	  str << endl;
	  count = 0;
	}
      }
    }
    break;
  }

  str << '>' << endl
      << "] setcolorspace" << endl
      << "<<" << endl
      << "/ImageType 1" << endl
      << "/Width " << dec << width << endl
      << "/Height " << dec << height << endl
      << "/BitsPerComponent 8" << endl
      << "/Decode [0 255]" << endl;

  str << "/ImageMatrix matrix" << endl
      << "/DataSource currentfile" << endl
      << "/ASCII85Decode filter" << endl
      << "/RunLengthDecode filter" << endl
      << ">>" << endl
      << "image" << endl
      << ends;

  Tcl_AppendResult(interp, str.str().c_str(), NULL);
}

void ColorbarPseudoColor8::psLevel2(PSColorSpace mode, int width, int height, 
				    float scale)
{
  if (!colorCells)
    return;

  psLevel2Head(mode, width, height);
  RLEAscii85 filter;

  for (int j=0; j<height; j++) {
    ostringstream str;

    for (int i=0; i<width; i++) {
      filter << colorIndex[(int)(double(i)/width*colorCount)];
      str << filter;
    }

    str << ends;
    psFix(str);
    Tcl_AppendResult(interp, str.str().c_str(), NULL);
  }

  ostringstream str;
  filter.flush(str);
  str << endl << "~>" << endl << ends;
  psFix(str);
  Tcl_AppendResult(interp, str.str().c_str(), NULL);
}

