// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "ps.h"
#include "util.h"

Filter::Filter()
{
  ptr = buf;
}

void Filter::flush(ostream& str)
{
  cflush();
  eflush(str);
}

// Compress

void NoCompress::in(unsigned char c)
{
  *ptr++ = c;
}

RLE::RLE()
{
  state = 0;
  num = 0;
}

void RLE::in(unsigned char c)
{
  current = c;

  switch (state) {
  case 0:                   // start state
    if (num == 0)           // not enough info yet
      rle[num++] = current;
    else {
      if (current != rle[num-1]) { // looks like non repeating
	rle[num++] = current;
	state = 1;
      }
      else {                // looks like repeating
	num++;
	state = 2;
      }
    }
    break;

  case 1:                   // non repeat state
    if (current != rle[num-1]) {
      rle[num++] = current;

      if (num > 127) {      // time to dump the rlefer
	dumpNonRepeat();    // dump the buffer
	num = 0;            // and reset counter
	state = 0;
      }
    }
    else {
      num--;                // decr
      dumpNonRepeat();      // dump the buffer

      state = 2;            // repeat state
      rle[0] = current;     // save repeat char
      num = 2;              // we aready have two now
    }
    break;

  case 2:                   // repeat state
    if (current == rle[0]) {
      if (++num > 127) {
	dumpRepeat();       // dump the repeat count
	state = 0;
	num = 0;
      }
    }
    else {
      dumpRepeat();         // dump the repeat count
      state = 1;            // back to non repeat state
      rle[0] = current;     // save first char
      num = 1;              // we have one now
    }
    break;
  }
}

void RLE::dumpNonRepeat()
{
  if (num) {
    *ptr++ = (unsigned char)(num-1);
    for (int i=0; i<num; i++)
      *ptr++ = rle[i];
  }
}

void RLE::dumpRepeat()
{
  if (num) {
    *ptr++ = (unsigned char)(257-num);
    *ptr++ = rle[0];
  }
}

void RLE::cflush()
{
  switch (state) {
  case 0:
  case 1:
    dumpNonRepeat();
    break;
  case 2:
    dumpRepeat();
    break;
  }

  *ptr++ = '\177'; // EOD Marker
}

// Encode

Encode::Encode()
{
  lineCount = 0;
}

void AsciiHex::out(ostream& str)
{
  unsigned char* p = buf;
  while (p < ptr) {
    unsigned short c = *p++;
    str << hex << setfill('0') << setw(2) << c;

    lineCount += 2;
    if (lineCount >= LINELIMIT) {
      str << endl;
      lineCount = 0;
    }
  }
  ptr = buf; // reset buffer
}

void AsciiHex::eflush(ostream& str)
{
  out(str);
}

Ascii85::Ascii85() : Encode()
{
  byteswap = (*(short *)"\001\000" & 0x0001);

  buf85.c = 0;
  index = 0;
}

BIT32 Ascii85::swap(unsigned BIT32* p)
{
  BIT32 r;
  swap4((char*)p, (char*)&r);
  return r;
}

void Ascii85::out(ostream& str)
{
  unsigned char* p = buf;
  while (p < ptr) {
    buf85.b[index++] = *p++;
    if (index==4)
      dump(str);
  }
  ptr = buf; // reset buffer
}

void Ascii85::eflush(ostream& str)
{
  if (index > 0) {

    // zero fill buffer

    for (int i=index; i<4; i++)
      buf85.b[index] = 0;

    // dump the remainder
    // we can't have any z's here
    // also, only dump index+1 chars

    unsigned BIT32 b;
    if (!byteswap)
      b = buf85.c;
    else
      b = swap(&buf85.c);

    {
      for (int i=4; i>=(4-index) ; i--) {
	unsigned BIT32 base = 1;
	for (int j=0; j<i; j++)
	  base *= 85;

	unsigned BIT32 a = b / base;
	b -= a * base;
	str << (char)(a + '!');

	if (lineCount++ > LINELIMIT) {
	  str << endl;
	  lineCount = 0;
	}
      }
    }
  }
}

void Ascii85::dump(ostream& str)
{
  if (buf85.c == 0) {
    str << 'z';
    if (lineCount++ > LINELIMIT) {
      str << endl;
      lineCount = 0;
    }
  }
  else {
    unsigned BIT32 b;
    if (!byteswap)
      b = buf85.c;
    else
      b = swap(&buf85.c);

    for (int i=4; i>=0 ; i--) {
      unsigned BIT32 base = 1;
      
      for (int j=0; j<i; j++)
	base *= 85;

      unsigned BIT32 a = b / base;
      b -= a * base;
      str << (char)(a + '!');
      if (lineCount++ > LINELIMIT) {
	str << endl;
	lineCount = 0;
      }
    }
  }

  index = 0;
  buf85.c = 0;
}
