// Copyright (C) 1999-2012
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "tcl.h"
#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include "frame3dbase.h"
#include "fitsimage.h"
#include "marker.h"
#include "context.h"
#include "ps.h"

#include "sigbus.h"

Frame3dBase::Frame3dBase(Tcl_Interp* i, Tk_Canvas c, Tk_Item* item) 
  : Base(i, c, item)
{
  zbufWidget_ = NULL;
  mkzbufWidget_ = NULL;
  zbufPanner_ = NULL;
  mkzbufPanner_ = NULL;
  zbufMagnifier_ = NULL;
  mkzbufMagnifier_ = NULL;
  zbufPS_ = NULL;
  mkzbufPS_ = NULL;

  zdepth_ = 100000;
  zzoom_ = 1;

  az_ =0;
  el_ =0;
  renderMethod_ = MIP;

  threedGC = NULL;

  highlite_ =1;
  highliteColorName_ = dupstr("cyan");

  cropsl_ =0;

  imageToData3d = Translate3d(-.5, -.5, -.5);
  dataToImage3d = Translate3d( .5,  .5, .5);
}

Frame3dBase::~Frame3dBase()
{
  // just in case, cancelImage() should clean this up
  if (zbufWidget_)
    delete [] zbufWidget_;
  if (mkzbufWidget_)
    delete [] mkzbufWidget_;

  if (zbufPanner_)
    delete [] zbufPanner_;
  if (mkzbufPanner_)
    delete [] mkzbufPanner_;

  if (zbufMagnifier_)
    delete [] zbufMagnifier_;
  if (mkzbufMagnifier_)
    delete [] mkzbufMagnifier_;

  if (zbufPS_)
    delete [] zbufPS_;
  if (mkzbufPS_)
    delete [] mkzbufPS_;

  if (threedGC)
    XFreeGC(display, threedGC);

  if (highliteColorName_)
    delete [] highliteColorName_;
}

void Frame3dBase::calcHighlite(Coord::InternalSystem sys, Vector* vv, int* rr)
{
  if (!keyContext->fits)
    return;

  FitsBound* params = 
    keyContext->fits->getDataParams(keyContext->frScale.scanMode());
  Vector ss(params->xmin,params->ymin);
  Vector tt(params->xmax,params->ymax);

  Vector ll = mapFromRef3d(ss,sys);
  Vector lr = mapFromRef3d(Vector(tt[0],ss[1]),sys);
  Vector ur = mapFromRef3d(tt,sys);
  Vector ul = mapFromRef3d(Vector(ss[0],tt[1]),sys);
    
  // context->slice() IMAGE (ranges 1-n)
  double sl = keyContext->slice(2)-.5;
  Vector3d ll1 = mapFromRef3d(ss,sys);
  Vector3d lr1 = mapFromRef3d(Vector(tt[0],ss[1]),sys);
  Vector3d ur1 = mapFromRef3d(tt,sys);
  Vector3d ul1 = mapFromRef3d(Vector3d(ss[0],tt[1]),sys);
  Vector3d ll0 = mapFromRef3d(ss,sys,sl-1);
  Vector3d lr0 = mapFromRef3d(Vector3d(tt[0],ss[1]),sys,sl-1);
  Vector3d ur0 = mapFromRef3d(tt,sys,sl-1);
  Vector3d ul0 = mapFromRef3d(Vector3d(ss[0],tt[1]),sys,sl-1);

  Vector3d aa1 = (ll0-ll1).normalize();
  Vector3d aa2 = (lr0-lr1).normalize();
  Vector3d aa3 = (ur0-ur1).normalize();
  Vector3d aa4 = (ul0-ul1).normalize();
  Vector3d bb1 = (lr1-ll1).normalize();
  Vector3d bb2 = (ur1-lr1).normalize();
  Vector3d bb3 = (ul1-ur1).normalize();
  Vector3d bb4 = (ll1-ul1).normalize();

  Vector3d cc1 = cross(bb1,aa1);
  Vector3d cc2 = cross(bb2,aa2);
  Vector3d cc3 = cross(bb3,aa3);
  Vector3d cc4 = cross(bb4,aa4);

  vv[0] = ll;
  vv[1] = lr;
  vv[2] = ur;
  vv[3] = ul;

  rr[0] = cc1[2]>0;
  rr[1] = cc2[2]>0;
  rr[2] = cc3[2]>0;
  rr[3] = cc4[2]>0;
}

double Frame3dBase::calcZoom3d(Vector3d src, Vector dest)
{
    Vector3d cc = src/2.;

    Vector3d llf(0,0,0);
    Vector3d lrf(0,src[1],0);
    Vector3d urf(src[0],src[1],0);
    Vector3d ulf(src[0],0,0);
 
    Vector3d llb(0,0,src[2]);
    Vector3d lrb(0,src[1],src[2]);
    Vector3d urb(src[0],src[1],src[2]);
    Vector3d ulb(src[0],0,src[2]);
 
    Matrix3d mx = 
      Translate3d(-cc) *
      RotateZ3d(-wcsRotation) *
      RotateZ3d(-rotation) *
      RotateY3d(az_) * 
      RotateX3d(el_);

    BBox3d bb(llf*mx);
    bb.bound(lrf*mx);
    bb.bound(urf*mx);
    bb.bound(ulf*mx);
    bb.bound(llb*mx);
    bb.bound(lrb*mx);
    bb.bound(urb*mx);
    bb.bound(ulb*mx);

    Vector3d bs = bb.size();
    double r0 = dest[0]/bs[0];
    double r1 = dest[1]/bs[1];

    return r0>r1 ? r1:r0;
}

double Frame3dBase::calcZoomPanner()
{
  if (!keyContext->fits)
    return 1;

  if (!pannerPixmap)
    return 1;

  Vector3d src = imageSize3d(keyContext->frScale.datasec() ? 
			     FrScale::DATASEC : FrScale::IMGSEC);
  Vector dest(pannerWidth,pannerHeight);

  Vector3d cc = src/2.;

  Vector3d llf = Vector3d(0,0,0);
  Vector3d lrf = Vector3d(0,src[1],0);
  Vector3d urf = Vector3d(src[0],src[1],0);
  Vector3d ulf = Vector3d(src[0],0,0);
 
  Vector3d llb = Vector3d(0,0,src[2]);
  Vector3d lrb = Vector3d(0,src[1],src[2]);
  Vector3d urb = Vector3d(src[0],src[1],src[2]);
  Vector3d ulb = Vector3d(src[0],0,src[2]);
 
  BBox3d bb;

  // 0, 0
  Matrix3d m0000 = 
    Translate3d(-cc) *
    RotateY3d(degToRad(0)) * 
    RotateX3d(degToRad(0));
  bb.bound(llf*m0000);
  bb.bound(llf*m0000);
  bb.bound(lrf*m0000);
  bb.bound(urf*m0000);
  bb.bound(ulf*m0000);
  bb.bound(llb*m0000);
  bb.bound(lrb*m0000);
  bb.bound(urb*m0000);
  bb.bound(ulb*m0000);

  // 0, 90
  Matrix3d m0090 = 
    Translate3d(-cc) *
    RotateY3d(degToRad(90)) * 
    RotateX3d(degToRad(0));
  bb.bound(llf*m0090);
  bb.bound(llf*m0090);
  bb.bound(lrf*m0090);
  bb.bound(urf*m0090);
  bb.bound(ulf*m0090);
  bb.bound(llb*m0090);
  bb.bound(lrb*m0090);
  bb.bound(urb*m0090);
  bb.bound(ulb*m0090);

  // 90, 0
  Matrix3d m9000 = 
    Translate3d(-cc) *
    RotateY3d(degToRad(0)) * 
    RotateX3d(-degToRad(90));
  bb.bound(llf*m9000);
  bb.bound(llf*m9000);
  bb.bound(lrf*m9000);
  bb.bound(urf*m9000);
  bb.bound(ulf*m9000);
  bb.bound(llb*m9000);
  bb.bound(lrb*m9000);
  bb.bound(urb*m9000);
  bb.bound(ulb*m9000);

  // 45, 45
  Matrix3d m4545 = 
    Translate3d(-cc) *
    RotateY3d(degToRad(45)) * 
    RotateX3d(-degToRad(45));
  bb.bound(llf*m4545);
  bb.bound(llf*m4545);
  bb.bound(lrf*m4545);
  bb.bound(urf*m4545);
  bb.bound(ulf*m4545);
  bb.bound(llb*m4545);
  bb.bound(lrb*m4545);
  bb.bound(urb*m4545);
  bb.bound(ulb*m4545);
 
  // 45, 90
  Matrix3d m4590 = 
    Translate3d(-cc) *
    RotateY3d(degToRad(90)) * 
    RotateX3d(-degToRad(45));
  bb.bound(llf*m4590);
  bb.bound(lrf*m4590);
  bb.bound(urf*m4590);
  bb.bound(ulf*m4590);
  bb.bound(llb*m4590);
  bb.bound(lrb*m4590);
  bb.bound(urb*m4590);
  bb.bound(ulb*m4590);

  // 90, 45
  Matrix3d m9045 = 
    Translate3d(-cc) *
    RotateY3d(degToRad(45)) * 
    RotateX3d(-degToRad(90));
  bb.bound(llf*m9045);
  bb.bound(lrf*m9045);
  bb.bound(urf*m9045);
  bb.bound(ulf*m9045);
  bb.bound(llb*m9045);
  bb.bound(lrb*m9045);
  bb.bound(urb*m9045);
  bb.bound(ulb*m9045);

  Vector3d bs = bb.size();
  double ll = bs[0] > bs[1] ? bs[0] : bs[1];
  double mm = dest[0] > dest[1] ? dest[0] : dest[1];

  return 1/ll*mm;
}

void Frame3dBase::centerImage()
{
  Base::centerImage();

  viewCursor_ = Vector();
  if (keyContext->fits) {
    // imageCenter is in IMAGE coords
    Vector3d aa = imageCenter3d(keyContext->frScale.scanMode());
    // always center to center of pixel, even for even sized image
    Vector3d bb = (aa*Translate3d(.5,.5,.5)).floor();
    // vp_ is in REF coords
    vp_ = bb*imageToData3d;
  }
  else
    vp_ = Vector();
}

Vector3d Frame3dBase::imageCenter3d(FrScale::ScanMode mode)
{
  if (!keyContext->fits)
    return Vector3d();

  // params is a BBOX in DATA coords 0-n
  FitsBound* pp = keyContext->fits->getDataParams(mode);
  // Note: imageCenter() is in IMAGE coords
  return Vector3d((pp->xmax-pp->xmin)/2.+pp->xmin, 
		  (pp->ymax-pp->ymin)/2.+pp->ymin,
		  (pp->zmax-pp->zmin)/2.+pp->zmin) * dataToImage3d;
}

Vector3d Frame3dBase::imageSize3d(FrScale::ScanMode mode )
{
  if (!keyContext->fits)
    return Vector3d();

  // params is a BBOX in DATA coords 0-n
  FitsBound* params = keyContext->fits->getDataParams(mode);

  // return in IMAGE coords and extends edge to edge
  return  Vector3d(params->xmax-params->xmin, params->ymax-params->ymin,
		   params->zmax-params->zmin);
}

void Frame3dBase::psHighlite(PSColorSpace mode)
{
  if (!highlite_)
    return;

  if (!keyContext->fits)
    return;

  Vector vv[4];
  int rr[4];
  calcHighlite(Coord::CANVAS,vv,rr);

  {
    ostringstream str;
    switch ((PSColorSpace)mode) {
    case BW:
    case GRAY:
      psColorGray(getXColor(highliteColorName_), str);
      str << " setgray";
      break;
    case RGB:
      psColorRGB(getXColor(highliteColorName_), str);
      str << " setrgbcolor";
      break;
    case CMYK:
      psColorCMYK(getXColor(highliteColorName_), str);
      str << " setcmykcolor";
      break;
    }
    str << endl << ends;
    Tcl_AppendResult(interp, str.str().c_str(), NULL);
  }  

  {
    ostringstream str;
    str << "1 setlinewidth" << endl << ends;
    Tcl_AppendResult(interp, str.str().c_str(), NULL);
  }

  {
    ostringstream str;
    if (rr[0])
      str << "[8 3] 0 setdash" << endl;
    else
      str << "[] 0 setdash" << endl;

    str << "newpath " 
	<< vv[0].TkCanvasPs(canvas) << " moveto" << endl
	<< vv[1].TkCanvasPs(canvas) << " lineto stroke" << endl << ends;

    Tcl_AppendResult(interp, str.str().c_str(), NULL);
  }

  {
    ostringstream str;
    if (rr[1])
      str << "[8 3] 0 setdash" << endl;
    else
      str << "[] 0 setdash" << endl;

    str << "newpath " 
	<< vv[1].TkCanvasPs(canvas) << " moveto" << endl
	<< vv[2].TkCanvasPs(canvas) << " lineto stroke" << endl << ends;

    Tcl_AppendResult(interp, str.str().c_str(), NULL);
  }

  {
    ostringstream str;
    if (rr[2])
      str << "[8 3] 0 setdash" << endl;
    else
      str << "[] 0 setdash" << endl;

    str << "newpath " 
	<< vv[2].TkCanvasPs(canvas) << " moveto" << endl
	<< vv[3].TkCanvasPs(canvas) << " lineto stroke" << endl << ends;

    Tcl_AppendResult(interp, str.str().c_str(), NULL);
  }

  {
    ostringstream str;
    if (rr[3])
      str << "[8 3] 0 setdash" << endl;
    else
      str << "[] 0 setdash" << endl;

    str << "newpath " 
	<< vv[3].TkCanvasPs(canvas) << " moveto" << endl
	<< vv[0].TkCanvasPs(canvas) << " lineto stroke" << endl << ends;

    Tcl_AppendResult(interp, str.str().c_str(), NULL);
  }
}

Matrix3d Frame3dBase::psMatrix(float scale, int width, int height)
{
  Matrix3d userToPS3d = 
    Matrix3d(wcsOrientationMatrix) *
    Matrix3d(orientationMatrix) *
    RotateZ3d(-wcsRotation) *
    RotateZ3d(-rotation) *

    RotateY3d(az_) * 
    RotateX3d(el_) *

    Translate3d(viewCursor_) *
    Scale3d(zoom_, zzoom_) *
    Scale3d(scale,1) *

    FlipY3d() *
    Translate3d(width/2., height/2., zdepth_/2.);

  return refToUser3d*userToPS3d;
}

void Frame3dBase::renderBBox(Pixmap pixmap, Vector3d bb, const Matrix3d& mm)
{
  Vector3d llf (0, 0, 0);
  Vector3d lrf (bb[0], 0, 0);
  Vector3d urf (bb[0],bb[1],0);
  Vector3d ulf (0,bb[1],0);

  Vector3d llb (0, 0, bb[2]);
  Vector3d lrb (bb[0], 0, bb[2]);
  Vector3d urb (bb[0],bb[1],bb[2]);
  Vector3d ulb (0,bb[1],bb[2]);

  Vector3d rllf = llf * mm;
  Vector3d rlrf = lrf * mm;
  Vector3d rurf = urf * mm;
  Vector3d rulf = ulf * mm;
  Vector3d rllb = llb * mm;
  Vector3d rlrb = lrb * mm;
  Vector3d rurb = urb * mm;
  Vector3d rulb = ulb * mm;

  // init for dash
  int dd[12];
  for (int ii=0; ii<12; ii++)
    dd[ii] =1;

  XSetForeground(display, pannerGC, getColor("black"));

  // front
  {
    Vector3d aa = rlrf-rllf;
    Vector3d cc = rulf-rllf;
    Vector3d ff = cross(aa,cc);
    for (int ii=0; ii<4; ii++)
      dd[ii] &= ff[2]>0;
  }

  // right
  {
    Vector3d aa = rlrb-rlrf;
    Vector3d cc = rurf-rlrf;
    Vector3d ff = cross(aa,cc);
    int ww = ff[2]>0;
    dd[1] &= ww;
    dd[9] &= ww;
    dd[5] &= ww;
    dd[10] &= ww;
  }

  // top
  {
    Vector3d aa = rurb-rurf;
    Vector3d cc = rulf-rurf;
    Vector3d ff = cross(aa,cc);
    int ww = ff[2]>0;
    dd[2] &= ww;
    dd[10] &= ww;
    dd[6] &= ww;
    dd[11] &= ww;
  }

  // left
  {
    Vector3d aa = rulb-rulf;
    Vector3d cc = rllf-rulf;
    Vector3d ff = cross(aa,cc);
    int ww = ff[2]>0;
    dd[3] &= ww;
    dd[8] &= ww;
    dd[7] &= ww;
    dd[11] &= ww;
  }

  // bottom
  {
    Vector3d aa = rllb-rllf;
    Vector3d cc = rlrf-rllf;
    Vector3d ff = cross(aa,cc);
    int ww = ff[2]>0;
    dd[0] &= ww;
    dd[9] &= ww;
    dd[4] &= ww;
    dd[8] &= ww;
  }

  // back
  {
    Vector3d aa = rllb-rlrb;
    Vector3d cc = rurb-rlrb;
    Vector3d ff = cross(aa,cc);
    for (int ii=4; ii<8; ii++)
      dd[ii] &= ff[2]>0;
  }

  // front
  renderDash(pannerGC, dd[0]);
  XDrawLine(display, pixmap, pannerGC, rllf[0], rllf[1], rlrf[0], rlrf[1]);
  renderDash(pannerGC, dd[1]);
  XDrawLine(display, pixmap, pannerGC, rlrf[0], rlrf[1], rurf[0], rurf[1]);
  renderDash(pannerGC, dd[2]);
  XDrawLine(display, pixmap, pannerGC, rurf[0], rurf[1], rulf[0], rulf[1]);
  renderDash(pannerGC, dd[3]);
  XDrawLine(display, pixmap, pannerGC, rulf[0], rulf[1], rllf[0], rllf[1]);

  // back
  renderDash(pannerGC, dd[4]);
  XDrawLine(display, pixmap, pannerGC, rllb[0], rllb[1], rlrb[0], rlrb[1]);
  renderDash(pannerGC, dd[5]);
  XDrawLine(display, pixmap, pannerGC, rlrb[0], rlrb[1], rurb[0], rurb[1]);
  renderDash(pannerGC, dd[6]);
  XDrawLine(display, pixmap, pannerGC, rurb[0], rurb[1], rulb[0], rulb[1]);
  renderDash(pannerGC, dd[7]);
  XDrawLine(display, pixmap, pannerGC, rulb[0], rulb[1], rllb[0], rllb[1]);

  // other
  renderDash(pannerGC, dd[8]);
  XDrawLine(display, pixmap, pannerGC, rllf[0], rllf[1], rllb[0], rllb[1]);
  renderDash(pannerGC, dd[9]);
  XDrawLine(display, pixmap, pannerGC, rlrf[0], rlrf[1], rlrb[0], rlrb[1]);
  renderDash(pannerGC, dd[10]);
  XDrawLine(display, pixmap, pannerGC, rurf[0], rurf[1], rurb[0], rurb[1]);
  renderDash(pannerGC, dd[11]);
  XDrawLine(display, pixmap, pannerGC, rulf[0], rulf[1], rulb[0], rulb[1]);
}

void Frame3dBase::renderHighlite()
{
  Base::renderHighlite();

  if (!highlite_)
    return;

  if (!keyContext->fits)
    return;

  Vector vv[4];
  int rr[4];
  calcHighlite(Coord::WIDGET,vv,rr);

  renderDash(threedGC,rr[0]);
  XDrawLine(display,pixmap,threedGC,vv[0][0],vv[0][1],vv[1][0],vv[1][1]);
  renderDash(threedGC,rr[1]);
  XDrawLine(display,pixmap,threedGC,vv[1][0],vv[1][1],vv[2][0],vv[2][1]);
  renderDash(threedGC,rr[2]);
  XDrawLine(display,pixmap,threedGC,vv[2][0],vv[2][1],vv[3][0],vv[3][1]);
  renderDash(threedGC,rr[3]);
  XDrawLine(display,pixmap,threedGC,vv[3][0],vv[3][1],vv[0][0],vv[0][1]);
}

void Frame3dBase::renderMagnifierCursor(const Vector& vv)
{
}

void Frame3dBase::updateBin(const Matrix& mx)
{
  centerImage();
  Base::updateBin(mx);
}

void Frame3dBase::updateGCs()
{
  Base::updateGCs();

    // widget clip region
  BBox bbWidget = BBox(0, 0, options->width, options->height);
  Vector sizeWidget = bbWidget.size();

  rectWidget[0].x = (int)bbWidget.ll[0];
  rectWidget[0].y = (int)bbWidget.ll[1];
  rectWidget[0].width = (int)sizeWidget[0];
  rectWidget[0].height = (int)sizeWidget[1];

// window clip region
  BBox bbWindow = bbWidget * widgetToWindow;
  Vector sizeWindow = bbWindow.size();

  rectWindow[0].x = (int)bbWindow.ll[0];
  rectWindow[0].y = (int)bbWindow.ll[1];
  rectWindow[0].width = (int)sizeWindow[0];
  rectWindow[0].height = (int)sizeWindow[1];

  // 3d highlite
  if (!threedGC) {
    threedGC = XCreateGC(display, Tk_WindowId(tkwin), 0, NULL);
    XSetForeground(display, threedGC, getColor(highliteColorName_));
    XSetLineAttributes(display, threedGC, 1, LineSolid, CapButt, JoinMiter);
  }
  XSetClipRectangles(display, threedGC, 0, 0, rectWidget, 1, Unsorted);
}

void Frame3dBase::updateMatrices()
{
  if (DebugPerf)
    cerr << "updateMatrices..." << endl;

  zzoom_ = (zoom_[0]+zoom_[1])/2.;
  if (zzoom_<1)
    zzoom_ = 1;

  // if othogonal, reset zzoom
  if ((teq(az_,0,.001) || 
       teq(fabs(az_),M_PI_2,.001) || 
       teq(fabs(az_),M_PI,.001)) &&
      (teq(el_,0,.001) || 
       teq(fabs(el_),M_PI_2,.001)))
      zzoom_ =1;

  // These are the basic tranformation matrices
  // Note: imageCenter() is in IMAGE coords
  refToUser3d = Translate3d(-vp_) * FlipY3d();
  userToRef3d = refToUser3d.invert();

  // userToWidget3d
  userToWidget3d =
    Matrix3d(wcsOrientationMatrix) *
    Matrix3d(orientationMatrix) *
    RotateZ3d(-wcsRotation) *
    RotateZ3d(-rotation) *

    RotateY3d(az_) * 
    RotateX3d(el_) *

    Translate3d(viewCursor_) *
    Scale3d(zoom_, zzoom_) *

    Translate3d(options->width/2., options->height/2.,zdepth_/2.);
  widgetToUser3d = userToWidget3d.invert();

  // widgetToCanvas
  widgetToCanvas3d = Translate3d(originX, originY, 0);
  canvasToWidget3d = widgetToCanvas3d.invert();

  // canvasToWindow
  short xx, yy;
  Tk_CanvasWindowCoords(canvas, 0, 0, &xx, &yy);
  canvasToWindow3d = Translate3d(xx, yy, 0);
  windowToCanvas3d = canvasToWindow3d.invert();

  // These are derived Transformation Matrices
  refToWidget3d = refToUser3d * userToWidget3d;
  widgetToRef3d = refToWidget3d.invert();

  refToCanvas3d = refToWidget3d * widgetToCanvas3d;
  canvasToRef3d = refToCanvas3d.invert();

  refToWindow3d = refToCanvas3d * canvasToWindow3d;
  windowToRef3d = refToWindow3d.invert();

  userToCanvas3d = userToWidget3d * widgetToCanvas3d;
  canvasToUser3d = userToCanvas3d.invert();

  widgetToWindow3d = widgetToCanvas3d * canvasToWindow3d;
  windowToWidget3d = widgetToWindow3d.invert();

  Base::updateMatrices();

  // delete current zbuffer since matrices have changed
  cancelImage();

  if (DebugPerf)
    cerr << "updateMatrices end" << endl;
}

void Frame3dBase::updateMagnifierMatrices()
{
  // vv is in CANVAS coords
  Vector ww = magnifierCursor*canvasToRef;

  // refToUser3d
  Matrix3d refToUser3d = Translate3d(Vector3d(-ww,-vp_[2])) * FlipY3d();

  // userToMagnifier
  userToMagnifier3d  = 
    Matrix3d(wcsOrientationMatrix) *
    Matrix3d(orientationMatrix) *
    RotateZ3d(-wcsRotation) *
    RotateZ3d(-rotation) *

    RotateY3d(az_) * 
    RotateX3d(el_) *

    Translate3d(viewCursor_) *
    Scale3d(zoom_, zzoom_) *
    Scale3d(magnifierZoom_,magnifierZoom_) *

    Translate3d(magnifierWidth/2., magnifierHeight/2., zdepth_/2.);
  magnifierToUser3d = userToMagnifier3d.invert();

  refToMagnifier3d = refToUser3d * userToMagnifier3d;
  magnifierToRef3d = refToMagnifier3d.invert();

  magnifierToWidget3d = magnifierToRef3d * refToWidget3d;
  widgetToMagnifier3d = magnifierToWidget3d.invert();

  Base::updateMagnifierMatrices();
}

void Frame3dBase::updatePannerMatrices()
{
  Vector3d center = imageCenter3d(FrScale::IMGSEC) * imageToData3d;

  // refToUser3d
  Matrix3d refToUser3d = Translate3d(-center) * FlipY3d();

  // userToPanner3d
  double pz = calcZoomPanner();
  double zz = zzoom_*pz;
  if (zz<1)
    zz =1;

  userToPanner3d =
    Matrix3d(wcsOrientationMatrix) *
    Matrix3d(orientationMatrix) *
    RotateZ3d(-wcsRotation) *
    RotateZ3d(-rotation) *

    RotateY3d(az_) * 
    RotateX3d(el_) *

    Scale3d(pz, zz) *

    Translate3d(pannerWidth/2., pannerHeight/2., zdepth_/2.);
  pannerToUser3d = userToPanner3d.invert();

  refToPanner3d = refToUser3d * userToPanner3d;
  pannerToRef3d = refToPanner3d.invert();

  pannerToWidget3d = pannerToRef3d * refToWidget3d;
  widgetToPanner3d = pannerToWidget3d.invert();

  Base::updatePannerMatrices();
}

void Frame3dBase::updatePanner()
{
  // do this first
  Base::updatePanner();

  // always render (to update panner background color)
  if (usePanner) {
    if (keyContext->fits)
      renderBBox(pannerPixmap, imageSize3d(FrScale::IMGSEC), 
		 keyContext->fits->dataToPanner3d);

    ostringstream str;
    str << pannerName << " update " << pannerPixmap << ";";

    // calculate bbox
    Vector ll = Vector(0,0) * widgetToPanner3d;
    Vector lr = Vector(options->width,0) * widgetToPanner3d;
    Vector ur = Vector(options->width,options->height) * widgetToPanner3d;
    Vector ul = Vector(0,options->height) * widgetToPanner3d;

    str << pannerName << " update bbox " << ll << lr << ur << ul << ";";

    // calculate image compass vectors
    Matrix3d mm = 
      Matrix3d(wcsOrientationMatrix) *
      Matrix3d(orientationMatrix) *
      RotateZ3d(wcsRotation) *
      RotateZ3d(rotation) *
      RotateY3d(az_) * 
      RotateX3d(-el_) * 
      FlipY3d();

    Vector xx = (Vector3d(1,0,0)*mm).normalize();
    Vector yy = (Vector3d(0,1,0)*mm).normalize();
    Vector zz = (Vector3d(0,0,1)*mm).normalize();

    str << pannerName << " update image compass " << xx << yy << zz << ";";

    if (keyContext->fits && keyContext->fits->hasWCS(pannerSystem)) {
      Vector orpix = keyContext->fits->center();
      Vector orval=keyContext->fits->pix2wcs(orpix, pannerSystem, pannerSky);
      Vector orpix2 = keyContext->fits->wcs2pix(orval, pannerSystem,pannerSky);
      Vector delta = keyContext->fits->getWCScdelt(pannerSystem).abs();

      // find normalized north
      Vector npix = keyContext->fits->wcs2pix(Vector(orval[0],orval[1]+delta[1]), pannerSystem,pannerSky);
      Vector north = (Vector3d(npix-orpix2)*mm).normalize();

      // find normalized east
      Vector epix = keyContext->fits->wcs2pix(Vector(orval[0]+delta[0],orval[1]), pannerSystem,pannerSky);
      Vector east = (Vector3d(epix-orpix2)*mm).normalize();

      // sanity check
      Vector diff = (north-east).abs();
      if ((north[0]==0 && north[1]==0) ||
	  (east[0]==0 && east[1]==0) ||
	  (diff[0]<.01 && diff[1]<.01)) {
	north = (Vector3d(0,1)*mm).normalize();
	east = (Vector3d(-1,0)*mm).normalize();
      }

      // and update the panner
      str << pannerName << " update wcs compass " << north << east << ends;
    }
    else
      str << pannerName << " update wcs compass invalid" << ends;

    Tcl_Eval(interp, str.str().c_str());
  }
}

void Frame3dBase::ximageToPixmapMagnifier()
{
  if (!basePixmap || !baseXImage || !magnifierPixmap || !magnifierXImage)
    return;

  // magnifier
  int& ww = magnifierXImage->width;
  int& hh = magnifierXImage->height;
  Vector wh(ww,hh);
  Vector cc = magnifierCursor * canvasToWidget;
  Vector ll =cc-wh/2.;
  Vector ur =cc+wh/2.;

  // clip to base
  BBox bb(0,0,baseXImage->width,baseXImage->height);
  Vector uu(ll);
  Vector vv(ur);
  uu.clip(bb);
  vv.clip(bb);
  Vector zz = vv-uu;
  Vector oo = uu-ll;

  // sanity check
  if (zz[0]<=0 || zz[1]<=0)
    return;

  XImage* srcXImage = XGetImage(display, basePixmap, uu[0], uu[1], 
				zz[0], zz[1], AllPlanes, ZPixmap);

  char* src = XImageData(srcXImage);
  int srcBytesPerLine =  srcXImage->bytes_per_line;

  char* dst = XImageData(magnifierXImage);
  int dstBytesPerLine =  magnifierXImage->bytes_per_line;
  int bytesPerPixel = magnifierXImage->bits_per_pixel/8;

  Matrix mx = Translate(-wh/2.) * 
    Translate(oo) * 
    Translate(-.5,-.5) *
    Scale(magnifierZoom_) * 
    Translate(wh/2.);
  Matrix mm = mx.invert();

  for (int jj=0; jj<hh; jj++) {
    char* dest = dst + jj*dstBytesPerLine;

    for (int ii=0; ii<ww; ii++, dest+=bytesPerPixel) {
      Vector vv = Vector(ii,jj)*mm;

      if (vv[0] >= 0 && vv[0] < zz[0] && vv[1] >= 0 && vv[1] < zz[1])
	memcpy(dest, src + ((int)vv[1])*srcBytesPerLine + ((int)vv[0])*bytesPerPixel, bytesPerPixel);
      else
	memcpy(dest, bgTrueColor_, bytesPerPixel);
    }
  }

  XPutImage(display, magnifierPixmap, gc, magnifierXImage, 0, 0, 0, 0, 
	    magnifierXImage->width, magnifierXImage->height);

  if (srcXImage)
    XDestroyImage(srcXImage);
}

#ifdef _MACOSX
void Frame3dBase::macosxHighlite()
{
  if (!highlite_)
    return;

  if (!keyContext->fits)
    return;

  Vector vv[4];
  int rr[4];
  calcHighlite(CANVAS,vv,rr);

  float dlist[2] = {8,3};
  macosxColor(getXColor(highliteColorName_));
  macosxWidth(1);

  if (rr[0])
    macosxDash(dlist,2);
  else
    macosxDash(NULL,0);
  macosxDrawLine(vv[0],vv[1]);

  if (rr[1])
    macosxDash(dlist,2);
  else
    macosxDash(NULL,0);
  macosxDrawLine(vv[1],vv[2]);

  if (rr[2])
    macosxDash(dlist,2);
  else
    macosxDash(NULL,0);
  macosxDrawLine(vv[2],vv[3]);

  if (rr[3])
    macosxDash(dlist,2);
  else
    macosxDash(NULL,0);
  macosxDrawLine(vv[3],vv[0]);
}
#endif

#ifdef _WIN32
void Frame3dBase::win32Highlite()
{
  if (!highlite_)
    return;

  if (!keyContext->fits)
    return;

  Vector vv[4];
  int rr[4];
  calcHighlite(Coord::CANVAS,vv,rr);

  float dlist[2] = {8,3};
  win32Color(getXColor(highliteColorName_));
  win32Width(1);

  if (rr[0])
    win32Dash(dlist,2);
  else
    win32Dash(NULL,0);
  win32DrawLine(vv[0],vv[1]);

  if (rr[1])
    win32Dash(dlist,2);
  else
    win32Dash(NULL,0);
  win32DrawLine(vv[1],vv[2]);

  if (rr[2])
    win32Dash(dlist,2);
  else
    win32Dash(NULL,0);
  win32DrawLine(vv[2],vv[3]);

  if (rr[3])
    win32Dash(dlist,2);
  else
    win32Dash(NULL,0);
  win32DrawLine(vv[3],vv[0]);
}
#endif
