// Copyright (c) 1996-1999 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	phil.wilsey@uc.edu
//          Dale E. Martin	dmartin@ece.uc.edu
//          Timothy J. McBrayer tmcbraye@ece.uc.edu
//          Umesh Kumar V. Rajasekaran urajasek@ece.uc.edu
//          Krishnan Subramani  skrish@ece.uc.edu
//          Malolan Chetlur     mal@ece.uc.edu

//---------------------------------------------------------------------------
// 
// $Id: IIRScram_FloatingPointLiteral.cc,v 1.2 1999/03/09 20:54:31 dmartin Exp $
// 
//---------------------------------------------------------------------------

#include <strstream.h>
#include <iomanip.h>
#include "IIRScram_FloatingPointLiteral.hh"
#include "set.hh"
#include "IIR_FloatingSubtypeDefinition.hh"
#include "StandardPackage.hh"

IIRScram_FloatingPointLiteral::~IIRScram_FloatingPointLiteral() {}

IIR_Boolean
IIRScram_FloatingPointLiteral::_is_negative()
{
  if (_get_value() < 0) {
    return TRUE;
  }

  return FALSE;
}

void 
IIRScram_FloatingPointLiteral::_publish_vhdl(ostream &_vhdl_out) {
  register int i;
  IIR_Char* man = _get_mantissa();
  IIR_Char* exp = _get_exponent();
  char force_paranthesis = 0;
  
  if (_is_negative() == TRUE) {
    force_paranthesis = 1;
    _vhdl_out << "(";
  }
  
  if (_get_base() != 10) {
    _vhdl_out << _get_base() << "#";
  }
  for (i = 0; i < _get_mantissa_length(); i++) {
    _vhdl_out << man[i];
  }
  if (_get_base() != 10) {
    _vhdl_out << "#";
  }
  if (_get_exponent() != NULL && _get_exponent_length() > 0 ) {
    _vhdl_out << "E";
    for (i = 0; i < _get_exponent_length(); i++) {
      _vhdl_out << exp[i];
    }
  }

  if (force_paranthesis == 1) {
    _vhdl_out << ")";
  }
}


void 
IIRScram_FloatingPointLiteral::_publish_cc_data() {
  IIR_Char* lit_string = print(10);
  _cc_out << lit_string;
  delete [] lit_string;
}

void 
IIRScram_FloatingPointLiteral::_publish_cc() {
  _cc_out << "RealType(ObjectBase::VARIABLE, ";
  _publish_cc_universal_value();
  _cc_out << ", SavantrealType_info)";
}

void 
IIRScram_FloatingPointLiteral::_publish_cc_state_object_init() {
  _cc_out << "new ";
  _publish_cc();
}

void 
IIRScram_FloatingPointLiteral::_publish_cc_value() {
  _publish_cc_data();
}

void 
IIRScram_FloatingPointLiteral::_publish_cc_universal_value() {
  _cc_out << "UniversalReal(";
  _publish_cc_data();
  _cc_out << ")";
}

void 
IIRScram_FloatingPointLiteral::_publish_cc_initialization_value() {
  _publish_cc_universal_value();
}

void 
IIRScram_FloatingPointLiteral::_publish_cc_elaborate() {
  _publish_cc_data();
}


void
IIRScram_FloatingPointLiteral::_publish_cc_bounds() {
  _cc_out << "defaultInfo";
}

void
IIRScram_FloatingPointLiteral::_publish_cc_headers() {
  _cc_out << "#include \"RealType.hh\"" << endl;
}

IIR_Char*
IIRScram_FloatingPointLiteral::print(IIR_Int32 length) {
  ostrstream retbuf;
  retbuf.setf(04000);
  retbuf << setprecision(length) << _get_value() << ends;

  // The user must delete this memory, or it will leak...
  return retbuf.str();
}


IIR_FP64
IIRScram_FloatingPointLiteral::_get_value() {
  register int i;
  IIR_FP64 value = 0.0;
  IIR_Int32 base = _get_base();
  IIR_Char* man = _get_mantissa();
  int sign = 1;

  ASSERT(_get_mantissa_length() > 0);

  if (man[0] == '-') {
    sign = -1;
    value = 0.0;
  } else {
    value = _char_to_int(man[0]);
  };

  // calculate part left of the point
  for (i = 1; i < _get_mantissa_length(); i++) {
    if (man[i] == '.') {
      i++;
      break;
    }
    value = value * base + sign * _char_to_int(man[i]);
  }

  // add on the part right of the point
  double divisor = base;
  for (/* continue with old value of i */; i < _get_mantissa_length(); i++) {
    value = value + sign * _char_to_int(man[i])/divisor;
    divisor = divisor * base;
  }

  if (_get_exponent_length() > 0) {
    IIR_Char* exp = _get_exponent();
    IIR_Int32 exp_val;

    if (exp[0] == '+' || exp[0] == '-') {
      i = 2;
      exp_val = _char_to_int(exp[1]);
    }
    else {
      i = 1;
      exp_val = _char_to_int(exp[0]);
    }
    for (/* i already set */; i < _get_exponent_length(); i++) {
      exp_val = exp_val * 10 + _char_to_int(exp[i]);
    }

    // exp_val is now the base-10 representation of the exponent.  The
    // value of this node is value * base^exp_val.  Care must be taken if
    // the exponent is < 0.
    if (exp[0] == '-') {
      for (i = 0; i < exp_val; i++) {
	value = value / base;
      }
    }
    else {
      for (i = 0; i < exp_val; i++) {
	value = value * base;
      }
    }
  }

  return value;
}


void 
IIRScram_FloatingPointLiteral::_type_check( set<IIR_TypeDefinition> * ){
  // We could make sure that the set contains a floating point type...
}


set<IIR_TypeDefinition> *
IIRScram_FloatingPointLiteral::_get_rval_set(IIR_Boolean(IIR::*constraint_function)()){
  set<IIR_TypeDefinition> *retval = new set<IIR_TypeDefinition>( StandardPackage::savant_universal_real );

  return retval;
}

ostream &
IIRScram_FloatingPointLiteral::_print( ostream &os ){
  os << print( 2 );

  return os;
}
