//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

// -----------------------------------------------------------
//> DV needed to change the initial orbit into an orbit with a low 
//> enough (adjustable) perigee
//> Y-axis : DV
//
// Auteur: A. Lamy
// -----------------------------------------------------------

// Declarations:
global %CL_eqRad;

// Code:

hmin = 500.e3;
hmax = 850.e3;
tab_hper = (400:50:650) * 1000;

desc_param = list(..
   CL_defParam("Altitude - min", hmin, units=['m', 'km'], id='$hmin', valid='$hmin>=0' ),..
   CL_defParam("Altitude - max", hmax, units=['m', 'km'], id='$hmax', valid='$hmax>$hmin' ),..
   CL_defParam("Perigee altitude", tab_hper, units=['m', 'km'], dim=-1, id='$hper', valid='$hper>=0' )..
);

[hmin,hmax,tab_hper] = CL_inputParam(desc_param);


nbpts = 100; // nombre de points en x
tab_h = linspace(hmin,hmax,nbpts);

ind = find(tab_hper < hmax);
tab_hper = tab_hper(ind);

// -----------------------------------------------------------
// results / plot
// -----------------------------------------------------------

f=scf();
f.visible="off";
f.immediate_drawing="off";

a=gca();

nb = length(tab_hper);
f.color_map = jetcolormap(min(max(nb,3),100)) * 0.95; // min 3 valeurs

Noir = addcolor([0,0,0]);
GrisF = addcolor([1,1,1]*0.4);

dvmin = %inf;
dvmax = -%inf;

for k = 1:length(tab_hper);
  hper = tab_hper(k);
  rper = %CL_eqRad + hper;
  dga_i =  %CL_eqRad + max(tab_h, hper);

  exc_i = zeros(dga_i);
  raf = dga_i;
  rpf = (%CL_eqRad + hper) * ones(raf);
  [dga_f,exc_f] = CL_op_rarp2ae(raf, rpf);

  [deltav] = CL_man_hohmannG(dga_i, exc_i, dga_f, exc_f);
  dvmin = min(min(deltav), dvmin);
  dvmax = max(max(deltav), dvmax);

  plot2d((dga_i-%CL_eqRad )/1000, deltav, style=k);
end


// legends
a1 = CL_g_legend(a, string(tab_hper/1000)+" km", header="Perigee altitude");

// general setting
CL_g_stdaxes(a, colg=GrisF)
a.data_bounds = [hmin/1000,dvmin; hmax/1000,dvmax];
a.tight_limits="on";
a.title.text = "Deorbit cost for a circular orbit";
a.x_label.text = "Initial altitude (km)";
a.y_label.text = "Dv (m/s)";

// change properties
h=CL_g_select(a, "Polyline");
h.thickness=2;

f.immediate_drawing="on";
f.visible="on";

