//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [pos_ter,vel_ter,jacob]=CL_fr_topoN2ter(orig,pos_topoN,vel_topoN,er,obl)
// Topocentric North frame to terrestrial frame vector transformation
//
// Calling Sequence
// [pos_ter[,vel_ter[,jacob]] = CL_fr_topoN2ter(orig[,pos_topoN[,vel_topoN[,er[,obl]]]])
//
// Description
// <itemizedlist><listitem>
// Converts position and (optionally) velocity vectors from the topocentric North ("Earth fixed") frame to the terrestrial ("Earth fixed") frame. 
// <para>The jacobian of the transformation is optionally computed.</para>
// </listitem>
// <listitem>
// Notes: 
// <para> - The jacobian of the transformation only depends on the origin of the topocentric reference Frame. It can then be computed even if the position and velocity vectors are omitted (in this case they are given the arbitrary value [0;0;0]). </para>
// <para> - The origin of the topocentric North frame is not the center of the planet. It is the location defined by <emphasis role="bold">orig</emphasis>. </para>
// <para> - This function can be used for another planet than the Earth. </para>
// </listitem>
// <listitem>
// See <link linkend="LocalFrames">Local Frames</link> for more details on the definition of local frames. 
// </listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-06-03 )</emphasis></para>
//
// Parameters
// orig: [lon;lat;alt] topocentric frame origin in elliptical (geodetic) coordinates [rad;rad;m] (3x1 or 3xN)
// pos_topoN: (optional) [X;Y;Z] position vector relative to topocentric North frame [m] (3xN)
// vel_topoN: (optional) [Vx;Vy;Vz] velocity vector relative to topocentric North frame [m/s] (3xN)
// er: (optional) planet equatorial radius (default is %CL_eqRad) [m]
// obl: (optional) planet oblateness (default is %CL_obla)
// pos_ter: [X;Y;Z] position vector relative to terrestrial frame [m] (3xN)
// vel_ter: (optional) [Vx;Vy;Vz] velocity vector relative to terrestrial frame [m/s] (3xN)
// jacob: (optional) transformation jacobian d(x,y,z,vx,vy,vz)_ter/d(x,y,z,vx,vy,vz)_topoN (6x6xsize(orig,1))
//
// Authors
// CNES - DCT/SB
//
// Bibliography
// CNES - MSLIB FORTRAN 90, Volume T (mt_topo_N_ref)
//
// See also
// CL_fr_topoNMat
// CL_fr_ter2topoN
//
// Examples
// orig = [%pi;%pi/2;100];
//
// // Compute only jacobian :
// [pos,vel,jacob] = CL_fr_topoN2ter(orig) 
//
// // Compute position and velocity :
// pos_topoN = [50;25;32];
// vel_topoN = [70;100;0];
// [pos,vel] = CL_fr_topoN2ter(orig,pos_topoN,vel_topoN) 
//
// // Compute only position (2 positions vectorized) :
// pos_topoN = [[50;25;32] , [40;12;38]];
// [pos] = CL_fr_topoN2ter(orig,pos_topoN) 
//

// Declarations:
if(~exists('%CL_eqRad')) then global %CL_eqRad; end;
if(~exists('%CL_obla')) then global %CL_obla; end;

// Code:

[lhs,rhs] = argn()

No = size(orig,2);
if exists('pos_topoN','local')
  Np = size(pos_topoN,2);
  if (No ~= 1 & No ~= Np) then CL__error('orig must be either 3x1 or the same size of pos_topoN (3xN)'); end
  N=Np;
else
  N=No;
end

select lhs
  case 3
    compute_jacob=%t
    compute_vel=%t
    compute_pos=%t
    if ~exists('vel_topoN','local')
      vel_ter = zeros(3,N)
      warning('Zero vel_ter vector given');
      compute_vel=%f
    end
    if ~exists('pos_topoN','local')
      pos_ter = zeros(3,N)
      warning('Zero pos_ter vector given');
      compute_pos=%f
    end
  case 2
    compute_jacob=%f
    compute_vel=%t
    compute_pos=%t
  case 1
    compute_jacob=%f
    compute_vel=%f
    compute_pos=%t
  else
    CL__error('bad number of arguments')
end

if ~exists('er','local') then er=%CL_eqRad; end
if ~exists('obl','local') then obl=%CL_obla; end

//topocentric frame vectors
lon = orig(1,:)*ones(1,N);
lat = orig(2,:)*ones(1,N);

//vect_i computation: horizontal towards norh
//vect_i = (-sin(lat)cos(long),-sin(lat)sin(long),cos(lat))
vect_i(1,:) = -sin(lat).*cos(lon);
vect_i(2,:) = -sin(lat).*sin(lon);
vect_i(3,:) = cos(lat);

//vect_j computation: local paralel tangent towards west
//vect_j = (sin(long),-cos(long),0)
vect_j(1,:) = sin(lon);
vect_j(2,:) = -cos(lon);
vect_j(3,:) = zeros(1,No);

//vect_k computation: local vertical
//vect_k = (cos(lat)cos(long),cos(lat)sin(long),sin(lat))
vect_k(1,:) = cos(lat).*cos(lon);
vect_k(2,:) = cos(lat).*sin(lon);
vect_k(3,:) = sin(lat);

if compute_pos
  //topocentric frame origin cartesian coordinates
  sta = CL_co_ell2car(orig,er,obl)
  if(No ==1)
  sta = sta.*.ones(1,N)
  end

  //satellite's position components in geocentric frame
  pos_tmp = [pos_topoN(1,:).*vect_i(1,:) + pos_topoN(2,:).*vect_j(1,:) + pos_topoN(3,:).*vect_k(1,:) ;
       pos_topoN(1,:).*vect_i(2,:) + pos_topoN(2,:).*vect_j(2,:) + pos_topoN(3,:).*vect_k(2,:) ;
       pos_topoN(1,:).*vect_i(3,:) + pos_topoN(2,:).*vect_j(3,:) + pos_topoN(3,:).*vect_k(3,:) ] ;
  pos_ter = pos_tmp + sta;
end

//JACOB
//partial derivates of position and velocity vectors
if compute_jacob

  if(No ==1)
  jacob = zeros(6,6);

  jacob(1:3,1) = vect_i(:,1)
  jacob(1:3,2) = vect_j(:,1)
  jacob(1:3,3) = vect_k(:,1)
  jacob(4:6,4) = vect_i(:,1)
  jacob(4:6,5) = vect_j(:,1)
  jacob(4:6,6) = vect_k(:,1)

  else
  jacob = hypermat([6,6,N])

  jacob(1:3,1,:) = vect_i
  jacob(1:3,2,:) = vect_j
  jacob(1:3,3,:) = vect_k
  jacob(4:6,4,:) = vect_i
  jacob(4:6,5,:) = vect_j
  jacob(4:6,6,:) = vect_k
  if N==1 then jacob = jacob(:,:,1); end
  end


end

//velocity on geocentric frame
if compute_vel
  vel_ter = [vel_topoN(1,:).*vect_i(1,:) + vel_topoN(2,:).*vect_j(1,:) + vel_topoN(3,:).*vect_k(1,:) ;
       vel_topoN(1,:).*vect_i(2,:) + vel_topoN(2,:).*vect_j(2,:) + vel_topoN(3,:).*vect_k(2,:) ;
       vel_topoN(1,:).*vect_i(3,:) + vel_topoN(2,:).*vect_j(3,:) + vel_topoN(3,:).*vect_k(3,:) ];
end

endfunction
