//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [ires] = CL_intervUnion(varargin)
// Union of sets of intervals
//
// Calling Sequence
// [ires] = CL_intervUnion(i1 [,i2,i3,...,ip])
//
// Description
// <itemizedlist><listitem>
// This function computes the union of sets of intervals.
// <para> The intervals are gathered in different sets (i1, i2,...,ip). </para>
// <para> A real number x belongs to the union of (i1, i2,...,ip) if it belongs to at least one interval in one of the sets. </para>
// <inlinemediaobject><imageobject><imagedata fileref="union.gif"/></imageobject></inlinemediaobject>
// </listitem>
// <listitem>
// Notes: 
// <para> - Only one set may be given. The result is then the union of the intervals that belong to the set. </para>
// <para> - Sets of intervals may be empty: CL_intervUnion(i1,[]) = CL_intervUnion(i1)</para>
// <para> - Resulting set of intervals are sorted in ascending order (first row) </para>
// </listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-06-03 )</emphasis></para>
//
// Parameters
// i1: Set of intervals [start ; end] (2xN)
// i2: Set of intervals [start ; end] (2xN2)
// ip: Set of intervals [start ; end] (2xNp)
// ires: Union of i1,i2...,iN (2xM)
//
// See also
// CL_intervInters
// CL_intervInv
//
// Authors
// CNES - DCT/SB
//
// Examples
// i1=[ [1;3] , [5;6] , [10;12]];
// i2=[ [2;4] , [5.5;5.7] , [5.6;15]];
// ires = CL_intervUnion(i1,i2);

// Declarations:


// Code:

for i=1:size(varargin)
  if(varargin(i) ~= [])
    if(size(varargin(i),1)~=2)
      CL__error('Interval '+string(i)+' must be of size 2xN');
    end
  end
end

itot = [];
for i=1:size(varargin)
    itot = [itot,varargin(i)];
end
isort = CL_matSort(itot,1,'r');

N = size(itot,2);

  // Union of intervals :
  ires = isort(:,1);
  j = 1; // indice pour parcourir ires
  k = 1; // indice pour parcourir isort
  while(k < N)
    ii = 0;
    while(ii ~= find(isort(1,:) <= ires(2,j)))
      ii = find(isort(1,:) <= ires(2,j)) ; // indices pour lesquels la date de debut est plus petite que la date de fin actuelle
      if(ii ~= [])
        ires(2,j) = max(isort(2,ii)) // la date de fin actuelle devient la plus grande date de fin parmi ces indices
      end
    end
    j=j+1;

    ind2 = find(isort(1,:) > ires(2,j-1));
    if(ind2 ~= [])
      ires = [ires,isort(:,ind2(1))];
      k = ind2(1);
    else
      k=N+1;
    end
  end


endfunction
