/*  Screem:  frameWizard.c,
 *  A wizard for adding framesets
 *
 *  Copyright (C) 2000  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>
#include <gmodule.h>
#include <gnome.h>
#include <glade/glade.h>

#include <dirent.h>
#include <sys/stat.h>
#include <unistd.h>

#include "site.h"
#include "page.h"
#include "editor.h"
#include "htmlfuncs.h"
#include "dtd.h"

extern GtkWidget *app;
extern Site *current_site;

static GladeXML *xml;
static GtkWidget *dialog = NULL;
static gboolean selecting = FALSE;

static GList *styles = NULL;

G_MODULE_EXPORT const gchar* g_module_check_init( GModule *module );
G_MODULE_EXPORT void g_module_unload( GModule *module );
G_MODULE_EXPORT void init(); 

void frameWizard( void );
void style_changed( GtkWidget *widget );
void frame_wizard_closed( GtkWidget *widget, GdkEvent *event );

typedef enum _FrameScrolling {
	FRAME_SCROLL_YES,
	FRAME_SCROLL_NO,
	FRAME_SCROLL_AUTO
} FrameScrolling;

/* each frame in the preview has a details property which contains an 
   instance of this struct */
typedef struct _FrameDetails {
	gchar          *src;
	gboolean        noresize;
	gboolean        border;
	gint            width;
	gint            height;
	FrameScrolling  scroll;
	gchar          *name;
} FrameDetails;

typedef enum _FrameType {
	FRAME_SET,
	FRAME
} FrameType;

/* used to construct the preview of the layout */
typedef struct _Frame {
	gint type;
	gint width;
	gint height;
	gint cols;           /* the number of columns the set has */
	gint rows;           /* the number of rows the set has */
	GList *children;     /* may be Frames, or Frame sets */
} Frame;

/* a list of these make up the styles available in the wizard */
typedef struct _FrameStyle {
	gchar *name;
	Frame *root;
} FrameStyle;

G_MODULE_EXPORT const gchar* g_module_check_init( GModule *module )
{
        g_print("frameWizard: check-init\n");
        return NULL;
}

G_MODULE_EXPORT void g_module_unload( GModule *module )
{
        g_print( "frameWizard: unloaded\n" );
}

G_MODULE_EXPORT void init() 
{
	GnomeUIInfo menuinfo[] = { 
                {
                        GNOME_APP_UI_ITEM, N_("Frames..."),
                        N_("Define a frameset"),
                        frameWizard, NULL, NULL,
                        GNOME_APP_PIXMAP_STOCK,
                        GNOME_STOCK_MENU_BLANK,
                        0,
                        GDK_CONTROL_MASK, NULL
                },
                GNOMEUIINFO_END
        };

      	/* place menu item after image under insert */
        gnome_app_insert_menus( GNOME_APP( app ),
                                _("_Insert/"), menuinfo);

	load_frame_styles();

	g_print( "frameWizard: initialised\n" );
}

void frameWizard()
{
	GtkWidget *combo;
	Page *page;
	GList *list;
	GList *list2;
	FrameStyle *style;

	page = screem_site_get_current_page( current_site );
	
	g_return_if_fail( page != NULL );
        
        if( dialog ) {
                gdk_window_raise( dialog->window );
                gdk_window_show( dialog->window );
                return;
        }

	xml = glade_xml_new( GLADE_PATH"/frameWizard.glade", "frame_wizard" );
	glade_xml_signal_autoconnect( xml );

	dialog = glade_xml_get_widget( xml, "frame_wizard" );

	combo = glade_xml_get_widget( xml, "styles" );

	/* populate the styles combo */
	for( list2 = NULL, list = styles; list; list = list->next ) {
		style = list->data;
		list2 = g_list_append( list2, style->name );
	}
	gtk_combo_set_popdown_strings( GTK_COMBO( combo ), list2 );
	g_list_free( list2 );
}

void style_changed( GtkWidget *widget )
{
	GList *list;
	gchar *name;
	FrameStyle *style;

	name = gtk_entry_get_text( GTK_ENTRY( widget ) );

	for( list = styles; list; list = list->next ) {
		style = list->data;
		if( ! strcmp( name, style->name ) )
			break;
	}
	if( ! list )
		return;

	/* display the frames in the preview canvas */

}

void frame_wizard_clicked( GtkWidget *widget, gint button )
{

	if( button == 2 || button == 0 ) { /* ok or close clicked */
                gtk_widget_destroy( widget );
                dialog = NULL;
	}
}

void frame_wizard_closed( GtkWidget *widget, GdkEvent *event )
{
        dialog = NULL;
}

FrameDetails* frame_details_new()
{
	FrameDetails *details;

	details = g_new( FrameDetails, 1 );

	details->src = g_strdup( "" );
	details->noresize = FALSE;
	details->border = FALSE;
	details->width = 100;
	details->height = 100;
	details->scroll = FRAME_SCROLL_AUTO;
	details->name = g_strdup( "" );

	return details;
}

void select_frame( GnomeCanvasItem *item )
{
	FrameDetails *details;
	GtkWidget *widget;

	details = gtk_object_get_data( GTK_OBJECT( item ), "details" );

	selecting = TRUE;

	widget = glade_xml_get_widget( xml, "src" );
	gtk_entry_set_text( GTK_ENTRY( widget ), details->src );

	widget = glade_xml_get_widget( xml, "noresize" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ),
				      details->noresize );

	widget = glade_xml_get_widget( xml, "border" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ),
				      details->border );

	widget = glade_xml_get_widget( xml, "width" );
	gtk_spin_button_set_value( GTK_SPIN_BUTTON( widget ),
				   details->width );

	widget = glade_xml_get_widget( xml, "height" );
	gtk_spin_button_set_value( GTK_SPIN_BUTTON( widget ),
				   details->height );

	switch( details->scroll ) {
	case FRAME_SCROLL_YES:
		widget = glade_xml_get_widget( xml, "scroll_yes" );
		break;
	case FRAME_SCROLL_NO:
		widget = glade_xml_get_widget( xml, "scroll_no" );
		break;
	default:
		widget = glade_xml_get_widget( xml, "scroll_auto" );
		break;
	}
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ), TRUE );

	widget = glade_xml_get_widget( xml, "name" );
	gtk_entry_set_text( GTK_ENTRY( widget ), details->name );

	selecting = FALSE;
}

GnomeCanvasItem* create_frame( GnomeCanvasGroup *root, 
			       gdouble x, gdouble y, gdouble x2, gdouble y2 )
{
	GnomeCanvasItem *item;
	GtkType type;

	FrameDetails *details;

	type = gnome_canvas_rect_get_type();

	item = gnome_canvas_item_new( root, type,
				      "x1", x,
				      "y1", y,
				      "x2", x2,
				      "y2", y2,
				      "fill_color", "white",
				      NULL );
	/* setup attribute values */
	details = frame_details_new();

	gtk_object_set_data( GTK_OBJECT( item ), "details", details );


	return item;
}

void load_frame_styles()
{
           struct dirent **namelist;
	   gint num;
	   const gchar *type;
	   gchar cwd[ 16384 ];
	   FrameStyle *frame_style;

	   getcwd( cwd, 16384 );

	   chdir( FRAMESPATH );

	   num = scandir(".", &namelist, 0, alphasort);

	   while( num > 0 ) {
		   num --;
		   type = gnome_mime_type_or_default( namelist[ num ]->d_name,
						      "no" );
		   if( ! strcmp( "text/html", type ) )
			   frame_style = read_frame( namelist[ num ]->d_name );
		   else
			   frame_style = NULL;
		   
		   if( frame_style )
			   styles = g_list_prepend( styles, frame_style );

		   g_free( namelist[ num ] );
	   }

	   chdir( cwd );
}

FrameStyle* read_frame( gchar *file )
{
	struct stat s;
	gchar *buffer;
	FILE *in;
	ScreemDTD *dtd;

	GNode *node;
	GNode *working;
	Node *n;
	FrameStyle *frame_style;

	if( stat( file, &s ) < 0 )
		return NULL;

	buffer = g_new0( gchar, s.st_size );
	in = fopen( file, "r" );

	g_return_val_if_fail( in != NULL, NULL );

	fread( buffer, 1, s.st_size, in );
	fclose( in );

	dtd = screem_get_doctype( buffer );
	node = screem_html_build_parse_tree( dtd, buffer, 0 );

	g_free( buffer );


	/* must have built the tree */
	if( ! node )
		return NULL;
	
	/* there must be a node with data */
	n = node->data;
	if( ! n ) {
		screem_html_destroy_parse_tree( node );
		return NULL;
	}
	/* the first node must be <title> as it is used as the style name */
	if( ! n->name || ( n->name && strcasecmp( "title", n->name ) ) ) {
		g_print( "Invalid Frame html file\n" );
		screem_html_destroy_parse_tree( node );
		return NULL;
	}

	/* set working to be the contents of the title, we won't bother
	   checking for other children as it should only be text */
	working = node->children;
	if( ! working ) {
		g_print( "Invalid Frame html file\n" );
		screem_html_destroy_parse_tree( node );
		return NULL;
	}
	n = working->data;
	if( ! n ) {
		g_print( "Invalid Frame html file\n" );
		screem_html_destroy_parse_tree( node );
		return NULL;
	}

	frame_style = g_new0( FrameStyle, 1 );
	frame_style->name = g_strdup( n->content );
	frame_style->root = g_new0( Frame, 1 );
	frame_style->root->type = FRAME_SET;

	/* now comes the frameset / frames info */
	working = node->next;
	if( ! working ) {
		g_print( "Invalid Frame html file\n" );
		screem_html_destroy_parse_tree( node );
		return NULL;
	}
	n = working->data;
	if( ! n || 
	    ( n && ( ! n->name || 
		     ( n->name && strcasecmp( "frameset", n->name ) ) ) ) ) {
		g_print( "Invalid Frame html file\n" );
		screem_html_destroy_parse_tree( node );
		return NULL;
	}
	/* working is the root <frameset> element, we can determine how
	   many frames we have by cols * rows, although some of those
	   frames may themselves be framesets */
	frame_style->root = parse_frameset( working, NULL );


	/* destroy the parse tree */
	screem_html_destroy_parse_tree( node );

	return frame_style;
}

Frame* parse_frameset( GNode *node, Frame *set )
{
	Node *n;
	Frame *frame;
	GList *attr;

	gchar *val;
	gchar *temp;

	if( ! node )
		    return NULL;

	while( TRUE ) {
		n = node->data;
		if( ! n->name )
			node = node->next;
		else
			break;

		if( ! node )
			return NULL;
	}

	frame = g_new0( Frame, 1 );
	if( set )
		set->children = g_list_append( set->children, frame );

	if( ! strcasecmp( "frame", n->name ) ) {
		/* its a normal frame */
		g_print( "new frame\n" );
		frame->type = FRAME;
		parse_frameset( node->next, set );
	} else {
		/* its a frameset, get the number of columns and rows */
		frame->cols = 1;
		frame->rows = 1;

		attr = screem_html_attribute_from_list( n->attributes,"cols" );
		if( attr ) {
			attr = attr->next;
			val = attr->data;
			while( ( temp = strchr( val, ',' ) ) ) {
				val = temp + 1;
				frame->cols ++;
			}
		}

		attr = screem_html_attribute_from_list( n->attributes,"rows" );
		if( attr ) {
			attr = attr->next;
			val = attr->data;
			val = attr->data;
			while( ( temp = strchr( val, ',' ) ) ) {
				val = temp + 1;
				frame->rows ++;
			}
		}

		g_print( "new frameset: %i columns\t%i rows\n",
			 frame->cols, frame->rows );
		frame->type = FRAME_SET;
		parse_frameset( node->children, frame );
	}

	return frame;
}
