/* 
   sitecopy WebDAV protocol driver module
   Copyright (C) 2000, Joe Orton <joe@orton.demon.co.uk>
                                                                     
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   $Id: davdriver.c,v 1.3 2001/02/21 20:45:04 davek Exp $
*/

#include <config.h>

#include <sys/stat.h> /* For S_IXUSR */

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#include <errno.h>

#include "common.h" /* for strerror */

#include <http_request.h>
#include <http_basic.h>
#include <dav_basic.h>
#include <dav_props.h>
#include <ne_alloc.h>
#include <uri.h>

#include "protocol.h"
#include "frontend.h"
#include "i18n.h"

struct fetch_context {
    struct proto_file **files;
    const char *root;
};

/* TODO:
 * not really sure whether we should be using an enum here... what
 * should the client do with resourcetypes it doesn't understand?
 * ignore them, or presume they have the same semantics as "normal"
 * resources. I really don't know.
 */
struct private {
    int iscollection;
};

#define ELM_resourcetype (DAV_ELM_207_UNUSED)
#define ELM_collection (DAV_ELM_207_UNUSED + 1)

/* The element definitinos for the complex prop handler. */
static const struct hip_xml_elm fetch_elms[] = {
    { "DAV:", "resourcetype", ELM_resourcetype, 0 },
    { "DAV:", "collection", ELM_collection, 0 },
    { NULL }
};

static const dav_propname complex_props[] = {
    { "DAV:", "resourcetype" },
    { NULL }
};

static const dav_propname flat_props[] = {
    { "DAV:", "getcontentlength" },
    { "DAV:", "getlastmodified" },
    { "http://apache.org/dav/props/", "executable" },
    { NULL }
};

static inline int get_depth(const char *href) {
    const char *pnt;
    int count = 0;
    for (pnt=href; *pnt != '\0'; pnt++) /* oneliner */
	if (*pnt == '/') count++;
    return count;
}

/* TODO: get rid of this? */
static void set_err(http_session *sess, const char *msg)
{
    char *err;
    CONCAT2(err, msg, strerror(errno));
    http_set_error(sess, err);
    free(err);
}

static int get_server_port(struct site *site)
{
    const char *svc;
    int port, defport;
    if (site->http_secure) {
	svc = "https";
        defport = 443;
    } else {
	svc = "http";
	defport = 80;
    }
    port = sock_service_lookup(svc);
    if (port == 0) {
	port = defport;
	DEBUG(DEBUG_HTTP, "Using default port for %s: %d\n", svc, port);
    }
    return port;
}

static int get_proxy_port(struct site *site)
{
    return 8080;
}

static int auth_common(void *userdata, fe_login_context ctx,
		       const char *realm, const char *hostname,
		       char **username, char **password)
{
    struct site_host *host = userdata;
    if (host->username && host->password) {
	*username = ne_strdup(host->username);
	*password = ne_strdup(host->password);
	return 0;
    } else {
	return fe_login(ctx, realm, hostname, username, password);
    }
}

static int 
server_auth_cb(void *userdata, const char *realm, const char *hostname,
	       char **username, char **password)
{
    return auth_common(userdata, fe_login_server, realm, hostname,
		       username, password);
}

static int 
proxy_auth_cb(void *userdata, const char *realm, const char *hostname,
	      char **username, char **password)
{
    return auth_common(userdata, fe_login_proxy, realm, hostname,
		       username, password);
}

static int h2s(http_session *sess, int errcode)
{
    switch (errcode) {
    case HTTP_OK:
	return SITE_OK;
    case HTTP_AUTH:
	return SITE_AUTH;
    case HTTP_AUTHPROXY:
	return SITE_PROXYAUTH;
    case HTTP_FAILED:
	return SITE_FAILED;
    case HTTP_CONNECT:
	return SITE_CONNECT;
    case HTTP_LOOKUP:
	return SITE_LOOKUP;
    case HTTP_TIMEOUT:
	http_set_error(sess, _("The connection timed out."));
	return SITE_ERRORS;
    case HTTP_SERVERAUTH:
	http_set_error(sess, 
		       _("The server did not authenticate itself correctly.\n"
			 "Report this error to your server administrator."));
	return SITE_ERRORS;
    case HTTP_PROXYAUTH:
	http_set_error(sess, 
		 _("The proxy server did not authenticate itself correctly.\n"
		   "Report this error to your proxy server administrator."));
	return SITE_ERRORS;
    case HTTP_ERROR:
    default:
	return SITE_ERRORS;
    }
}

/* Special h2s() wrapper for init() functions. */
static int h2s_init(http_session *sess, int errcode)
{
    int ret = h2s(sess, errcode);
    if (ret == SITE_ERRORS)
	ret = SITE_FAILED;
    return ret;
}

static int 
init(void **session, struct site *site)
{
    http_session *sess = http_session_create();
    http_server_capabilities caps = {0};
    int ret;

    *session = sess;

    if (site->http_use_expect)
 	http_set_expect100(sess, 1);
    else
	http_set_expect100(sess, 0);

    if (site->http_limit)
	http_set_persist(sess, 0);

    if (site->http_secure) {
	if (http_set_secure(sess, 1)) {
	    http_set_error(sess, _("SSL support has not be compiled in."));
	    return SITE_FAILED;
	}
    }

    /* Note, this won't differentiate between xsitecopy and
     * sitecopy... maybe we should put a comment in as well. */
    http_set_useragent(sess, PACKAGE "/" VERSION);

    if (site->proxy.hostname) {
	http_set_proxy_auth(sess, proxy_auth_cb, &site->proxy);
	ret = http_session_proxy(sess, site->proxy.hostname, site->proxy.port);
	if (ret == HTTP_LOOKUP)
	    return SITE_PROXYLOOKUP;
	else if (ret != HTTP_OK) {
	    return h2s_init(sess, ret);
	}
    }

    http_set_server_auth(sess, server_auth_cb, &site->server);
    ret = http_session_server(sess, site->server.hostname, site->server.port);
    if (ret != HTTP_OK) {
	return h2s_init(sess, ret);
    }
    
    ret = http_options(sess, site->remote_root, &caps);
    if (ret == HTTP_OK) {
	if (!caps.dav_class1) {
	    http_set_error(sess, 
			    _("The server does not appear to be a WebDAV server."));
	    return SITE_FAILED;
	} else if (site->perms != sitep_ignore && !caps.dav_executable) {
	    /* Need to set permissions, but the server can't do that */
	    http_set_error(sess, 
			    _("The server does not support the executable live property."));
	    return SITE_FAILED;
	}
    } else {
	return h2s_init(sess, ret);
    }


    return SITE_OK;
}

static void finish(void *session) 
{
    http_session *sess = session;
    http_session_destroy(sess);
}

static int file_move(void *session, const char *from, const char *to) 
{
    http_session *sess = session;
    return h2s(sess, dav_move(sess, 0, from, to));
}

static int file_upload(void *session, const char *local, const char *remote, 
		       int ascii)
{
    FILE *f = fopen(local, "r" FOPEN_BINARY_FLAGS);
    http_session *sess = session;
    int ret;

    if (f == NULL) {
	set_err(sess, _("Could not open file: "));
	return SITE_ERRORS;
    }
    
    ret = http_put(sess, remote, f);

    (void) fclose(f);

    return h2s(sess, ret);
}

static int 
file_upload_cond(void *session, const char *local, const char *remote,
		 int ascii, time_t time)
{
    http_session *sess = session;
    FILE *f = fopen(local, "r" FOPEN_BINARY_FLAGS);
    int ret;

    if (f == NULL) {
	set_err(sess, _("Could not open file: "));
	return SITE_ERRORS;
    }
    
    ret = h2s(sess, http_put_if_unmodified(sess, remote, f, time));
    
    if (ferror(f)) {
	if (ret != SITE_OK) {
	    set_err(sess, _("Could not write to file: "));
	    ret = SITE_ERRORS;
	}
    } 
    if (fclose(f)) {
	ret = SITE_ERRORS;
    }
    
    return ret;
}

static int file_get_modtime(void *session, const char *remote, time_t *modtime)
{
    http_session *sess = session;
    return h2s(sess, http_getmodtime(sess,remote,modtime));
}
    
static int file_download(void *session, const char *local, const char *remote,
			 int ascii) 
{
    http_session *sess = session;
    FILE *f = fopen(local, "w" FOPEN_BINARY_FLAGS);
    int ret;

    if (f == NULL) {
	set_err(sess, _("Could not open file: "));
	return SITE_ERRORS;
    }
    
    ret = h2s(sess, http_get(sess, remote, f));
    
    if (ferror(f)) {
	set_err(sess, _("Could not write to file: "));
	ret = SITE_ERRORS;
    }
    if (fclose(f)) {
	ret = SITE_ERRORS;
    }
    
    return ret;
}

static int file_read(void *session, const char *remote, 
		     sock_block_reader reader, void *userdata) 
{
    http_session *sess = session;
    return h2s(sess, http_read_file(sess, remote, reader, userdata));
}

static int file_delete(void *session, const char *filename) 
{
    http_session *sess = session;
    return h2s(sess, dav_delete(sess, filename));
}

static int file_chmod(void *session, const char *filename, mode_t mode) 
{
    http_session *sess = session;
    static const dav_propname execprop = 
    { "http://apache.org/dav/props/", "executable" };
    /* Use a single operation; set the executable property to... */
    dav_proppatch_operation ops[] = { 
	{ &execprop, dav_propset, NULL }, { NULL } 
    };
    
    /* True or false, depending... */
    if (mode & S_IXUSR) {
	ops[0].value = "T";
    } else {
	ops[0].value = "F";
    }
    return h2s(sess, dav_proppatch(sess, filename, ops));
}

static int dir_create(void *session, const char *dirname)
{
    http_session *sess = session;
    return h2s(sess, dav_mkcol(sess, dirname));
}

static int dir_remove(void *session, const char *dirname)
{
    http_session *sess = session;
    /* TODO: check whether it is empty first */
    return h2s(sess, dav_delete(sess, dirname));
}

/* Insert the file in the list in the appropriate position (keeping it
 * sorted). */
static void insert_file(struct proto_file **list, struct proto_file *file)
{
    struct proto_file *previous, *current;
    previous = NULL;
    current = *list;
    while (current != NULL && current->depth < file->depth) {
	previous = current;
	current = current->next;
    }
    if (previous == NULL) {
	*list = file;
    } else {
	previous->next = file;
    }
    file->next = current;
}

static void pfind_results(void *userdata, const char *href,
			  const dav_prop_result_set *set)
{
    struct fetch_context *ctx = userdata;
    struct private *private = dav_propset_private(set);
    const char *clength = NULL, *modtime = NULL, *isexec = NULL;
    struct proto_file *file;
    int iscoll;

    /* Find out whether this is a collection or not, then
     * free the private structure so we don't have to worry 
     * about it any more. */
    iscoll = private->iscollection;
    free(private);

    if (strncmp(href, "http://", 7) == 0) {
	/* Absolute URI */
	href = strchr(href+7, '/');
	if (href == NULL) {
	    DEBUG(DEBUG_HTTP, "Invalid URI???");
	    return;
	}
    }
    if (!uri_childof(ctx->root, href)) {
	/* URI not a child of the root collection... 
	 * ignore this resource */
	return;
    }
   
    if (!iscoll) {
	const http_status *status = NULL;

	clength = dav_propset_value(set, &flat_props[0]);    
	modtime = dav_propset_value(set, &flat_props[1]);
	isexec = dav_propset_value(set, &flat_props[2]);
	
	if (clength == NULL)
	    status = dav_propset_status(set, &flat_props[0]);
	if (modtime == NULL)
	    status = dav_propset_status(set, &flat_props[1]);

	if (status != NULL) {
	    fe_warning(_("Could not access resource"), href,
		       status->reason_phrase);
	    return;
	}

    }

    file = ne_calloc(sizeof(struct proto_file));
    file->filename = ne_strdup(href+strlen(ctx->root));
    file->depth = get_depth(file->filename);
    
    if (iscoll) {
	file->type = proto_dir;

	/* Strip the trailing slash if it has one. */
	if (uri_has_trailing_slash(file->filename)) {
	    file->filename[strlen(file->filename) - 1] = '\0';
	}

    } else {
	file->type = proto_file;
	file->size = atoi(clength);
	file->modtime = http_dateparse(modtime);
	if (isexec && strcasecmp(isexec, "T") == 0) {
	    file->mode = 0755;
	} else {
	    file->mode = 0644;
	}
    }

    /* Insert the file into the files list. */
    insert_file(ctx->files, file);

}

static int check_context(hip_xml_elmid parent, hip_xml_elmid child)
{
    if (parent == DAV_ELM_prop && child == ELM_resourcetype)
	return HIP_XML_VALID;

    if (parent == ELM_resourcetype && child == ELM_collection)
	return HIP_XML_VALID;

    return HIP_XML_DECLINE;
}

static int end_element(void *userdata, 
		       const struct hip_xml_elm *elm, const char *cdata)
{
    dav_propfind_handler *handler = userdata;
    struct private *private = dav_propfind_current_private(handler);

    if (private == NULL) {
	return 0;
    }

    if (elm->id == ELM_collection)
	private->iscollection = 1;
    
    return 0;
}

/* Creates the private structure. */
static void *create_private(void *userdata, const char *uri)
{
    struct private *private = ne_calloc(sizeof *private);
    return private;
}

/* TODO: optimize: only ask for lastmod + executable when we really
 * need them: it does waste bandwidth and time to ask for executable
 * when we don't want it, since it forces a 404 propstat for each
 * non-collection resource if it is not defined.  */
static int fetch_list(void *session, const char *dirname, int need_modtimes,
		       struct proto_file **files) 
{
    http_session *sess = session;
    int ret;
    struct fetch_context ctx;
    dav_propfind_handler *ph;

    ctx.root = dirname;
    ctx.files = files;
    ph = dav_propfind_create(sess, dirname, DAV_DEPTH_INFINITE);
    
    /* The flat props */    
    dav_propfind_set_flat(ph, flat_props);

    /* The complex props. */
    dav_propfind_set_complex(ph, complex_props, create_private, NULL);

    /* Register the handler for the complex props. */
    hip_xml_push_handler(dav_propfind_get_parser(ph), fetch_elms,
			 check_context, NULL, end_element, ph);

    ret = dav_propfind_named(ph, pfind_results, &ctx);

    return h2s(sess,ret);
}

static int unimp_link2(void *session, const char *link, const char *target)
{
    http_session *sess = session;
    http_set_error(sess, "Operation not supported");
    return SITE_UNSUPPORTED;
}
 
static int unimp_link1(void *session, const char *link)
{
    http_session *sess = session;
    http_set_error(sess, "Operation not supported");
    return SITE_UNSUPPORTED;
}


static const char *error(void *session) 
{
    http_session *sess = session;
    return http_get_error(sess);
}

/* The WebDAV protocol driver */
const struct proto_driver dav_driver = {
    init,
    finish,
    file_move,
    file_upload,
    file_upload_cond,
    file_get_modtime,
    file_download,
    file_read,
    file_delete,
    file_chmod,
    dir_create,
    dir_remove,
    unimp_link2, /* create link */
    unimp_link2, /* change link target */
    unimp_link1, /* delete link */
    fetch_list,
    error,
    get_server_port,
    get_proxy_port,
    "WebDAV"
};
