/*  Screem:  site.c,
 *  contains the real site struct definition, along with access functions.
 *  also contains functions for loading, creating, saving sites and the
 *  site project file
 * 
 *  Copyright (C) 1999, 2000  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>
#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <glib.h>
#include <string.h>
#include <unistd.h>
#include <gnome-xml/debugXML.h>
#include <gnome-xml/tree.h>
#include <gnome-xml/parser.h>
#include <libgnome/libgnome.h>
#include <sys/stat.h>

#ifdef HAVE_GNOME_VFS
#include <libgnomevfs/gnome-vfs-mime.h>
#endif

#include "fileops.h"
#include "page.h"
#include "site.h"
#include "todo.h"
#include "xml.h"

#include "htmlfuncs.h"

#include "editMenu.h"

const gchar *permission_strings[] = {
	"p_exec",
	"p_ignore",
	"p_all"
};
const gchar *symlink_strings[] = {
	"s_follow",
	"s_ignore",
	"s_maintain"
};
const gchar *upload_strings[] = {
	"local",
	"ftp",
	"webdav",
	"rsh",
	"ssh"
};

/* This is the real site structure.  Site is just a typedef to a gpointer,
   the reason being that it hides the details of the struct which allows
   us to change it without needing to recompile everything and all the
   plugins */
typedef struct RealSite {
	gchar *name;
	gchar *pathname;
	gchar *remote_url;
	UploadMethods remote_method;
	gchar *remote_path;
	gchar *remote_user;
	gchar *remote_pass;
	gboolean passive_ftp;	/* sitecopy : use passive ftp  */
	gboolean no_delete;	/* sitecopy : delete remote files */
	gboolean check_moved;	/* sitecopy : check moved files */
	gboolean no_overwrite;	/* sitecopy : overwrite remote files */
	SitePerms permissions;	/* sitecopy : permissions handling */
	SiteSymlinks symlinks;	/* sitecopy : symlinks handling */
	gchar *http_url;
	gchar *cvs_root;
	gchar *template_path;

	GList *pages;
	GList *tasks;
	Page  *current;      /* the current page */

	gboolean is_import;  /* flag to signify if we are importing a
				non screem site */

	gint open_pages;     /* the number of open pages */
	
	gboolean fake_site;  /* indicates if we are just a bunch of random
				open pages rather than a screem site */

	GList *exclude;      /* list of paths to exclude */
	GList *ignore;       /* list of paths to ignore */
	GList *ascii;        /* list of files to upload in ascii mode */

	GList *auto_open;    /* list of files to be auto opened */

	gboolean auto_update; /* auto update from CVS upon opening the site */
	gboolean auto_update_dont_ask;
} RealSite;

static void screem_site_set_pages( Site *site, GList *list );
static void screem_site_set_tasks( Site *site, GList *list );

/**
 * screem_site_new:
 *
 * Creates and initialises a new site struct
 *
 * return values: a Site
 */
Site* screem_site_new()
{
	RealSite *site;

	site = (RealSite *)g_new0( RealSite, 1 );

	site->name = NULL;
	site->pathname = NULL;
	site->remote_url = NULL;
	site->remote_method = LOCAL;
	site->remote_path = NULL;
	site->remote_user = NULL;
	site->remote_pass = NULL;
        site->passive_ftp = TRUE;
        site->no_delete = FALSE;
        site->check_moved = TRUE;
        site->no_overwrite = FALSE;
	site->permissions = PERMS_IGNORE;
        site->symlinks = SYM_FOLLOW;
	site->cvs_root = NULL;
	site->template_path = NULL;
	site->pages = NULL;
	site->tasks = NULL;
	site->current = NULL;
	site->http_url = NULL;
	site->is_import = FALSE;
	site->open_pages = -1;
	site->fake_site = FALSE;

	site->exclude = NULL;
	site->ignore = NULL;
	site->ascii = NULL;
	site->auto_open = NULL;
	
	site->auto_update = FALSE;
	site->auto_update_dont_ask = FALSE;

	return (Site*)site;
}

/**
 * screem_site_destroy:
 * @site: the site to be destroyed
 *
 * Destroys a site struct
 *
 * return values: none
 */

void screem_site_destroy( Site *site )
{
	RealSite *s;
	g_return_if_fail( site );

	s = (RealSite*)site;

	screem_site_purge( site );

	g_free( s->name );
	g_free( s->pathname );
	g_free( s->remote_url );
    	g_free( s->remote_path );
	g_free( s->remote_user );
	g_free( s->remote_pass );
	g_free( s->cvs_root );
	g_free( s->template_path );
	g_list_free( s->pages );
	g_list_free( s->tasks );
	g_free( s->http_url );
	g_free( s );
}

/**
 * screem_site_purge:
 * @site: the site to be purged
 *
 * removes all the pages and tasks from a site freeing up any memory 
 * they are using
 *
 * return values: none
 */
void screem_site_purge( Site *site )
{
	RealSite *s;
	GList *list;
	Page  *p;
	TodoItem *item;

	g_return_if_fail( site );

	s = (RealSite*) site;

	for( list = s->pages; list; list = list->next ) {
		p = (Page*)list->data;
		screem_page_destroy( p );
	}
	g_list_free( s->pages );
	s->pages = NULL;
	screem_site_set_current_page( site, NULL );

	for( list = s->tasks; list; list = list->next ) {
		item = (TodoItem*)list->data;
		screem_todo_item_destroy( item );
	}
}

/**
 * screem_site_set_name:
 * @site: the site
 * @name: the name to give the site
 *
 * Sets / changes the name of the site
 *
 * return values: none
 */
void screem_site_set_name( Site *site, const gchar *name )
{
	RealSite *s;
	g_return_if_fail( site );
	g_return_if_fail( name );

	s = (RealSite*)site;

	/* does the site already have a name set? */
	if( s->name )
		g_free( s->name );
	
	s->name = g_strdup( name );
}

const gchar* screem_site_get_name( Site *site )
{
	g_return_val_if_fail( site != NULL, NULL );

	return ((RealSite*)site)->name;
}

/**
 * screem_site_set_pathname:
 * @site: the site
 * @pathname: the pathname that contains the site
 *
 * Sets / changes the pathname of the site
 * It does NOT copy the site to the new pathname if changing, its
 * also does NOT alter the page pathnames to use the new site directory
 *
 * return values: none
 */
void screem_site_set_pathname( Site *site, const gchar *pathname )
{
	RealSite *s;

	g_return_if_fail( site );
	g_return_if_fail( pathname );

	s = (RealSite*)site;

	/* does the site already have a pathname set? */
	if( s->pathname )
		g_free( s->pathname );

	/* the pathname must end with a G_DIR_SEPARATOR */
	if( pathname[ strlen( pathname ) - 1 ] != G_DIR_SEPARATOR ) {
		s->pathname = g_strconcat( pathname, G_DIR_SEPARATOR_S, 
					      NULL );
	} else {
		s->pathname = g_strdup( pathname );
	}
}


const gchar* screem_site_get_pathname( Site *site )
{
	g_return_val_if_fail( site != NULL, NULL );

	return ((RealSite*)site)->pathname;
}

/**
 * screem_site_set_remote_url:
 * @site: the site
 * @remote_url: the url of the remote site
 *
 * Sets / changes the address of the remote site for uploading to
 *
 * return values: none
 */
void screem_site_set_remote_url( Site *site, const gchar *remote_url )
{
	RealSite *s;

	g_return_if_fail( site );
	g_return_if_fail( remote_url );

	s = (RealSite*)site;

	/* does the site already have a remote url set? */
	if( s->remote_url )
		g_free( s->remote_url );

	s->remote_url = g_strdup( remote_url );
}

const gchar* screem_site_get_remote_url( Site *site )
{
	g_return_val_if_fail( site != NULL, NULL );

	return ((RealSite*)site)->remote_url;
}

/**
 * screem_site_set_remote_method:
 * @site: the site
 * @remote_method: the upload method for the site
 *
 * Sets / changes the method used to copy the site to the remote url
 *
 * return values: none
 */
void screem_site_set_remote_method( Site *site, UploadMethods remote_method )
{
	RealSite *s;

	g_return_if_fail( site );

	s = (RealSite*)site;

	s->remote_method = remote_method;
}

UploadMethods screem_site_get_remote_method( Site *site )
{
	g_return_val_if_fail( site != NULL, LOCAL );

	return ((RealSite*)site)->remote_method;
}

/**
 * screem_site_set_remote_path:
 * @site: the site
 * @remote_path: the path at the remote url in which the site will be placed
 *
 * Sets / changes the remote location of the site on remote url
 *
 * return values: none
 */
void screem_site_set_remote_path( Site *site, const gchar *remote_path )
{
	RealSite *s;

	g_return_if_fail( site );
	g_return_if_fail( remote_path );

	s = (RealSite*)site;

	/* does the site already have a remote path set? */
	if( s->remote_path )
		g_free( s->remote_path );

	s->remote_path = g_strdup( remote_path );
}

const gchar* screem_site_get_remote_path( Site *site )
{
	g_return_val_if_fail( site != NULL, NULL );

	return ((RealSite*)site)->remote_path;
}

/**
 * screem_site_set_remote_user
 * @site: the site
 * @remote_user: the username to login to the remote url as
 *
 * Sets / changes the username with which to login to the remote url as
 *
 * return values: none
 */
void screem_site_set_remote_user( Site *site, const gchar *remote_user )
{
	RealSite *s;

	g_return_if_fail( site );
	g_return_if_fail( remote_user );

	s = (RealSite*)site;
	
	/* does the site already have a remote usernmae set? */
	if( s->remote_user )
		g_free( s->remote_user );

	s->remote_user = g_strdup( remote_user );
}

const gchar* screem_site_get_remote_user( Site *site )
{
	g_return_val_if_fail( site != NULL, NULL );

	return ((RealSite*)site)->remote_user;
}

/**
 * screem_site_set_remote_pass
 * @site: the site
 * @remote_pass: the password to login to the remote url
 *
 * Sets / changes the password with which to login to the remote url
 *
 * return values: none
 */
void screem_site_set_remote_pass( Site *site, const gchar *remote_pass )
{
	RealSite *s;

	g_return_if_fail( site );
	g_return_if_fail( remote_pass );

	s = (RealSite*)site;

	/* does the site already have a remote password set? */
	if( s->remote_pass )
		g_free( s->remote_pass );

	s->remote_pass = g_strdup( remote_pass );
}

const gchar* screem_site_get_remote_pass( Site *site )
{
	g_return_val_if_fail( site != NULL, NULL );

	return ((RealSite*)site)->remote_pass;
}

/**
 * screem_site_set_cvs_root:
 * @site: the site
 * @cvs_root: the location of the CVS repository for the site to use
 *
 * Sets / changes the location of the CVS repository that will be used by
 * the site
 *
 * return values: none
 */
void screem_site_set_cvs_root( Site *site, const gchar *cvs_root )
{
	RealSite *s;
	g_return_if_fail( site );

	s = (RealSite*)site;

	/* does the site already have a cvs root set? */
	if( s->cvs_root )
		g_free( s->cvs_root );

	if( cvs_root )
		s->cvs_root = g_strdup( cvs_root );
	else
		s->cvs_root = NULL;
}

const gchar* screem_site_get_cvs_root( Site *site )
{
	g_return_val_if_fail( site != NULL, NULL );

	return ((RealSite*)site)->cvs_root;
}

void screem_site_set_auto_update_ask( Site *site, gboolean value )
{
	g_return_if_fail( site != NULL );

	((RealSite*)site)->auto_update_dont_ask = ! value;
}

gboolean screem_site_get_auto_update_ask( Site *site ) 
{
	g_return_val_if_fail( site != NULL, FALSE );

	return !((RealSite*)site)->auto_update_dont_ask;
}

void screem_site_set_auto_update( Site *site, gboolean value )
{
	g_return_if_fail( site != NULL );

	((RealSite*)site)->auto_update = value;
}

gboolean screem_site_get_auto_update( Site *site ) 
{
	g_return_val_if_fail( site != NULL, FALSE );

	return ((RealSite*)site)->auto_update;
}

/**
 * screem_site_set_template_path:
 * @site: the site
 * @template_path: the location of the site's standard page template
 *
 * Sets / changes the location of the site's standard page template, which
 * will be used when adding new pages to the site.
 *
 * return values: none
 */
void screem_site_set_template_path( Site *site, const gchar *template_path )
{
	RealSite *s;
	g_return_if_fail( site );

	s = (RealSite*)site;

	/* does the site already have a cvs root set? */
	if( s->template_path )
		g_free( s->template_path );

	if( template_path )
		s->template_path = g_strdup( template_path );
	else
		s->template_path = NULL;
}

const gchar* screem_site_get_template_path( Site *site )
{
	g_return_val_if_fail( site != NULL, NULL );

	return ((RealSite*)site)->template_path;
}

/**
 * screem_site_set_http_url:
 * @site: the site
 * @http_url: the url that the pages can be viewed at
 *
 * return values: none
 */
void screem_site_set_http_url( Site *site, const gchar *http_url )
{
	RealSite *s;
	g_return_if_fail( site );
	g_return_if_fail( http_url );

	s = (RealSite*)site;

	/* does the site already have a remote usernmae set? */
	if( s->http_url )
		g_free( s->http_url );

	s->http_url = g_strdup( http_url );
}

const gchar* screem_site_get_http_url( Site *site )
{
	g_return_val_if_fail( site != NULL, NULL );

	return ((RealSite*)site)->http_url;
}

/**
 * screem_site_set_fake_flag:
 * @site: the site
 * @flag: flag value
 *
 * return values: none
 */
void screem_site_set_fake_flag( Site *site, gboolean flag )
{
	RealSite *s;
	g_return_if_fail( site );
 
	s = (RealSite*)site;

	s->fake_site = flag;

	if( flag ) {
		/* the site is a fake, so we want fake pathnames etc */
		screem_site_set_name( site, _("Single Pages") );
	}
}

gboolean screem_site_get_fake_flag( Site *site )
{
	g_return_val_if_fail( site != NULL, TRUE );

	return ((RealSite*)site)->fake_site;
}


/**
 * screem_site_set_current_page:
 * @site: the site
 * @page: the page
 *
 * return values: none
 */
void screem_site_set_current_page( Site *site, Page *page )
{
	g_return_if_fail( site != NULL );

	((RealSite*)site)->current = page;
}

/**
 * screem_site_get_current_page:
 * @site: the site
 *
 * return values: the current page
 */
Page *screem_site_get_current_page( Site *site )
{
	g_return_val_if_fail( site != NULL, NULL );

	return ((RealSite*)site)->current;
}

/**
 * screem_site_set_is_import:
 * @site: the site
 * @val: the status to set
 *
 * return values: none
 */
void screem_site_set_is_import( Site *site, gboolean val )
{
	g_return_if_fail( site != NULL );

	((RealSite*)site)->is_import = val;
}
/**
 * screem_site_get_is_import:
 * @site: the site
 *
 * return values: boolean
 */
gboolean screem_site_get_is_import( Site *site )
{
	g_return_val_if_fail( site != NULL, FALSE );

	return ((RealSite*)site)->is_import;
}

void screem_site_set_open_pages( Site *site, gint num )
{
	g_return_if_fail( site != NULL );

	if( num < -1 )
		num = -1;

	((RealSite*)site)->open_pages = num;
}

gint screem_site_get_open_pages( Site *site )
{
	g_return_val_if_fail( site != NULL, -1 );

	return ((RealSite*)site)->open_pages;
}

Page* screem_site_get_page_from_tab( Site *site, gint tab )
{
	GList *list;
	RealSite *s;
	Page *page;

	g_return_val_if_fail( site != NULL, NULL );

	s = (RealSite*)site;

	page = NULL;
	for( list = s->pages; list; list = list->next ) {
		page = (Page*)list->data;
		if( screem_page_get_tab( page ) == tab )
			break;
		page = NULL;
	}

	return page;
}

/**
 * screem_site_set_passive_ftp:
 * @site: the site
 * @flag: flag value
 *
 * return values: none
 */
void screem_site_set_passive_ftp( Site *site, gboolean flag )
{
	RealSite *s;
	g_return_if_fail( site );
 
	s = (RealSite*)site;

	s->passive_ftp = flag;
}

gboolean screem_site_get_passive_ftp( Site *site )
{
	g_return_val_if_fail( site != NULL, TRUE );

	return ((RealSite*)site)->passive_ftp;
}

/**
 * screem_site_set_no_delete:
 * @site: the site
 * @flag: flag value
 *
 * return values: none
 */
void screem_site_set_no_delete( Site *site, gboolean flag )
{
	RealSite *s;
	g_return_if_fail( site );
 
	s = (RealSite*)site;

	s->no_delete = flag;
}

gboolean screem_site_get_no_delete( Site *site )
{
	g_return_val_if_fail( site != NULL, TRUE );

	return ((RealSite*)site)->no_delete;
}

/**
 * screem_site_set_check_moved:
 * @site: the site
 * @flag: flag value
 *
 * return values: none
 */
void screem_site_set_check_moved( Site *site, gboolean flag )
{
	RealSite *s;
	g_return_if_fail( site );
 
	s = (RealSite*)site;

	s->check_moved = flag;
}

gboolean screem_site_get_check_moved( Site *site )
{
	g_return_val_if_fail( site != NULL, TRUE );

	return ((RealSite*)site)->check_moved;
}

/**
 * screem_site_set_no_overwrite:
 * @site: the site
 * @flag: flag value
 *
 * return values: none
 */
void screem_site_set_no_overwrite( Site *site, gboolean flag )
{
	RealSite *s;
	g_return_if_fail( site );
 
	s = (RealSite*)site;

	s->no_overwrite = flag;
}

gboolean screem_site_get_no_overwrite( Site *site )
{
	g_return_val_if_fail( site != NULL, TRUE );

	return ((RealSite*)site)->no_overwrite;
}

/**
 * screem_site_set_permissions:
 * @site: the site
 * @permissions: file permission handling
 *
 * Sets / changes the method used to handle file permissions during upload
 *
 * return values: none
 */
void screem_site_set_permissions( Site *site, SitePerms permissions )
{
	RealSite *s;

	g_return_if_fail( site );

	s = (RealSite*)site;

	s->permissions = permissions;
}

SitePerms screem_site_get_permissions( Site *site )
{
	g_return_val_if_fail( site != NULL, PERMS_IGNORE );

	return ((RealSite*)site)->permissions;
}

/**
 * screem_site_set_symlinks:
 * @site: the site
 * @symlinks: symbolic link handling
 *
 * Sets / changes the method used to handle symbolic links during upload
 *
 * return values: none
 */
void screem_site_set_symlinks( Site *site, SiteSymlinks symlinks )
{
	RealSite *s;

	g_return_if_fail( site );

	s = (RealSite*)site;

	s->symlinks = symlinks;
}

SiteSymlinks screem_site_get_symlinks( Site *site )
{
	g_return_val_if_fail( site != NULL, SYM_IGNORE );

	return ((RealSite*)site)->symlinks;
}

gboolean screem_site_is_excluded( Site *site, const gchar *path )
{
	RealSite *rs;

	g_return_val_if_fail( site != NULL, FALSE );
	g_return_val_if_fail( path != NULL, FALSE );

	rs = (RealSite*)site;

	return (gboolean) g_list_find_custom( rs->exclude, (gpointer)path,
					      (GCompareFunc)strcmp );
}
void screem_site_add_exclude( Site *site, const gchar *path )
{
	RealSite *rs;

	g_return_if_fail( site != NULL );
	g_return_if_fail( path != NULL );
	
	rs = (RealSite*)site;

	rs->exclude = g_list_append( rs->exclude, g_strdup( path ) );
}
void screem_site_remove_exclude( Site *site, const gchar *path )
{
	GList *list;
	RealSite *rs;

	g_return_if_fail( site != NULL );
	g_return_if_fail( path != NULL );
	
	rs = (RealSite*)site;

	list = g_list_find_custom( rs->exclude, (gpointer)path, 
				   (GCompareFunc)strcmp );

	g_return_if_fail( list != NULL );

	rs->exclude = g_list_remove( rs->exclude, list->data );
}

gboolean screem_site_is_ascii( Site *site, const gchar *path )
{
	RealSite *rs;

	g_return_val_if_fail( site != NULL, FALSE );
	g_return_val_if_fail( path != NULL, FALSE );

	rs = (RealSite*)site;

	return (gboolean) g_list_find_custom( rs->ascii, (gpointer)path,
					      (GCompareFunc)strcmp );
}
void screem_site_add_ascii( Site *site, const gchar *path )
{
	RealSite *rs;

	g_return_if_fail( site != NULL );
	g_return_if_fail( path != NULL );
	
	rs = (RealSite*)site;

	rs->ascii = g_list_append( rs->ascii, g_strdup( path ) );
}
void screem_site_remove_ascii( Site *site, const gchar *path )
{
	GList *list;
	RealSite *rs;

	g_return_if_fail( site != NULL );
	g_return_if_fail( path != NULL );
	
	rs = (RealSite*)site;

	list = g_list_find_custom( rs->ascii, (gpointer)path, 
				   (GCompareFunc)strcmp );

	g_return_if_fail( list != NULL );

	rs->ascii = g_list_remove( rs->ascii, list->data );
}

gboolean screem_site_is_ignored( Site *site, const gchar *path )
{
	RealSite *rs;

	g_return_val_if_fail( site != NULL, FALSE );
	g_return_val_if_fail( path != NULL, FALSE );

	rs = (RealSite*)site;

	return (gboolean) g_list_find_custom( rs->ignore, (gpointer)path,
					      (GCompareFunc)strcmp );
}
void screem_site_add_ignore( Site *site, const gchar *path )
{
	RealSite *rs;

	g_return_if_fail( site != NULL );
	g_return_if_fail( path != NULL );
	
	rs = (RealSite*)site;

	rs->ignore = g_list_append( rs->ignore, g_strdup( path ) );
}
void screem_site_remove_ignore( Site *site, const gchar *path )
{
	GList *list;
	RealSite *rs;

	g_return_if_fail( site != NULL );
	g_return_if_fail( path != NULL );
	
	rs = (RealSite*)site;

	list = g_list_find_custom( rs->ignore, (gpointer)path, 
				   (GCompareFunc)strcmp );

	g_return_if_fail( list != NULL );

	rs->ignore = g_list_remove( rs->ignore, list->data );
}

GList *screem_site_get_excludes( Site *site )
{
	RealSite *rs;

	g_return_val_if_fail( site != NULL, NULL );

	rs = (RealSite*)site;

	return rs->exclude;
}

GList *screem_site_get_ignores( Site *site )
{
	RealSite *rs;

	g_return_val_if_fail( site != NULL, NULL );

	rs = (RealSite*)site;

	return rs->ignore;
}

GList *screem_site_get_asciis( Site *site )
{
	RealSite *rs;

	g_return_val_if_fail( site != NULL, NULL );

	rs = (RealSite*)site;

	return rs->ascii;
}

void screem_site_add_auto_open( Site *site, gchar *file )
{
	RealSite *rs;

	g_return_if_fail( site != NULL );
	g_return_if_fail( file != NULL );
	
	rs = (RealSite*)site;

	rs->auto_open = g_list_append( rs->auto_open, g_strdup( file ) );
}

GList *screem_site_get_auto_open( Site *site )
{
	RealSite *rs;

	g_return_val_if_fail( site != NULL, NULL );

	rs = (RealSite*)site;

	return rs->auto_open;
}

/**
 * screem_site_save:
 * @site: the site
 *
 * Saves all the open pages in the site along with the site's project file
 *
 * return values: none
 */
void screem_site_save( Site *site )
{
	GList *pages;

	g_return_if_fail( site );

	/* first the project file */
	screem_site_write_project_file( site );

	/* now the pages */
	for( pages = screem_site_get_pages( site ); pages; pages=pages->next )
		screem_page_save( (Page*)pages->data );
}

/**
 * screem_site_create:
 * @site: the site
 *
 * Creates the site at site->pathname, return TRUE if successful
 * FALSE otherwise.
 * This will NOT load the site (ie fill in site->pages etc)
 * to do this call screem_site_load() after a successfull screem_site_create()
 *
 * return values: an boolean
 */
gboolean screem_site_create( Site *site )
{
	gchar *temp;
	gchar cwd[ 16384 ];

	struct stat s;
	const gchar *pathname;

	g_return_val_if_fail( site != NULL, FALSE );

	pathname = screem_site_get_pathname( site );

	g_return_val_if_fail( pathname != NULL, FALSE );

	/* get current directory */
	getcwd( cwd, 16384 );

	/* does the pathname exist as a directory? */
	if( stat( pathname, &s ) < 0 ) {
      		if( errno != ENOENT )
			return FALSE;
		if( ! mkdir_recursive( pathname ) )
			return FALSE;
	}

	/* we must ensure that the pathname ends with a G_DIR_SEPARATOR */
	if( pathname[ strlen( pathname ) - 1] != G_DIR_SEPARATOR ) {
		temp = g_strdup_printf( "%s%c", pathname, 
					G_DIR_SEPARATOR );
		screem_site_set_pathname( site, temp );
		g_free( temp );
	}

	/* write the initial project file */
	screem_site_write_project_file( site );

	chdir( screem_site_get_pathname( site ) );

	return TRUE;
}

/**
 * screem_site_load:
 * @site: the site
 *
 * loads the site and project file
 *
 * return values: an boolean
 */
gboolean screem_site_load( Site *site )
{
	gchar *project_file;
	xmlDocPtr doc = NULL;
	DIR *dir;
	struct dirent *entry;
	const gchar *pathname;
	const gchar *mime_type;

	GList *project_files = NULL;
	GList *file;
    
	g_return_val_if_fail( site != NULL, FALSE );

	pathname = screem_site_get_pathname( site );

	g_return_val_if_fail( pathname != NULL, FALSE );

	if( ! g_file_test( pathname, G_FILE_TEST_ISDIR ) )
		return FALSE;

	if( ( dir = opendir( pathname ) ) ) {
		/* build a list of *.screem files */
		while( ( entry = readdir( dir ) ) ) {
#ifndef HAVE_GNOME_VFS
			mime_type = gnome_mime_type( entry->d_name );
#else
			mime_type = gnome_vfs_mime_type_from_name( entry->d_name );
#endif
			if( strcmp( "application/x-screem", mime_type ) )
				continue;

			/* add it to the list */
			project_files = 
				g_list_append( project_files,
					       g_strdup( entry->d_name ) );
		}
		closedir( dir );
	}
	
	/* now check the list.  The priority we give is the following:
	   .project.screem
	   project.screem

	   Perhaps the project file could be named  .<sitename>.screem
	*/
	file = g_list_find_custom( project_files, (gpointer)".project.screem",
				   (GCompareFunc)strcmp );
	if( ! file )
		file = g_list_find_custom( project_files, 
					   (gpointer)"project.screem",
					   (GCompareFunc)strcmp );
	if( file ) {
		project_file = g_strconcat( pathname,
					    file->data, NULL );
		doc = xmlParseFile( project_file );
		g_free( project_file );
	}
	g_list_foreach( project_files, (GFunc)g_free, NULL );

	if( ! doc ) {
		/* we failed to load the project file, we must be importing */
		screem_site_set_is_import( site, TRUE );
	} else {
		screem_site_parse_project_file( site, doc );
		xmlFreeDoc( doc );
	}
	
	chdir( pathname );

	return TRUE;
}

/**
 * screem_site_write_project_file:
 * @site: the site
 *
 * creates and writes a site's project file
 *
 * return values: a boolean
 */
gboolean screem_site_write_project_file( Site *site )
{
	gchar *project_file;
	xmlDocPtr doc;
	xmlNsPtr ns;
	xmlNodePtr tree;
	xmlNodePtr subtree;
	gint retval;
	TodoItem *item;
	GList *list;

	const gchar *pathname;
	const gchar *name;
	const gchar *remote_url;
	UploadMethods remote_method;
	const gchar *remote_user;
	const gchar *remote_pass;
	const gchar *remote_path;
	gboolean passive_ftp;
	gboolean no_delete;
	gboolean check_moved;
	gboolean no_overwrite;

	SitePerms permissions;
	SiteSymlinks symlinks;

	const gchar *http_url;
	const gchar *cvs_root;
	const gchar *template_path;

	Page *page;
	const gchar *page_path;

	g_return_val_if_fail( site != NULL, FALSE );

	/* if we are a fake site pretend we wrote the file with no problems */
	if( screem_site_get_fake_flag( site ) )
		return TRUE;

	pathname = screem_site_get_pathname( site );

	g_return_val_if_fail( pathname != NULL, FALSE );

	name = screem_site_get_name( site );
	remote_url = screem_site_get_remote_url( site );
	remote_method = screem_site_get_remote_method( site );
	remote_user = screem_site_get_remote_user( site );
	remote_pass = screem_site_get_remote_pass( site );
	remote_path = screem_site_get_remote_path( site );
	passive_ftp = screem_site_get_passive_ftp( site );
	no_delete = screem_site_get_no_delete( site );
	check_moved = screem_site_get_check_moved( site );
	no_overwrite = screem_site_get_no_overwrite( site );
	permissions = screem_site_get_permissions( site );
	symlinks = screem_site_get_symlinks( site );
	http_url = screem_site_get_http_url( site );
	cvs_root = screem_site_get_cvs_root( site );
	template_path = screem_site_get_template_path( site );

	doc = xmlNewDoc( XML_DEFAULT_VERSION );
	doc->root = xmlNewDocNode( doc, NULL, "SCREEM_PROJECT", NULL );
	ns = xmlNewNs( doc->root, "http://www.screem.org/", "screem" );
	xmlNewChild( doc->root, ns, "title", name );

	tree = xmlNewChild( doc->root, ns, "remote", remote_url );
	xmlSetProp( tree, "method", upload_strings[ remote_method ] );

	xmlSetProp( tree, "username", remote_user );
	xmlSetProp( tree, "password", remote_pass );
	xmlSetProp( tree, "path", remote_path );
	tree = xmlNewChild( doc->root, ns, "options", NULL);
	xmlSetProp( tree, "passive_ftp", passive_ftp?"TRUE":"FALSE" );
	xmlSetProp( tree, "no_delete", no_delete?"TRUE":"FALSE" );
	xmlSetProp( tree, "check_moved", check_moved?"TRUE":"FALSE" );
	xmlSetProp( tree, "no_overwrite", no_overwrite?"TRUE":"FALSE" );

	xmlSetProp( tree, "permissions",  permission_strings[ permissions ] );

	xmlSetProp( tree, "symlinks",  symlink_strings[ symlinks ] );

	xmlNewChild( doc->root, ns, "http", http_url );

	tree = xmlNewChild( doc->root, ns, "cvs", cvs_root );
	if( cvs_root )
		xmlSetProp( tree, "use", "true" );
	else
		xmlSetProp( tree, "use", "false" );

	if( screem_site_get_auto_update( site ) )
		xmlSetProp( tree, "auto_update", "true" );
	else
		xmlSetProp( tree, "auto_update", "false" );

	if( screem_site_get_auto_update_ask( site ) )
		xmlSetProp( tree, "dont_ask", "true" );
       
	xmlNewChild( doc->root, ns, "template", template_path );

	tree = xmlNewChild( doc->root, ns, "tasks", NULL );
        for( list = screem_site_get_tasks( site ); list; list = list->next ) {
                item = (TodoItem*)list->data;
		subtree = xmlNewChild( tree, ns, "task", NULL );
		xmlSetProp( subtree, "name", item->task );
		xmlSetProp( subtree, "assigned", item->assigned );
		xmlSetProp( subtree, "priority", item->priority );
		xmlSetProp( subtree, "linkedTo", item->linked_to );
		xmlSetProp( subtree, "description", item->description);
	}

	/* write exclude list */
	tree = xmlNewChild( doc->root, ns, "excludes", NULL );
	for( list = screem_site_get_excludes( site ); list; list = list->next )
		subtree = xmlNewChild(tree, ns, "exclude", (gchar*)list->data);
	/* write ignore list */
	tree = xmlNewChild( doc->root, ns, "ignores", NULL );
	for( list = screem_site_get_ignores( site ); list; list = list->next )
		subtree = xmlNewChild(tree, ns, "ignore", (gchar*)list->data);
	/* write ascii list */
	tree = xmlNewChild( doc->root, ns, "asciis", NULL );
	for( list = screem_site_get_asciis( site ); list; list = list->next )
		subtree = xmlNewChild(tree, ns, "ascii", (gchar*)list->data);

	/* write a list of open files */
	tree = xmlNewChild( doc->root, ns, "open_files", NULL );
	for( list = screem_site_get_pages(site);list;list = list->next ) {
		page = (Page*)list->data;
		page_path = screem_page_get_pathname( page );
		if( screem_page_get_tab( page ) != -1 )
			subtree = xmlNewChild( tree, ns, "file", page_path );
	}

	if( FALSE && name && strlen( name ) )
		project_file = g_strconcat( pathname, name, ".screem", NULL );
	else
		project_file = g_strconcat( pathname,".project.screem", NULL );


	retval = xmlSaveFile( project_file, doc );
	chmod( project_file, S_IRUSR | S_IWUSR );
	
	xmlFreeDoc( doc );
	g_free( project_file );

	return (retval < 0);
}

/**
 * screem_site_parse_project_file:
 * @site: the site
 * @doc: the xml tree of the project file
 *
 * parses the given sites project file
 *
 * return values: an boolean
 */
gboolean screem_site_parse_project_file( Site *site, xmlDocPtr doc )
{
	gchar *string;
	xmlNodePtr node;
	TodoItem *item;

	g_return_val_if_fail( site, FALSE );
	g_return_val_if_fail( doc, FALSE );

 	/* get the site name */
	string = xml_get_value( doc->root, "title" );
	if( string )
		screem_site_set_name( site, string );
	g_free( string );

	/* get the remote url */
	node = xml_search_child( doc->root, "remote" );
	string = xml_get_value( doc->root, "remote" );
	if( string )
		screem_site_set_remote_url( site, string );
	g_free( string );

	if( node ) {
		string = xml_get_value( node, "method" );
		if( string ) {
			if( ! strcmp( "ftp", string ) )
				screem_site_set_remote_method( site, FTP );
			else if( ! strcmp( "webdav", string ) )
				screem_site_set_remote_method( site, WEBDAV );
			else if( ! strcmp( "rsh", string ) )
				screem_site_set_remote_method( site, RSH );
			else if( ! strcmp( "ssh", string ) ) {
				screem_site_set_remote_method( site, SSH );
			} else
				screem_site_set_remote_method( site, LOCAL );
		}
		g_free( string );
      		string = xml_get_value( node, "path" );
		if( string )
			screem_site_set_remote_path( site, string );
		g_free( string );
		string = xml_get_value( node, "username" );
		if( string )
			screem_site_set_remote_user( site, string );
		g_free( string );
		string = xml_get_value( node, "password" );
		if( string )
			screem_site_set_remote_pass( site, string );
		g_free( string );
	}

	/* get the transfer options */
	node = xml_search_child( doc->root, "options" );
	if( node ) {
		string = xml_get_value( node, "passive_ftp" );
		if( string ) {
			if( ! g_strcasecmp( string , "TRUE" ) )
				screem_site_set_passive_ftp( site, TRUE );
			if( ! g_strcasecmp( string , "FALSE" ) )
				screem_site_set_passive_ftp( site, FALSE );
		}
		g_free( string );
		string = xml_get_value( node, "no_delete" );
		if( string ) {
			if( ! g_strcasecmp( string , "TRUE" ) )
				screem_site_set_no_delete( site, TRUE );
			if( ! g_strcasecmp( string , "FALSE" ) )
				screem_site_set_no_delete( site, FALSE );
		}
		g_free( string );
		string = xml_get_value( node, "check_moved" );
		if( string ) {
			if( ! g_strcasecmp( string , "TRUE" ) )
				screem_site_set_check_moved( site, TRUE );
			if( ! g_strcasecmp( string , "FALSE" ) )
				screem_site_set_check_moved( site, FALSE );
		}
		g_free( string );
		string = xml_get_value( node, "no_overwrite" );
		if( string ) {
			if( ! g_strcasecmp( string , "TRUE" ) )
				screem_site_set_no_overwrite( site, TRUE );
			if( ! g_strcasecmp( string , "FALSE" ) )
				screem_site_set_no_overwrite( site, FALSE );
		}
		g_free( string );
		string = xml_get_value( node, "permissions" );
		if( string ) {
			if( ! strcmp( permission_strings[ PERMS_ALL ], 
				      string ) )
				screem_site_set_permissions( site, PERMS_ALL );
			else if( ! strcmp( permission_strings[ PERMS_IGNORE ],
					   string ) )
				screem_site_set_permissions(site,PERMS_IGNORE);
			else 
				screem_site_set_permissions( site,PERMS_EXEC );
		} else
			screem_site_set_permissions( site, PERMS_EXEC );
		g_free( string );

		string = xml_get_value( node, "symlinks" );
		if( string ) {
			if( ! strcmp( symlink_strings[ SYM_FOLLOW ],
				      string ) )
				screem_site_set_symlinks( site, SYM_FOLLOW );
			else if( ! strcmp( symlink_strings[ SYM_MAINTAIN ],
					   string ) )
				screem_site_set_symlinks( site, SYM_MAINTAIN );
			else
				screem_site_set_symlinks( site, SYM_IGNORE );
		} else
			screem_site_set_symlinks( site, SYM_IGNORE );
		g_free( string );
	}

	/* get http url */
   	string = xml_get_value( doc->root, "http" );
	if( string )
		screem_site_set_http_url( site, string );
	g_free( string );

	/* get the cvs repository path */
	node = xml_search_child( doc->root, "cvs" );
	string = xml_get_value( node, "use" );
	if( strcmp( "use", string ) ) {
		g_free( string );
		string = xml_get_value( doc->root, "cvs" );
		if( string )
			screem_site_set_cvs_root( site, string );
	}
	g_free( string );
	string = xml_get_value( node, "auto_update" );
	if( string )
		screem_site_set_auto_update( site, !strcmp( "true", string ) );
	g_free( string );

	string = xml_get_value( node, "dont_ask" );
	if( string )
		screem_site_set_auto_update_ask( site, FALSE );
	g_free( string );

	/* get the template path */
	string = xml_get_value( doc->root, "template" );
	if( string )
		screem_site_set_template_path( site, string );
	g_free( string );

	/* get the tasks */
	node = xml_search_child( doc->root, "tasks" );
	if( node ) {
		node = node->childs;
                while( node ) {
			item = screem_todo_item_new();
                        item->task = xml_get_value( node, "name" );
                        item->assigned = xml_get_value( node, "assigned" );
                        item->priority = xml_get_value( node, "priority" );
                        item->linked_to = xml_get_value( node, "linkedTo" );
                        item->description = xml_get_value( node, 
							   "description" );
                        screem_site_add_task( site, item );
                        node = node->next;
		}
	}

	/* get the excludes */
	node = xml_search_child( doc->root, "excludes" );
	if( node ) {
		node = node->childs;
		while( node ) {
			string = xmlNodeGetContent( node );
			screem_site_add_exclude( site, string );
			g_free( string );
			node = node->next;
		}
	}
	/* get the ignores */
	node = xml_search_child( doc->root, "ignores" );
	if( node ) {
		node = node->childs;
		while( node ) {
			string = xmlNodeGetContent( node );
			screem_site_add_ignore( site, string );
			g_free( string );
			node = node->next;
		}
	}
	/* get the asciis */
	node = xml_search_child( doc->root, "asciis" );
	if( node ) {
		node = node->childs;
		while( node ) {
			string = xmlNodeGetContent( node );
			screem_site_add_ascii( site, string ); 
			g_free( string );
 			node = node->next;
		}
	}

	/* get auto open */
	node = xml_search_child( doc->root, "open_files" );
	if( node ) {
		node = node->childs;
		while( node ) {
			string = xmlNodeGetContent( node );
			screem_site_add_auto_open( site, string );
			g_free( string );
			node = node->next;
		}
	}

	return TRUE;
}


/**
 * screem_site_locate_page:
 * @site: the site
 * @path: the pages pathname
 *
 * search the site for a page with the given pathname and return the page
 * if found
 *
 * return values: a Page
 */
Page* screem_site_locate_page( Site *site, const gchar *path )
{
	GList *list;
	Page *page = NULL;
	const gchar *pathname;

	g_return_val_if_fail( site, NULL );
	g_return_val_if_fail( path, NULL );

	for( list = screem_site_get_pages(site ); list && ( ! page );
	     list = list->next ) {
		page = (Page*)list->data;
		pathname = screem_page_get_pathname( page );
		if( ! pathname || (pathname && strcmp(path, pathname)) )
			page = NULL;
	}

	return page;
}

/**
 * screem_site_add_page:
 * @site: the site
 * @path: the pages pathname
 *
 * Adds a new page to the site
 *
 * return values: none
 */
void screem_site_add_page( Site *site, const gchar *path )
{
	Page *page;
	GList *list;
	struct stat s;

	g_return_if_fail( site != NULL );
	g_return_if_fail( path != NULL );

	list = screem_site_get_pages( site );

	page = screem_page_new();
	screem_page_set_pathname( page, path );

	/* the page should exist on the filesystem, so we can stat
	   it and set the mod time correctly, if we can't stat the
	   file set it to the current time, we have to do something and this
	   is as good as anything else */
	if( ! stat( path, &s ) )
		screem_page_set_mod_time( page, s.st_mtime );
	else
		screem_page_set_mod_time( page, time( NULL ) );

	list = g_list_append( list, page );
	screem_site_set_pages( site, list );
}

static void screem_site_set_pages( Site *site, GList *list )
{
	g_return_if_fail( site != NULL );

	((RealSite*)site)->pages = list;
}

/**
 * screem_site_remove_page:
 * @site: the site
 * @path: the pages pathname
 *
 * Removes the page with the given pathname
 *
 * return values: a Page
 */
Page* screem_site_remove_page( Site *site, const gchar *path )
{
	Page *page;
	GList *list;
	gint len;
	
	g_return_val_if_fail( site != NULL, NULL );
	g_return_val_if_fail( path != NULL, NULL );

	list = screem_site_get_pages( site );

	len = g_list_length( list );

	page = screem_site_locate_page( site, path );

	if( page ) {
		list = g_list_remove( list, page );
		screem_site_set_pages( site, list );
	}
	return page;
}

/**
 * screem_site_get_pages:
 * @site: the site
 *
 * return values: a list of all the pages in the site
 */
GList* screem_site_get_pages( Site *site )
{
	return ((RealSite*)site)->pages;
}

/**
 * screem_site_locate_task:
 * @site: the site
 * @task: the name of the task to locate
 *
 * finds the task with the given name.
 *
 * return values: a Todo_item pointer
 */
TodoItem* screem_site_locate_task( Site *site, const gchar *task )
{
	GList *list;
	TodoItem *item = NULL;

	g_return_val_if_fail( site, NULL );
	g_return_val_if_fail( task, NULL );

	for( list = screem_site_get_tasks( site ); ( list ) && ( ! item ) ; 
	     list = list->next ) {
		item = (TodoItem*)list->data;
		if( strcmp( task, item->task ) )
			item = NULL;
       	}

	return item;
}

/**
 * screem_site_add_task:
 * @site: the site
 * @item: the item to add
 *
 * adds the given todo item to the sites list of tasks
 *
 * return values: none
 */
void screem_site_add_task( Site *site, TodoItem *item )
{
	GList *list;

	g_return_if_fail( site != NULL );
	g_return_if_fail( item != NULL );

	list = screem_site_get_tasks( site );

	list = g_list_append( list, item );

	screem_site_set_tasks( site, list );
}

static void screem_site_set_tasks( Site *site, GList *list )
{
	g_return_if_fail( site != NULL );

	((RealSite*)site)->tasks = list;
}


/**
 * screem_site_remove_task:
 * @site: the site
 * @task: the name of the task to remove
 *
 * finds the task with the given name and removes it
 *
 * return values: a Todo_item pointer
 */
TodoItem* screem_site_remove_task( Site *site, const gchar *task )
{
	TodoItem *item;
	GList *list;

	g_return_val_if_fail( site, NULL );
	g_return_val_if_fail( task, NULL );

	list = screem_site_get_tasks( site );

	item = screem_site_locate_task( site, task );
	
	if( item ) {
		list = g_list_remove( list, item );
		screem_site_set_tasks( site, list );
	}

	return item;
}

/**
 * screem_site_get_tasks:
 * @site: the site
 *
 * return values: a list of all the tasks in the site
 */
GList* screem_site_get_tasks( Site *site )
{
	return ((RealSite*)site)->tasks;
}

/**
 * screem_site_file_change:
 * @site: the site
 * @src:  the original file
 * @dest: the files new location
 *
 * scans all pages in the site that have links to src and changes them
 * to link to dest
 *
 * return values: none
 */
void screem_site_file_change( Site *site, gchar *src, gchar *dest )
{
	GList *list;

	gchar cwd[ 16384 ];

	Page *page;
	const gchar *page_path;

	gchar *data;

	/* if we are a fake site we do not want to do this */
	if( screem_site_get_fake_flag( site ) )
		return;

	getcwd( cwd, 16384 );

	/* go through all pages in the site and update any links */
	for( list = screem_site_get_pages( site ); list; list = list->next ) {
		page = (Page*)list->data;

		screem_page_load( page );

		page_path = screem_page_get_pathname( page );

		data = screem_html_fix_links( screem_page_get_data( page ),
					      page_path,
					      screem_site_get_pathname( site ),
					      src, dest );
		screem_page_set_data( page, data );
		g_free( data );
	}

	chdir( cwd );
}
