/*  Screem: fileops.c,
 *  This file provides file/directory copying/moving/deletion
 * 
 *  Copyright (C) 1999, 2000 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>
#include <dirent.h>
#include <errno.h>
#include <gnome.h>
#include <time.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>

#include <glade/glade.h>

#include "fileops.h"

gboolean copy_file( const gchar *source, const gchar *dest )
{
	char buffer[ BUFSIZ + 1 ];
	FILE *r;
	FILE *w;
	gint size;

	if( ! overwrite( dest ) )
		return FALSE;

  	if( ! ( r = fopen( source, "r" ) ) )
		return FALSE;
	
	if( ! ( w = fopen( dest, "w" ) ) ) {
		fclose( r );
		return FALSE;
	}

	while( ( size = fread( buffer, 1, BUFSIZ, r ) ) )	{
		if( fwrite( buffer, 1, size, w ) != size ) {
			fclose( r );
			fclose( w );
			return FALSE;
		}
	}

	fclose( r );
	fclose( w );
	return TRUE;
}
/*****************************************************************************/
gboolean copy_dir( const gchar *source, const gchar *dest, gboolean move )
{
	struct stat s;
	DIR *dir;
        struct dirent *entry = NULL;
	gchar *orig;
	gchar *new_path;

	gboolean made;

	made = FALSE;

	if( stat( dest, &s ) < 0 ) {
		if( errno != ENOENT )
			return FALSE;
		if( ! mkdir_recursive( dest ) )
			return FALSE;
		made = TRUE;
	}

	if( ! made && ! S_ISDIR(s.st_mode) )
		return FALSE;

	if( ! ( dir = opendir( source ) ) )
		return FALSE;

	while( ( entry = readdir( dir ) ) ) {
		if( strcmp( "..", entry->d_name ) &&
		    strcmp( ".", entry->d_name ) ) {
			orig = g_strdup_printf("%s%c%s", source,
					       G_DIR_SEPARATOR, entry->d_name);
			new_path = g_strdup_printf( "%s%c%s", dest, 
						    G_DIR_SEPARATOR,
						    entry->d_name );
			if( ! g_file_test( orig, G_FILE_TEST_ISDIR ) ) {
				if( move )
					move_file( orig, new_path );
				else
					copy_file( orig ,new_path );
			} else
				copy_dir( orig, new_path, move );
		
			g_free( orig );
			g_free( new_path );
		}
	}

	closedir( dir );

	if( move )
		rmdir( source );

	return TRUE;
}
/*****************************************************************************/
gboolean move_file( const gchar *source, const gchar *dest )
{
	if( ! overwrite( dest ) )
		return FALSE;

	if( rename( source, dest ) < 0 ) {
		if( copy_file( source, dest ) ) {
			return ( unlink( source ) < 0 );
		} else
			return FALSE;
	}

	return TRUE;
}
/*****************************************************************************/
gboolean delete_file( const gchar *file )
{
	if( unlink( file ) < 0 )
		return FALSE;

	return TRUE;
}
/*****************************************************************************/
gboolean delete_dir( const gchar *path )
{
	DIR *dir;
        struct dirent *entry = NULL;
	gchar *new_path;

	dir = opendir( path );
	if( ! dir )
		return FALSE;

	while( ( entry = readdir( dir ) ) ) {
		if( strcmp( "..", entry->d_name ) &&
		    strcmp( ".", entry->d_name ) ) {
			new_path = g_strdup_printf("%s%c%s", path,
						   G_DIR_SEPARATOR, 
						   entry->d_name);
			if( ! g_file_test( new_path, G_FILE_TEST_ISDIR ) ) {
				delete_file( new_path );
			} else
				delete_dir( new_path );
			g_free( new_path );
		}
	}

	closedir( dir );
	rmdir( path );

	return TRUE;
}
/*****************************************************************************/
/* recusivley makes directories (if needed), starting at current path */
gboolean mkdir_recursive( const gchar *path )
{
	gchar **dir = NULL;
	gint pos;

	gchar cwd[ 16384 ];

	dir = g_strsplit( path, G_DIR_SEPARATOR_S, 255 );

	getcwd( cwd, 16384 );

	if( g_path_is_absolute( path ) )
		chdir( G_DIR_SEPARATOR_S );

    	for( pos = 0; dir[ pos ]; pos ++ ) {
		if( strlen( dir[ pos ] ) ) {
			if( chdir( dir[ pos ] ) < 0 ) {
				if( errno != ENOENT ) {
					perror( "chdir" );
					g_strfreev( dir );
					chdir( cwd );
					return FALSE;
				} else if( mkdir( dir[ pos ], 0x1e4 ) < 0 ) {
					perror( "mkdir" );
					g_strfreev( dir );
					chdir( cwd );
					return FALSE;
				}
				pos --;  /* decrease pos so that we try to
					    enter the directory we just made
					    on the next loop */
			}
		}
	}

	g_strfreev( dir );
	chdir( cwd );

	return TRUE;
}
/*****************************************************************************/
/* converts the given path into one relative to the passed root, or
   the current directory if no root is given */
gchar *relative_path( const gchar *text, const gchar *root )
{
      	gchar *tmp;
	gchar *temp;
	gchar *temp2;
	gchar **current;
	gchar **dropped;
	gboolean set = FALSE;
	gint count = 0;
	gint prev = 0;

	gchar *ret;

	const gchar *pathname;
	gchar cwd[ 16384 ];
    
	getcwd( cwd, 16384 );

	tmp = g_strdup( cwd );

	if( root )
		pathname = root;
	else
		pathname = tmp;


	if( ! strncmp( text, pathname, strlen( pathname ) ) ) {
		current = g_strsplit( tmp, G_DIR_SEPARATOR_S, 255 );
		dropped = g_strsplit( text, G_DIR_SEPARATOR_S, 255 );

		while( dropped[ count ] && current[ count ] && ! set ){
			if( ! strcmp( dropped[ count ], current[ count ] ) )
				count ++;
			else    /* didn't match at position */
				set = TRUE;
		}

		for( prev = count; current[ prev ]; prev ++ )
			current[ prev ] = g_strdup( ".." );
		
		if( current[ count ] )
			temp = g_strjoinv( "/", &current[ count ] );
		else
			temp = g_strdup( "." );

		temp2 = g_strjoinv("/", &dropped[ count ] );
		ret = g_strdup_printf( "%s/%s", temp, temp2 );
		g_free( temp );
		g_free( temp2 );
		g_strfreev( dropped );
		g_strfreev( current );
	} else
		ret = g_strdup( text );

	g_free( tmp );

	return ret;
}
/***************************************************************************/
/* converts the given relative path to a full pathname,
   we should be in the path that it is relative from before calling the
   function */
gchar *relative_to_full( const gchar *relPath )
{
	gchar **rel;
	gint count;
	gchar cwd[ 16384 ];
	gchar scwd[ 16384 ];
	gchar *path = NULL;

	rel = g_strsplit( relPath, G_DIR_SEPARATOR_S, 255 );

	getcwd( scwd, 16384 );

	for( count = 0; rel[ count ]; count ++ ) {
		if( g_file_test( rel[ count ], G_FILE_TEST_ISDIR ) )
			chdir( rel[ count ] );
		else
			path = rel[ count ];
	}

	getcwd( cwd, 16384 );
	path = g_strjoin( G_DIR_SEPARATOR_S, cwd, path, NULL );

	g_strfreev( rel );

	chdir( scwd );

	return path;
}


gboolean overwrite( const gchar *filename )
{
	static GtkWidget *prompt;
	static gint button;
	gchar *message;

	struct stat s;

	/* does dest already exist? */
	if( ! stat( filename, &s ) ) {
		message = g_strconcat( filename,
				       _(": already exists\noverwrite?"),
				       NULL );
		prompt = gnome_message_box_new( message,
						GNOME_MESSAGE_BOX_WARNING,
						GNOME_STOCK_BUTTON_YES,
						GNOME_STOCK_BUTTON_NO,
						NULL );
		g_free( message );
		button = gnome_dialog_run_and_close( GNOME_DIALOG( prompt ) );
			
		if( button == 1 )
			return FALSE;

		/* if button == 0 then it was yes so we just continue */
	} else if( errno != ENOENT ) {
		/* FIXME: report error reason */
		return FALSE;
	}

	return TRUE;
}

/* does path1 match path2, or is it a subdir going from current directory,
   path2 must be absolute */
gchar* paths_match( const gchar *path1, const gchar *path2 )
{
	gchar cwd[ 16384 ];
	gchar *abs_path1;
	
	getcwd( cwd, 16384 );

	abs_path1 = relative_to_full( path1 );

	chdir( cwd );

	if( ! strncmp( abs_path1, path2, strlen( path2 ) ) )
		return abs_path1;

	g_free( abs_path1 );
	return NULL;
}
