/*  Screem:  siteUI.c,
 *  a rewrite of Site.c to make it neater and overall better code,
 *  this file contains GUI code that deals with the site
 * 
 *  Copyright (C) 1999, 2000  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>
#include <gnome.h>
#include <sys/stat.h>
#include <unistd.h>

#include <gtk/gtkinvisible.h>

#include <glade/glade.h>

#include "editor.h"
#include "fileops.h"
#include "page.h"
#include "pageUI.h"
#include "preferences.h"
#include "preview.h"
#include "site.h"
#include "siteTree.h"
#include "siteTreeUI.h"
#include "siteUI.h"
#include "support.h"
#include "xml.h"

extern Site *current_site;
extern GtkWidget *app;
extern Preferences *cfg;

extern GList *loaded_sites;

GladeXML *xml;
static GtkWidget *dialog1 = NULL;

static gboolean ignore_change = FALSE;

typedef struct Stats {
	gchar *mime_type;
	gint number;
	gint total_size;
} Stats;

void screem_site_settings_dialog_destroy_cb( GtkWidget *widget, 
					     GdkEvent *event );
void screem_site_settings_dialog_cb( GtkWidget *widget, gint button );
void screem_site_settings_change_url( GtkWidget *widget, gpointer data );


static void recent_proxy( GtkWidget *widget, gchar *filename );

static void screem_site_gather_stats( GladeXML *xml );
static GList* calc_stats( xmlNodePtr node, GList *types );


/**
 * screem_site_save_confirmation:
 * @site:  the site which we want confirmation for
 *
 * inquires if the site is to be saved or not
 *
 * return values: none
 */
void screem_site_save_confirmation( Site *site )
{
	GtkWidget *box;
	gint button;

	box = gnome_message_box_new( _("Save current site?"),
				     GNOME_MESSAGE_BOX_QUESTION,
				     GNOME_STOCK_BUTTON_YES,
				     GNOME_STOCK_BUTTON_NO, NULL );
	gnome_dialog_set_default( GNOME_DIALOG( box ), 0 );

	button = gnome_dialog_run_and_close( GNOME_DIALOG( box ) );

	if( button == 0 )
		screem_site_save( site );
}

/**
 * screem_site_close_site:
 *
 * closes the site
 *
 * return values: none
 */
void screem_site_close_site()
{
	Site *site;
	GtkWidget *tree;
	GtkWidget *hackish;
	GList *list;
	Page *page;
	gint num;
	GtkWidget *pagebook = gtk_object_get_data( GTK_OBJECT( app ),
						   "pagebook" );

	site = current_site;

	g_return_if_fail( site != NULL );

	/* if this is the fake site then it can't be closed */
	if( screem_site_get_fake_flag( site ) )
		return;

	screem_site_save_confirmation( site );

	/* remove any open page tabs */
	list = screem_site_get_pages( site );
	for( ; list; list = list->next ) {
		page = list->data;
		if( ( num = screem_page_get_tab( page ) ) != -1 ) {
			/* we have an open page */
			screem_page_close_page( page );
		}
	}

	/* remove from the list of loaded sites */
	loaded_sites = g_list_remove( loaded_sites, site );

	tree = gtk_object_get_data( GTK_OBJECT( app ), "file_tree" );
	gtk_clist_clear( GTK_CLIST( tree ) );
     
	/* update the site combo box */
	screem_build_sites_combo_list();

	screem_editor_clear();
	screem_editor_disable();
	screem_preview_display( "" );

	screem_site_destroy( site );

	hackish = gtk_object_get_data( GTK_OBJECT( app ), "sitecombo" );
	screem_site_change( GTK_COMBO( hackish )->entry, NULL );
}

/**
 * screem_site_close_all:
 *
 * closes all the loaded sites
 *
 * return values: none
 */
void screem_site_close_all()
{
	GList *list;

	while( ( list = loaded_sites ) ) {
		if( ! ( current_site = list->data ) )
			break;
		screem_site_close_site();
	}
}

/**
 * screem_site_create_site:
 *
 * creates a new site
 *
 * return values: none
 */
void screem_site_create_site()
{
	Site *site;

	site = screem_site_new();
	screem_site_druid( site );
}

/**
 * screem_site_open:
 *
 * displays a file selecter to choose which site to open
 *
 * return values: none
 */
void screem_site_open()
{
	gchar *filename;

	filename = file_select( _("Select directory to open as a site") );

	/* we didn't select one */
	if( ! filename )
		return;

	screem_site_open_with_filename( filename );
}

/**
 * screem_site_open_with_filename:
 * @filename:  the path name of the site
 *
 * opens the site with the given filename
 *
 * return values: gboolean
 */
gboolean screem_site_open_with_filename( const gchar *filename )
{
	Site *site;
	gchar *path = NULL;
	const gchar *page_path;
	gchar *temp;
	gchar cwd[ 16384 ];

	GList *list;
	Page *page;

	GladeXML *xml;
	GtkWidget *widget;
	gint button;
	GtkWidget *toggle;
	gboolean ask;

	GtkWidget *combo = gtk_object_get_data(GTK_OBJECT( app ), "sitecombo");

	g_return_val_if_fail( filename != NULL, FALSE );

	/* is the filename absolute? */
	if( filename[ 0 ] != G_DIR_SEPARATOR ) {
		/* no */
		getcwd( cwd, 16384 );
		path = g_strconcat( cwd, G_DIR_SEPARATOR_S, filename, NULL );
	}

	if( ! g_file_test( filename, G_FILE_TEST_ISDIR ) ) {
		if( ! path )
			path = g_dirname( filename );
		else {
			temp = g_dirname( path );
			g_free( path );
			path = temp;
		}
	} else if( ! path ) {
		path = g_strdup( filename );
	}

	site = screem_site_new();
	screem_site_set_pathname( site, path );
	screem_site_set_name( site, "imported_site" );

	g_free( path );

	if( ! screem_site_load( site ) ) {
		/* we failed to load the site */
		screem_site_destroy( site );
		return FALSE;
	}

	/* add this site to the list of loaded sites */
	loaded_sites = g_list_append( loaded_sites, site );
     	/* build the combo options */
	screem_build_sites_combo_list();
	/* set the combo entry to be the name of the current site */
	gtk_entry_set_text( GTK_ENTRY( GTK_COMBO( combo )->entry ),
			    screem_site_get_name( site ) );

	/* add to recent sites list */
	page_path = screem_site_get_pathname( site );
	cfg->recent_sites = add_recent( cfg->recent_sites, page_path );
	build_recent_sites_menu();

	current_site = site;

	/* site loaded, check for cvs update,
	   only do this if the site auto update flag is false,
	   and the user hasn't disabled displaying of the prompting dialog */
	if( screem_site_get_cvs_root( site ) && 
	    ( ! screem_site_get_auto_update( site ) ) &&  
	    screem_site_get_auto_update_ask( site ) ) {
		xml = glade_xml_new( cfg->glade_path, "auto_cvs_update" );
		widget = glade_xml_get_widget( xml, "auto_cvs_update" );
		toggle = glade_xml_get_widget( xml, "dont_ask" );
		button = gnome_dialog_run( GNOME_DIALOG( widget ) );
		ask = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(toggle));
		screem_site_set_auto_update_ask( site, ask );
		screem_site_set_auto_update( site, button == 0 );
		gtk_widget_destroy( widget );
	}

	if( screem_site_get_auto_update( site ) ) {
		/* do a cvs update */
		cvs_update_site();
	}

	/* create the site file tree */
	refresh_file_tree();
	
	/* if we are importing a site open the site settings dialog */
	if( screem_site_get_is_import( site ) ) {
		screem_site_settings_dialog();
		screem_site_set_is_import( site, FALSE );
	}


	/* do we auto load pages? */
	if( cfg->mpage->auto_open ) {
		/* yes we do */
		list = screem_site_get_auto_open( site );
		while( list ) {
			page_path = (gchar*)list->data;
			page = screem_site_locate_page( site, page_path );
			if( page )
				screem_page_insert( page );
			list = list->next;
		}
	}

	return TRUE;
}

/**
 * screem_site_save_as:
 *
 * save the current site in a directory other than site->pathname
 *
 * return values: none
 */
void screem_site_save_as()
{
	GList *list;
	gchar *filename;
	gchar *new;
	Page *page;
	Site *site;

	const gchar *pathname;
	const gchar *src;

	site = current_site;

	g_return_if_fail( site != NULL );

	/* you can't save the fake site as anything */
	if( screem_site_get_fake_flag( site ) )
		return;

	filename = file_select( _("Select directory to save site as") );

	/* directory was not selected */
	if( ! filename )
		return;

	src = screem_site_get_pathname( site );

	if( ! copy_dir( src, filename, FALSE ) ) {
		/* failed to copy the site to the new location */

	}

	/* modify the pathnames of all the pages */
	for( list = screem_site_get_pages( site ); list; list = list->next ) {
		page = (Page*)list->data;
		pathname = screem_page_get_pathname( page );
		new = g_strdup_printf( "%s%s", filename, 
				       pathname + strlen( src ) );
		screem_page_set_pathname( page, new );
		g_free( new );
	}

	/* save the site */
	screem_site_set_pathname( site, filename );
	screem_site_save( site );

	refresh_file_tree();
}

/**
 * screem_site_save_proxy
 *
 * callback for toolbar/menu items to save the current site
 *
 * return values: none
 */
void screem_site_save_proxy()
{
	g_return_if_fail( current_site != NULL );

	if( screem_site_get_current_page( current_site ) )
		screem_editor_buffer_text();
	screem_site_save( current_site );
}

/**
 * screem_site_settings_dialog:
 * opens the site settings dialog
 *
 * return values: none
 */
void screem_site_settings_dialog()
{
     	GtkWidget *entry;
	GtkWidget *toggle;
	Site *site;
	UploadMethods method;
	SitePerms permissions;
	SiteSymlinks symlinks;
	const gchar *text;

	site = current_site;

	g_return_if_fail( site );

	/* can't alter settings for the fake site */
	if( screem_site_get_fake_flag( site ) )
		return;
	
	if( dialog1 ) {
		gdk_window_raise( dialog1->window );
                gdk_window_show( dialog1->window );
                return;
	}

	xml = glade_xml_new( cfg->glade_path, "site_settings_dialog" );

	glade_xml_signal_autoconnect( xml );

	screem_site_gather_stats( xml );

	dialog1 = glade_xml_get_widget( xml, "site_settings_dialog" );

	/* fill in the details */
	/* name */
	entry = glade_xml_get_widget( xml, "site_name_entry" );
	text = screem_site_get_name( site );
	if( ! text )
		text = "";
	gtk_entry_set_text( GTK_ENTRY( entry ), text );
	
	/* path */
	entry = glade_xml_get_widget( xml, "site_path_entry" );
	text = screem_site_get_pathname( site );
	if( ! text )
		text = "";
	gtk_entry_set_text( GTK_ENTRY( entry ), text );

	/* cvs root */
	entry = glade_xml_get_widget( xml, "cvs_rep_path" );
	if( ( text = screem_site_get_cvs_root( site ) ) ) {
		toggle = glade_xml_get_widget( xml, "cvs_check" );
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON( toggle ),
					     TRUE );
		gtk_entry_set_text( GTK_ENTRY( entry ), text );
	}

	/* auto cvs update */
	toggle = glade_xml_get_widget( xml, "auto_cvs_update" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( toggle ),
				      screem_site_get_auto_update( site ) );

	/* template path */
	entry = glade_xml_get_widget( xml, "template_entry" );
	if( ( text = screem_site_get_template_path( site ) ) ) {
		toggle = glade_xml_get_widget( xml, "template_check" );
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON( toggle ),
					     TRUE );
		gtk_entry_set_text(GTK_ENTRY( entry ), text );
	}

	/* remote url */
	entry = glade_xml_get_widget( xml, "remote_url_entry" );
	text = screem_site_get_remote_url( site );
	if( ! text )
		text = "";
       	gtk_entry_set_text( GTK_ENTRY( entry ), text );

	/* http url */
	entry = glade_xml_get_widget( xml, "http_url_entry" );
	text = screem_site_get_http_url( site );
	if( ! text )
		text = "";
       	gtk_entry_set_text( GTK_ENTRY( entry ), text );

	/* remote method */
	method = screem_site_get_remote_method( site );
	g_print( "Method: %i\n", method );
	toggle = glade_xml_get_widget( xml, upload_strings[ method ] );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( toggle ), TRUE );

	/* passive ftp */
	if( screem_site_get_passive_ftp( site ) ) {
		toggle = glade_xml_get_widget( xml, "ftpmode" );
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( toggle ), TRUE );
	}

	/* no file delete */
	if( screem_site_get_no_delete( site ) ) {
		toggle = glade_xml_get_widget( xml, "nodelete" );
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( toggle ), TRUE );
	}

	/* check for moved files */
	if( screem_site_get_check_moved( site ) ) {
		toggle = glade_xml_get_widget( xml, "checkmoved" );
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( toggle ), TRUE );
	}

	/* no file overwrite */
	if( screem_site_get_no_overwrite( site ) ) {
		toggle = glade_xml_get_widget( xml, "nooverwrite" );
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( toggle ), TRUE );
	}

	/* permissions handling */
	permissions = screem_site_get_permissions( site );
       	toggle = glade_xml_get_widget( xml, permission_strings[permissions] );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( toggle ), TRUE );

	/* symbolic link handling */
	symlinks = screem_site_get_symlinks( site );
      	toggle = glade_xml_get_widget( xml, symlink_strings[ symlinks ] );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( toggle ), TRUE );

	/* remote path */
	entry = glade_xml_get_widget( xml, "remote_path_entry" );
	text = screem_site_get_remote_path( site );
	if( ! text )
		text = "";
	gtk_entry_set_text( GTK_ENTRY( entry ), text );

	/* remote user */
	entry = glade_xml_get_widget( xml, "remote_user_entry" );
	text = screem_site_get_remote_user( site );
	if( ! text )
		text = "";
	gtk_entry_set_text( GTK_ENTRY( entry ), text );
	
	/* remote password */
	entry = glade_xml_get_widget( xml, "remote_password_entry" );
	text = screem_site_get_remote_pass( site );
	if( ! text )
		text = "";
	gtk_entry_set_text( GTK_ENTRY( entry ), text );

	/* attach the site to the dialog */
	gtk_object_set_data( GTK_OBJECT( dialog1 ), "site", site );

}

void screem_site_settings_change_url( GtkWidget *widget, gpointer data )
{
	gchar *host;
	GtkWidget *url;
	GtkWidget *rhost;
	GtkWidget *path;
	GtkWidget *webdav;
	GladeXML *xml;
	gchar *pathname;
	gchar *baseuri;

	xml = glade_get_widget_tree( widget );

	webdav = glade_xml_get_widget( xml, upload_strings[ WEBDAV ] );
	if( ! gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( webdav ) ) )
		return;

	url = glade_xml_get_widget( xml, "http_url_entry" );
	path = glade_xml_get_widget( xml, "remote_path_entry" );
	rhost = glade_xml_get_widget( xml, "remote_url_entry" );

	host = gtk_entry_get_text( GTK_ENTRY( rhost ) );
	pathname = gtk_entry_get_text( GTK_ENTRY( path ) );

	baseuri = g_strconcat( "http://", host, pathname, NULL );
	gtk_entry_set_text( GTK_ENTRY( url ), baseuri );
	g_free( baseuri );
}

void screem_site_settings_dialog_destroy_cb( GtkWidget *widget, 
					     GdkEvent *event )
{
	dialog1 = NULL;
}

void screem_site_settings_dialog_cb( GtkWidget *widget, gint button ) 
{
	GtkWidget *entry;
	GtkWidget *toggle;
	Site *site;

	if( button < 2 ) { /* ok or apply clicked */
		site = gtk_object_get_data(GTK_OBJECT( widget ), "site");

		/* name */
		entry = glade_xml_get_widget( xml, "site_name_entry" );
		screem_site_set_name( site, gtk_entry_get_text( GTK_ENTRY( entry ) ) );
		/* cvs root */
		toggle = glade_xml_get_widget( xml, "cvs_check" );
		if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( toggle ) ) ) {
			entry = glade_xml_get_widget(xml,"cvs_rep_path");
			screem_site_set_cvs_root( site, gtk_entry_get_text( GTK_ENTRY( entry ) ) );
		} else {
			screem_site_set_cvs_root( site, NULL );
		}

		/* template */
		toggle = glade_xml_get_widget( xml, "template_check" );
		if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( toggle ) ) ) {
			entry = glade_xml_get_widget( xml, "template_entry" );
			screem_site_set_template_path( site, gtk_entry_get_text( GTK_ENTRY( entry ) ) );
		} else {
			screem_site_set_template_path( site, NULL );
		}

		/* remote url */
		entry = glade_xml_get_widget( xml, "remote_url_entry" );
		screem_site_set_remote_url( site, gtk_entry_get_text( GTK_ENTRY( entry ) ) );
		/* http url */
		entry = glade_xml_get_widget( xml, "http_url_entry" );
		screem_site_set_http_url( site, gtk_entry_get_text( GTK_ENTRY( entry ) ) );

		/* remote method */
		toggle = glade_xml_get_widget( xml, "local" );
		if(gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(toggle) )) {
			screem_site_set_remote_method( site, LOCAL );
		}
		toggle = glade_xml_get_widget( xml, "ftp" );
		if(gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(toggle) )) {
			screem_site_set_remote_method( site, FTP );
		}
		toggle = glade_xml_get_widget( xml, "webdav" );
		if(gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(toggle) )) {
			screem_site_set_remote_method( site, WEBDAV );
		}
		toggle = glade_xml_get_widget( xml, "rsh" );
		if(gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(toggle) )) {
			screem_site_set_remote_method( site, RSH );
		}
		toggle = glade_xml_get_widget( xml, "ssh" );
                if(gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(toggle) )) {
                        screem_site_set_remote_method( site, SSH );
                }


		/* passive ftp */
		toggle = glade_xml_get_widget( xml, "ftpmode" );
		screem_site_set_passive_ftp( site, gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(toggle) ));

		/* no file delete */
		toggle = glade_xml_get_widget( xml, "nodelete" );
		screem_site_set_no_delete( site, gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(toggle) ));

		/* check for moved files */
		toggle = glade_xml_get_widget( xml, "checkmoved" );
		screem_site_set_check_moved( site, gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(toggle) ));

		/* no file overwrite */
		toggle = glade_xml_get_widget( xml, "nooverwrite" );
		screem_site_set_no_overwrite( site, gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(toggle) ));

		/* permissions handling */
		toggle = glade_xml_get_widget( xml, "p_ignore" );
		if(gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(toggle) )) {
			screem_site_set_permissions( site, PERMS_IGNORE );
		}
		toggle = glade_xml_get_widget( xml, "p_exec" );
		if(gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(toggle) )) {
			screem_site_set_permissions( site, PERMS_EXEC );
		}
		toggle = glade_xml_get_widget( xml, "p_all" );
		if(gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(toggle) )) {
			screem_site_set_permissions( site, PERMS_ALL );
		}

		/* symbolic link handling */
		toggle = glade_xml_get_widget( xml, "s_ignore" );
		if(gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(toggle) )) {
			screem_site_set_symlinks( site, SYM_IGNORE );
		}
		toggle = glade_xml_get_widget( xml, "s_follow" );
		if(gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(toggle) )) {
			screem_site_set_symlinks( site, SYM_FOLLOW );
		}
		toggle = glade_xml_get_widget( xml, "s_maintain" );
		if(gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(toggle) )) {
			screem_site_set_symlinks( site, SYM_MAINTAIN );
		}

		/* remote path */
		entry = glade_xml_get_widget( xml, "remote_path_entry" );
		screem_site_set_remote_path( site, gtk_entry_get_text( GTK_ENTRY( entry ) ) );

		/* remote user */
		entry = glade_xml_get_widget( xml, "remote_user_entry" );
		screem_site_set_remote_user( site, gtk_entry_get_text( GTK_ENTRY( entry ) ) );

		/* remote password */
		entry = glade_xml_get_widget( xml, 
					      "remote_password_entry");
		screem_site_set_remote_pass( site, gtk_entry_get_text( GTK_ENTRY( entry ) ) );
	}

	if( ( button == 0 ) || ( button == 2 ) ) {
		/* close the dialog */
		gtk_widget_destroy( widget );
		dialog1 = NULL;
	}
}

void build_recent_sites_menu()
{

	GnomeUIInfo *menu;
	gchar *path = _("_File/_Recent Sites/");
        int i;
     
	GList *list = cfg->recent_sites;
	GList *prev = NULL;

	if( ( i = g_list_length( list ) ) ) {
                gnome_app_remove_menu_range( GNOME_APP( app ), path, 0, i );
        } else {
		return;
	}

	list = g_list_last( list );
	for( i = 3; list && i > 0; i --, list = list->prev )
		prev = list;
 
	if( ! list )
		list = prev;

	cfg->recent_sites = list;
	list->prev = NULL;

	for( list = cfg->recent_sites; list; list = list->next ) {
		menu = g_malloc0 (2 * sizeof (GnomeUIInfo));
		menu->label = escape_path( list->data );
		
		menu->type = GNOME_APP_UI_ITEM;
		menu->hint = NULL;
		
		menu->moreinfo = (gpointer)recent_proxy;
		
		menu->user_data = list->data;
		menu->unused_data = NULL;
		menu->pixmap_type = 0;
		menu->pixmap_info = NULL;
		menu->accelerator_key = 0;
		
		(menu + 1)->type = GNOME_APP_UI_ENDOFINFO;
		
		gnome_app_insert_menus( GNOME_APP(app), path, menu );

		g_free( menu->label );
		g_free( menu );
	}

	save_preferences();
}

static void recent_proxy( GtkWidget *widget, gchar *filename )
{
      	screem_site_open_with_filename( filename );
}

void screem_build_sites_combo_list()
{
	GtkWidget *combo = 
		gtk_object_get_data(GTK_OBJECT( app ), "sitecombo");

	GList *li;
	GList *li2;
	Site *site;
	const gchar *name;

	ignore_change = TRUE;

	li = NULL;
	for( li2 = loaded_sites; li2; li2 = li2->next ) {
		site = li2->data;
		name = screem_site_get_name( site );
		/* cast to gchar* from const gchar* to avoid compiler
		   warning - name will not be freed so its ok, if a 
		   little dirty */
		li = g_list_append( li, (gchar*)name );
	}

	gtk_combo_set_popdown_strings( GTK_COMBO( combo ), li );
	g_list_free( li );
}

void screem_site_change( GtkWidget *widget, gpointer data )
{
	gchar *name;
	GList *list;
	Site *site = NULL;

	const gchar *n;

	const gchar *page_path;
	gchar *pname;
	GtkWidget *label;
	GtkWidget *junk;
	GtkWidget *ebox;

	Page *page;

	gint num;
	gint c;
	GtkWidget *book = gtk_object_get_data( GTK_OBJECT( app ), "pagebook" );

	if( ignore_change ) {
		ignore_change = FALSE;
		return;
	}

	name = gtk_entry_get_text( GTK_ENTRY( widget ) );

	/* find name in the list of loaded sites */
	for( list = loaded_sites; list && ! site; list = list->next ) {
		n = screem_site_get_name( list->data );
		if( ! strcmp( n, name ) )
			site = list->data;
	}

	if( ! site ) {
		screem_show_warning( "Error: Unable to locate a site!\n" );
		return;
	}
	if( site == current_site )
		return;

	screem_editor_display_page( NULL );

	/* remove old file tabs */
	num = screem_site_get_open_pages( current_site );
	gtk_object_set_data( GTK_OBJECT( book ), "switching",
			     GINT_TO_POINTER( 1 ) );
	while( num >= 0 )
		gtk_notebook_remove_page( GTK_NOTEBOOK( book ), num -- );

	/* create new ones */
	num = screem_site_get_open_pages( site );
	for( c = 0; num >= 0; num --, c++ ) {
		page = screem_site_get_page_from_tab( site, c );
		page_path = screem_page_get_pathname( page );
		pname = g_dirname( page_path );
		label = gtk_label_new( page_path +
				       strlen( pname ) + 1 );
		g_free( pname );
		junk = gtk_invisible_new();
		gtk_widget_show( junk );
		gtk_widget_set_usize( junk, 0, 0 );
		gtk_widget_show( label );
		ebox = gtk_event_box_new();
		gtk_container_add( GTK_CONTAINER( ebox ), label );
		gtk_notebook_append_page( GTK_NOTEBOOK( book ), junk, ebox );

		gtk_signal_connect( GTK_OBJECT( ebox ),
				    "button_press_event",
				    GTK_SIGNAL_FUNC( screem_page_tab_clicked ),
				    0 );
	}


	/* set current site to be this site */
	current_site = site;
	
	/* show the file tree */
	refresh_file_tree();

	page = screem_site_get_current_page( current_site );
	if( page ) {
		screem_editor_display_page( page );
		num = screem_page_get_tab( page );
		gtk_notebook_set_page( GTK_NOTEBOOK( book ), num );
	}
	gtk_object_set_data( GTK_OBJECT( book ), "switching",
			     GINT_TO_POINTER( 0 ) );
}

void screem_site_rename_pages_with_path( Site *site, gchar *path, 
					 gchar *new_path )
{
	GList *pages;
	Page *page;
	gint num;
	GtkWidget *book;
	GtkWidget *note;
	GtkWidget *label;
	const gchar *page_path;
	gchar *new_page_path;
	gchar *pname;

	g_return_if_fail( site != NULL );
	g_return_if_fail( path != NULL );
	g_return_if_fail( new_path != NULL );

	book = gtk_object_get_data( GTK_OBJECT( app ), "pagebook" );
      
	for(pages = screem_site_get_pages(site); pages; pages = pages->next) {
		page = pages->data;
		page_path = screem_page_get_pathname( page );
		if( ! strncmp( page_path, path, strlen( path ) ) ) {
			/* match found, rename page path */
			new_page_path = g_strconcat( new_path,
						     page_path + strlen(path),
						     NULL );
			screem_page_set_pathname( page, new_page_path );
			if( ( num = screem_page_get_tab( page ) ) > -1 ) {
				/* page is open, rename the tab */
				note = gtk_notebook_get_nth_page( GTK_NOTEBOOK( book ), num );
				pname = g_dirname( new_page_path );
				label = gtk_label_new( new_page_path + 
						       strlen( pname ) + 1 );
				g_free( pname );
				gtk_notebook_set_tab_label( GTK_NOTEBOOK(book),
							    note, label );
			}
			g_free( new_page_path );
		}
	}
}

void screem_site_purge_pages_with_path( Site *site, gchar *path )
{
	GList *pages;
	Page *page;
	gint num;
	GtkWidget *book;
	const gchar *page_path;

	g_return_if_fail( site != NULL );
	g_return_if_fail( path != NULL );

	book = gtk_object_get_data( GTK_OBJECT( app ), "pagebook" );
	pages = screem_site_get_pages( site );

	while( pages ) {
		page = pages->data;
		page_path = screem_page_get_pathname( page );
		if( ! strncmp( page_path, path, strlen( path ) ) ) {
			/* match found, remove the page */
			if( ( num = screem_page_get_tab( page ) ) > -1 ) {
				/* page is open */
				gtk_notebook_remove_page( GTK_NOTEBOOK( book ),
							  num );
			}
			/* remove page from site */
			page = screem_site_remove_page( site, page_path );
			screem_page_destroy( page );
			pages = screem_site_get_pages( site );
			if( num > -1 )
				screem_page_update_tab_nums( pages, num );

		} else {
			pages = pages->next;
		}
	}
}

void screem_site_switch_to_site( const gchar *name )
{
	GtkWidget *combo;
	GList *in_list;
    	const gchar *temp_name;

	g_return_if_fail( name != NULL );

        combo = gtk_object_get_data(GTK_OBJECT( app ), "sitecombo");

	for( in_list = loaded_sites; in_list; in_list = in_list->next ) {
		temp_name = screem_site_get_name( in_list->data );
		if( ! strcmp( name, temp_name ) )
			break;
	}

	g_return_if_fail( in_list != NULL );

	gtk_entry_set_text( GTK_ENTRY( GTK_COMBO( combo )->entry ), name );
}

static void screem_site_gather_stats( GladeXML *xml )
{
	GtkWidget *last_upload;
	GtkWidget *files_list;
	GtkWidget *total_size;
	const gchar *path;
	xmlDocPtr doc;
	xmlNodePtr node;
	GList *list;
	Stats *s;
	gchar *num;
	gchar *avg;
	gint total;
	gchar *items[ 4 ] = { NULL, NULL, NULL, NULL };
	gchar *sitecopy_file;
	const gchar *name;
	struct stat st;

	last_upload = glade_xml_get_widget( xml, "last_upload" );
	files_list = glade_xml_get_widget( xml, "files_list" );
	total_size = glade_xml_get_widget( xml, "total_size" );

	name = screem_site_get_name( current_site );

	sitecopy_file = g_strconcat( g_get_home_dir(), G_DIR_SEPARATOR_S,
				     ".sitecopy", G_DIR_SEPARATOR_S, name,
				     NULL );
	if( ! stat( sitecopy_file, &st ) ) {
		gtk_entry_set_text( GTK_ENTRY( last_upload ),
				    ctime( &st.st_mtime ) );
	} else {
		gtk_entry_set_text( GTK_ENTRY( last_upload ), _("Unknown") );
	}
	g_free( sitecopy_file );

	path = screem_site_get_pathname( current_site );
	doc = build_directory_tree( current_site, path, NULL );
	node = doc->root;
	node = node->childs;
	node = xml_search_child( node, "node" );
	list = calc_stats( node, NULL );

	gtk_clist_freeze( GTK_CLIST( files_list ) );
	gtk_clist_clear( GTK_CLIST( files_list ) );

	/* we have the list of stats */
      	for( total = 0; list; list = list->next ) {
		s = list->data;
		num = g_strdup_printf( "%i", s->number );
		avg = g_strdup_printf( "%i", s->total_size / s->number );
		total += s->total_size;
		items[ 0 ] = s->mime_type;
		items[ 1 ] = num;
		items[ 2 ] = avg;
		gtk_clist_append( GTK_CLIST( files_list ), items );
		g_free( s->mime_type );
		g_free( num );
		g_free( avg );
		g_free( s );
	}
	g_list_free( list );

	gtk_clist_columns_autosize( GTK_CLIST( files_list ) );
	gtk_clist_thaw( GTK_CLIST( files_list ) );

	num = g_strdup_printf( _("%i bytes"), total );
	gtk_entry_set_text( GTK_ENTRY( total_size ), num );
	g_free( num );

	xmlFreeDoc( doc );
}

static GList* calc_stats( xmlNodePtr node, GList *types )
{
	Stats *s;
	gchar *type;
	gchar *size;
	GList *list;
	xmlNodePtr child;

	if( ! node )
		return types;

	s = NULL;
	type = xml_get_value( node, "type" );

	if( ! strcmp( type, "directory" ) ) {
		/* another directory to scan */
		child = xml_search_child( node, "node" );
		types = calc_stats( child, types );
	} else {
		/* a file, try and locate its mime type in the list of
		   types already found */
		for( list = types; list; list = list->next ) {
			s = list->data;
			if( ! strcmp( s->mime_type, type ) )
				break;
		}
		/* if list then we broke out and so found one */
		size = xml_get_value( node, "size" );
		if( list ) {
			s->number ++;
			s->total_size += atoi( size );
		} else {
			s = g_new0( Stats, 1 );
			s->number = 1;
			s->total_size = atoi( size );
			s->mime_type = g_strdup( type );
			types = g_list_append( types, s );
		}
	}

	list = calc_stats( node->next, types );

	return list;
}
