;*=====================================================================*/
;*    serrano/prgm/project/scribe/scribeapi/biblio.scm                 */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Fri Dec  7 06:12:29 2001                          */
;*    Last change :  Sun Jan 13 10:43:31 2002 (serrano)                */
;*    Copyright   :  2001-02 Manuel Serrano                            */
;*    -------------------------------------------------------------    */
;*    Bibtex handling                                                  */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    The module                                                       */
;*---------------------------------------------------------------------*/
(module __scribeapi_biblio

   (import __scribeapi_io
	   __scribeapi_ast
	   __scribeapi_api
	   __scribeapi_param)

   (export (biblio-load ::bstring)
	   (biblio-add . entries)
	   (biblio-find ::bstring)
	   (biblio-all)
	   (biblio-print-bib)))

;*---------------------------------------------------------------------*/
;*    *biblio-table*                                                   */
;*---------------------------------------------------------------------*/
(define *biblio-table* #unspecified)

;*---------------------------------------------------------------------*/
;*    *biblio-used* ...                                                */
;*---------------------------------------------------------------------*/
(define *biblio-used* '())

;*---------------------------------------------------------------------*/
;*    *biblio-string-table* ...                                        */
;*---------------------------------------------------------------------*/
(define *biblio-string-table* #unspecified)

;*---------------------------------------------------------------------*/
;*    biblio-init! ...                                                 */
;*---------------------------------------------------------------------*/
(define (biblio-init!)
   (if (not (hashtable? *biblio-table*))
       (set! *biblio-table* (make-hashtable)))
   (if (not (hashtable? *biblio-used*))
       (set! *biblio-used* (make-hashtable))))

;*---------------------------------------------------------------------*/
;*    make-biblio-hashtable ...                                        */
;*---------------------------------------------------------------------*/
(define (make-biblio-hashtable)
   (let ((table (make-hashtable)))
      (for-each (lambda (k)
		   (let ((cp (string-capitalize k)))
		      (hashtable-put! table k cp)
		      (hashtable-put! table cp cp)))
		'("jan" "feb" "mar" "apr" "may" "jun" "jul"
			"aug" "sep" "oct" "nov" "dec"))
      table))

;*---------------------------------------------------------------------*/
;*    biblio-find ...                                                  */
;*---------------------------------------------------------------------*/
(define (biblio-find ident)
   (if (hashtable? *biblio-table*)
       (let ((en (hashtable-get *biblio-table* ident)))
	  (if (%bibentry? en)
	      (begin
		 (hashtable-update! *biblio-used* ident (lambda (x) x) en)
		 en)
	      #f))))

;*---------------------------------------------------------------------*/
;*    biblio-all ...                                                   */
;*---------------------------------------------------------------------*/
(define (biblio-all)
   (if (hashtable? *biblio-table*)
       (hashtable-for-each *biblio-table*
			   (lambda (k e)
			      (hashtable-update! *biblio-used*
						 k
						 (lambda (x) x)
						 e)))))

;*---------------------------------------------------------------------*/
;*    biblio-load ...                                                  */
;*---------------------------------------------------------------------*/
(define (biblio-load filename::bstring)
   ;; initialize the global bib hashtable
   (biblio-init!)
   ;; read the file
   (let ((p (open-bib-file filename)))
      (if (not (input-port? p))
	  (error "bibliograph" "Can't open data base" filename)
	  (unwind-protect
	     (parse-biblio p)
	     (close-input-port p)))))

;*---------------------------------------------------------------------*/
;*    biblio-add ...                                                   */
;*---------------------------------------------------------------------*/
(define (biblio-add . entries)
   ;; initialize the global bib hashtable
   (biblio-init!)
   ;; add the entries
   (for-each (lambda (entry)
		(match-case entry
		   (((and (? symbol?) ?kind) (and (? symbol?) ?ident) . ?fields)
		    (let ((ident (symbol->string ident)))
		       (hashtable-put! *biblio-table*
				       ident
				       (make-biblio-entry kind ident fields))))
		   (else
		    (biblio-error entry))))
	     entries))

;*---------------------------------------------------------------------*/
;*    biblio-error ...                                                 */
;*---------------------------------------------------------------------*/
(define (biblio-error entry)
   (if (epair? entry)
       (match-case (cer entry)
	  ((at ?fname ?pos ?-)
	   (error/location "parse-biblio"
			   "bibliography syntax error"
			   entry
			   fname
			   pos))
	  (else
	   (error "parse-biblio" "bibliography syntax error" entry)))
       (error "parse-biblio" "bibliography syntax error" entry)))

;*---------------------------------------------------------------------*/
;*    parse-biblio ...                                                 */
;*---------------------------------------------------------------------*/
(define (parse-biblio port)
   (let loop ((entry (read port #t)))
      (if (not (eof-object? entry))
	  (match-case entry
	     (((and (? symbol?) ?kind) (and (? symbol?) ?ident) . ?fields)
	      (let ((ident (symbol->string ident)))
		 (hashtable-put! *biblio-table*
				 ident
				 (make-biblio-entry kind ident fields)))
	      (loop (read port #t)))
	     (else
	      (biblio-error entry))))))

;*---------------------------------------------------------------------*/
;*    make-biblio-entry ...                                            */
;*---------------------------------------------------------------------*/
(define (make-biblio-entry kind ident fields)
   (instantiate::%bibentry
      (kind kind)
      (id ident)
      (fields (map (lambda (f)
		      (match-case f
			 ((?key ?val)
			  f)
			 (else
			  (biblio-error f))))
		   fields))))

;*---------------------------------------------------------------------*/
;*    biblio-print-bib ...                                             */
;*---------------------------------------------------------------------*/
(define (biblio-print-bib)
   (if (hashtable? *biblio-used*)
       (let ((all (sort (hashtable->list *biblio-used*)
			(lambda (e1 e2)
			   (string<? (%bibentry-id e1)
				     (%bibentry-id e2))))))
	  ;; allocates a number of each bibentry and print them
	  (let ((bib (let loop ((all all)
				(i 1)
				(res '()))
			(if (pair? all)
			    (with-access::%bibentry (car all) (number id)
			       (set! number i)
			       (loop (cdr all)
				     (+fx i 1)
				     (cons (biblio-render-entry (car all))
					   res)))
			    (reverse! res)))))
	     (if (scribe-format? 'tex)
		 (hook :before (lambda ()
				  (print "\\begin{thebibliography}{"
					 (length all)
					 "}"))
		       bib
		       :after (lambda ()
				 (print "\n\\end{thebibliography}")))
		 (apply paragraph bib))))))

;*---------------------------------------------------------------------*/
;*    biblio-render-entry ...                                          */
;*    -------------------------------------------------------------    */
;*    It is important to introduce the %BIBENTRY instance inself       */
;*    in the Scribe result of this function. %BIBENTRY is not          */
;*    rendered by back-ends (it inherits from classes that have        */
;*    no rendering) but it is mandatory to introduce it so that        */
;*    it is marked by the CONTAINER-SET! (see __scribeapi_container)   */
;*    function and so the %BIBENTRY will be correctly point to         */
;*    by %BIBLIO-REFs.                                                 */
;*---------------------------------------------------------------------*/
(define (biblio-render-entry e::%bibentry)
   (with-access::%bibentry e (number fields kind)
      (define (find-field id::symbol default pref suf proc)
	 (let ((field (assoc id fields)))
	    (if (and (pair? field)
		     (or (not (string? (cadr field)))
			 (>fx (string-length (cadr field)) 0)))
		(list pref (proc (cadr field)) suf)
		default)))
      (define (tex-author->scribe val)
	 val)
      (define (author)
	 (let ((field (find-field 'author #f "" " -- " *scribe-bib-author*)))
	    (if (not field)
		""
		(tex-author->scribe field))))
      (define (title)
	 (let* ((durl (find-field 'documenturl #f "" "" nop))
		(url (find-field 'url durl "" "" nop))
		(proctitle (lambda (t)
			      (*scribe-bib-title* t url))))
	    (find-field 'title "" "" " -- " proctitle)))
      (define (nop x)
	 x)
      (define (do-entry . bib-entry)
	 (if (scribe-format? 'tex)
	     (hook :before (lambda ()
			      (print "\\bibitem{"
				     (%bibentry-stamp e)
				     "}"))
		   bib-entry)
	     (apply flush :side 'left
		    e
		    (string-append "[" (number->string number) "] ")
		    bib-entry)))
      (define (techreport)
	 (do-entry (author)
		   (title)
		   (find-field 'number "" "" ", " nop)
		   (find-field 'institution "" "" ", " *scribe-bib-edition*)
		   (find-field 'address "" "" ", " *scribe-bib-address*)
		   (find-field 'month "" "" ", " *scribe-bib-month*)
		   (find-field 'year "" "" "" *scribe-bib-year*)
		   (find-field 'pages "" ", pp. " "" nop)
		   ".\n"))
      (define (article)
	 (do-entry (author)
		   (title)
		   (find-field 'journal "" "" ", " *scribe-bib-edition*)
		   (find-field 'volume "" "" "" nop)
		   (find-field 'number "" "(" "), " nop)
		   (find-field 'address "" "" ", " *scribe-bib-address*)
		   (find-field 'month "" "" ", " *scribe-bib-month*)
		   (find-field 'year "" "" "" *scribe-bib-year*)
		   (find-field 'pages "" ", pp. " "" nop)
		   ".\n"))
      (define (inproceedings)
	 (do-entry (author)
		   (title)
		   (find-field 'booktitle "" "" ", " *scribe-bib-edition*)
		   (find-field 'series "" "" " " *scribe-bib-edition*)
		   (find-field 'number "" "(" "), " *scribe-bib-edition*)
		   (find-field 'address "" "" ", " *scribe-bib-address*)
		   (find-field 'month "" "" ", " *scribe-bib-month*)
		   (find-field 'year "" "" "" *scribe-bib-year*)
		   (find-field 'pages "" ", pp. " "" nop)
		   ".\n"))
      (define (book)
	 (do-entry (author)
		   (title)
		   (find-field 'publisher "" "" ", " *scribe-bib-edition*)
		   (find-field 'address "" "" ", " *scribe-bib-address*)
		   (find-field 'month "" "" ", " *scribe-bib-month*)
		   (find-field 'year "" "" "" *scribe-bib-year*)
		   (find-field 'pages "" ", pp. " "" nop)
		   ".\n"))
      (define (default)
	 (do-entry (author)
		   (title)
		   (find-field 'month "" "" ", " *scribe-bib-month*)
		   (find-field 'year "" "" "" *scribe-bib-year*)
		   ".\n"))
      (case kind
	 ((techreport) (techreport))
	 ((article) (article))
	 ((book) (book))
	 ((inproceedings) (inproceedings))
	 (else (default)))))
