#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "../include/os.h"

#if defined(__MSW__)
# include <windows.h>
#else
# include <X11/X.h>
# include <X11/Xlib.h>
# include <X11/Xutil.h>
# include <X11/keysym.h>
# include <X11/Xproto.h>
# include <X11/Xatom.h>
# include <X11/cursorfont.h>
# include <X11/extensions/shape.h>
# ifdef HAVE_XF86_VIDMODE
#  include <X11/extensions/xf86vmode.h>
#  ifndef XF86VIDMODE_H
#   define XF86VIDMODE_H
#  endif
#  ifndef XF86VIDMODE_EVENTS
#   define XF86VIDMODE_EVENTS
#  endif
# endif	/* HAVE_XF86_VIDMODE */
# ifdef HAVE_LIBXPM
#  include <X11/xpm.h>
#  ifndef XPM_H
#   define XPM_H
#  endif
#  ifndef XpmDefaultColorCloseness
#   define XpmDefaultColorCloseness      40000
#  endif
# endif	/* HAVE_LIBXPM */
#endif

#include <GL/gl.h>
#include <GL/glu.h>

#include "../include/string.h"
#include "../include/tga.h"

#include "gw.h"
#include "stategl.h"

#include "gwstdfont.h"


#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))


#ifdef X_H
int GWXErrorHandler(Display *xdpy, XErrorEvent *xerrorevent);
int GWGetCharacterFromKeyCode(
	gw_display_struct *display, KeyCode keycode
);
void GWSetWindowIconFile(
        gw_display_struct *display,
        const char *icon_path, const char *icon_name
);
Window GWCreateWindow(
	gw_display_struct *display,
	Window parent,
	int x, int y,
	int width, int height,
        const char *title
);
#endif	/* X_H */

#ifdef X_H
gw_display_struct *GWInit(int argc, char **argv);
void GWManage(gw_display_struct *display);
void GWShutdown(gw_display_struct *display);
#endif	/* X_H */

void GWOutputMessage(
	gw_display_struct *display,
	int type,
	const char *subject,
	const char *message,
	const char *help_details
);
int GWConfirmation(
	gw_display_struct *display,
	int type,
	const char *subject,
	const char *message,
	const char *help_details
);
int GWConfirmationSimple(
	gw_display_struct *display, const char *message
);

void GWSetDrawCB(
	gw_display_struct *display,
	void (*func)(void *),
	void *data
);
void GWSetResizeCB(
	gw_display_struct *display,
        void (*func)(void *, int, int, int, int),
        void *data
);
void GWSetKeyboardCB(
	gw_display_struct *display,
	void (*func)(void *, int, Boolean, long),
        void *data
);
void GWSetPointerCB(
	gw_display_struct *display,
	void (*func)(void *, int, int, int, int, long),
	void *data
);
void GWSetVisibilityCB(
	gw_display_struct *display,
	void (*func)(void *, int),
	void *data
);
void GWSetSaveYourselfCB(
        gw_display_struct *display,
        void (*func)(void *),
        void *data
);
void GWSetCloseCB(
        gw_display_struct *display,
        void (*func)(void *, void *),
        void *data
);
void GWSetTimeoutCB(
	gw_display_struct *display,
	void (*func)(void *),
	void *data
);

void GWPostRedraw(gw_display_struct *display);
void GWSwapBuffer(gw_display_struct *display);

void GWOrtho2D(gw_display_struct *display);
void GWKeyboardAutoRepeat(gw_display_struct *display, Boolean b);

gw_accelerator_struct *GWAcceleratorNew(
	int key, int modifier
);
void GWAcceleratorDelete(gw_accelerator_struct *a);
int GWAcceleratorListAdd(
        gw_accelerator_struct ***a, int *total,
        int key, int modifier
);
void GWAcceleratorListDelete(
        gw_accelerator_struct ***a, int *total
);
Boolean GWAcceleratorListCheck(
	gw_accelerator_struct **a, int total,
	int key, int modifier
);

void GWSetPointerCursor(
	gw_display_struct *display,
	int cursor	/* One of GWPointerCursor*. */
);
void GWSetInputBusy(gw_display_struct *display);
void GWSetInputReady(gw_display_struct *display);

GWFont *GWLoadFont(gw_display_struct *display, char *name);
void GWSetFont(gw_display_struct *display, GWFont *font);
int GWGetFontSize(
	GWFont *font,
	int *width, int *height,
	int *character_spacing, int *line_spacing
);
void GWUnloadFont(gw_display_struct *display, GWFont *font);
GWColor *GWLoadColor(
	gw_display_struct *display,
	double a, double r, double g, double b
);
void GWSetColor(gw_display_struct *display, GWColor *color);
void GWUnloadColor(gw_display_struct *display, GWColor *color);

void GWDrawString(
	gw_display_struct *display,
	int x, int y,
	const char *string
);
void GWDrawCharacter(
	gw_display_struct *display,
	int x, int y,
	char c
);
void GWDrawLine(
	gw_display_struct *display,
	int x1, int y1,
	int x2, int y2
);


/* Graphics wrapped debugging flag. */
static Boolean gw_debug = False;
/* Prefix to all GW debugging messages. */
#define GW_DEBUG_PREFIX	"GW Debug: "


#ifdef X_H
/*
 *	Modified by GWXErrorHandler() to indicate error or none.
 *
 *	This will be set to a non-zero value when GWXErrorHandler()
 *	is called, GWXErrorHandler() is called whenever the X server
 *	generates an error.
 */
static int xerrval;
#endif	/* X_H */


#ifdef X_H
/*
 *	Visual attributes list (for picking Visuals), last member must
 *	be None to mark end of array.
 */
static int attributeListSglRGB[] = {
	GLX_RGBA,		/* Use only TrueColor or DirectColor. */
	GLX_RED_SIZE,	1,	/* Get deepest buffer with >= 1 red bit */
	GLX_GREEN_SIZE,	1,
	GLX_BLUE_SIZE,	1,
	GLX_DEPTH_SIZE,	1,	/* Z buffer depth, not bit depth. */
	GLX_STENCIL_SIZE, 1,
	None
};

static int attributeListSglRGBA[] = {
	GLX_RGBA,		/* Use only TrueColor or DirectColor. */
	GLX_RED_SIZE,	1,	/* Get deepest buffer with >= 1 red bit */
	GLX_GREEN_SIZE,	1,
	GLX_BLUE_SIZE,	1,
	GLX_ALPHA_SIZE,	1,
	GLX_DEPTH_SIZE,	1,
        GLX_STENCIL_SIZE, 1,
	None
};

static int attributeListDblRGB[] = {
	GLX_RGBA,		/* Use only TrueColor or DirectColor. */
	GLX_DOUBLEBUFFER,	/* Double buffer. */
	GLX_RED_SIZE,	1,
	GLX_GREEN_SIZE,	1,
	GLX_BLUE_SIZE,	1,
	GLX_DEPTH_SIZE,	1,
        GLX_STENCIL_SIZE, 1,
	None
};

static int attributeListDblRGBA[] = {
	GLX_RGBA,		/* Use only TrueColor or DirectColor. */
	GLX_DOUBLEBUFFER,	/* Double buffer. */
	GLX_RED_SIZE,	1,
	GLX_GREEN_SIZE,	1,
	GLX_BLUE_SIZE,	1,
	GLX_ALPHA_SIZE,	1,
	GLX_DEPTH_SIZE,	1,
        GLX_STENCIL_SIZE, 1,
	None
};
#endif	/* X_H */


/*
 *	Keysym name to character value table, even numbers are pointers
 *	to strings while odd numbers are pointers who's value is a
 *	character value.
 */
static char *keytable[] = {
	"Alt_L",	(char *)GWKeyAlt,
	"Alt_R",        (char *)GWKeyAlt,
        "Control_L",    (char *)GWKeyCtrl,
        "Control_R",    (char *)GWKeyCtrl,
        "Shift_L",      (char *)GWKeyShift,
        "Shift_R",      (char *)GWKeyShift,

        "BackSpace",    (char *)GWKeyBackSpace,
        "Tab",          (char *)'\t',
	"ISO_Left_Tab",	(char *)'\t',
        "Linefeed",     (char *)'\f',
        "Clear",        (char *)'\0',
        "Return",       (char *)'\n',
        "Pause",        (char *)GWKeyPause,
        "Scroll_Lock",  (char *)GWKeyScrollLock,
        "Sys_Req",      (char *)GWKeySysReq,
        "Escape",       (char *)0x1b,
        "Delete",       (char *)GWKeyDelete,

	"Home",		(char *)GWKeyHome,
        "Left",         (char *)GWKeyLeft,
        "Up",           (char *)GWKeyUp,
        "Right",        (char *)GWKeyRight,
        "Down",         (char *)GWKeyDown,
        "Prior",        (char *)GWKeyPageUp,	/* ??? */
        "Page_Up",      (char *)GWKeyPageUp,
        "Next",         (char *)GWKeyPageDown,	/* ??? */
        "Page_Down",    (char *)GWKeyPageDown,
        "End",          (char *)GWKeyEnd,
        "Begin",        (char *)GWKeyHome,	/* ??? */

        "KP_0",         (char *)'0',
        "KP_1",         (char *)'1',
        "KP_2",         (char *)'2',
        "KP_3",         (char *)'3',
        "KP_4",         (char *)'4',
        "KP_5",         (char *)'5',
        "KP_6",         (char *)'6',
        "KP_7",         (char *)'7',
        "KP_8",         (char *)'8',
        "KP_9",         (char *)'9',

        "KP_Space",     (char *)' ',
        "KP_Tab",       (char *)'\t',
        "KP_Enter",     (char *)'\n',
        "KP_F1",        (char *)GWKeyF1,
        "KP_F2",        (char *)GWKeyF2,
        "KP_F3",        (char *)GWKeyF3,
        "KP_F4",        (char *)GWKeyF4,
        "KP_Home",      (char *)GWKeyHome,
        "KP_Left",      (char *)GWKeyLeft,
        "KP_Up",        (char *)GWKeyUp,
        "KP_Right",     (char *)GWKeyRight,
        "KP_Down",      (char *)GWKeyDown,
        "KP_Prior",     (char *)GWKeyPageUp,
        "KP_Page_Up",   (char *)GWKeyPageUp,
        "KP_Next",      (char *)GWKeyPageDown,
        "KP_Page_Down", (char *)GWKeyPageDown,
        "KP_End",       (char *)GWKeyEnd,
        "KP_Begin",     (char *)GWKeyHome,
        "KP_Insert",    (char *)GWKeyInsert,
        "KP_Delete",    (char *)GWKeyDelete,
        "KP_Equal",     (char *)'=',
        "KP_Multiply",  (char *)'*',
        "KP_Add",       (char *)'+',
        "KP_Separator", (char *)'\0',
        "KP_Subtract",  (char *)'-',
        "KP_Decimal",   (char *)'.',
        "KP_Divide",    (char *)'/',

	"F1",		(char *)GWKeyF1,
        "F2",           (char *)GWKeyF2,
        "F3",           (char *)GWKeyF3,
        "F4",           (char *)GWKeyF4,
        "F5",           (char *)GWKeyF5,
        "F6",           (char *)GWKeyF6,
        "F7",           (char *)GWKeyF7,
        "F8",           (char *)GWKeyF8,
        "F9",           (char *)GWKeyF9,
        "F10",          (char *)GWKeyF10,
        "F11",          (char *)GWKeyF11,
        "F12",          (char *)GWKeyF12,
        "F13",          (char *)GWKeyF13,
        "F14",          (char *)GWKeyF14,
        "F15",          (char *)GWKeyF15,
        "F16",          (char *)GWKeyF16,
        "F17",          (char *)GWKeyF17,
        "F18",          (char *)GWKeyF18,
        "F19",          (char *)GWKeyF19,
        "F20",          (char *)GWKeyF20,

        "space",        (char *)' ',
        "exclam",       (char *)'!',
        "quotedbl",     (char *)'"',
        "numbersign",   (char *)'#',
        "dollar",       (char *)'$',
        "percent",      (char *)'%',
        "ampersand",    (char *)'&',
        "apostrophe",   (char *)'\'',
        "quoteright",   (char *)'"',
        "parenleft",    (char *)'(',
        "parenright",   (char *)')',
        "asterisk",     (char *)'*',
        "plus",         (char *)'+',
        "comma",        (char *)',',
        "minus",        (char *)'-',
        "period",       (char *)'.',
        "slash",        (char *)'/',

	"0",		(char *)'0',
        "1",            (char *)'1',
        "2",            (char *)'2',
        "3",            (char *)'3',
        "4",            (char *)'4',
        "5",            (char *)'5',
        "6",            (char *)'6',
        "7",            (char *)'7',
        "8",            (char *)'8',
        "9",            (char *)'9',

        "colon",        (char *)':',
        "semicolon",    (char *)';',
        "less",         (char *)'\0',
        "equal",        (char *)'=',
        "greater",      (char *)'>',
        "question",     (char *)'?',
        "at",           (char *)'@',

        "A",            (char *)'A',
        "B",            (char *)'B',
        "C",            (char *)'C',
        "D",            (char *)'D',
        "E",            (char *)'E',
        "F",            (char *)'F',
        "G",            (char *)'G',
        "H",            (char *)'H',
        "I",            (char *)'I',
        "J",            (char *)'J',
        "K",            (char *)'K',
        "L",            (char *)'L',
        "M",            (char *)'M',
        "N",            (char *)'N',
        "O",            (char *)'O',
        "P",            (char *)'P',
        "Q",            (char *)'Q',
        "R",            (char *)'R',
        "S",            (char *)'S',
        "T",            (char *)'T',
        "U",            (char *)'U',
        "V",            (char *)'V',
        "W",            (char *)'W',
        "X",            (char *)'X',
        "Y",            (char *)'Y',
        "Z",            (char *)'Z',

        "bracketleft",  (char *)'[',
        "backslash",    (char *)'\\',
        "bracketright", (char *)']',
        "asciicircum",  (char *)'^',
        "underscore",   (char *)'_',
        "grave",        (char *)'\0',
        "quoteleft",    (char *)'"',

        "a",            (char *)'a',
        "b",            (char *)'b',
        "c",            (char *)'c',
        "d",            (char *)'d',
        "e",            (char *)'e',
        "f",            (char *)'f',
        "g",            (char *)'g',
        "h",            (char *)'h',
        "i",            (char *)'i',
        "j",            (char *)'j',
        "k",            (char *)'k',
        "l",            (char *)'l',
        "m",            (char *)'m',
        "n",            (char *)'n',
        "o",            (char *)'o',
        "p",            (char *)'p',
        "q",            (char *)'q',
        "r",            (char *)'r',
        "s",            (char *)'s',
        "t",            (char *)'t',
        "u",            (char *)'u',
        "v",            (char *)'v',
        "w",            (char *)'w',
        "x",            (char *)'x',
        "y",            (char *)'y',
        "z",            (char *)'z',

        "braceleft",    (char *)'{',
        "bar",          (char *)'|',
        "braceright",   (char *)'}',
        "asciitilde",   (char *)'~'

};


#ifdef X_H
/*
 *	X error handler:
 */
int GWXErrorHandler(Display *dpy, XErrorEvent *ev)
{
	const char *error_code_name;


        if((dpy == NULL) ||
           (ev == NULL)
        )
            return(0);

	/* Reset global xerror value. */
	xerrval = 0;

	/* ******************************************************* */
	/* Ignore these error codes. */

        /* Success. */
        if(ev->error_code == Success)
            return(0);

        /* Calls to XGetImage() often return error when the error is
	 * non-critical, we need to ignore that here.
	 */
        if(ev->request_code == X_GetImage)
            return(0);

        /* ******************************************************** */
        /* If this point is reached, it means we got a definate
	 * (serious) error from the X server.
	 */

	/* Set global xerrval to -1, indicating a serious error. */
	xerrval = -1;

	switch(ev->error_code)
	{
	  case Success:
	    error_code_name = "Success";
	    break;
          case BadRequest:
            error_code_name = "BadRequest";
            break;
	  case BadValue:
	    error_code_name = "BadValue";
	    break;
          case BadWindow:
            error_code_name = "BadWindow";
            break;
          case BadPixmap:
            error_code_name = "BadPixmap";
            break;
          case BadAtom:
            error_code_name = "BadAtom";
            break;
          case BadCursor:
            error_code_name = "BadCursor";
            break;
          case BadFont:
            error_code_name = "BadFont";
            break;
          case BadMatch:
            error_code_name = "BadMatch";
            break;
          case BadDrawable:
            error_code_name = "BadDrawable";
            break;
          case BadAccess:
            error_code_name = "BadAccess";
            break;
	  case BadAlloc:
            error_code_name = "BadAlloc";
            break;
          case BadColor:
            error_code_name = "BadColor";
            break;
          case BadGC:
            error_code_name = "BadGC";
            break;
          case BadIDChoice:
            error_code_name = "BadIDChoice";
            break;
          case BadName:
            error_code_name = "BadName";
            break;
          case BadLength:
            error_code_name = "BadLength"; 
            break;
          case BadImplementation:
            error_code_name = "BadImplementation";
            break;
	  default:
	    error_code_name = "Unknown";
	    break;
	}

	/* Print out the information about this error. */
	fprintf(
	    stderr,
 "X Server error #%ld:  type: %s  major_op_code: %i  minor_op_code: %i\n",
	    ev->serial,
	    error_code_name,
	    ev->request_code,
	    ev->minor_code
	);

	return(0);
}
#endif	/* X_H */


#ifdef X_H
/*
 *	Returns the character or key value for the given keycode
 *	looked up on the keytable.
 */
int GWGetCharacterFromKeyCode(
	gw_display_struct *display,
	KeyCode keycode
)
{
	int i;
	const int total = sizeof(keytable) / sizeof(char *);
	KeySym keysym;
	char *strptr;


	if(display == NULL)
	    return((int)'\0');

	if(display->display == NULL)
	    return((int)'\0');

	xerrval = 0;
	keysym = XKeycodeToKeysym(
	    display->display,
	    keycode,
/*	    (int)((display->shift_key_state) ? 1 : 0) */
	    0
	);
	if(xerrval)
	    return((int)'\0');

	xerrval = 0;
        if((keysym != XK_Shift_R) && (keysym != XK_Shift_L) &&
           display->shift_key_state
	)
        {
	    keysym = XKeycodeToKeysym(
		display->display,
		keycode,
		1
	    );
	    if(xerrval)
		return((int)'\0');
	}

	xerrval = 0;
	strptr = XKeysymToString(keysym);
	if(xerrval || (strptr == NULL))
	    return((int)'\0');

	for(i = 0; i < total; i += 2)
	{
	    if(!strcmp(keytable[i], strptr))
		return(
		    (int)(keytable[i + 1])
		);
	}

	return('\0');
}
#endif	/* X_H */


#ifdef X_H
/*
 *	Sets the WM icon for the given toplevel Window in display.
 *
 *	The path to the icon data icon_path needs to be an xpm file.
 *
 *	The icon_name will be set as the icon's icon name which appears
 *	next to the icon (or however the WM has it displayed).
 */
void GWSetWindowIconFile(
        gw_display_struct *display,
        const char *icon_path, const char *icon_name
)
{
#ifdef XPM_H
        int depth;
        char *tmp_filename;
	Display *dpy;
        Window w, root, icon_w = None;
        Pixmap icon_pm, icon_mask_pm;
        int xpm_status;
        XpmAttributes xpm_attr;
        XWMHints *wm_hints;


	if((display == NULL) || (icon_path == NULL))
	    return;

	dpy = display->display;
	w = display->toplevel;
	root = display->root;
	depth = display->depth;
        if((dpy == NULL) || (w == None) || (root == None) || (depth < 1))
            return;

        /* Set up XPM attributes for XPM loading. */
        xpm_attr.valuemask = XpmSize | XpmCloseness | XpmDepth;
        xpm_attr.closeness = XpmDefaultColorCloseness;
        xpm_attr.depth = depth;
        xpm_attr.width = 0;             /* Reset size returns. */
        xpm_attr.height = 0;
        tmp_filename = strdup(icon_path);
        xpm_status = XpmReadFileToPixmap(
            dpy, root, tmp_filename,
            &icon_pm, &icon_mask_pm,
            &xpm_attr
        );
        free(tmp_filename);
        if(xpm_status == XpmSuccess)
        {
            int count;
            int width = xpm_attr.width;
            int height = xpm_attr.height;
            Pixel black_pix = display->black_pix;
            Pixel white_pix = display->white_pix;
            XIconSize *icon_sizes;

            /* Check if loaded icon is a valid size by confirming with
             * X server.
             */
	    xerrval = 0;
            if(XGetIconSizes(dpy, w, &icon_sizes, &count))
            {
                int i; 
        
                for(i = 0; i < count; i++)
                {
                    if((width >= icon_sizes[i].min_width) &&
                       (width <= icon_sizes[i].max_width) &&
                       (height >= icon_sizes[i].min_height) &&
                       (height <= icon_sizes[i].max_height)
                    )
                        break; 
                }
                XFree(icon_sizes);
        
                /* Did not get valid icon size bounds? */
                if(i >= count)
                {
                    fprintf(
                        stderr,
                        "GWSetWindowIconFile(): Invalid icon size %ix%i.\n",
                        width, height
                    );
		    /* Deallocate resources before returning. */
                    if(icon_mask_pm != None)
                        XFreePixmap(dpy, icon_mask_pm);
                    if(icon_pm != None)
                        XFreePixmap(dpy, icon_pm);
                    return;
                }
            }


            /* Create icon window. */
	    xerrval = 0;
            icon_w = XCreateSimpleWindow(
                dpy, root,
                0, 0,           /* Coordinates. */
                width, height,  /* Size. */
                0,              /* Border width. */
                white_pix,      /* Border color. */
                black_pix       /* Background color. */
            );
	    if(xerrval || (icon_w == None))
	    {
		/* Error creating icon window. */
                /* Deallocate resources before returning. */
                if(icon_mask_pm != None)
                    XFreePixmap(dpy, icon_mask_pm);
                if(icon_pm != None)
                    XFreePixmap(dpy, icon_pm);
                return;
	    }
	    /* If icon mask is available mask the newly created Window
	     * to the shape of the icon mask.
	     */
	    xerrval = 0;
            if(icon_mask_pm != None)
                XShapeCombineMask(
                    dpy, icon_w,
                    ShapeBounding,
                    0, 0,
                    icon_mask_pm,
                    ShapeSet  
                );
	    if(xerrval && gw_debug)
		printf(GW_DEBUG_PREFIX
"XShapeCombineMask(): Cannot set shape of icon window 0x%.8x.\n",
		    (u_int32_t)icon_w
		);
        }
	else
	{
	    /* Could not load XPM file. */
	    fprintf(stderr, "%s: Cannot load.\n", icon_path);
	    return;
	}

        /* Allocate a WM hints structure. */
        wm_hints = XAllocWMHints();
        if(wm_hints != NULL)
        {
            /* Set up WM hints structure to set icon on the window. */
            wm_hints->flags = (IconPixmapHint | IconMaskHint |
                               IconWindowHint
            );
            wm_hints->initial_state = NormalState;
            wm_hints->icon_pixmap = icon_pm;
            wm_hints->icon_mask = icon_mask_pm;
            wm_hints->icon_window = icon_w;

	    xerrval = 0;
            XSetWMHints(dpy, w, wm_hints);
	    if(xerrval && gw_debug)
		printf(GW_DEBUG_PREFIX
"XSetWMHints(): Cannot set icon w=0x%.8x p=0x%.8x m=0x%.8x for window 0x%.8x.\n",
		    (u_int32_t)icon_w, (u_int32_t)icon_pm,
		    (u_int32_t)icon_mask_pm, (u_int32_t)w
		);

            XFree(wm_hints);
        }

        /* Set up text property for setting of icon name. */
	if(icon_name != NULL)
	{
	    XTextProperty text_prop;

	    text_prop.value = (unsigned char *)strdup(icon_name);
	    text_prop.encoding = XA_STRING;
	    text_prop.format = 8;
	    text_prop.nitems = strlen(icon_name);

	    /* Set icon name. */
	    xerrval = 0;
	    XSetWMIconName(dpy, w, &text_prop);

	    if(xerrval && gw_debug)
                printf(GW_DEBUG_PREFIX
"XSetWMIconName(): Cannot set name of icon window 0x%.8x for window 0x%.8x.\n",
                    (u_int32_t)icon_w, (u_int32_t)w
                );

	    free(text_prop.value);
	}


        return;
#else
        return;
#endif  /* XPM_H */
}
#endif	/* X_H */


#ifdef X_H
/*
 *	Creates a new Window.
 */
Window GWCreateWindow(
	gw_display_struct *display,
	Window parent,
	int x, int y,
	int width, int height,
	const char *title
)
{
	unsigned long swa_flags;
	XSizeHints *sz_hints;
	Window w = None;
        XSetWindowAttributes swa;


	if((display == NULL) ||
           (width == 0) ||
	   (height == 0)
	)
	    return(w);

	if((display->display == NULL) ||
           (display->visual_info == NULL)
	)
	    return(w);

	/* If given parent is None, use root window (desktop) as
	 * parent.
	 */
	if(parent == None)
	{
	    parent = display->root;

	    /* Root window not available? */
	    if(parent == None)
	    {
		if(gw_debug)
		    printf(GW_DEBUG_PREFIX
 "GWCreateWindow(): Root window not available as parent to create window.\n"
		    );
		return(w);
	    }
	}

	/* Set values that the new window is to have upon creation. */
	swa_flags = CWBorderPixel | CWColormap | CWEventMask;
        swa.colormap = display->colormap;
        swa.border_pixel = display->black_pix;
        swa.event_mask = StructureNotifyMask | ExposureMask | \
                         KeyPressMask | KeyReleaseMask |
                         ButtonPressMask | ButtonReleaseMask |
                         PointerMotionMask | VisibilityChangeMask;

	if(gw_debug)
	    printf(GW_DEBUG_PREFIX
 "Creating X window parented to 0x%.8x of geometry %ix%i%s%i%s%i at Depth %i bits\n",
 (unsigned int)parent,
 width, height,
 (x < 0) ? "" : "+", x,
 (y < 0) ? "" : "+", y,
 display->depth
	    );

	xerrval = 0;
	w = XCreateWindow(
            display->display,
            parent,
            x, y,
            width, height,
            0,
            display->depth,
            InputOutput,
            display->visual_info->visual,
            swa_flags,
            &swa
        );
	if(xerrval || (w == None))
	{
	    if(gw_debug)
		printf(GW_DEBUG_PREFIX
 "XCreateWindow(): Could not create Window.\n"
		);
	    return(w);
	}
	else
	{
	    if(gw_debug)
		printf(GW_DEBUG_PREFIX
		    "Created Window 0x%.8x.\n",
		    (unsigned int)w
		);
	}

	/* Set size hints. */
	sz_hints = XAllocSizeHints();
	if(sz_hints != NULL)
	{
	    /* Set size hint values. */
	    sz_hints->flags = USPosition | PSize | PBaseSize;
	    sz_hints->x = x;
	    sz_hints->y = y;
	    sz_hints->width = width;
	    sz_hints->height = height;
	    sz_hints->base_width = 0;
	    sz_hints->base_height = 0;

	    /* Set min size if our window is a toplevel window. */
	    if(parent == display->root)
	    {
		sz_hints->flags |= PMinSize;
		sz_hints->min_width = GW_MIN_TOPLEVEL_WIDTH;
		sz_hints->min_width = GW_MIN_TOPLEVEL_HEIGHT;
	    }

	    /* WM size hints for our window. */
	    xerrval = 0;
	    XSetWMNormalHints(display->display, w, sz_hints);
	    if(xerrval && gw_debug)
		printf(GW_DEBUG_PREFIX
"XSetWMNormalHints(): Cannot set WM size hints for window 0x%.8x.\n",
		    (u_int32_t)w
		);

	    XFree(sz_hints);
	    sz_hints = NULL;
	}

	/* Set title. */
	if(title != NULL)
	{
	    xerrval = 0;
	    XStoreName(display->display, w, title);
	    if(xerrval && gw_debug)
		printf(GW_DEBUG_PREFIX
"XStoreName(): Cannot set name for window 0x%.8x.\n",
		    (u_int32_t)w
		);
	}

	/* Is this new window parented to root (is it a toplevel
	 * window)?
	 */
	if(parent == display->root)
	{
	    /* Set atoms. */
	    Atom a[6];

	    a[0] = display->atom_wm_close_window;
	    a[1] = display->atom_wm_delete_window;
	    a[2] = display->atom_wm_ping;
	    a[3] = display->atom_wm_save_yourself;
	    a[4] = display->atom_wm_take_focus;
	    a[5] = display->atom_wm_workarea;

	    xerrval = 0;
	    XSetWMProtocols(
		display->display,
		w,			/* Window. */
		a,			/* Atoms array. */
		6			/* Number of atoms in a. */
	    );
	    if(xerrval && gw_debug)
		printf(GW_DEBUG_PREFIX
"XSetWMProtocols(): Cannot set atoms for window 0x%.8x.\n",
		    (u_int32_t)w
		);
	}

	return(w);
}
#endif	/* X_H */


#ifdef X_H
/*
 *	Initializes the graphics wrapper.
 */
gw_display_struct *GWInit(int argc, char **argv)
{
	Boolean direct_rendering = True;
	double aspect_offset = 0.0;
	int no_windows = 0;
	int i, status;
	int major_version, minor_version;
#ifdef XF86VIDMODE_H
	int vm_event_base, vm_error_base;
#endif	/* XF86VIDMODE_H */
	const char *cstrptr;
	XVisualInfo *vi;
	int x = 0, y = 0, off_screen_margin = 10;
	int width = 320, height = 240;
	gw_display_struct *display = NULL;
	const char *host = NULL;
	const char *title = "Untitled";
	const char *icon_path = NULL;
	const char *icon_name = NULL;
	const char *def_xfont_name =
	    "-adobe-helvetica-medium-r-normal-*-12-*-*-*-p-*-iso8859-1";


	/* Parse arguments. */
	for(i = 0; i < argc; i++)
	{
	    cstrptr = (const char *)argv[i];
	    if(cstrptr == NULL)
		continue;

	    /* Print GW debug messages to stdout? */
	    if(!strcasecmp(cstrptr, "--gw_debug") ||
               !strcasecmp(cstrptr, "-gw_debug")
	    )
	    {
		gw_debug = True;
	    }
	    /* Software rendering? */
	    else if(!strcasecmp(cstrptr, "--software") ||
                    !strcasecmp(cstrptr, "--software_rendering") ||
                    !strcasecmp(cstrptr, "--softwarerendering") ||
                    !strcasecmp(cstrptr, "-software") ||
                    !strcasecmp(cstrptr, "-software_rendering") ||
                    !strcasecmp(cstrptr, "-softwarerendering")
            )
            {
		direct_rendering = False;
	    }
            /* Hardware rendering? */ 
            else if(!strcasecmp(cstrptr, "--hardware") ||
                    !strcasecmp(cstrptr, "--hardware_rendering") ||
                    !strcasecmp(cstrptr, "--hardwarerendering") ||
                    !strcasecmp(cstrptr, "-hardware") ||
                    !strcasecmp(cstrptr, "-hardware_rendering") ||
                    !strcasecmp(cstrptr, "-hardwarerendering") ||
                    !strcasecmp(cstrptr, "--direct_rendering") ||
                    !strcasecmp(cstrptr, "--directrendering") ||
                    !strcasecmp(cstrptr, "-direct_rendering") ||
                    !strcasecmp(cstrptr, "-directrendering")
            )
            {
		direct_rendering = True;
	    }
	    /* Address to X server. */
	    else if(!strcasecmp(cstrptr, "--display") ||
                    !strcasecmp(cstrptr, "--dpy") ||
                    !strcasecmp(cstrptr, "-display") ||
                    !strcasecmp(cstrptr, "--dpy")
	    )
	    {
		i++;
		if(i < argc)
		{
		    host = (const char *)argv[i];
		}
		else
		{
		    fprintf(stderr,
			"%s: Requires argument.\n",
			argv[i - 1]
		    );
		}
	    }
	    /* Default XFont name. */
	    else if(!strcasecmp(cstrptr, "--font") ||
                    !strcasecmp(cstrptr, "--fn") ||
                    !strcasecmp(cstrptr, "-font") ||
                    !strcasecmp(cstrptr, "-fn")
            )
            {
                i++;
                if(i < argc)
                {
                    def_xfont_name = (const char *)argv[i];
                }
                else
                {
                    fprintf(stderr,
                        "%s: Requires argument.\n",
                        argv[i - 1]
                    );
                }
	    }
	    /* Title of toplevel window. */
	    else if(!strcasecmp(cstrptr, "--title") ||
                    !strcasecmp(cstrptr, "-title")
	    )
	    {
                i++;
                if(i < argc)
                {   
		    title = (const char *)argv[i];
		}
                else
                {
                    fprintf(stderr,
                        "%s: Requires argument.\n",
                        argv[i - 1]
                    );
                }
	    }
            /* Toplevel window's WM icon path. */
            else if(!strcasecmp(cstrptr, "--icon_path") ||
                    !strcasecmp(cstrptr, "-icon_path") ||
                    !strcasecmp(cstrptr, "--icon") ||
                    !strcasecmp(cstrptr, "-icon")
            )
            {
                i++;
                if(i < argc)
                {
                    icon_path = (const char *)argv[i];
                }
                else
                {
                    fprintf(stderr,
                        "%s: Requires argument.\n",
                        argv[i - 1]
                    );
                }
            }
            /* Toplevel window's WM icon name. */
            else if(!strcasecmp(cstrptr, "--icon_name") ||
                    !strcasecmp(cstrptr, "-icon_name")
            )
            {
                i++;
                if(i < argc)
                {
                    icon_name = (const char *)argv[i];
                }
                else
                {
                    fprintf(stderr,
                        "%s: Requires argument.\n",
                        argv[i - 1]
                    );
                }
            }
	    /* Position and size of toplevel window. */
	    else if(!strcasecmp(cstrptr, "--geometry") ||
                    !strcasecmp(cstrptr, "-geometry")
            )
            {
                i++;
                if(i < argc)
                {
		    int start_x = 0, start_y = 0;
		    unsigned int start_width = 320, start_height = 240;

		    if(argv[i] == NULL)
			continue;

                    XParseGeometry(
			argv[i],
			&start_x, &start_y,
			&start_width, &start_height
		    );

		    x = start_x;
		    y = start_y;
		    width = MAX((int)start_width, off_screen_margin);
		    height = MAX((int)start_height, off_screen_margin);
                }
                else
                {
                    fprintf(stderr,
                        "%s: Requires argument.\n",
                        argv[i - 1]
                    );
                }
	    }
	    /* Aspect offset. */
            else if(!strcasecmp(cstrptr, "--aspect_offset") ||
                    !strcasecmp(cstrptr, "-aspect_offset") ||
                    !strcasecmp(cstrptr, "--aspectoffset") ||
                    !strcasecmp(cstrptr, "-aspectoffset")
            )
	    {
                i++;
                if(i < argc)
                {
                    aspect_offset = atof(argv[i]);
                }
                else
                {
                    fprintf(stderr,
                        "%s: Requires argument.\n",
                        argv[i - 1]
                    );
                }
	    }
	    /* No window, this is a special argument to specify that no
	     * windows (or dialogs) be created.
	     */
            else if(!strcasecmp(cstrptr, "--no_windows") ||
                    !strcasecmp(cstrptr, "-no_windows") ||
                    !strcasecmp(cstrptr, "--nowindows") ||
		    !strcasecmp(cstrptr, "-nowindows")
            )
            {
		no_windows = 1;
	    }
	}

	/* Is GW debugging enabled? */
	if(gw_debug)
	    printf(GW_DEBUG_PREFIX
"Debugging enabled, messages will be sent to stdout.\n"
	    );


	/* Allocate a new graphics wrapper display structure. */
	display = (gw_display_struct *)calloc(
	    1,
	    sizeof(gw_display_struct)
	);
	if(display == NULL)
            return(display);


        /* Open connection to display. */
        status = -1;
        while(1)
        {
            /* First try given input host address to connect to. */
            display->display = XOpenDisplay(host);
            if(display->display != NULL)
            {
                status = 0;     /* Success. */
		if(gw_debug)
		    printf(GW_DEBUG_PREFIX
 "Connected to X server display: %s\n",
			host
		    );
                break;
            }

            /* Check if host address was NULL (unspecified). */
            if(host == NULL)
            {
                /* Try enviroment valriable. */
                display->display = XOpenDisplay(getenv("DISPLAY"));
                if(display->display != NULL)
                {
                    status = 0; /* Success. */
		    if(gw_debug)
			printf(GW_DEBUG_PREFIX
 "Connected to X server display (specified by `DISPLAY'): %s\n",
			    getenv("DISPLAY")
			);
                    break;
                }

                /* All else fails, try explicitly setting localhost. */
                display->display = XOpenDisplay("127.0.0.1:0.0");
                if(display->display != NULL)
                {
                    status = 0; /* Success. */
		    if(gw_debug)
			printf(GW_DEBUG_PREFIX
 "Connected to X server display (default local): %s\n",
			    "127.0.0.1:0.0"
			);
                    break;
                }
            }

            /* Failed. */
            break;
        }
        /* Connect failed? */
        if(status)
        {
	    if(gw_debug)
		printf(GW_DEBUG_PREFIX
 "Connection to X server failed, see stderr output for details.\n"
		);

            /* Print reason for failed connect. */
            fprintf(stderr, "Cannot connect to X server: ");
            if(getenv("DISPLAY") == NULL)
                fprintf(stderr, "`DISPLAY' environment variable not set.");
            else
                fprintf(stderr, "%s", XDisplayName(NULL));
            fprintf(stderr, "\n");

            /* Print verbose help. */
            fprintf(stderr,
"\nCheck to make sure that your X server is running and that your\n\
enviroment variable `DISPLAY' is set properly.\n"
            );

	    free(display);
            return(NULL);
        }

        /* Set X server error handler. */
        XSetErrorHandler(GWXErrorHandler);


	/* Check version numbers of the glX extension. */
	if(glXQueryVersion(display->display, &major_version, &minor_version))
	{
	    if(gw_debug)
		printf(GW_DEBUG_PREFIX
 "glX version: major=%i minor=%i\n",
		    major_version, minor_version
		);
	    /* Store glX version. */
	    display->glx_version_major = major_version;
	    display->glx_version_minor = minor_version;
	}
	else
	{
	    /* Could not get glX version, implying glX not available. */

	    if(gw_debug)
		printf(GW_DEBUG_PREFIX
 "glXQueryVersion(): Failed, implying glX is not available.\n"
		);

	    fprintf(stderr,
 "Warning: Unable to find glX, this is used for X and OpenGL communications.\n"
	    );
	}

        /* Get appropriate visual. */
	vi = NULL;
	display->alpha_channel_bits = 0;
	display->has_double_buffer = False;
	for(i = 0; i < 4; i++)
	{
	    switch(i)
	    {
	      case 0:	/* First try double buffer, rgba. */
		vi = glXChooseVisual(
		    display->display,
		    DefaultScreen(display->display),
		    attributeListDblRGBA
		);
		if(vi == NULL)
		{
		    if(gw_debug)
			printf(GW_DEBUG_PREFIX
 "glXChooseVisual(): RGBA double buffer X Visual not available.\n"
			);
		}
		break;

              case 1:	/* Second try single buffer, rgba. */
                vi = glXChooseVisual(
                    display->display,
                    DefaultScreen(display->display),
                    attributeListSglRGBA
                );
                if(vi == NULL)
                { 
                    if(gw_debug)
                        printf(GW_DEBUG_PREFIX
 "glXChooseVisual(): RGBA single buffer X Visual not available.\n"
                        );
                }
                break;

              case 2:	/* Third try double buffer, rgb. */
                vi = glXChooseVisual(
                    display->display,
                    DefaultScreen(display->display),
                    attributeListDblRGB
                );
                if(vi == NULL)
                { 
                    if(gw_debug)
                        printf(GW_DEBUG_PREFIX
 "glXChooseVisual(): RGB double buffer X Visual not available.\n"
                        );
                }
                break;

              case 3:	/* Fourth try single buffer, rgb. */
                vi = glXChooseVisual(
                    display->display,
                    DefaultScreen(display->display),
                    attributeListSglRGB
                );
                if(vi == NULL)
                {
                    if(gw_debug)
                        printf(GW_DEBUG_PREFIX
 "glXChooseVisual(): RGB single buffer X Visual not available.\n"
                        );
                }
                break;
	    }
	    if(vi != NULL)
		break;
	}
	/* Record matched XVisualInfo from above, even if it is NULL. */
        display->visual_info = vi;

	/* Get values from matched visual. */
	if(vi != NULL)
	{
	    int vi_val;

	    if(!glXGetConfig(
		display->display, vi,
		GLX_DOUBLEBUFFER, &vi_val
	    ))
		display->has_double_buffer = (vi_val ? True : False);

	    if(!glXGetConfig(
		display->display, vi,
		GLX_ALPHA_SIZE, &vi_val
	    ))
		display->alpha_channel_bits = vi_val;
	}

	/* Failed to match visual? */
	if(vi == NULL)
	{
	    fprintf(
		stderr,
		"Unable to find useable X Visual.\n"
	    );
	    GWShutdown(display);
	    return(NULL);
	}


	/* Query X extensions. */
	/* Shape. */
	if(XShapeQueryVersion(
	    display->display, &major_version, &minor_version
	))
	{
           if(gw_debug)
                printf(GW_DEBUG_PREFIX
 "Shape extension version: major=%i minor=%i\n",
		    major_version, minor_version
		);
	}
	else
	{
            if(gw_debug)
                printf(GW_DEBUG_PREFIX
 "XShapeQueryVersion(): Shape extension not available.\n"
                );
	}

	/* Screen saver. */
/*
	Status XScreenSaverQueryVersion(
    Display *dpy,
    int *major,       
    int *minor
	);
 */

	/* XFree86 VidMode (proprietery to XFree86). */
#ifdef XF86VIDMODE_H
	if(XF86VidModeQueryExtension(
	    display->display, &vm_event_base, &vm_error_base
	))
	{
	    if(XF86VidModeQueryVersion(
		display->display, &major_version, &minor_version
	    ))
	    {
		if(gw_debug)
		    printf(GW_DEBUG_PREFIX
 "XF86VidMode extension version: major=%i minor=%i\n",
			major_version, minor_version
		    );
	    }
	    if(gw_debug)
	    {
		XF86VidModeModeInfo **mode_line, *mode_line_ptr;
		int total_mode_lines;

		if(XF86VidModeGetAllModeLines(
		    display->display,
		    DefaultScreen(display->display),
		    &total_mode_lines, &mode_line
		))
		{
		    for(i = 0; i < total_mode_lines; i++)
		    {
			mode_line_ptr = mode_line[i];
			if(mode_line_ptr == NULL)
			    continue;

/* Need to work on printing these sync rates. */

			/* Do not free each structure. */
/*			XFree(mode_line_ptr); */
		    }
		    XFree(mode_line);
		}
	    }
	    display->has_vidmode_ext = True;
	    display->vidmode_ext_event_offset = vm_event_base;
	}
	else
	{
	    /* Not available. */
	    if(gw_debug)
		printf(GW_DEBUG_PREFIX
 "XF86VidModeQueryExtension(): XF86 VidMode extension not available.\n"
		);
	    display->has_vidmode_ext = False;
	    display->vidmode_ext_event_offset = 0;
	}
#else
	display->has_vidmode_ext = False;
	display->vidmode_ext_event_offset = 0;
#endif	/* XF86VIDMODE_H */


	/* Get other default values. */
	display->aspect_offset = aspect_offset;
	display->direct_rendering = direct_rendering;
	display->white_pix = WhitePixel(display->display, vi->screen);
        display->black_pix = BlackPixel(display->display, vi->screen);
	display->depth = vi->depth;
        if(gw_debug)
            printf(GW_DEBUG_PREFIX
 "Using X Depth %i bits.\n", display->depth
                );                  

        display->gc = DefaultGC(display->display, vi->screen);
        display->current_font = NULL;
/* We don't use X fonts to draw strings, so don't set any of this.
 * Farther below, the default xfont name will be coppied to the display
 * structure.
 */
/*	 = XLoadFont(display->display, def_xfont_name); */
/*	XSetFont(display->display, display->gc, font); */

	XSetForeground(display->display, display->gc, display->black_pix);
	display->current_color = NULL;
	display->current_font = gw_std_font;
	display->atom_wm_motif_all_clients = XInternAtom(
            display->display,
            "_MOTIF_WM_ALL_CLIENTS", 
            False
        );
        display->atom_wm_motif_hints = XInternAtom(
            display->display,
            "_MOTIF_WM_HINTS",
            False
        );
        display->atom_wm_motif_info = XInternAtom(
            display->display,
            "_MOTIF_WM_INFO",
            False
        );
        display->atom_wm_motif_menu = XInternAtom(
            display->display,
            "_MOTIF_WM_MENU",
            False
        );
        display->atom_wm_motif_messages = XInternAtom(
            display->display,
            "_MOTIF_WM_MESSAGES",
            False
        );
        display->atom_wm_motif_offset = XInternAtom(
            display->display,
            "_MOTIF_WM_OFFSET",
            False
        );
        display->atom_wm_motif_query = XInternAtom(
            display->display,
            "_MOTIF_WM_QUERY",
            False
        );
        display->atom_wm_close_window = XInternAtom(
            display->display,
            "_NET_CLOSE_WINDOW",
            False
        );
        display->atom_wm_delete_window = XInternAtom(
            display->display,
            "WM_DELETE_WINDOW", 
            False
        );
	display->atom_wm_ping = XInternAtom(
            display->display,
	    "_NET_WM_PING",
	    False
	);
        display->atom_wm_save_yourself = XInternAtom(
            display->display,
            "WM_SAVE_YOURSELF", 
            False
        );
        display->atom_wm_state = XInternAtom(
            display->display,
            "WM_STATE",
            False
        );
        display->atom_wm_take_focus = XInternAtom(
            display->display,
            "WM_TAKE_FOCUS",
            False
        );
	display->atom_wm_workarea = XInternAtom(
            display->display,
            "_NET_WORKAREA",
            False
        );


	display->root = RootWindow(display->display, vi->screen);
	display->root_width = DisplayWidth(display->display, vi->screen);
        display->root_height = DisplayHeight(display->display, vi->screen);
	if(display->root == None)
	{
	    if(gw_debug)
		printf(GW_DEBUG_PREFIX
 "Cannot find root window (the desktop).\n"
		);
	}


	/* Sanitize x, y, width, and height based on root window size. */
	if((x + width) < off_screen_margin)
	    x = off_screen_margin - width;
	if((y + height) < off_screen_margin)
	    y = off_screen_margin - height;

	if(x > (display->root_width - off_screen_margin))
	    x = display->root_width - off_screen_margin;
        if(y > (display->root_height - off_screen_margin))
            y = display->root_height - off_screen_margin;


        /* Create glX rendering context */
        display->glx_context = glXCreateContext(
	    display->display,
	    vi,			/* Visual info. */
	    NULL,		/* Share list (none). */
	    (display->direct_rendering) ? GL_TRUE : GL_FALSE	/* Direct rendering? */
	);
	if(display->glx_context == NULL)
	{
	    if(gw_debug)
		printf(GW_DEBUG_PREFIX
 "glXCreateContext(): Cannot create glX %s rendering context.\n",
		    (display->direct_rendering) ? "hardware" : "software"
		);

	    fprintf(
		stderr,
		"Error creating glX %s rendering context.\n",
		(display->direct_rendering) ? "hardware" : "software"
	    );
	    if(display->direct_rendering)
		fprintf(
		    stderr,
 "Try software rendering instead (add argument \"--software\")?\n"
		);
	}
	else
	{
	    if(gw_debug)
                printf(GW_DEBUG_PREFIX
 "glXCreateContext(): Created glX %s rendering context.\n",
                    (display->direct_rendering) ? "hardware" : "software"
                );
	}

        /* Create a colormap. */
	xerrval = 0;
        display->colormap = XCreateColormap(
	    display->display,
            display->root,
            vi->visual, AllocNone
        );
	if(xerrval || (display->colormap == None))
	{
	    if(gw_debug)
		printf(GW_DEBUG_PREFIX
 "XCreateColormap(): Cannot create X colormap.\n"
		);
	}

	/* Create toplevel window and buffer. */
	if(!no_windows)
	{
	    display->x = x;
	    display->y = y;
	    display->width = width;
	    display->height = height;
	    display->toplevel = GWCreateWindow(
		display,
		display->root,
		x, y,
		width, height,
		title
	    );
	    if(display->toplevel == None)
	    {
		fprintf(stderr, "Cannot create toplevel window.\n");
		free(display);
		return(NULL);
	    }
	    else
	    {
		/* Set icon and icon name. */
		GWSetWindowIconFile(display, icon_path, icon_name);

		/* Map newly created toplevel window. */
		xerrval = 0;
		XMapRaised(display->display, display->toplevel);
		if(xerrval && gw_debug)
		    printf(GW_DEBUG_PREFIX
"XMapRaised(): Cannot map and raise window 0x%.8x.\n",
			(u_int32_t)display->toplevel
		    );

		/* Have GL wait untill X has caught up with Window
		 * creation.
		 */
		glXWaitX();
	    }

	    /* Set the glX context with the X Window we just
	     * created.
	     */
	    glXMakeCurrent(
		display->display,
		display->toplevel,
		display->glx_context
	    );
	    glXWaitGL();
	}

	/* At this point we have the glX context created and set to our
	 * main toplevel window. Now we can get the gl version.
	 */
	cstrptr = (const char *)glGetString(GL_VERSION);
	if(cstrptr != NULL)
	{
	    char *gl_vs_ptr;
	    char *gl_vs = strdup(cstrptr);
	    if(gl_vs != NULL)
	    {
		/* Deliminate at space which separates version and vendor
		 * name.
		 */
		gl_vs_ptr = strchr(gl_vs, ' ');
		if(gl_vs_ptr != NULL)
		    (*gl_vs_ptr) = '\0';

		gl_vs_ptr = gl_vs;
		if(gl_vs_ptr != NULL)
		{
		    display->gl_version_major = atoi(gl_vs_ptr);
		    gl_vs_ptr = strchr(gl_vs_ptr, '.');
		    if(gl_vs_ptr != NULL)
			gl_vs_ptr++;
		}
                if(gl_vs_ptr != NULL)
                {
                    display->gl_version_minor = atoi(gl_vs_ptr);
                    gl_vs_ptr = strchr(gl_vs_ptr, '.');
                    if(gl_vs_ptr != NULL)
                        gl_vs_ptr++;
                }

		free(gl_vs);
	    }

            if(gw_debug)
                printf(GW_DEBUG_PREFIX
 "OpenGL implementation version: major=%i minor=%i\n",
		    display->gl_version_major, display->gl_version_minor
                );
	}
	else
	{
	    if(gw_debug)
		printf(GW_DEBUG_PREFIX
"glGetString(GL_VERSION): Returned NULL, cannot get OpenGL implementation version.\n"
		);
	    fprintf(stderr, "Cannot obtain OpenGL implementation version.\n");
	}

	/* Check if the default (probably fancy) XFont exists. */
	if(def_xfont_name != NULL)
	{
	    XFontStruct *xfont = XLoadQueryFont(
		display->display, def_xfont_name
	    );
	    if(xfont == NULL)
	    {
		/* Fall back to something safe. */
		def_xfont_name =
		    "-*-clean-*-*-*-*-*-*-*-*-*-*-*-*";
	    }
	    else
	    {
		XFreeFont(display->display, xfont);
	    }
	    /* Set default XFont name. */
	    display->def_xfont_name = strdup(def_xfont_name);
	}

	/* Turn on keyboard autorepeat. */
	GWKeyboardAutoRepeat(display, True);

        /* Reset all other OpenGL states on record. */
        StateGLResetAll(&display->state_gl);

	/* Set pixel storage format (for rastering font bitmaps). */
        glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
	glPixelStorei(GL_PACK_ALIGNMENT, 1);

	/* Set other GL defaults that we want. Some of these might default
	 * to on but we want them off to improve performance.
	 */
	StateGLDisable(&display->state_gl, GL_DITHER);


	/* Load cursors. */
	display->cursor_standard = None;
	xerrval = 0;
	display->cursor_busy = XCreateFontCursor(
	    display->display,
	    XC_watch
	);
	xerrval = 0;
	display->cursor_translate = XCreateFontCursor(
            display->display,
            XC_fleur
        );
	display->cursor_zoom = None;

	/* Initialize our own dialog widgets. */
	if(!no_windows)
	{
	    GWXDialogCreate(
	        display, &display->mesg_dialog,
		GWX_DIALOG_TYPE_MESG
	    );
	    GWXDialogCreate(
		display, &display->conf_dialog,
		GWX_DIALOG_TYPE_CONF
	    );
	}

        if(gw_debug)
            printf(GW_DEBUG_PREFIX
"Initialization done.\n"
	    );

	return(display);
}
#endif  /* X_H */

#ifdef X_H
/*
 *	Graphics wrapper management function, called once per loop.
 */
void GWManage(gw_display_struct *display)
{
	XEvent event;
	XClientMessageEvent *cm;
	Window w;
	int key, events_handled;


        if(display == NULL)
            return;

	if(display->display != NULL)
	{
	    /* Fetch events. */
	    while(
	XPending(display->display)
/*	XEventsQueued(display->display, QueuedAlready) > 0 */
/*	XEventsQueued(display->display, QueuedAfterFlush) > 0 */
	    )
	    {
		/* Get event. */
		xerrval = 0;
		XNextEvent(display->display, &event);
		if(xerrval)
		    break;

		/* Get window id from event structure. */
		w = event.xany.window;

		events_handled = 0;

		/* Handle event by type. */
		switch(event.type)
		{
		  case Expose:
		    if(display->toplevel == w)
                    {
		        if(display->func_draw != NULL)
			    display->func_draw(display->func_draw_data);
			events_handled++;
		    }
		    break;

		  case ConfigureNotify:
		    if(display->toplevel == w)
		    {
			glXWaitGL();
			glXWaitX();

			display->x = event.xconfigure.x;
			display->y = event.xconfigure.y;
                        display->width = (int)event.xconfigure.width;
                        display->height = (int)event.xconfigure.height;

			/* Call resize callback. */
		        if(display->func_resize != NULL)
                            display->func_resize(
			        display->func_resize_data,
				display->x, display->y,
			        display->width,
			        display->height
			    );
                        events_handled++;
		    }
                    break;

		  case KeyPress:
		    key = GWGetCharacterFromKeyCode(
                        display,
                        event.xkey.keycode
                    );
		    switch(key)
		    {
		      case GWKeyAlt:
			display->alt_key_state = True;
			break;

                      case GWKeyCtrl:
                        display->ctrl_key_state = True;
                        break;

                      case GWKeyShift:
                        display->shift_key_state = True;
                        break;

		      default:
			break;
		    }
                    if(display->func_keyboard != NULL)
                        display->func_keyboard(
			    display->func_keyboard_data,
			    key,
			    True,
			    event.xkey.time
			);
                    events_handled++;
		    break;

                  case KeyRelease:
                    key = GWGetCharacterFromKeyCode(
                        display,
                        event.xkey.keycode
                    );
                    switch(key)
                    {
                      case GWKeyAlt:
                        display->alt_key_state = False;
                        break;

                      case GWKeyCtrl:
                        display->ctrl_key_state = False;
                        break;

                      case GWKeyShift:
                        display->shift_key_state = False;
                        break;
                    }
                    if(display->func_keyboard != NULL)
                        display->func_keyboard(
                            display->func_keyboard_data,
			    key,
			    False,
			    event.xkey.time
                        );
                    events_handled++;
                    break;

                  case ButtonPress:
                    if((display->func_pointer != NULL) &&
                       (display->toplevel == w)
		    )
		    {
			int btn_num = 0;

			switch(event.xbutton.button)
			{
			  case Button1:
			    btn_num = 1;
			    break;

                          case Button2:
                            btn_num = 2;
                            break;

                          case Button3:
                            btn_num = 3;
                            break;

                          case Button4:
                            btn_num = 4;
                            break;

                          case Button5:
                            btn_num = 5;
                            break;
			}
                        display->func_pointer(
                            display->func_pointer_data,
			    event.xbutton.x,
                            event.xbutton.y,
                            GWEventTypeButtonPress,
			    btn_num,
			    event.xbutton.time
                        );
                        events_handled++;
		    }
		    break;

                  case ButtonRelease:
                    if((display->func_pointer != NULL) &&
                       (display->toplevel == w)
                    )
                    {
                        int btn_num = 0;

                        switch(event.xbutton.button)
                        {
                          case Button1:
                            btn_num = 1;
                            break;

                          case Button2:
                            btn_num = 2;
                            break;

                          case Button3:
                            btn_num = 3;
                            break;

                          case Button4:
                            btn_num = 4;
                            break;

                          case Button5:
                            btn_num = 5;
                            break;
                        }
                        display->func_pointer(
                            display->func_pointer_data,
                            event.xbutton.x,
                            event.xbutton.y,
                            GWEventTypeButtonRelease,
			    btn_num,
			    event.xbutton.time
                        );
                        events_handled++;
                    }
                    break;

                  case MotionNotify:
                    if((display->func_pointer != NULL) &&
                       (display->toplevel == w)
		    )
		    {
                        display->func_pointer(
                            display->func_pointer_data,
                            event.xmotion.x,
                            event.xmotion.y,
                            GWEventTypePointerMotion,
			    0,
			    event.xmotion.time
                        );
                        events_handled++;
		    }
		    break;

		  case VisibilityNotify:
		    if((display->func_visibility != NULL) &&
                       (display->toplevel == w)
		    )
		    {
                        glXWaitGL();
                        glXWaitX();

			switch(event.xvisibility.state)
			{
			  case VisibilityFullyObscured:
			    display->func_visibility(
				display->func_visibility_data,
				GWVisibilityFullyObscured
			    );
			    break;

			  case VisibilityPartiallyObscured:
                            display->func_visibility(
                                display->func_visibility_data,
                                GWVisibilityPartiallyObscured
                            );
                            break;

			  default:
                            display->func_visibility(
                                display->func_visibility_data,
                                GWVisibilityUnobscured
                            );
                            break;
			}
		    }
		    break;

		  case ClientMessage:
		    cm = &event.xclient;

		    if(gw_debug && (cm->format == 32))
		    {
			char *atom_name = XGetAtomName(
			    display->display,
			    cm->message_type
			);
			printf(GW_DEBUG_PREFIX
"Got client message: %s for window: 0x%.8x\n",
			    atom_name,
			    (unsigned int)cm->window
			);
			XFree(atom_name);
		    }

		    /* Note, NEWER client messages have member
		     * message_type set to the atom to determine the type
		     * of message. While OLDER client messages use member
		     * data.l[0] to determine message type and 
		     * message_type is always set to WM_PROTOCOLS.
		     */

		    /* WM close (from WM_DELETE_WINDOW). */
		    if((cm->format == 32) &&
                       (cm->data.l[0] == display->atom_wm_delete_window) &&
                       (cm->window == display->toplevel)
		    )
		    {
                        glXWaitGL();
                        glXWaitX();

			/* Call close callback. */
			if(display->func_close != NULL)
			    display->func_close(
				display->func_close_data,
				(void *)w
			    );
		    }
                    /* WM close (from _NET_CLOSE_WINDOW). */
                    else if((cm->format == 32) &&
                            (cm->message_type == display->atom_wm_close_window) &&
                            (cm->window == display->toplevel)
                    )
                    {
                        glXWaitGL();
                        glXWaitX();

                        /* Call close callback. */
                        if(display->func_close != NULL)
                            display->func_close(
                                display->func_close_data,
                                (void *)w
                            );
                    }
                    /* WM ping. */
		    else if((cm->format == 32) &&
                            (cm->data.l[0] == display->atom_wm_ping) &&
                            (cm->window == display->toplevel)
                    )
                    {
/*
			long timestamp = cm->data.l[1];
 */

			/* Respond to ping. */
			event.xany.window = display->root;
			event.xclient.window = display->root;
			XSendEvent(
			    display->display,
			    display->root,
			    False,
			    SubstructureNotifyMask | SubstructureRedirectMask,
			    &event
			);
                    }
		    /* Save yourself. */
		    else if((cm->format == 32) &&
                            (cm->data.l[0] == display->atom_wm_save_yourself) &&
                            (cm->window == display->toplevel)
                    )
		    {
                        glXWaitGL();
                        glXWaitX();

			/* Call save yourself callback. */
			if(display->func_save_yourself != NULL)
			{
			    display->func_save_yourself(
				display->func_save_yourself_data
			    );
			}
		    }
		    /* Take focus. */
                    else if((cm->format == 32) &&
                            (cm->data.l[0] == display->atom_wm_take_focus) &&
                            (cm->window == display->toplevel)
                    )
		    {
/* Don't need to rebind the GL context.
			glXWaitX();
			glXMakeCurrent(
			    display->display,
			    display->toplevel,
			    display->glx_context
			);
			glXWaitGL();
 */
		    }
                    /* Work area (viewport, from _NET_WORKAREA). */
                    else if((cm->format == 32) &&
                            (cm->message_type == display->atom_wm_workarea) &&
                            (cm->window == display->toplevel)
                    )
                    {
			display->viewport_x = (int)cm->data.l[0];
                        display->viewport_y = (int)cm->data.l[1];
                        display->viewport_width = (int)cm->data.l[2];
                        display->viewport_height = (int)cm->data.l[3];
                    }
		    break;

		  default:
		    /* Some extension type event. */
#ifdef XF86VIDMODE_H
		    /* XF86 VidMode change? */
		    if(display->has_vidmode_ext)
		    {
#ifdef XF86VidModeNotify
			if(event.type == (display->vidmode_ext_event_offset +
			    XF86VidModeNotify)
			)
			{
			    if(gw_debug)
				printf(GW_DEBUG_PREFIX
"Got event %i XF86VidModeNotify\n", event.type
				);
			}
#endif	/* XF86VidModeNotify */
#ifdef XF86VidModeModeChange
                        if(event.type == (display->vidmode_ext_event_offset +   
                            XF86VidModeModeChange)
                        )
                        {

			    if(gw_debug)
                                printf(GW_DEBUG_PREFIX
"Got event %i XF86VidModeModeChange\n", event.type
				);
                        }
#endif	/* XF86VidModeModeChange */
		    }
#endif	/* XF86VIDMODE_H */
		    break;
		}

		/* Hand event message off to dialog management functions. */
		GWXDialogManage(
		    display, &display->mesg_dialog, &event
		);
                GWXDialogManage(
                    display, &display->conf_dialog, &event
                );

	    }
	}


	/* Always call the timeout function once per call. */
	if(display->func_timeout != NULL)
	    display->func_timeout(display->func_timeout_data);

	return;
}
#endif	/* X_H */

#ifdef X_H
/*
 *	Graphics wrapper shutdown.
 */
void GWShutdown(gw_display_struct *display)
{
	Display *dpy;

        if(display == NULL)
            return;

	if(gw_debug)
	    printf(GW_DEBUG_PREFIX
"Beginning shutdown...\n"
	    );

	/* Get pointer to X display structure. */
	dpy = display->display;
	if(dpy != NULL)
	{
	    Cursor *cursor;
	    Window w;

	    /* Destroy dialog widgets and resources. */
	    GWXDialogDestroy(
		display, &display->mesg_dialog
	    );
            GWXDialogDestroy(
                display, &display->conf_dialog
            );


	    /* Begin unreferancing GL context. */
	    if(gw_debug)
		printf(GW_DEBUG_PREFIX
"Waiting for X operations to complete and unreferancing GL context...\n"
		);

	    /* Have GL wait for above X resource destroying to finish. */ 
            glXWaitX();

	    /* Unreferance current GL context. */
            glXMakeCurrent(dpy, None, NULL);

	    /* Wait for GL to unreferance the current context. */
	    glXWaitGL();


	    /* Destroy toplevel Window. */
	    w = display->toplevel;
	    if(w != None)
	    {
                XWMHints *wm_hints;

		/* Get WM hints of window. */
		xerrval = 0;
		wm_hints = XGetWMHints(dpy, w);
                if(!xerrval && (wm_hints != NULL))
                {
		    /* Deallocate resources of window specified by the WM
		     * hints.
		     */
                    if(wm_hints->flags & IconWindowHint)
                        if(wm_hints->icon_window != None)
                            XDestroyWindow(dpy, wm_hints->icon_window);
                    if(wm_hints->flags & IconPixmapHint)
                        if(wm_hints->icon_pixmap != None)
                            XFreePixmap(dpy, wm_hints->icon_pixmap);
                    if(wm_hints->flags & IconMaskHint)
                        if(wm_hints->icon_mask != None)
                            XFreePixmap(dpy, wm_hints->icon_mask);

		    /* Deallocate recieved WM hints structure. */
		    XFree(wm_hints);
		}

		/* Destroy the toplevel Window. */
		xerrval = 0;
		XDestroyWindow(dpy, w);
		if(xerrval && gw_debug)
		    printf(GW_DEBUG_PREFIX
"XDestroyWindow(): Error destroying window 0x%.8x.\n",
			(u_int32_t)w
		    );
		else if(gw_debug)
		    printf(GW_DEBUG_PREFIX
"Destroyed window 0x%.8x.\n",
                        (u_int32_t)w
                    );
                display->toplevel = None;
		w = None;
	    }

	    /* Wait for X to destroy the toplevel Window and then
	     * destroy the GLX context.
	     */
	    if(gw_debug)
		printf(GW_DEBUG_PREFIX
"Waiting for X operations to complete and destroying glX context.\n"
		);

            glXWaitX();
	    if(display->glx_context != NULL)
	    {
		glXDestroyContext(dpy, display->glx_context);
		if(gw_debug)
                    printf(GW_DEBUG_PREFIX
"Destroyed glX context 0x%.8x.\n",
			(u_int32_t)display->glx_context
		    );
		display->glx_context = NULL;
	    }

	    /* Wait for GL to destroy the context. */
            glXWaitGL();


	    /* Destroy cursors. */
#define DO_DESTROY_CURSOR	\
if(cursor != NULL) \
{ \
 if((*cursor) != None) \
 { \
  xerrval = 0; \
  XFreeCursor(dpy, *cursor); \
  if(xerrval && gw_debug) \
   printf(GW_DEBUG_PREFIX \
"XFreeCursor(): Error destroying Cursor 0x%.8x.\n", \
    (u_int32_t)(*cursor) \
   ); \
  (*cursor) = None; \
 } \
}
            cursor = &display->cursor_standard;
            DO_DESTROY_CURSOR
	    cursor = &display->cursor_busy;
	    DO_DESTROY_CURSOR
            cursor = &display->cursor_translate;
            DO_DESTROY_CURSOR
            cursor = &display->cursor_zoom;
            DO_DESTROY_CURSOR

#undef DO_DESTROY_CURSOR

            /* Turn on keyboard autorepeat. */
            GWKeyboardAutoRepeat(display, True);

            if(gw_debug)
                printf(GW_DEBUG_PREFIX
"Closing X display...\n" 
                );

	    /* Destroy XVisualInfo structure. */
	    if(display->visual_info != NULL)
	    {
		XFree(display->visual_info);
		display->visual_info = NULL;
	    }

	    /* Close connection to X server. */
	    XCloseDisplay(dpy);
	    display->display = NULL;
	    dpy = NULL;
	}

	/* Deallocate other display structure resources. */
	free(display->def_xfont_name);

	/* Deallocate graphics wrapper display structure itself. */
	free(display);

        if(gw_debug)
            printf(GW_DEBUG_PREFIX
"Shutdown done.\n"
	    );

	return;
}
#endif  /* X_H */

#ifdef X_H
/*
 *	Output message.
 */
void GWOutputMessage(
        gw_display_struct *display,
        int type,       /* One of GWOutputMessageType*. */
        const char *subject,
        const char *message,
        const char *help_details
)
{
	gwx_dialog_struct *dialog;


	if(display == NULL)
	    return;

	dialog = &display->mesg_dialog;

	switch(type)
	{
	  case GWOutputMessageTypeWarning:
	    GWXDialogLoadIcon(
		display, dialog,
		GWX_ICON_WARNING
	    );
	    break;

          case GWOutputMessageTypeError:
            GWXDialogLoadIcon(
                display, dialog,
                GWX_ICON_ERROR
            );
            break;

          case GWOutputMessageTypeQuestion:
            GWXDialogLoadIcon(
                display, dialog,
                GWX_ICON_QUESTION
            );
            break;

          default:
            GWXDialogLoadIcon(
                display, dialog,
                GWX_ICON_INFO
            );
            break;
	}

        GWXDialogSetMesg(
            display, dialog,
            subject, message, help_details
        );
	GWXDialogMap(display, dialog);

	return;
}
#endif  /* X_H */

#ifdef X_H
/*
 *	Blocks client execution untill confirmation is returned.
 *	Returns one of GWConfirmation*.
 *
 *	If this function is called while already blocking then
 *	GWConfirmationNotAvailable will be returned immediatly.
 */
int GWConfirmation(
	gw_display_struct *display,
        int type,       /* One of GWOutputMessageType*. */
	const char *subject,
        const char *message,
        const char *help_details
)
{
	static Boolean reenterant = False;
	int status;
        gwx_dialog_struct *dialog;


	if(reenterant)
	    return(GWConfirmationNotAvailable);
	else
	    reenterant = True;

	/* Is display valid? */
        if(display == NULL)
	{
	    /* No display, so return with GWConfirmationNotAvailable. */
	    reenterant = False;
            return(GWConfirmationNotAvailable);
	}
	else
	{
	    /* Get pointer to dialog. */
	    dialog = &display->conf_dialog;
	}

	/* Load icon. */
	switch(type)
	{
          case GWOutputMessageTypeWarning:
            GWXDialogLoadIcon(
                display, dialog,
                GWX_ICON_WARNING
            );
            break;

          case GWOutputMessageTypeError:
            GWXDialogLoadIcon(
                display, dialog,
                GWX_ICON_ERROR
            );
            break; 

	  case GWOutputMessageTypeQuestion:
	    GWXDialogLoadIcon(
                display, dialog,
                GWX_ICON_QUESTION
            );
            break;

          default:
            GWXDialogLoadIcon(
                display, dialog,
                GWX_ICON_INFO
            );
            break;
        }

	/* Set message. */
        GWXDialogSetMesg(
            display, dialog,
            subject, message, help_details
        );
        GWXDialogMap(display, dialog);

	/* Block untill confirmation is recieved. */
	status = GWXDoBlockUntilConf(display);

	/* Unmap confirmation dialog. */
	GWXDialogUnmap(display, dialog);

	reenterant = False;
	return(status);
}
#endif	/* X_H */

#ifdef X_H
/*
 *      Block client execution untill confirmation is returned.
 *      Returns one of GWConfirmation*.
 */
int GWConfirmationSimple(
        gw_display_struct *display, const char *message 
)
{
	return(GWConfirmation(
            display,
	    GWOutputMessageTypeQuestion,
	    "Confirmation",
	    message,
	    NULL
	));
}
#endif	/* X_H */

#ifdef X_H
/*
 *	Sets redraw (expose) callback function:
 */
void GWSetDrawCB(
        gw_display_struct *display,
        void (*func)(void *),
        void *data
)
{
	if(display == NULL)
	    return;

	display->func_draw = func;
	display->func_draw_data = data;

	return;
}

/*
 *	Sets resize (configure notify) callback function:
 */
void GWSetResizeCB(
        gw_display_struct *display, 
        void (*func)(void *, int, int, int, int),
        void *data
)
{
        if(display == NULL)
            return;

        display->func_resize = func;
	display->func_resize_data = data;

        return;
}

/*
 *	Sets keyboard event callback function:
 */
void GWSetKeyboardCB(
        gw_display_struct *display,
        void (*func)(void *, int, Boolean, long),
        void *data
)
{
        if(display == NULL)
            return;

        display->func_keyboard = func;
	display->func_keyboard_data = data;

        return;
}

/*
 *	Sets visibility notify callback function:
 */
void GWSetVisibilityCB(
        gw_display_struct *display,
        void (*func)(void *, int),
	void *data
)
{
        if(display == NULL)
            return;

        display->func_visibility = func;
	display->func_visibility_data = data;

        return;
}

/*
 *	Sets pointer event callback function:
 */
void GWSetPointerCB(
        gw_display_struct *display,
        void (*func)(void *, int, int, int, int, long),
        void *data 
)
{
        if(display == NULL)
            return;
 
        display->func_pointer = func;
        display->func_pointer_data = data;

	return;
}

/*
 *	Sets save yourself callback function:
 */
void GWSetSaveYourselfCB(
        gw_display_struct *display,
        void (*func)(void *),
        void *data
)
{
        if(display == NULL)
            return;

        display->func_save_yourself = func;
        display->func_save_yourself_data = data;

	return;
}

/*
 *	Sets window close (delete event) callback function:
 */
void GWSetCloseCB(
        gw_display_struct *display,
        void (*func)(void *, void *),
        void *data
)
{
        if(display == NULL)
            return;

        display->func_close = func;
	display->func_close_data = data;

        return;
}

/*
 *	Sets timeout (idle) callback function:
 */
void GWSetTimeoutCB(
        gw_display_struct *display,
        void (*func)(void *),
	void *data
)
{
        if(display == NULL)
            return;

        display->func_timeout = func;
	display->func_timeout_data = data;

        return;
}
#endif	/* X_H */

#ifdef X_H
/*
 *	Sends a (synthetic) redraw event:
 */
void GWPostRedraw(gw_display_struct *display)
{
	if(display == NULL)
	    return;

	/* Call redraw function. */
	if(display->func_draw != NULL)
	    display->func_draw(display->func_draw_data);

	return;
}
#endif	/* X_H */

#ifdef X_H
/*
 *	Swaps buffers (if using double buffers), all GL operations are
 *	gauranteed to be completed after this call.
 */
void GWSwapBuffer(gw_display_struct *display)
{
	if(display == NULL)
	    return;

	if(display->display == NULL)
	    return;

	if(display->has_double_buffer)
	    glXSwapBuffers(display->display, display->toplevel);

	return;
}
#endif  /* X_H */
  
#ifdef X_H
/*
 *	Turns keyboard auto repeat on/off.
 */
void GWKeyboardAutoRepeat(gw_display_struct *display, Boolean b)
{
	if(display == NULL)
	    return;
	if(display->display == NULL)
	    return;

	xerrval = 0;
	if(b)
	    XAutoRepeatOn(display->display);
	else
	    XAutoRepeatOff(display->display);
	if(xerrval && gw_debug)
	    printf(GW_DEBUG_PREFIX
"Cannot switch keyboard auto repeat.\n"
	    );

	return;
}
#endif  /* X_H */

/*
 *	Creates a new accelerator structure, can return NULL on error.
 */
gw_accelerator_struct *GWAcceleratorNew(
        int key, int modifier
)
{
	gw_accelerator_struct *a = (gw_accelerator_struct *)calloc(
	    1, sizeof(gw_accelerator_struct)
	);
	if(a == NULL)
	    return(NULL);

	a->key = key;
	a->modifier = modifier;

	return(a);
}

/*
 *	Deallocates the given accelerator structure.
 */
void GWAcceleratorDelete(gw_accelerator_struct *a)
{
	if(a == NULL)
	    return;

	free(a);

	return;
}

/*
 *	Appends a new accelerator structure to the list. Returns the
 *	index to the new accelerator structure or -1 on error.
 */
int GWAcceleratorListAdd(
	gw_accelerator_struct ***a, int *total,
        int key, int modifier
)
{
	int n;

	if((a == NULL) || (total == NULL))
	    return(-1);

	if((*total) < 0)
	    (*total) = 0;

	n = (*total);
	(*total) = n + 1;
	(*a) = (gw_accelerator_struct **)realloc(
	    *a,
	    (*total) * sizeof(gw_accelerator_struct *)
	);
	if((*a) == NULL)
	{
	    (*total) = 0;
	    return(-1);
	}

	(*a)[n] = GWAcceleratorNew(key, modifier);
	if((*a)[n] == NULL)
	    return(-1);
	else
	    return(n);
}

/*
 *	Deallocates the list of accelerators.
 */
void GWAcceleratorListDelete(
        gw_accelerator_struct ***a, int *total
)
{
	int i;

        if((a == NULL) || (total == NULL))
            return;

	for(i = 0; i < (*total); i++)
	    GWAcceleratorDelete((*a)[i]);

	free(*a);
	(*a) = NULL;
	(*total) = 0;

	return;
}

/*
 *	Returns True if the given key and modifier match one of the
 *	accelerators in the given list.
 */
Boolean GWAcceleratorListCheck(    
        gw_accelerator_struct **a, int total,
        int key, int modifier
)
{
	int i;
	gw_accelerator_struct *a_ptr;

	if((a == NULL) || (total <= 0))
	    return(False);

	for(i = 0; i < total; i++)
	{
	    a_ptr = a[i];
	    if(a_ptr == NULL)
		continue;

	    if((a_ptr->key == key) &&
               (a_ptr->modifier == modifier)
	    )
		return(True);
	}

	return(False);
}


#ifdef X_H
/*
 *	Sets the poitner cursor to the one specified by cursor.
 */
void GWSetPointerCursor(
	gw_display_struct *display,
	int cursor	/* One of GWPointerCursor*. */
)
{
        if(display == NULL)
            return;

        if((display->display != NULL) &&
           (display->toplevel != None)
	)
	{
	    Cursor cur = None;

	    switch(cursor)
	    {
	      case GWPointerCursorStandard:
		cur = display->cursor_standard;
		break;

              case GWPointerCursorBusy:
                cur = display->cursor_busy;
                break;

              case GWPointerCursorTranslate:
                cur = display->cursor_translate;
                break;

              case GWPointerCursorZoom:
                cur = display->cursor_zoom;
                break;
	    }

	    if(cur == None)
		XUndefineCursor(display->display, display->toplevel);
	    else
                XDefineCursor(
                    display->display,
                    display->toplevel,
		    cur
		);
	    XFlush(display->display);
	}

	return;
}
#endif	/* X_H */
  
#ifdef X_H
/*
 *	Sets pointer cursor as busy and ignores input.
 */
void GWSetInputBusy(gw_display_struct *display)
{
	GWSetPointerCursor(display, GWPointerCursorBusy);
        return;
}
#endif	/* X_H */

#ifdef X_H
/*
 *	Sets pointer cursor as ready and resumes allowing input.
 */
void GWSetInputReady(gw_display_struct *display)
{
        GWSetPointerCursor(display, GWPointerCursorStandard);
	return;
}
#endif  /* X_H */


/*
 *	Disables depth testing, transforms matrixes and model
 *	view to 2D.
 */
void GWOrtho2D(gw_display_struct *display)
{
	if(display == NULL)
	    return;

	StateGLDisable(&display->state_gl, GL_DEPTH_TEST);
	StateGLDepthFunc(&display->state_gl, GL_ALWAYS);

        glMatrixMode(GL_PROJECTION);
        glLoadIdentity();
        gluOrtho2D(  
            0, display->width,
            0, display->height
        );      
        glMatrixMode(GL_MODELVIEW);   
        glLoadIdentity();

	return;
}


/*
 *	Loads a font.
 */
GWFont *GWLoadFont(gw_display_struct *display, char *path)
{
	if((display == NULL) ||
           (path == NULL)
	)
	    return(NULL);

/* Need to write code to load font. */

	return(NULL);
}

/*
 *	Sets font to the foreground of the graphics context.
 */
void GWSetFont(gw_display_struct *display, GWFont *font)
{
	if(display == NULL)
	    return;

	display->current_font = font;

	return;
}

/*
 *	Gets font sizes.
 *
 *	If an input is NULL, then its value will not be set.
 *
 *	Returns 0 on success or non-zero on error.
 */
int GWGetFontSize(
        GWFont *font, 
        int *width, int *height,  
        int *character_spacing, int *line_spacing  
)
{
	if(font == NULL)
	    return(-1);

        /* Format, header is first 32 bytes:
         *
         * Address       Desc
         * 0             Width of each font
         * 1             Height of each font
         * 2             Character width spacing
         * 3             Line spacing
         * 4             Bytes per line
         */

	if(width != NULL)
	    (*width) = (int)((u_int8_t)font[0]);

        if(height != NULL)
            (*height) = (int)((u_int8_t)font[1]);

        if(character_spacing != NULL)
            (*character_spacing) = (int)((u_int8_t)font[2]);

        if(line_spacing != NULL)
            (*line_spacing) = (int)((u_int8_t)font[3]);

	return(0);
}

/*
 *	Unloads a font previously loaded by GWLoadFont().
 */
void GWUnloadFont(gw_display_struct *display, GWFont *font)
{
return;	/* Don't do anything for now. */
	if(display != NULL)
	{
	    if(display->current_font == font)
		display->current_font = NULL;
	}

	free(font);

	return;
}

/*
 *	Loads a color, returns a pointer to a dynamically allocated
 *	GWColor structure which needs to be deallocated with a call to
 *	GWUnloadColor().
 */
GWColor *GWLoadColor(
	gw_display_struct *display,
	double a, double r, double g, double b
)
{
	GWColor *color;


        if(display == NULL)
            return(NULL);

        color = (GWColor *)calloc(1, sizeof(GWColor));
        if(color == NULL)
            return(NULL);

	color->a = a;
	color->r = r;
	color->g = g;
	color->b = b;

	return(color);
}

/*
 *	Sets color to foreground of graphics context.
 */
void GWSetColor(gw_display_struct *display, GWColor *color)
{
        if(display == NULL)
            return;

	if(color != NULL)
	    glColor4d(
		color->r,
                color->g,
                color->b,
                color->a
	    );

        display->current_color = color;

        return;
}

/*
 *	Unloads and deallocates a color previously loaded by
 *	GWLoadColor().
 */
void GWUnloadColor(gw_display_struct *display, GWColor *color)
{
        if(display == NULL)
            return;

	/* Check if current color has changed. */
	if(display->current_color == color)
	    display->current_color = NULL;

	free(color);

	return;
}


/*
 *	Draws string.
 *
 *      GWOrtho2D() must be called to transform orientations before 
 *      using this function.
 */
#ifdef USE_OLD_GWDRAWSTRING
void GWDrawString(
	gw_display_struct *display,
	int x, int y,
	const char *string
)
{
	u_int8_t *font_header, *font_data;

	int font_width, font_height;		/* In pixels. */
	int width_spacing, line_spacing;	/* In pixels. */
	int bytes_per_line;	/* Bytes per line (per width). */
	int bytes_per_char_len;	/* Bytes per character. */

	int win_height;


	if((display == NULL) ||
           (string == NULL)
	)
	    return;

	win_height = display->height;

	/* Get pointer to font header data. */
	font_header = display->current_font;
	if(font_header == NULL)
	    return;

	/* Seek past header and get pointer to start of font data. */
	font_data = font_header + 32;

	/* Format, header is first 32 bytes:
	 *
	 * Address       Desc
	 * 0             Width of each font
	 * 1             Height of each font
	 * 2             Character width spacing
	 * 3             Line spacing
	 * 4             Bytes per line
	 */

	font_width = font_header[0];
	font_height = font_header[1];
	width_spacing = font_header[2];
	line_spacing = font_header[3];
	bytes_per_line = font_header[4];

	/* Calculate bytes per character. */
	bytes_per_char_len = bytes_per_line * font_height;

	/* Convert y position. */
	y = win_height - y - font_height;

	while(*string != '\0')
	{
            glRasterPos2i(x, y);
            glBitmap(
		font_width, font_height,
		0.0, 0.0,
		width_spacing, 0.0,
		&font_data[
		    bytes_per_char_len * (char)(*string)
		]
	    );

	    x += width_spacing;
	    string++;
	}

	return;
}
#else	/* USE_OLD_GWDRAWSTRING */
void GWDrawString(
	gw_display_struct *display,
	int x, int y,
	const char *string
)
{
        u_int8_t *font_header, *font_data;

        int font_width, font_height;            /* In pixels. */
        int width_spacing, line_spacing;        /* In pixels. */
        int bytes_per_line;     /* Bytes per line (per width). */
        int bytes_per_char;     /* Bytes per character. */
        int char_offset;        /* Current char's offset in the font data. */

        int chars_hidden = 0;	/* # of chars entirely outside viewport */
        double xorig = 0.0;
        double yorig = 0.0;


        if((display == NULL) ||
           (string == NULL)
	)
            return;

        /* Get pointer to font header data. */
        font_header = display->current_font;
        if(font_header == NULL)
            return;

        /* Seek past header and get pointer to start of font data. */
        font_data = font_header + 32;

        /* Format, header is first 32 bytes:
         *
         * Address       Desc
         * 0             Width of each font
         * 1             Height of each font
         * 2             Character width spacing
         * 3             Line spacing
         * 4             Bytes per line   
         */     

        font_width = font_header[0];
        font_height = font_header[1];
        width_spacing = font_header[2];
        line_spacing = font_header[3];
        bytes_per_line = font_header[4];

	/* Flip y values. */
        y = display->height - line_spacing - y + 1;

        /* Calculate bytes per character. */
        bytes_per_char = bytes_per_line * font_height;

        /* Adjust string index in case x < 0. */
        if(x < 0)
	{
            chars_hidden = (-x) / width_spacing;
            xorig = (-x) - (chars_hidden * width_spacing);

            if(chars_hidden >= (int)strlen(string))
                return;

            x = 0;
            string += chars_hidden;
        } 
        if(y < 0)
	{
	    if(y < -line_spacing)
		return;

	    yorig = -y;
	    y = 0;
        }

        glRasterPos2i(x, y);
        for(; *string; string++)
	{
	    char_offset = bytes_per_char * (*string);

	    glBitmap(
		font_width, font_height,
		(GLfloat)xorig, (GLfloat)yorig,
		(GLfloat)width_spacing, 0.0,
		font_data + char_offset
	    );
        }

	return;
}
#endif	/* USE_OLD_GWDRAWSTRING */

/*
 *	Same as GWDrawString() except that it draws just the specified
 *	character.
 */
void GWDrawCharacter(
        gw_display_struct *display,
        int x, int y,   
        char c
)
{
        u_int8_t *font_header, *font_data;

        int font_width, font_height;            /* In pixels. */
        int width_spacing, line_spacing;        /* In pixels. */
        int bytes_per_line;     /* Bytes per line (per width). */
        int bytes_per_char_len; /* Bytes per character. */

        int win_height;


        if((display == NULL) ||
           (c == '\0')
	)
	    return;

        win_height = display->height;
            
        /* Get pointer to font header data. */
        font_header = display->current_font;
        if(font_header == NULL)
            return;
                
        /* Seek past header and get pointer to start of font data. */
        font_data = font_header + 32;

        /* Format, header is first 32 bytes:
         *
         * Address       Desc
         * 0             Width of each font
         * 1             Height of each font
         * 2             Character width spacing
         * 3             Line spacing
         * 4             Bytes per line
         */

        font_width = font_header[0];
        font_height = font_header[1];
        width_spacing = font_header[2];
        line_spacing = font_header[3];
        bytes_per_line = font_header[4];

        /* Calculate bytes per character. */
        bytes_per_char_len = bytes_per_line * font_height;

        /* Convert y position. */
        y = win_height - y - font_height;

        glRasterPos2i(x, y);
        glBitmap(
            font_width, font_height,
            0.0, 0.0,
            (GLfloat)width_spacing, 0.0,
            &font_data[
		bytes_per_char_len * c
            ]
        );

	return;
}

/*
 *	Draws a line in 2D.
 *
 *	GWOrtho2D() must be called to transform orientations before
 *	using this function.
 */
void GWDrawLine(
	gw_display_struct *display,
	int x1, int y1,
	int x2, int y2
)
{
	if(display == NULL)
	    return;

	y1 = display->height - y1;
	y2 = display->height - y2;

        glBegin(GL_LINES);
        {
            glVertex2d(x1, y1);
            glVertex2d(x2, y2);
        }
	glEnd();

        return;
}       
