#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <limits.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "../include/os.h"

#ifdef __MSW__
# include <windows.h>
#endif

#include <GL/gl.h>

#include "../include/string.h"
#include "../include/disk.h"
#include "../include/tga.h"

#include "gw.h"
#include "stategl.h"

#include "image.h"

#include "sar.h"


sar_image_struct *SARImageLoadFromFile(const char *file);
void SARImageDestroy(sar_image_struct *image);

void SARImageDraw(
	gw_display_struct *display,
	const sar_image_struct *image,
	int x, int y,
	int width, int height
);


/*
 *	Loads and allocates a new image structure, returns the
 *	newly allocated structure or NULL on error.
 *
 *	If the specified file path is not absolute then the
 *	file will be looked for in the local and global data directories.
 */
sar_image_struct *SARImageLoadFromFile(const char *file)
{
	char *strptr;
	int status, bytes_per_pixel, total_bytes, total_pixels;
	sar_image_struct *image;
	u_int8_t r, g, b;
	u_int32_t *src_ptr32, *src_end32;
	u_int32_t *tar_ptr32, *tar_end32;
	tga_data_struct td;
	struct stat stat_buf;
	char tmp_path[PATH_MAX + NAME_MAX];


	if(file == NULL)
	    return(NULL);

	if(ISPATHABSOLUTE(file))
	{
	    strncpy(tmp_path, file, PATH_MAX + NAME_MAX);
	}
	else
	{
	    strptr = PrefixPaths(dname.local_data, file);
	    if(strptr != NULL)
	    {
		if(stat(strptr, &stat_buf))
		    strptr = PrefixPaths(dname.global_data, file);
	    }

	    strncpy(
		tmp_path,
		((strptr == NULL) ? file : strptr),
		PATH_MAX + NAME_MAX
	    );
	}
	tmp_path[PATH_MAX + NAME_MAX - 1] = '\0';


	/* File exists? */
#ifndef __MSW__
	if(stat(tmp_path, &stat_buf))
	{
            fprintf(stderr,
                "%s: No such file.\n",
                file
            );
            return(NULL);
	}
	if(!S_ISREG(stat_buf.st_mode))
	{
	    fprintf(stderr,
                "%s: Not a file.\n",
		file
            );
            return(NULL);
	}
#endif

        /* Load data from file, read as 32 bits. */
        status = TgaReadFromFile(
            tmp_path,
            &td,
            32          /* Read to 32 bits. */
        );
        if(status != TgaSuccess)
        {
            TgaDestroyData(&td);
            return(NULL);
        }

	/* Data loaded? */
        if(td.data == NULL)
        {
            TgaDestroyData(&td);
            return(NULL);
        }


	/* Allocate image structure. */
	image = (sar_image_struct *)calloc(1, sizeof(sar_image_struct));
	if(image == NULL)
	    return(NULL);

	/* Get values from loaded data. */
	image->type = SAR_IMAGE_TYPE_RGBA;
	image->width = td.width;
	image->height = td.height;

        bytes_per_pixel = 4;    /* Set bytes per pixel to 4. */
	total_pixels = image->width * image->height;
	total_bytes = total_pixels * bytes_per_pixel;

	src_ptr32 = (u_int32_t *)td.data;
	src_end32 = src_ptr32 + total_pixels;


	/* Allocate memory for target image data. */
	image->data = (u_int8_t *)malloc(total_bytes);
	if(image->data == NULL)
	{
	    TgaDestroyData(&td);
	    free(image);
	    return(NULL);
	}
	tar_ptr32 = (u_int32_t *)image->data;
	tar_end32 = tar_ptr32 + total_pixels;

	/* Copy over image data. */
        while(src_ptr32 < src_end32)
        {
/*          a = (u_int8_t)(((*src_ptr32) & 0xff000000) >> 24); */
            r = (u_int8_t)(((*src_ptr32) & 0x00ff0000) >> 16);
            g = (u_int8_t)(((*src_ptr32) & 0x0000ff00) >> 8);
            b = (u_int8_t)(((*src_ptr32) & 0x000000ff) >> 0);

            if(*src_ptr32 == 0x00000000)
                *tar_ptr32++ = PACK8TO32(0x00, b, g, r);
            else
                *tar_ptr32++ = PACK8TO32(0xff, b, g, r);

	    src_ptr32++;
        }


	/* Unload image. */
	TgaDestroyData(&td);

	return(image);
}

/*
 *	Deallocates the resources of the image and the image structure
 *	itself.
 */
void SARImageDestroy(sar_image_struct *image)
{
	if(image != NULL)
	{
	    free(image->data);
	    free(image);
	}

	return;
}

/*
 *	Draws the image to the screen at the specified width and
 *	height.
 */
void SARImageDraw(
	gw_display_struct *display,
	const sar_image_struct *image,
	int x, int y,
	int width, int height
)
{
	GLint x2, y2;
	GLfloat x_zoom_factor, y_zoom_factor;


	if((display == NULL) || (image == NULL))
	    return;

	if((image->width < 1) ||
           (image->height < 1) ||
           (image->data == NULL)
	)
	    return;

	/* Calculate GL converted coordinate positions. */
	x2 = (GLint)x;
	y2 = (GLint)(display->height - y - 1);

	/* Sanitize positions. */
	if(x2 < 0)
	    x2 = 0;
	if(y2 < 0)
	    y2 = 0;

	/* Calculate zoom factors. */
        x_zoom_factor = (GLfloat)width / (GLfloat)image->width;
        y_zoom_factor = -(GLfloat)height / (GLfloat)image->height;
 

	/* Set starting drawing position. */
	glRasterPos2i(x2, y2);

	/* Set zoom factor. */
	glPixelZoom(x_zoom_factor, y_zoom_factor);

	/* Enable GL alpha testing (some images have transparent pixels). */
	StateGLEnable(&display->state_gl, GL_ALPHA_TEST);
	StateGLAlphaFunc(&display->state_gl, GL_GREATER, 0.5);

	/* Draw the image. */
	glDrawPixels(
	    image->width, image->height,
	    GL_RGBA, GL_UNSIGNED_BYTE,
	    (const GLvoid *)image->data
	);

	/* Disable GL alpha testing. */
	StateGLDisable(&display->state_gl, GL_ALPHA_TEST);

	/* Restore zoom. */
        glPixelZoom(1.0, 1.0);
}

