#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <sys/types.h>

#include "../include/os.h"
#ifdef __MSW__
# include <windows.h>
#endif

#include <GL/gl.h>

#include "../include/string.h"
#include "v3dtex.h"

#include "gw.h"
#include "stategl.h"
#include "image.h"
#include "menu.h"

#include "sar.h"


void SARMenuGLStateReset(gw_display_struct *display);
int SARMenuIsObjectAllocated(sar_menu_struct *m, int n);
sar_menu_struct *SARMenuCreate(
	int type, const char *name, const char *bg_image
);
void SARMenuLoadBackgroundImage(sar_menu_struct *m);
void SARMenuUnloadBackgroundImage(sar_menu_struct *m);
void SARMenuDestroy(sar_menu_struct *m);

int SARMenuCreateLabel(
	sar_menu_struct *m,
	double x, double y,
	int width, int height,
	const char *label,
	sar_menu_color_struct *fg_color, GWFont *font,
	const sar_image_struct *image
);

int SARMenuCreateButton(
        sar_menu_struct *m,
        double x, double y,   
        int width, int height,
        const char *label,
        sar_menu_color_struct *fg_color, GWFont *font,
	const sar_image_struct *unarmed_image,
	const sar_image_struct *armed_image,
	const sar_image_struct *highlighted_image,
        void *client_data, int id_code,
        void (*func_cb)(void *, void *, int)
);

int SARMenuCreateProgress(
        sar_menu_struct *m,
        double x, double y,
        int width, int height,
        const char *label,
        sar_menu_color_struct *fg_color,
        GWFont *font,
	const sar_image_struct *bg_image,
	const sar_image_struct *fg_image,
        double progress
);

int SARMenuCreateMessageBox(
	sar_menu_struct *m,
	double x, double y,
	double width, double height,
	sar_menu_color_struct *fg_color, GWFont *font,
        const sar_image_struct **bg_image,	/* Pointer to 9 images. */
        const char *message
);

int SARMenuCreateList(
        sar_menu_struct *m,
        double x, double y,
        double width, double height,
        sar_menu_color_struct *fg_color, GWFont *font,
        const char *label,
	const sar_image_struct **bg_image,	/* Pointer to 9 images. */
        void (*select_cb)(void *, void *, int, void *)
);
int SARMenuListAppendItem(
        sar_menu_struct *m,
        int n,                  /* List object number on m. */
        const char *name,
        void *client_data, 
        sar_menu_flags_t flags
);
sar_menu_list_item_struct *SARMenuListGetItemByNumber(
	sar_menu_list_struct *list_ptr, int n
);
void SARMenuListDeleteAllItems(
	sar_menu_struct *m,
	int n			/* List object number on m. */
);

int SARMenuCreateSwitch(
	sar_menu_struct *m,
	double x, double y,
	int width, int height,
	sar_menu_color_struct *fg_color, GWFont *font,
	const char *label,
	const sar_image_struct *bg_image,
	const sar_image_struct *switch_off_image,
	const sar_image_struct *switch_on_image,
	Boolean state,
	void *client_data, int id_code,
	void (*switch_cb)(void *, void *, int, Boolean)
);

int SARMenuCreateSpin(
	sar_menu_struct *m,
	double x, double y,
	double width, double height,
	sar_menu_color_struct *label_color,
	sar_menu_color_struct *value_color,
	GWFont *font,
	GWFont *value_font,
	const char *label,
	const sar_image_struct *label_image,
	const sar_image_struct *value_image,
        void *client_data, int id_code,
        void (*change_cb)(void *, void *, int, char *)
);
void SARMenuSpinSetValueType(
	sar_menu_struct *m,
	int n,			/* Spin object number on m. */
        int type
);
int SARMenuSpinAddValue(
	sar_menu_struct *m,
	int n,			/* Spin object number on m. */
	const char *value
);
char *SARMenuSpinGetCurrentValue(
	sar_menu_struct *m,
	int n,			/* Spin object number on m. */
	int *sel_num
);
void SARMenuSpinDeleteAllValues(
	sar_menu_struct *m,
	int n			/* Spin object number on m. */
);
static void SARMenuSpinDoInc(
	gw_display_struct *display, sar_menu_struct *m,
	int n			/* Spin object number on m. */
);
static void SARMenuSpinDoDec(
	gw_display_struct *display, sar_menu_struct *m,
	int n			/* Spin object number on m. */
);

void SARMenuDestroyObject(
	sar_menu_struct *m,
	int n			/* Object number on m. */
);

void SARMenuSetProgressSimple(
	gw_display_struct *display,
	sar_menu_struct *m,
        double progress,
	int redraw
);
void SARMenuSetMessageBoxSimple(
	gw_display_struct *display,
	sar_menu_struct *m,
	const char *message,
        int redraw
);

static char *SARMenuMessageBoxDrawString(
	gw_display_struct *display,
	int x, int y,
	char *buf_ptr,
	int visible_columns, int font_width,
	Boolean draw_line,
	sar_menu_color_struct *default_color,
	sar_menu_color_struct *bold_color,
	sar_menu_color_struct *underline_color
);
static int SARMenuMessageBoxTotalLines(char *buf_ptr, int visible_columns);
static void SARMenuDrawWindowBG(
	gw_display_struct *display,
	const sar_image_struct **bg_image,	/* Total 9 images. */
	int x, int y,
	int width, int height,
	Boolean draw_base,
	Boolean draw_shadow
);
static void SARMenuDoDrawObject(
        gw_display_struct *display, sar_menu_struct *m,
        int n,			/* Object number on m. */
	Boolean draw_shadows
);
void SARMenuDrawObject(
	gw_display_struct *display, sar_menu_struct *m,
	int n			/* Object number on m. */
);
void SARMenuDrawAll(
	gw_display_struct *display, sar_menu_struct *m
);

int SARMenuManagePointer(
	gw_display_struct *display, sar_menu_struct *m,
	int x, int y, int state, int btn_num
);
int SARMenuManageKey(
	gw_display_struct *display,
	sar_menu_struct *m,
	int key, int state
);


#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))

#define ISCR(c)		(((c) == '\n') || ((c) == '\r'))

#define IS_CHAR_ESC_START(c)	((c) == '<')
#define IS_CHAR_ESC_END(c)	((c) == '>')




/*
 *	Default spacings (in pixels):
 */
#define DEF_XMARGIN		5
#define DEF_YMARGIN		5

#define DEF_LIST_XMARGIN	25
#define DEF_LIST_YMARGIN	20

#define DEF_MB_XMARGIN		25	/* Message box. */
#define DEF_MB_YMARGIN		20

#define DEF_WIDTH		100
#define DEF_HEIGHT		100

#define DEF_PROGRESS_HEIGHT	20

#define DEF_SCROLL_CURSOR_WIDTH		16
#define DEF_SCROLL_CURSOR_HEIGHT	16

#define DEF_SWITCH_WIDTH	48
#define DEF_SWITCH_HEIGHT	48


#define DO_REDRAW_MENU	\
{ \
 SARMenuDrawAll(display, m); \
 GWSwapBuffer(display); \
}

#define DO_REDRAW_OBJECT(d,m,n)	\
{ \
 SARMenuDrawObject(display,m,n); \
 GWSwapBuffer(display); \
}


/*
 *	Resets GL states for menu system drawing, this function
 *	should be called at startup and whenever the menu system is
 *	re-entered.
 */
void SARMenuGLStateReset(gw_display_struct *display)
{
	state_gl_struct *s;

	if(display == NULL)
	    return;

	s = &display->state_gl;

	/* Switch to 2D. */
	GWOrtho2D(display);

	/* Reset GL states. */
	StateGLDepthMask(s, GL_TRUE);
	StateGLDisableF(s, GL_DEPTH_TEST, True);
	StateGLDisableF(s, GL_BLEND, True);
	StateGLDisableF(s, GL_FOG, True);
        StateGLDisableF(s, GL_ALPHA_TEST, True);
        StateGLDisableF(s, GL_TEXTURE_1D, True);
        StateGLDisableF(s, GL_TEXTURE_2D, True);
        StateGLDisableF(s, GL_COLOR_MATERIAL, True);
        StateGLDisableF(s, GL_LIGHTING, True);
        StateGLDisableF(s, GL_LIGHT0, True);
	StateGLDisableF(s, GL_LIGHT1, True);
	StateGLDisableF(s, GL_LIGHT2, True);
        StateGLDisableF(s, GL_LIGHT3, True);
        StateGLDisableF(s, GL_LIGHT4, True);
        StateGLDisableF(s, GL_LIGHT5, True);
        StateGLDisableF(s, GL_LIGHT6, True);
        StateGLDisableF(s, GL_LIGHT7, True);
        StateGLDisableF(s, GL_POINT_SMOOTH, True);
        StateGLDisableF(s, GL_POLYGON_OFFSET_FILL, True);
        StateGLDisableF(s, GL_POLYGON_OFFSET_LINE, True);
        StateGLDisableF(s, GL_POLYGON_OFFSET_POINT, True);
        StateGLDisableF(s, GL_SCISSOR_TEST, True);

	glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
	glPixelStorei(GL_PACK_ALIGNMENT, 1);

	return;
}


/*
 *	Checks if object n is allocated on menu m.
 */
int SARMenuIsObjectAllocated(sar_menu_struct *m, int n)
{
	if(m == NULL)
	    return(0);
	else if((n < 0) || (n >= m->total_objects))
	    return(0);
	else if(m->object[n] == NULL)
	    return(0);
	else
	    return(1);
}

/*
 *	Allocates a new menu structure.
 */
sar_menu_struct *SARMenuCreate(
        int type, const char *name, const char *bg_image
)
{
	sar_menu_struct *m;


	/* Allocate a new menu structure. */
	m = (sar_menu_struct *)calloc(1, sizeof(sar_menu_struct));
	if(m == NULL)
	    return(NULL);

	/* Set name. */
	m->name = StringCopyAlloc(name);

	/* Set background image. */
	m->bg_image_path = StringCopyAlloc(bg_image);
	m->bg_image = NULL;

	return(m);
}

/*
 *	Loads background image as needed on menu m.
 */
void SARMenuLoadBackgroundImage(sar_menu_struct *m)
{
	if(m == NULL)
	    return;

	if(m->bg_image_path == NULL)
	    return;

	if(m->bg_image != NULL)
	    return;

	m->bg_image = SARImageLoadFromFile(m->bg_image_path);

	return;
}

/*
 *      Unloads background image as needed on menu m.
 */
void SARMenuUnloadBackgroundImage(sar_menu_struct *m)
{
        if(m == NULL)
            return;
 
        if(m->bg_image != NULL)
	{
	    SARImageDestroy(m->bg_image);
	    m->bg_image = NULL;
	}
            
        return;
}

/*
 *	Deallocates menu structure m and all its resources.
 */
void SARMenuDestroy(sar_menu_struct *m)
{
	int i;

	if(m == NULL)
	    return;

	/* Destroy all menu objects on this menu. */
	for(i = 0; i < m->total_objects; i++)
	    SARMenuDestroyObject(m, i);
	free(m->object);
	m->object = NULL;
	m->total_objects = 0;

	/* Deallocate name of this menu. */
	free(m->name);
	m->name = NULL;

	/* Deallocate path to menu's background image. */
	free(m->bg_image_path);
	m->bg_image_path = NULL;

	/* Deallocate menu's background image. */
	SARImageDestroy(m->bg_image);
	m->bg_image = NULL;

	/* Deallocate menu structure itself. */
	free(m);
}

/*
 *	Allocates a new label on menu m.
 *
 *	Returns its index number or -1 on error.
 */
int SARMenuCreateLabel(
        sar_menu_struct *m,
        double x, double y,
	int width, int height,
        const char *label,
        sar_menu_color_struct *fg_color,
	GWFont *font,
	const sar_image_struct *image
)
{
	int i, n;
	sar_menu_label_struct *label_ptr;


	if(m == NULL)
	    return(-1);

	if(m->total_objects < 0)
	    m->total_objects = 0;

	for(i = 0; i < m->total_objects; i++)
	{
	    if(m->object[i] == NULL)
		break;
	}
	if(i < m->total_objects)
	{
	    n = i;
	}
	else
	{
	    n = m->total_objects;
	    m->total_objects++;

	    m->object = (void **)realloc(
		m->object,
		m->total_objects * sizeof(void *)
	    );
	    if(m->object == NULL)
	    {
		m->total_objects = 0;
		return(-1);
	    }

	    m->object[n] = NULL;
	}

	/* Allocate structure. */
	label_ptr = (sar_menu_label_struct *)calloc(
	    1,
	    sizeof(sar_menu_label_struct)
	);
	if(label_ptr == NULL)
	{
	    return(-1);
	}

	m->object[n] = (void *)label_ptr;


	/* Load label values. */
	label_ptr->type = SAR_MENU_OBJECT_TYPE_LABEL;
	label_ptr->x = x;
	label_ptr->y = y;
	label_ptr->width = width;
	label_ptr->height = height;
	label_ptr->image = image;
        memcpy(&label_ptr->color, fg_color, sizeof(sar_menu_color_struct));
	label_ptr->font = font;
	label_ptr->label = StringCopyAlloc(label);
	label_ptr->align = SAR_MENU_LABEL_ALIGN_CENTER;

	/* Select this object if it is the first one. */
	if(m->total_objects == 1)
	    m->selected_object = 0;

	return(n);
}

/*
 *	Allocates a new button on menu m.
 *
 *	Returns its index number or -1 on error.
 */
int SARMenuCreateButton(
	sar_menu_struct *m,
	double x, double y,
	int width, int height,
	const char *label,
        sar_menu_color_struct *fg_color, GWFont *font,
        const sar_image_struct *unarmed_image,
	const sar_image_struct *armed_image,
	const sar_image_struct *highlighted_image,
	void *client_data, int id_code,
	void (*func_cb)(void *, void *, int)
)
{
	int i, n;
	sar_menu_button_struct *button_ptr;


        if(m == NULL)
            return(-1);
              
        if(m->total_objects < 0)
            m->total_objects = 0;
        
        for(i = 0; i < m->total_objects; i++)
        {
            if(m->object[i] == NULL)
                break;
        }
        if(i < m->total_objects)
        {
            n = i;
        }
        else
        {
            n = m->total_objects;
            m->total_objects++;
 
            m->object = (void **)realloc(
                m->object,
                m->total_objects * sizeof(void *)
            );

            if(m->object == NULL)
            {
                m->total_objects = 0;
                return(-1);
            }

            m->object[n] = NULL;
        }

        /* Allocate structure. */
        button_ptr = (sar_menu_button_struct *)calloc(
            1,
            sizeof(sar_menu_button_struct)
        );
        if(button_ptr == NULL)
        {
            return(-1);
        }

        m->object[n] = (void *)button_ptr;


        /* Load button values. */
	button_ptr->type = SAR_MENU_OBJECT_TYPE_BUTTON;
	button_ptr->x = x;
	button_ptr->y = y;
	button_ptr->width = width;
	button_ptr->height = height;
	button_ptr->unarmed_image = unarmed_image;
        button_ptr->armed_image = armed_image;
        button_ptr->highlighted_image = highlighted_image;
	memcpy(&button_ptr->color, fg_color, sizeof(sar_menu_color_struct));
        button_ptr->font = font;
	button_ptr->label = StringCopyAlloc(label);
	button_ptr->client_data = client_data;
	button_ptr->id_code = id_code;
	button_ptr->func_cb = func_cb;

	/* Select this object if it is the first one. */
	if(m->total_objects == 1)
	    m->selected_object = 0;

	return(n);
}

/*
 *      Allocates a new progress bar on menu m.
 *
 *      Returns its index number or -1 on error.
 */
int SARMenuCreateProgress( 
	sar_menu_struct *m,
	double x, double y,
	int width, int height,
	const char *label,
        sar_menu_color_struct *fg_color, GWFont *font,
	const sar_image_struct *bg_image,
	const sar_image_struct *fg_image,
	double progress
)
{
        int i, n;
        sar_menu_progress_struct *progress_ptr;


        if(m == NULL)
            return(-1);

        if(m->total_objects < 0)
            m->total_objects = 0;

        for(i = 0; i < m->total_objects; i++)
        {
            if(m->object[i] == NULL)
                break;
        }
        if(i < m->total_objects)
        {
            n = i;
        }
        else
        {
            n = m->total_objects;
            m->total_objects++;
        
            m->object = (void **)realloc(
                m->object,
                m->total_objects * sizeof(void *)
            );
        
            if(m->object == NULL)
            {
                m->total_objects = 0;
                return(-1);
            }
  
            m->object[n] = NULL;
        }

        /* Allocate structure. */
	progress_ptr = (sar_menu_progress_struct *)calloc(
            1,
            sizeof(sar_menu_progress_struct)
        );
        if(progress_ptr == NULL)
        {
            return(-1);
        }

        m->object[n] = (void *)progress_ptr;


        /* Load progress bar values. */
	progress_ptr->type = SAR_MENU_OBJECT_TYPE_PROGRESS;
	progress_ptr->x = x;
	progress_ptr->y = y;
	progress_ptr->width = width;
	progress_ptr->height = height;
        progress_ptr->bg_image = bg_image;
        progress_ptr->fg_image = fg_image;
        memcpy(&progress_ptr->color, fg_color, sizeof(sar_menu_color_struct));
        progress_ptr->font = font;
        progress_ptr->label = StringCopyAlloc(label);
	progress_ptr->progress = progress;

	/* Select this object if it is the first one. */
	if(m->total_objects == 1)
	    m->selected_object = 0;

	return(n);
}

/*
 *	Create message box on menu m.
 */
int SARMenuCreateMessageBox(
	sar_menu_struct *m,
        double x, double y,
        double width, double height,
        sar_menu_color_struct *fg_color,
        GWFont *font,
        const sar_image_struct **bg_image,	/* Pointer to 9 images. */
	const char *message
)
{
        int i, n;
        sar_menu_message_box_struct *mesgbox_ptr;


        if(m == NULL)
            return(-1);

        if(m->total_objects < 0)
            m->total_objects = 0;

        for(i = 0; i < m->total_objects; i++)
        {
            if(m->object[i] == NULL)
                break;
        }
        if(i < m->total_objects)
        {
            n = i;
        }
        else
        {
            n = m->total_objects;
            m->total_objects++;

            m->object = (void **)realloc(
                m->object,
                m->total_objects * sizeof(void *)
            );

            if(m->object == NULL)
            {
                m->total_objects = 0;
                return(-1);
            }

            m->object[n] = NULL;
        }

        /* Allocate structure. */
        mesgbox_ptr = (sar_menu_message_box_struct *)calloc(
            1,
            sizeof(sar_menu_message_box_struct)
        );
        if(mesgbox_ptr == NULL)
        {
            return(-1);
        }

        m->object[n] = (void *)mesgbox_ptr;

        /* Load message box values. */
        mesgbox_ptr->type = SAR_MENU_OBJECT_TYPE_MESSAGE_BOX;
        mesgbox_ptr->x = x;
        mesgbox_ptr->y = y;
        mesgbox_ptr->width = width;
        mesgbox_ptr->height = height;
        mesgbox_ptr->bg_image = bg_image;
        memcpy(&mesgbox_ptr->color, fg_color, sizeof(sar_menu_color_struct));
        mesgbox_ptr->font = font;
        mesgbox_ptr->message = StringCopyAlloc(message);
        mesgbox_ptr->scrolled_line = 0;

	/* Select this object if it is the first one. */
	if(m->total_objects == 1)
	    m->selected_object = 0;

	return(n);
}

/*
 *	Create list on menu m.
 *
 *	Returns its index number or -1 on error.
 */
int SARMenuCreateList(
        sar_menu_struct *m,
        double x, double y,
        double width, double height,
        sar_menu_color_struct *fg_color,
        GWFont *font,
	const char *label,
        const sar_image_struct **bg_image,	/* Pointer to 9 images. */
	void (*select_cb)(void *, void *, int, void *)
)
{
        int i, n;
        sar_menu_list_struct *list_ptr;


        if(m == NULL)
            return(-1);

        if(m->total_objects < 0)
            m->total_objects = 0;

        for(i = 0; i < m->total_objects; i++)
        {
            if(m->object[i] == NULL)
                break;
        }
        if(i < m->total_objects)
        {
            n = i;
        }   
        else
        {
            n = m->total_objects;
            m->total_objects++;

            m->object = (void **)realloc(
                m->object,
                m->total_objects * sizeof(void *)
            );
            if(m->object == NULL)
            {
                m->total_objects = 0;
                return(-1);
            }

            m->object[n] = NULL;
        }

        /* Allocate structure. */
        list_ptr = (sar_menu_list_struct *)calloc(
            1,
            sizeof(sar_menu_list_struct)
        );
        if(list_ptr == NULL)
            return(-1);
	else
	    m->object[n] = (void *)list_ptr;


        /* Load list values. */
        list_ptr->type = SAR_MENU_OBJECT_TYPE_LIST;
        list_ptr->x = x;
        list_ptr->y = y;
        list_ptr->width = width;
        list_ptr->height = height;
        list_ptr->bg_image = bg_image;
        memcpy(&list_ptr->color, fg_color, sizeof(sar_menu_color_struct));
        list_ptr->font = font;
	list_ptr->label = StringCopyAlloc(label);
        list_ptr->item = NULL;
        list_ptr->total_items = 0;
	list_ptr->items_visable = 0;	/* Calculate when drawn or managed. */
        list_ptr->scrolled_item = 0;
	list_ptr->selected_item = -1;
        list_ptr->select_cb = select_cb;


	/* Select this object if it is the first one. */
	if(m->total_objects == 1)
	    m->selected_object = 0;

	return(n);
}

/*
 *	Appends item to the list object n on menu m.
 *
 *	Returns the item number or -1 on error.
 */
int SARMenuListAppendItem(
        sar_menu_struct *m,
        int n,                  /* List object number on m. */
        const char *name,
        void *client_data,
        sar_menu_flags_t flags
)
{
        int item_num;
        sar_menu_list_struct *list_ptr;
        sar_menu_list_item_struct *item_ptr;


        if(!SARMenuIsObjectAllocated(m, n))
            return(-1);

        if(*(int *)(m->object[n]) != SAR_MENU_OBJECT_TYPE_LIST)
            return(-1);

        list_ptr = m->object[n];

	if(list_ptr->total_items < 0)
	    list_ptr->total_items = 0;

	item_num = list_ptr->total_items;

	list_ptr->total_items++;
	list_ptr->item = (sar_menu_list_item_struct **)realloc(
	    list_ptr->item,
	    list_ptr->total_items * sizeof(sar_menu_list_item_struct *)
	);
	if(list_ptr->item == NULL)
	{
	    list_ptr->total_items = 0;
	    return(-1);
	}

	list_ptr->item[item_num] = (sar_menu_list_item_struct *)calloc(
	    1,
	    sizeof(sar_menu_list_item_struct)
	);
	if(list_ptr->item[item_num] == NULL)
	{
	    return(-1);
	}

	item_ptr = list_ptr->item[item_num];

	/* Set item values. */
	item_ptr->flags = flags;
	item_ptr->name = StringCopyAlloc(name);
	item_ptr->client_data = client_data;

	return(item_num);
}

/*
 *	Returns the pointer to the item number n or -1 on error.
 */
sar_menu_list_item_struct *SARMenuListGetItemByNumber(
	sar_menu_list_struct *list_ptr,
	int n
)
{
	if(list_ptr == NULL)
	    return(NULL);

	if((n < 0) || (n >= list_ptr->total_items))
	    return(NULL);
	else
	    return(list_ptr->item[n]);
}

/*
 *	Deletse all items allocated on list object n on menu m.
 *
 *	Client data on each item should have been taken care of
 *	by the calling function.
 */
void SARMenuListDeleteAllItems(
        sar_menu_struct *m,
        int n                   /* List object number on m. */
)
{
	int i;
	sar_menu_list_struct *list_ptr;
	sar_menu_list_item_struct *item_ptr;


	if(!SARMenuIsObjectAllocated(m, n))
	    return;

	if(*(int *)(m->object[n]) != SAR_MENU_OBJECT_TYPE_LIST)
	    return;

	list_ptr = m->object[n];

        for(i = 0; i < list_ptr->total_items; i++)
        {
	    item_ptr = list_ptr->item[i];
            if(item_ptr == NULL)
                continue;

            free(item_ptr->name);
            /* Calling function needs to free client_data. */
	    free(item_ptr);
        }
	free(list_ptr->item);
	list_ptr->item = NULL;

	list_ptr->total_items = 0;
	list_ptr->scrolled_item = 0;
	list_ptr->selected_item = -1;

	return;
}

/*
 *	Create switch object on menu m.
 *
 *	Returns its index number or -1 on error.
 */
int SARMenuCreateSwitch(
	sar_menu_struct *m,
	double x, double y,
	int width, int height,
	sar_menu_color_struct *fg_color, GWFont *font,
	const char *label,
	const sar_image_struct *bg_image,
	const sar_image_struct *switch_off_image,
	const sar_image_struct *switch_on_image,
	Boolean state,
	void *client_data, int id_code,
	void (*switch_cb)(void *, void *, int, Boolean)
)
{
	int i, n;
        sar_menu_switch_struct *switch_ptr;


        if(m == NULL)
            return(-1);

        if(m->total_objects < 0)
            m->total_objects = 0;

        for(i = 0; i < m->total_objects; i++)
        {
            if(m->object[i] == NULL)
                break;
        }
        if(i < m->total_objects)
        {
            n = i;
        }
        else
        {
            n = m->total_objects;
            m->total_objects++;

            m->object = (void **)realloc(
                m->object,
                m->total_objects * sizeof(void *)
            );
            if(m->object == NULL)
            {
                m->total_objects = 0;
                return(-1);
            }

            m->object[n] = NULL;
        }

        /* Allocate structure. */ 
        switch_ptr = (sar_menu_switch_struct *)calloc(
            1,
            sizeof(sar_menu_switch_struct)
        );
        if(switch_ptr == NULL)  
            return(-1);
	else
	    m->object[n] = (void *)switch_ptr;


        /* Set switch values. */
        switch_ptr->type = SAR_MENU_OBJECT_TYPE_SWITCH;
        switch_ptr->x = x;
        switch_ptr->y = y;
        switch_ptr->width = width;
        switch_ptr->height = height;
        switch_ptr->bg_image = bg_image;
        switch_ptr->switch_off_image = switch_off_image;
        switch_ptr->switch_on_image = switch_on_image;
        memcpy(&switch_ptr->color, fg_color, sizeof(sar_menu_color_struct));
        switch_ptr->font = font;
        switch_ptr->label = StringCopyAlloc(label);
	switch_ptr->state = state;
	switch_ptr->client_data = client_data;
	switch_ptr->id_code = id_code;
	switch_ptr->switch_cb = switch_cb;

	/* Select this object if it is the first one. */
	if(m->total_objects == 1)
	    m->selected_object = 0;

	return(n);
}

/*
 *	Create spin object on menu m.
 *
 *	Returns the object's index or -1 on error.
 */
int SARMenuCreateSpin(
	sar_menu_struct *m,
        double x, double y,
        double width, double height,
        sar_menu_color_struct *label_color,
        sar_menu_color_struct *value_color,
        GWFont *font, GWFont *value_font,
	const char *label,
        const sar_image_struct *label_image,
	const sar_image_struct *value_image,
        void *client_data, int id_code,
        void (*change_cb)(void *, void *, int, char *)
)
{
        int i, n;
        sar_menu_spin_struct *spin_ptr;


        if(m == NULL)
            return(-1);

        if(m->total_objects < 0)
            m->total_objects = 0;

        for(i = 0; i < m->total_objects; i++)
        {
            if(m->object[i] == NULL) 
                break;
        }
        if(i < m->total_objects)
        {
            n = i;
        }
        else
        {
            n = m->total_objects;
            m->total_objects++;

            m->object = (void **)realloc(
                m->object,
                m->total_objects * sizeof(void *)
            );
            if(m->object == NULL)
            {
                m->total_objects = 0;
                return(-1);
            }

            m->object[n] = NULL;
        }

        /* Allocate structure. */
        spin_ptr = (sar_menu_spin_struct *)calloc(
            1,
            sizeof(sar_menu_spin_struct)
        );
        if(spin_ptr == NULL)
            return(-1);
        else
            m->object[n] = (void *)spin_ptr;
        
         
        /* Set spin values. */
        spin_ptr->type = SAR_MENU_OBJECT_TYPE_SPIN;
        spin_ptr->x = x;
        spin_ptr->y = y;
        spin_ptr->width = width;
        spin_ptr->height = height;
        spin_ptr->label_image = label_image;
        spin_ptr->value_image = value_image;
        memcpy(&spin_ptr->label_color, label_color, sizeof(sar_menu_color_struct));
        memcpy(&spin_ptr->value_color, value_color, sizeof(sar_menu_color_struct));
        spin_ptr->font = font;
	spin_ptr->value_font = value_font;
        spin_ptr->label = StringCopyAlloc(label);
        spin_ptr->value_type = SAR_MENU_SPIN_TYPE_STRING;
	spin_ptr->allow_warp = True;
	spin_ptr->step = 1.0;
	spin_ptr->value = NULL;
	spin_ptr->total_values = 0;
	spin_ptr->cur_value = -1;
        spin_ptr->client_data = client_data;
        spin_ptr->id_code = id_code;
        spin_ptr->change_cb = change_cb;

	/* Select this object if it is the first one. */
	if(m->total_objects == 1)
	    m->selected_object = 0;

        return(n);
}

/*
 *	Set value type for spin object n on menu m.
 */
void SARMenuSpinSetValueType(
	sar_menu_struct *m,
	int n, int type
)
{
        sar_menu_spin_struct *spin_ptr;


        if(!SARMenuIsObjectAllocated(m, n))
            return;

        if(*(int *)(m->object[n]) == SAR_MENU_OBJECT_TYPE_SPIN)
            spin_ptr = m->object[n];
        else
            return;

	spin_ptr->value_type = type;

	return;
}


/*
 *	Adds value to spin object n on menu m.
 *
 *	If the spin object is set to SAR_MENU_SPIN_TYPE_NUMERIC then
 *	this will update its current value, otherwise a new value will
 *	be appended to the spin object's list of values.
 */
int SARMenuSpinAddValue(
	sar_menu_struct *m,
	int n,
	const char *value
)
{
	int value_num;
	sar_menu_spin_struct *spin_ptr;


        if(!SARMenuIsObjectAllocated(m, n))
            return(-1);

        if(*(int *)(m->object[n]) == SAR_MENU_OBJECT_TYPE_SPIN)
	    spin_ptr = m->object[n];
	else
            return(-1);

        if(spin_ptr->total_values < 0)
            spin_ptr->total_values = 0;

	/* Handle by value type. */
	switch(spin_ptr->value_type)
	{
	  case SAR_MENU_SPIN_TYPE_STRING:
	    /* Allocate one more pointer. */
            value_num = spin_ptr->total_values;
	    spin_ptr->total_values++;
	    spin_ptr->value = (char **)realloc(
		spin_ptr->value,
		spin_ptr->total_values * sizeof(char *)
	    );
	    if(spin_ptr->value == NULL)
	    {
		spin_ptr->total_values = 0;
		spin_ptr->cur_value = -1;
		return(-1);
	    }
	    /* Now value_num should be a valid pointer, allocate a new
	     * value for it.
	     */
	    spin_ptr->value[value_num] = StringCopyAlloc(value);
	    break;

	  case SAR_MENU_SPIN_TYPE_NUMERIC:
	    /* Have atleast one value? */
	    if(spin_ptr->total_values < 1)
	    {
		/* Need to allocate one value. */
		value_num = 0;
		spin_ptr->total_values = value_num + 1;
		spin_ptr->value = (char **)realloc(
		    spin_ptr->value,
		    spin_ptr->total_values * sizeof(char *)
		);
		if(spin_ptr->value == NULL)
		{
		    spin_ptr->total_values = 0;
		    spin_ptr->cur_value = -1;
		    return(-1);
		}

		spin_ptr->value[value_num] = NULL;
	    }
	    else
	    {
		value_num = 0;
	    }
	    /* Now value_num should be a valid pointer, allocate a new
	     * value string for it.
	     */
	    free(spin_ptr->value[value_num]);
	    spin_ptr->value[value_num] = StringCopyAlloc(value);
	    break;
	}

	/* Update current value. */
	if(spin_ptr->total_values > 0)
	{
	    if(spin_ptr->cur_value < 0)
		spin_ptr->cur_value = 0;
	}

	return(0);
}

/*
 *	Fetches current value, can return NULL on error.
 *
 *	The returned pointer must not be deallocated as it is a pointer
 *	value from the spin object's value list.
 */
char *SARMenuSpinGetCurrentValue(
	sar_menu_struct *m,
	int n,
	int *sel_num
)
{
        int value_num;
        sar_menu_spin_struct *spin_ptr;


	if(sel_num != NULL)
	    (*sel_num) = -1;

        if(!SARMenuIsObjectAllocated(m, n))
            return(NULL);

        if(*(int *)(m->object[n]) == SAR_MENU_OBJECT_TYPE_SPIN)
            spin_ptr = m->object[n];
        else
            return(NULL);

	value_num = spin_ptr->cur_value;
	if((value_num < 0) || (value_num >= spin_ptr->total_values))
	{
	    return(NULL);
	}
	else
	{
	    if(sel_num != NULL)
		(*sel_num) = value_num;

	    return(spin_ptr->value[value_num]);
	}
}

/*
 *	Deletes all values on spin n on menu m.
 */
void SARMenuSpinDeleteAllValues(
	sar_menu_struct *m,
	int n
)
{
        sar_menu_spin_struct *spin_ptr;

        if(!SARMenuIsObjectAllocated(m, n))
            return;

        if(*(int *)(m->object[n]) == SAR_MENU_OBJECT_TYPE_SPIN)
            spin_ptr = m->object[n];
        else
            return;

	StringFreeArray(spin_ptr->value, spin_ptr->total_values);
	spin_ptr->value = NULL;
	spin_ptr->total_values = 0;
	spin_ptr->cur_value = -1;

	return;
}

/*
 *	Spin increment procedure.
 */
static void SARMenuSpinDoInc(
	gw_display_struct *display, sar_menu_struct *m,
	int n                   /* Spin object number on m. */
)
{
	int prev_value_num;
	char *value_ptr;
        sar_menu_spin_struct *spin_ptr;

        if((display == NULL) || !SARMenuIsObjectAllocated(m, n))
            return;

        if(*(int *)(m->object[n]) == SAR_MENU_OBJECT_TYPE_SPIN)
            spin_ptr = m->object[n];
        else
            return;

	/* Record previous value number. */
	prev_value_num = spin_ptr->cur_value;

	/* Increment by value type. */
	switch(spin_ptr->value_type)
	{
	  case SAR_MENU_SPIN_TYPE_STRING:
	    if(spin_ptr->allow_warp ? True :
		(spin_ptr->cur_value < (spin_ptr->total_values - 1))
	    )
	    {
		spin_ptr->cur_value++;
		if(spin_ptr->cur_value >= spin_ptr->total_values)
		    spin_ptr->cur_value = 0;

                DO_REDRAW_OBJECT(display, m, n);

		if((spin_ptr->cur_value >= 0) &&
		   (spin_ptr->cur_value < spin_ptr->total_values)
		)
		    value_ptr = spin_ptr->value[spin_ptr->cur_value];
		else
		    value_ptr = NULL;

		/* Call change callback? */
		if((spin_ptr->change_cb != NULL) &&
		   (prev_value_num != spin_ptr->cur_value) &&
                   (value_ptr != NULL)
		)
		    spin_ptr->change_cb(
			spin_ptr,		/* Object. */
			spin_ptr->client_data,	/* Client data. */
			spin_ptr->id_code,	/* ID code. */
			value_ptr
		    );
	    }
	    break;

          case SAR_MENU_SPIN_TYPE_NUMERIC:
/* Need to work on this. */
	    break;
	}

	return;
}

/*
 *	Spin increment procedure.
 */
static void SARMenuSpinDoDec(
	gw_display_struct *display, sar_menu_struct *m,
	int n                   /* Spin object number on m. */ 
)
{
        int prev_value_num;
        char *value_ptr;
        sar_menu_spin_struct *spin_ptr;

        if((display == NULL) || !SARMenuIsObjectAllocated(m, n))
            return;

        if(*(int *)(m->object[n]) == SAR_MENU_OBJECT_TYPE_SPIN)
            spin_ptr = m->object[n];
        else
            return;

        /* Record previous value number. */
        prev_value_num = spin_ptr->cur_value;

        /* Decrement by value type. */
        switch(spin_ptr->value_type)
        {
          case SAR_MENU_SPIN_TYPE_STRING:
            if(spin_ptr->allow_warp ? True :
                (spin_ptr->cur_value > 0)
            )
            {
                spin_ptr->cur_value--;
                if(spin_ptr->cur_value < 0)
                    spin_ptr->cur_value = spin_ptr->total_values - 1;

		DO_REDRAW_OBJECT(display, m, n);

                if((spin_ptr->cur_value >= 0) &&
                   (spin_ptr->cur_value < spin_ptr->total_values)
                )
                    value_ptr = spin_ptr->value[spin_ptr->cur_value];
                else
                    value_ptr = NULL;

                /* Call change callback? */
                if((spin_ptr->change_cb != NULL) &&
                   (prev_value_num != spin_ptr->cur_value) &&
                   (value_ptr != NULL)
                )
                    spin_ptr->change_cb(
                        spin_ptr,               /* Object. */
                        spin_ptr->client_data,  /* Client data. */
                        spin_ptr->id_code,      /* ID code. */
                        value_ptr
                    );
            }
            break;

          case SAR_MENU_SPIN_TYPE_NUMERIC:
/* Need to work on this. */
            break;
        }

        return;
}


/*
 *	Deallocates object n on menu m.
 */
void SARMenuDestroyObject(
	sar_menu_struct *m,
	int n
)
{
	if(m == NULL)
	    return;

	if(SARMenuIsObjectAllocated(m, n))
	{
	    int type;
	    void *ptr;
	    sar_menu_label_struct *label_ptr;
	    sar_menu_button_struct *button_ptr;
	    sar_menu_progress_struct *progress_ptr;
	    sar_menu_message_box_struct *mesgbox_ptr;
	    sar_menu_list_struct *list_ptr;
	    sar_menu_switch_struct *switch_ptr;
	    sar_menu_spin_struct *spin_ptr;
	    sar_menu_map_struct *map_ptr;


	    ptr = m->object[n];
	    type = (*(int *)ptr);
	    switch(type)
	    {
	      case SAR_MENU_OBJECT_TYPE_LABEL:
		label_ptr = ptr;
/* Shared
		SARImageDestroy(label_ptr->image);
 */
		free(label_ptr->label);
		free(label_ptr);
                break;

              case SAR_MENU_OBJECT_TYPE_BUTTON:
                button_ptr = ptr;
/* Shared
		SARImageDestroy(button_ptr->unarmed_image);
                SARImageDestroy(button_ptr->armed_image);
                SARImageDestroy(button_ptr->highlighted_image);
 */
                free(button_ptr->label);
		free(button_ptr);
                break;

              case SAR_MENU_OBJECT_TYPE_PROGRESS:
		progress_ptr = ptr;
/* Shared
                SARImageDestroy(progress_ptr->bg_image);
                SARImageDestroy(progress_ptr->fg_image);
 */
                free(progress_ptr->label);
                free(progress_ptr);
                break;

	      case SAR_MENU_OBJECT_TYPE_MESSAGE_BOX:
		mesgbox_ptr = ptr;
/* Shared
                SARImageDestroy(mesgbox_ptr->image);
 */
		free(mesgbox_ptr->message);
		free(mesgbox_ptr);
		break;

	      case SAR_MENU_OBJECT_TYPE_LIST:
                list_ptr = ptr;
		SARMenuListDeleteAllItems(m, n);
/* Shared
                SARImageDestroy(list_ptr->bg_image);
 */
                free(list_ptr->item);
		free(list_ptr->label);
                free(list_ptr);
                break;

	      case SAR_MENU_OBJECT_TYPE_SWITCH:
		switch_ptr = ptr;

		free(switch_ptr->label);
		free(switch_ptr);
		break;

	      case SAR_MENU_OBJECT_TYPE_SPIN:
		spin_ptr = ptr;

		SARMenuSpinDeleteAllValues(m, n);
		free(spin_ptr->label);
		free(spin_ptr);
		break;

	      case SAR_MENU_OBJECT_TYPE_MAP:
		map_ptr = ptr;
		SARMenuMapDeleteAllMarkings(map_ptr);
		V3DTextureDestroy(map_ptr->bg_tex);
		map_ptr->bg_tex = NULL;
		free(map_ptr);
		break;

	      default:
		fprintf(
		    stderr,
 "SARMenuDestroyObject(): Destroying unknown type `%i'\n",
		    type
		);
		free(ptr);
		break;
	    }

	    m->object[n] = NULL;
	}

	return;
}


/*
 *	Updates the first progress bar object on the menu (if any)
 *	with the specified progress position.
 */
void SARMenuSetProgressSimple(
	gw_display_struct *display,
	sar_menu_struct *m,
	double progress,
	int redraw
)
{
	int i;
	void *ptr;
	int type;
        sar_menu_progress_struct *progress_ptr;


	if((display == NULL) ||
           (m == NULL)
	)
	    return;

	if(progress < 0)
	    progress = 0;
	else if(progress > 1)
	    progress = 1;

	for(i = 0; i < m->total_objects; i++)
	{
	    ptr = m->object[i];
	    if(ptr == NULL)
		continue;

	    type = *(int *)ptr;
	    if(type == SAR_MENU_OBJECT_TYPE_PROGRESS)
	    {
		progress_ptr = ptr;
		progress_ptr->progress = progress;

		if(redraw)
		    SARMenuDrawObject(display, m, i);

		break;
	    }
	}
}

/*
 *      Updates the first message box object on the menu (if any)
 *      with the specified message.
 */
void SARMenuSetMessageBoxSimple(
        gw_display_struct *display,
        sar_menu_struct *m,
        const char *message,
        int redraw
)
{
        int i;
        void *ptr;
        int type;
	sar_menu_message_box_struct *mesgbox_ptr;


        if((display == NULL) ||
           (m == NULL)
        )
            return;

        for(i = 0; i < m->total_objects; i++)
        {
            ptr = m->object[i];
            if(ptr == NULL)
                continue;

            type = *(int *)ptr;
            if(type == SAR_MENU_OBJECT_TYPE_MESSAGE_BOX)
            {
                mesgbox_ptr = ptr;

		free(mesgbox_ptr->message);
                mesgbox_ptr->message = StringCopyAlloc(message);

		mesgbox_ptr->scrolled_line = 0;

                if(redraw)
                    SARMenuDrawObject(display, m, i);

                break;
            }
        }
}


/*
 *	Draws one line of string starting at buf_ptr and does not
 *	draw more than the number of whole word characters specified by
 *	visible_columns.
 *
 *	Returns the pointer to the next word that lies within buf_ptr
 *	or NULL if a '\0' character was encountered while drawing the 
 *	string.
 */
static char *SARMenuMessageBoxDrawString(
        gw_display_struct *display,
        int x, int y,
        char *buf_ptr,
	int visible_columns, int font_width,
	Boolean draw_line,
        sar_menu_color_struct *default_color,
        sar_menu_color_struct *bold_color,
        sar_menu_color_struct *underline_color
)
{
	char c;
	int characters_drawn;
	char *strptr, *buf_stop;


	if((display == NULL) || (buf_ptr == NULL) || (visible_columns < 1))
	    return(NULL);

	/* If given buffer points to a end of buffer then return NULL. */
	if((*buf_ptr) == '\0')
	    return(NULL);

	/* If given string points to a newline then just return the 
	 * pointer to the next character. This will effectivly draw one
	 * empty line.
	 */
	if(ISCR(*buf_ptr))
	    return(buf_ptr + 1);

        /* Seek past initial spaces. */
        while(ISBLANK(*buf_ptr))
            buf_ptr++;

	/* Probe to end of line and mark it by setting buf_stop to point 
	 * to that end.
	 */
	buf_stop = NULL;
	strptr = buf_ptr;
	for(characters_drawn = 0; characters_drawn < visible_columns;)
	{
	    c = *strptr;

	    /* End of buffer? */
	    if(c == '\0')
	    {
		buf_stop = strptr;
		break;
	    }
	    /* Blank character? */
	    else if(ISBLANK(c))
	    {
		/* Mark end of buffer at this blank character and count it
		 * as one character drawn (regardless of how many blank
		 * characters follow).
		 */
		buf_stop = strptr;
		characters_drawn++;

		/* Seek past this and any following blank characters. */
		while(ISBLANK(*strptr))
		    strptr++;
	    }
	    /* New line? */
	    else if(ISCR(c))
	    {
		buf_stop = strptr;
		break;
	    }
	    /* Escape sequence? */
	    else if(IS_CHAR_ESC_START(c))
	    {
		/* Skip this character. */
		strptr++;

		/* Seek to end of escape sequence or end of buffer. */
		while(((*strptr) != '\0') &&
                      !IS_CHAR_ESC_END(*strptr)
		)
		    strptr++;
		if((*strptr) != '\0')
		    strptr++;
	    }
	    /* Regular character. */
	    else
	    {
		/* Mark this as a character to be drawn and increment to
		 * next character.
		 */
		characters_drawn++;
		strptr++;
	    }
	}

	/* If end of buffer was not found then assume it is at strptr. */
	if(buf_stop == NULL)
	    buf_stop = strptr;


	/* Draw from buf_ptr to (but not including) buf_stop. */
	characters_drawn = 0;
	strptr = buf_ptr;
	while(strptr < buf_stop)
	{
	    c = *strptr;

	    /* End of buffer? */
            if(c == '\0')
            {
                break;
            }
            /* Blank character? */
            else if(ISBLANK(c))
            {
                /* Draw one blank character and seek to next non-blank
		 * character (regardless of how many blank characters
		 * follow).
                 */
                if(draw_line)
                    GWDrawCharacter(
                        display,
                        x + (characters_drawn * font_width),
                        y,
                        c
                    );
                characters_drawn++;

                /* Seek past this and any following blank characters. */
                while(ISBLANK(*strptr))
                    strptr++;
            }
	    /* Escape sequence? */
            else if(IS_CHAR_ESC_START(c))
            {
		int n;
		char tag_name[256];


                /* Skip this character. */
                strptr++;

                /* Seek to end of escape sequence or end of buffer. */
		n = 0;
                while(((*strptr) != '\0') &&
                      !IS_CHAR_ESC_END(*strptr)
                )
		{
		    if(n < 254)
		    {
			tag_name[n] = *strptr;
			n++;
		    }
                    strptr++;
		}
                if((*strptr) != '\0')
                    strptr++;

		/* Null terminate tag_name. */
		tag_name[n] = '\0';

		/* Handle tag. */
		/* Default? */
		if(!strcasecmp(tag_name, "default") ||
		   !strcasecmp(tag_name, "plain") ||
                   !strcasecmp(tag_name, "regular")
		)
		{
		    if(default_color != NULL)
			glColor4d(
			    default_color->r,
                            default_color->g,
                            default_color->b,
                            default_color->a
			);
		}
		/* Bold? */
		else if(!strcasecmp(tag_name, "bold"))
		{
                    if(bold_color != NULL)
                        glColor4d(
                            bold_color->r,
                            bold_color->g,
                            bold_color->b,
                            bold_color->a
                        );
		}
                /* Underline? */
                else if(!strcasecmp(tag_name, "underline") ||
		        !strcasecmp(tag_name, "underlined")
		)
                {
                    if(underline_color != NULL)
                        glColor4d(
                            underline_color->r,
                            underline_color->g,
                            underline_color->b,
                            underline_color->a
                        );
                }

            }
	    else
	    {
		/* Actually draw this character? */
		if(draw_line)
		    GWDrawCharacter(
			display,
			x + (characters_drawn * font_width),
			y,
			c
		    );
		characters_drawn++;
		strptr++;
	    }
	}

	/* Return the pointer to the next character that was not
	 * drawn.
	 */
	return(buf_stop);
}

/*
 *	Returns the total number of lines by seeking through the given
 *	buffer.
 */
static int SARMenuMessageBoxTotalLines(char *buf_ptr, int visible_columns)
{
	char c;
	int line_count = 0, characters_drawn;
	char *strptr, *buf_stop;


	if((buf_ptr == NULL) || (visible_columns < 1))
	    return(line_count);

	/* Itterate untill end of buffer. */
	while((buf_ptr != NULL) ? ((*buf_ptr) != '\0') : False)
	{
	    /* Set strptr to point to the start of current line. */
	    strptr = buf_ptr;

	    /* If strptr is on a newline then seek to next character and
	     * count this as one line.
	     */
	    if(ISCR(*strptr))
	    {
		buf_ptr = strptr + 1;
		line_count++;
		continue;
	    }

	    /* Seek past initial spaces. */
	    while(ISBLANK(*strptr))
		strptr++;

	    /* Reset stop buffer and seek to end of `line'. Remember that
	     * characters_drawn represents the number of characters that
	     * would be drawn to fit on one line with whole words. Not the
	     * the actual characters drawn since nothing is being drawn in
	     * this function.
	     */
	    buf_stop = NULL;
	    for(characters_drawn = 0; characters_drawn < visible_columns;)
	    {
		c = *strptr;

                /* End of buffer? */
                if(c == '\0')
                {
		    buf_stop = strptr;
		    break;
		}
		/* Blank character? */
		else if(ISBLANK(c))
		{
		    /* Count blank character as one character drawn and 
		     * seek to next non-blank character.
		     */
		    buf_stop = strptr;
		    characters_drawn++;

		    /* Seek past this and any following blank characters. */
		    while(ISBLANK(*strptr))
			strptr++;
		}
		/* New line? */
		else if(ISCR(c))
		{
		    buf_stop = strptr;
		    break;
		}
                /* Escape sequence? */
                else if(IS_CHAR_ESC_START(c))
                {
                    /* Skip this character. */
                    strptr++;

                    /* Seek to end of escape sequence or end of buffer. */
                    while(((*strptr) != '\0') &&
                          !IS_CHAR_ESC_END(*strptr)
                    )
                        strptr++;
                    if((*strptr) != '\0')
                        strptr++;
                }
                /* Regular character. */
                else
                {
		    /* Mark this as a character to be drawn and increment
		     * to next character.
		     */
		    characters_drawn++;
		    strptr++;
		}
	    }
	    /* If not able to find end of line pointer then assume end of
	     * line is at current position of strptr.
	     */
	    if(buf_stop == NULL)
		buf_stop = strptr;

	    /* End of line reached, count this as one line and set buffer
	     * to start of next line.
	     */
	    line_count++;
	    buf_ptr = buf_stop;
	}

	return(line_count);
}

/*
 *	Redraws a `window' comprised of 9 images where the first image
 *	(index 0) is the background and the second image (index 1) through
 *	the 9th image (index 8) are the 8 decoration edges starting from
 *	the upper left corner.
 */
static void SARMenuDrawWindowBG(
        gw_display_struct *display,
        const sar_image_struct **bg_image,      /* Total 9 images. */
        int x, int y,
        int width, int height,
	Boolean draw_base,
	Boolean draw_shadow
)
{
	int corner_width = 0, corner_height = 0;
	int vframe_width = 0, hframe_height = 0;
	const sar_image_struct *img_ptr;


	if((display == NULL) || (bg_image == NULL))
	    return;

	/* Get upper left corner image. */
	img_ptr = bg_image[1];
	if(img_ptr != NULL)
	{
	    /* Get size of each corner by assuming they're all the same
	     * and take the size of the upper left corner image.
	     */
	    corner_width = MAX(img_ptr->width, 0);
	    corner_height = MAX(img_ptr->height, 0);
	}

	/* Get upper horizontal frame image. */
	img_ptr = bg_image[2];
        if(img_ptr != NULL)
        {
	    /* Get height of both horizontal frames by taking the size of
	     * the upper horizontal frame image.
	     */
	    hframe_height = MAX(img_ptr->height, 0);
	}

        /* Get right vertical frame image. */
        img_ptr = bg_image[4];
        if(img_ptr != NULL)
        {
            /* Get width of both vertical frames by taking the size of
             * the right vertical frame image.
             */
            vframe_width = MAX(img_ptr->width, 0);
        }

	/* Adjust given geometry to fit `within' the decorations. */
	x += corner_width;
	y += corner_height;
	width = MAX(width - (2 * corner_width), 0);
	height = MAX(height - (2 * corner_height), 0);

        if((width < 1) || (height < 1))
            return;

	/* Draw shadow. */
	if(draw_shadow)
	{
	    StateGLBoolean	last_alpha_test = display->state_gl.alpha_test,
				last_blend = display->state_gl.blend;

	    StateGLDisable(&display->state_gl, GL_ALPHA_TEST);
	    StateGLEnable(&display->state_gl, GL_BLEND);
	    StateGLBlendFunc(&display->state_gl, GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

	    glColor4d(0.0, 0.0, 0.0, 0.25);

	    glBegin(GL_QUADS);
	    {
		glVertex2i(
		    (GLint)(x + width + (0.5 * corner_width)),
		    (GLint)(display->height - (y - (0.5 * corner_height)))
		);
                glVertex2i(
		    (GLint)(x + width + (1.5 * corner_width)),
		    (GLint)(display->height - (y - (0.5 * corner_height)))
		);
                glVertex2i(
                    (GLint)(x + width + (1.5 * corner_width)),
                    (GLint)(display->height - (y + height + (1.5 * corner_height)))
                );
                glVertex2i(
                    (GLint)(x + width + (0.5 * corner_width)),
                    (GLint)(display->height - (y + height + (1.5 * corner_height)))
                );

                glVertex2i(
                    (GLint)(x - (0.5 * corner_width)),
                    (GLint)(display->height - (y + height + (0.5 * corner_height)))
                );
                glVertex2i(
                    (GLint)(x + width + (0.5 * corner_width)),
                    (GLint)(display->height - (y + height + (0.5 * corner_height)))
                );
                glVertex2i(
                    (GLint)(x + width + (0.5 * corner_width)),
                    (GLint)(display->height - (y + height + (1.5 * corner_height)))
                );
                glVertex2i(
                    (GLint)(x - (0.5 * corner_width)),
                    (GLint)(display->height - (y + height + (1.5 * corner_height)))
                );
	    }
	    glEnd();



	    if(last_alpha_test)
		StateGLEnable(&display->state_gl, GL_ALPHA_TEST);
	    if(!last_blend)
		StateGLDisable(&display->state_gl, GL_BLEND);
	}

	/* Draw base image at center, scaled to fit the width and height
	 * regardless of the actual image size.
	 */
	if(draw_base)
	{
	    img_ptr = bg_image[0];
	    SARImageDraw(
		display, img_ptr,
		x - 1, y - 1, width + 2, height + 2
	    );
	}

	/* Draw frame decorations (not including corners) next. */
        img_ptr = bg_image[2];
        SARImageDraw(
            display, img_ptr,
            x - 1, y - hframe_height,
            width + 2, hframe_height
        );

        img_ptr = bg_image[4];
        SARImageDraw(
            display, img_ptr,
            x + width, y - 1,
            vframe_width, height + 2
        );

        img_ptr = bg_image[6];
        SARImageDraw(
            display, img_ptr,
            x - 1, y + height,
            width + 2, hframe_height
        );

        img_ptr = bg_image[8];
        SARImageDraw(
            display, img_ptr,
            x - vframe_width, y - 1,
            vframe_width, height + 2
        );


	/* Draw corners. */
        img_ptr = bg_image[1];
        SARImageDraw(
            display, img_ptr,
            x - corner_width, y - corner_height,
            corner_width, corner_height
        );

        img_ptr = bg_image[3];
        SARImageDraw(
            display, img_ptr,
            x + width, y - corner_height,
            corner_width, corner_height
        );

        img_ptr = bg_image[5];
        SARImageDraw(
            display, img_ptr,
            x + width, y + height,
            corner_width, corner_height
        );

        img_ptr = bg_image[7];
        SARImageDraw(
            display, img_ptr,
            x - corner_width, y + height,
            corner_width, corner_height
        );
}


/*
 *	Draws the specified object n.
 */
static void SARMenuDoDrawObject(
        gw_display_struct *display, sar_menu_struct *m,
        int n,                  /* Object number on m. */
        Boolean draw_shadows
)
{
	int i, x, y, xc, yc, w = 10, h = 10, width, height, len, type;
	int fw, fh;
	int items_visable, chars_per_row;
	char *strptr;
	void *ptr;
	sar_menu_label_struct *label_ptr;
	sar_menu_button_struct *button_ptr;
	sar_menu_progress_struct *progress_ptr;
	sar_menu_message_box_struct *mesgbox_ptr;
	sar_menu_list_struct *list_ptr;
	sar_menu_list_item_struct *item_ptr;
	sar_menu_switch_struct *switch_ptr;
	sar_menu_spin_struct *spin_ptr;
	sar_menu_map_struct *map_ptr;

#define item_name_len	256
	char item_name[item_name_len];

	u_int8_t scroll_cursor_up_bm[] = {
0xff, 0xff, 
0x80, 0x01, 
0xbf, 0xfd, 
0xbf, 0xfd, 
0x9f, 0xf9, 
0x9f, 0xf9, 
0x8f, 0xf1, 
0x8f, 0xf1, 
0x87, 0xe1, 
0x87, 0xe1, 
0x83, 0xc1, 
0x83, 0xc1, 
0x81, 0x81, 
0x81, 0x81, 
0x80, 0x01,
0xff, 0xff
	}; 

        u_int8_t scroll_cursor_down_bm[] = {
0xff, 0xff,
0x80, 0x01,
0x81, 0x81,
0x81, 0x81, 
0x83, 0xc1, 
0x83, 0xc1,
0x87, 0xe1,
0x87, 0xe1,
0x8f, 0xf1,
0x8f, 0xf1, 
0x9f, 0xf9,
0x9f, 0xf9,
0xbf, 0xfd, 
0xbf, 0xfd,
0x80, 0x01,         
0xff, 0xff
        };          

        u_int8_t scroll_cursor_left_bm[] = {
0xff, 0xff, 
0x80, 0x01, 
0x80, 0x0d, 
0x80, 0x3d, 
0x80, 0xfd, 
0x83, 0xfd, 
0x8f, 0xfd, 
0xbf, 0xfd, 
0xbf, 0xfd, 
0x8f, 0xfd, 
0x83, 0xfd, 
0x80, 0xfd, 
0x80, 0x3d, 
0x80, 0x0d, 
0x80, 0x01, 
0xff, 0xff
        };

        u_int8_t scroll_cursor_right_bm[] = {
0xff, 0xff,
0x80, 0x01,
0xb0, 0x01,
0xbc, 0x01,
0xbf, 0x01,
0xbf, 0xc1,
0xbf, 0xf1,
0xbf, 0xfd,
0xbf, 0xfd,
0xbf, 0xf1,
0xbf, 0xc1,
0xbf, 0x01,
0xbc, 0x01,
0xb0, 0x01,
0x80, 0x01,
0xff, 0xff
	};

	const sar_image_struct *image;


        if((display == NULL) || (m == NULL))
            return;

	if(SARMenuIsObjectAllocated(m, n))
	    ptr = m->object[n];
	else
	    return;

	type = *(int *)ptr;

        width = display->width;
        height = display->height;

#define DO_DRAW_IMAGE	\
if(image != NULL) \
{ \
 SARImageDraw( \
  display, \
  image, \
  (int)(x - (w / 2)), \
  (int)(y - (h / 2)), \
  w, h \
 ); \
}

	switch(type)
	{
	  /* ******************************************************** */
          case SAR_MENU_OBJECT_TYPE_LABEL:
	    label_ptr = ptr;
	    w = label_ptr->width;
	    h = label_ptr->height;
	    x = (int)(label_ptr->x * width) + label_ptr->offset_x;
	    y = (int)(label_ptr->y * height) + label_ptr->offset_y;
	    image = label_ptr->image;
	    DO_DRAW_IMAGE	    

	    strptr = label_ptr->label;
	    if(strptr != NULL)
	    {
                GWGetFontSize(label_ptr->font, NULL, NULL, &fw, &fh);
	        GWSetFont(display, label_ptr->font);
		glColor4d(
		    label_ptr->color.r,
                    label_ptr->color.g,
                    label_ptr->color.b,
                    label_ptr->color.a
		);
		len = strlen(strptr);
		switch(label_ptr->align)
		{
		  case SAR_MENU_LABEL_ALIGN_RIGHT:
                    GWDrawString(
                        display,
                        x + ((w / 2) - fw - (len * fw)),
			y - (fh / 2),
                        strptr
                    );
		  case SAR_MENU_LABEL_ALIGN_LEFT:
                    GWDrawString(
                        display,
                        x - ((w / 2) + fw),
			y - (fh / 2),
                        strptr
                    );
		    break;
		  default:	/* SAR_MENU_LABEL_ALIGN_CENTER */
		    GWDrawString(
                        display,
                        x - (fw * len / 2),
			y - (fh / 2),
                        strptr
                    );
		    break;
		}
	    }
            break;

          /* ******************************************************** */
          case SAR_MENU_OBJECT_TYPE_BUTTON:
            button_ptr = ptr;
	    w = button_ptr->width;
	    h = button_ptr->height;
            x = (int)(button_ptr->x * width) + button_ptr->offset_x;
            y = (int)(button_ptr->y * height) + button_ptr->offset_y;
	    /* Select image to use based on button state. */
	    switch(button_ptr->state)
	    {
              case SAR_MENU_BUTTON_STATE_HIGHLIGHTED: 
                image = button_ptr->highlighted_image;
                break;

	      case SAR_MENU_BUTTON_STATE_ARMED:
                image = button_ptr->armed_image;
		break;

	      default:
		image = button_ptr->unarmed_image;
                break;
	    }
	    /* If image is not available, then fall back to unarmed
	     * image.
	     */
	    if(image == NULL)
		image = button_ptr->unarmed_image;
	    DO_DRAW_IMAGE

	    /* Draw label. */
            strptr = button_ptr->label;
            if(strptr != NULL)
            {
		int x_min, x_max, y_min, y_max;

		x_min = x - (w / 2);
		x_max = x_min + w;
		y_min = y - (h / 2);
		y_max = y_min + h;

		len = strlen(strptr);
                GWGetFontSize(button_ptr->font, NULL, NULL, &fw, &fh);
                GWSetFont(display, button_ptr->font);

                glColor4d(
                    button_ptr->color.r,
                    button_ptr->color.g,
                    button_ptr->color.b,
                    button_ptr->color.a
                );
 
                GWDrawString(
                    display,
                    x - (len * fw / 2),
		    y - (fh / 2),
                    strptr
                );

		/* Do not draw outlines for buttons. */
            }
            break;

          /* ******************************************************** */
          case SAR_MENU_OBJECT_TYPE_PROGRESS:
            progress_ptr = ptr;

	    x = (int)(progress_ptr->x * width);
	    y = (int)(progress_ptr->y * height);

	    if(progress_ptr->width > 0)
		w = progress_ptr->width;
	    else
		w = width - (2 * DEF_XMARGIN);

            if(progress_ptr->height > 0)
                h = progress_ptr->height;
            else
                h = DEF_PROGRESS_HEIGHT;

            image = progress_ptr->bg_image;
	    if(image != NULL)
	    {
		SARImageDraw(
		    display,
		    image,
		    x - (w / 2),
		    y - (h / 2),
		    w, h
		);
	    }
            image = progress_ptr->fg_image;
            if(image != NULL)
            {
                SARImageDraw(
                    display,
                    image,
                    x - (w / 2),
                    y - (h / 2),
		    (int)(w * progress_ptr->progress), (int)h
                );
            }
            break;

          /* ******************************************************** */
          case SAR_MENU_OBJECT_TYPE_MESSAGE_BOX:
            mesgbox_ptr = (sar_menu_message_box_struct *)ptr;

            x = (int)(mesgbox_ptr->x * width);
            y = (int)(mesgbox_ptr->y * height);

            w = (int)(mesgbox_ptr->width * width);
            if(w <= 0)
                w = width - (2 * DEF_MB_XMARGIN);
            if(w <= 0)
                w = DEF_MB_XMARGIN;

            if(mesgbox_ptr->height > 0)
                h = (int)(mesgbox_ptr->height * height);
            else
                h = DEF_HEIGHT;
            if(h <= 0)
                h = DEF_MB_YMARGIN;

            xc = MAX(x - (w / 2), 0);
            yc = MAX(y - (h / 2), 0);

	    SARMenuDrawWindowBG(
		display, mesgbox_ptr->bg_image,
		xc, yc,
                w, h,
		True,		/* Draw base. */
		draw_shadows	/* Draw shadow. */
	    );

            GWGetFontSize(
                mesgbox_ptr->font,
                NULL, NULL,
                &fw, &fh
            );

            GWSetFont(display, mesgbox_ptr->font);
            glColor4d(
                mesgbox_ptr->color.r,
                mesgbox_ptr->color.g,
                mesgbox_ptr->color.b,
		mesgbox_ptr->color.a
            );

	    /* Has message to be drawn? */
	    if(mesgbox_ptr->message != NULL)
	    {
		Boolean actually_draw_line;
		char *buf_ptr;
		int lines_processed = 0, lines_drawn = 0;
		int lines_to_skip = mesgbox_ptr->scrolled_line;
		int	mesg_x, mesg_y,
			lines_visable, columns_visable, scroll_width;
		sar_menu_color_struct bold_color, underline_color;


		/* Calculate width (height) of scroll indicator. */
		scroll_width = fh + (2 * DEF_MB_YMARGIN);

		/* Calculate lines visable. */
		if(fh > 0)
		    lines_visable = (h - (2 * DEF_MB_YMARGIN)) / fh;
		else
		    lines_visable = 0;
		if(lines_visable < 0)
		    lines_visable = 0;

		/* Calculate columns visable (include margin for arrows). */
		if(fw > 0)
		    columns_visable = (w - DEF_SCROLL_CURSOR_WIDTH
			- (2 * DEF_MB_XMARGIN)) / fw;
		else
		    columns_visable = 1;
		/* Must have atleast one column visible. */
		if(columns_visable < 1)
		    columns_visable = 1;

                /* Draw scroll up and scroll down arrows. */
                glRasterPos2i(
                    xc + w - DEF_SCROLL_CURSOR_WIDTH - DEF_MB_XMARGIN,
                    height - (yc + DEF_SCROLL_CURSOR_HEIGHT + DEF_MB_YMARGIN)
                );
                glBitmap(
                    DEF_SCROLL_CURSOR_WIDTH,
                    DEF_SCROLL_CURSOR_HEIGHT,
                    0.0, 0.0,
                    DEF_SCROLL_CURSOR_WIDTH, 0.0,
                    scroll_cursor_up_bm 
                );
                glRasterPos2i(
                    xc + w - DEF_SCROLL_CURSOR_WIDTH - DEF_MB_XMARGIN,
                    height - (y + (h / 2) - DEF_MB_YMARGIN)
                );
                glBitmap(
                    DEF_SCROLL_CURSOR_WIDTH,
                    DEF_SCROLL_CURSOR_HEIGHT,
                    0.0, 0.0,
                    DEF_SCROLL_CURSOR_WIDTH, 0.0,
                    scroll_cursor_down_bm
                );

		/* Set up additional font colors. */
		bold_color.a = 1.0;
		bold_color.r = 1.0;
		bold_color.g = 1.0;
		bold_color.b = 0.0;

		underline_color.a = 1.0;
		underline_color.r = 0.5;
		underline_color.g = 0.5;
		underline_color.b = 1.0;

		/* Get pointer to start of message buffer and begin
		 * drawing lines.
		 */
		buf_ptr = mesgbox_ptr->message;
		lines_processed = 0;
		mesg_x = xc + DEF_MB_XMARGIN;
		mesg_y = yc + DEF_MB_YMARGIN;
		do
		{
		    actually_draw_line = ((lines_processed >= lines_to_skip) ?
			True : False
		    );

		    buf_ptr = SARMenuMessageBoxDrawString(
			display,
			mesg_x, mesg_y,
			buf_ptr,
			columns_visable, fw, actually_draw_line,
			&mesgbox_ptr->color,
			&bold_color,
			&underline_color
		    );
		    lines_processed++;

		    if(actually_draw_line)
		    {
			lines_drawn++;
			mesg_y += fh;
		    }
		}
		while((buf_ptr != NULL) && (lines_drawn < lines_visable));
	    }

            /* Draw outline if selected. */
            if(m->selected_object == n)
            {  
                int x_min, x_max, y_min, y_max;
                  
                x_min = x - (w / 2);
                x_max = x_min + w;
                y_min = y - (h / 2);
                y_max = y_min + h;

                glColor4d(0.0, 0.0, 0.0, 1.0);

                GWDrawLine(display, x_min, y_max, x_min, y_min);
                GWDrawLine(display, x_min, y_min, x_max, y_min);
                GWDrawLine(display, x_max, y_min, x_max, y_max);
                GWDrawLine(display, x_min, y_max, x_max, y_max);
            }
	    break;

          /* ******************************************************** */
          case SAR_MENU_OBJECT_TYPE_LIST:
            list_ptr = ptr;

            x = (int)(list_ptr->x * width);
            y = (int)(list_ptr->y * height);

            w = (int)(list_ptr->width * width);
	    if(w <= 0)
		w = width - (2 * DEF_LIST_XMARGIN);
	    if(w <= 0)
		w = DEF_LIST_XMARGIN;

            if(list_ptr->height > 0)
                h = (int)(list_ptr->height * height);
            else
                h = DEF_HEIGHT;
	    if(h <= 0)
		h = DEF_LIST_YMARGIN;

	    xc = MAX(x - (w / 2), 0);
	    yc = MAX(y - (h / 2), 0);

            SARMenuDrawWindowBG(
                display, list_ptr->bg_image,
                xc, yc,
                w, h,
		True,		/* Draw base. */
		draw_shadows	/* Draw shadow. */
            );

	    GWGetFontSize(
		list_ptr->font,
		NULL, NULL,
		&fw, &fh
	    );
            /* Calculate items visible, remember to subtract two
             * items for borders and heading.
             */
            if(fh > 0)
                items_visable = MAX(
                    (h / fh) - 2,
                    0
                );
            else  
                items_visable = 0;
	    /* Update items visable on list. */
	    list_ptr->items_visable = items_visable;


	    /* Recalculate scroll position if too great. */
	    if((list_ptr->total_items - list_ptr->scrolled_item) <
		items_visable
	    )
	        list_ptr->scrolled_item = MAX(
		    list_ptr->total_items - items_visable,
		    0
		);


            /* Calculate len as lines visable + yc. */
	    len = yc + h;

	    /* Calculate characters per row. */
	    if(fw > 0)
		chars_per_row = MAX(
		    (w - DEF_SCROLL_CURSOR_WIDTH -
			(3 * DEF_LIST_XMARGIN)) / fw,
		    1
		);
	    else
		chars_per_row = 1;

	    /* Draw heading label. */
	    yc += DEF_LIST_YMARGIN;
            GWSetFont(display, list_ptr->font);
            glColor4d(
                list_ptr->color.r,
                list_ptr->color.g,
                list_ptr->color.b,
                list_ptr->color.a
            );
            GWDrawString(   
                display,  
                xc + DEF_LIST_XMARGIN, yc,
                list_ptr->label
            );
            yc += fh;

	    /* Draw scroll up and down arrows. */
	    glRasterPos2i(
		xc + w - DEF_SCROLL_CURSOR_WIDTH - DEF_LIST_XMARGIN,
		height - (yc + DEF_SCROLL_CURSOR_HEIGHT)
	    );
            glBitmap(
                DEF_SCROLL_CURSOR_WIDTH,
                DEF_SCROLL_CURSOR_HEIGHT,
                0.0, 0.0,
                DEF_SCROLL_CURSOR_WIDTH, 0.0,
                scroll_cursor_up_bm
            );
            glRasterPos2i(
                xc + w - DEF_SCROLL_CURSOR_WIDTH - DEF_LIST_XMARGIN,
                height - (y + (h / 2) - DEF_LIST_YMARGIN)
            );
            glBitmap(
                DEF_SCROLL_CURSOR_WIDTH,
                DEF_SCROLL_CURSOR_HEIGHT,
                0.0, 0.0,
                DEF_SCROLL_CURSOR_WIDTH, 0.0,
                scroll_cursor_down_bm
            );

	    /* Sanitize scrolled item position. */
	    if(list_ptr->scrolled_item < 0)
		list_ptr->scrolled_item = 0;

	    /* Draw each item. */
	    for(i = list_ptr->scrolled_item; i < list_ptr->total_items; i++)
	    {
		item_ptr = list_ptr->item[i];
		if(item_ptr == NULL)
		    continue;

		if((yc + fh) >= len)
		    break;

		if(item_ptr->name == NULL)
		    continue;

		strncpy(item_name, item_ptr->name, item_name_len);
		if(chars_per_row < item_name_len)
		    item_name[chars_per_row] = '\0';
		else
		    item_name[item_name_len - 1] = '\0';

                GWDrawString(
                    display,
                    xc + (1 * fw) + DEF_LIST_XMARGIN,
		    yc,
		    item_name
		);
		if(i == list_ptr->selected_item)
		    GWDrawString(
                    display, 
                    xc + DEF_LIST_XMARGIN, yc,
                    ">"
                );

		yc += fh;
	    }

	    /* Draw outline if selected. */
            if(m->selected_object == n)
            {
                int x_min, x_max, y_min, y_max;

                x_min = x - (w / 2);
                x_max = x_min + w;
                y_min = y - (h / 2);
                y_max = y_min + h + 1;

		glColor4d(0.0, 0.0, 0.0, 1.0);

                GWDrawLine(display, x_min, y_max, x_min, y_min);
                GWDrawLine(display, x_min, y_min, x_max, y_min);
                GWDrawLine(display, x_max, y_min, x_max, y_max);
                GWDrawLine(display, x_min, y_max, x_max, y_max);
	    }
            break;

          /* ******************************************************** */
          case SAR_MENU_OBJECT_TYPE_SWITCH:
            switch_ptr = ptr;
            w = switch_ptr->width;
            h = switch_ptr->height;

            GWGetFontSize(
                switch_ptr->font,
                NULL, NULL,
                &fw, &fh
            );

            if(w <= 0)
                w = MAX(
		    (2 * DEF_XMARGIN) + DEF_SWITCH_WIDTH,
		    (2 * DEF_XMARGIN) + ((switch_ptr->label == NULL) ?
			0 : strlen(switch_ptr->label) * fw
		    )
		);
	    if(h <= 0)
		h = (3 * DEF_YMARGIN) + DEF_SWITCH_HEIGHT +
		    ((switch_ptr->label == NULL) ? 0 : fh);

            x = (int)(switch_ptr->x * width);
            y = (int)(switch_ptr->y * height);

	    image = switch_ptr->bg_image;
            DO_DRAW_IMAGE 

	    if(switch_ptr->label != NULL)
	    {
                GWSetFont(display, switch_ptr->font);
                glColor4d(
                    switch_ptr->color.r,   
                    switch_ptr->color.g,
                    switch_ptr->color.b,
                    switch_ptr->color.a
                );
                GWDrawString(
                    display,
                    x - (w / 2) + DEF_XMARGIN,
                    y - (h / 2) + (2 * DEF_YMARGIN) + DEF_SWITCH_HEIGHT,
                    switch_ptr->label
                );
	    }

	    /* Check state of switch and choose the right switch image.
	     * to draw.
	     */
            if(switch_ptr->state)
		image = switch_ptr->switch_on_image;
	    else
		image = switch_ptr->switch_off_image;

	    /* Draw the switch image. */
            SARImageDraw(
                display,
                image,
                x - (DEF_SWITCH_WIDTH / 2),
		y - (h / 2) + DEF_YMARGIN,
                DEF_SWITCH_WIDTH, DEF_SWITCH_HEIGHT
            );

            /* Draw outline if selected. */
            if(m->selected_object == n)
            {
                int x_min, x_max, y_min, y_max;
            
                x_min = x - (w / 2);
                x_max = x_min + w;
                y_min = y - (h / 2);
                y_max = y_min + h;
         
                glColor4d(0.0, 0.0, 0.0, 1.0);
            
                GWDrawLine(display, x_min, y_max, x_min, y_min);
                GWDrawLine(display, x_min, y_min, x_max, y_min);
                GWDrawLine(display, x_max, y_min, x_max, y_max);
                GWDrawLine(display, x_min, y_max, x_max, y_max);
            }
 	    break;

          /* ******************************************************** */
          case SAR_MENU_OBJECT_TYPE_SPIN:
            spin_ptr = ptr;

            GWGetFontSize(
                spin_ptr->font,
                NULL, NULL,
                &fw, &fh
            );

            w = (int)(width * spin_ptr->width);
            h = (int)MAX(
		height * spin_ptr->height,
		(4 * DEF_YMARGIN) + fh
	    );

	    x = (int)(width * spin_ptr->x);
	    y = (int)(height * spin_ptr->y);

	    if(1)
	    {
		int label_len, label_width, x_start, y_start;


		x_start = x - (w / 2);
		y_start = y - (h / 2);
		label_len = ((spin_ptr->label == NULL) ?
		    0 : strlen(spin_ptr->label)
		);

		/* Calculate width of left side label. */
		label_width = (2 * DEF_XMARGIN) + (label_len * fw);

		/* Draw label background. */
                SARImageDraw(
                    display,   
                    spin_ptr->label_image,
                    x_start,
		    y_start,
		    label_width,
		    h
		);
                GWSetFont(display, spin_ptr->font);
                glColor4d(
                    spin_ptr->label_color.r,
                    spin_ptr->label_color.g,
                    spin_ptr->label_color.b,
                    spin_ptr->label_color.a
                );
                GWDrawString(
                    display,
                    x_start + DEF_XMARGIN,
                    y - (fh / 2),
                    spin_ptr->label
                );


		/* Begin drawing value just to the right of the label. */

		/* Get attributes of and switch to value font. */
                GWGetFontSize(  
                    spin_ptr->value_font,
                    NULL, NULL,
                    &fw, &fh
                );
		GWSetFont(display, spin_ptr->value_font);
		/* Draw right side value background image. */
                SARImageDraw(
                    display,
                    spin_ptr->value_image,
                    x_start + label_width - 1,	/* Overlap one pixel to the left. */
                    y_start,
                    MAX(w - label_width, 0),
                    h
                );

		/* Is current spin value referancing a valid pointer in
		 * its value list?
		 */
		if((spin_ptr->cur_value >= 0) &&
		   (spin_ptr->cur_value < spin_ptr->total_values)
		)
		{
		    int value_len;
		    const char *value_ptr = (const char *)spin_ptr->value[
			spin_ptr->cur_value
		    ];

		    /* Value allocated? */
		    if(value_ptr != NULL)
		    {
			value_len = strlen(value_ptr);

                        glColor4d(
                            spin_ptr->value_color.r,
                            spin_ptr->value_color.g,
                            spin_ptr->value_color.b,
                            spin_ptr->value_color.a 
                        );
                        GWDrawString(
                            display,
			    x_start + label_width +
			    ((w - label_width) / 2) -
				(value_len * fw / 2),
                            y - (fh / 2),
                            value_ptr
                        );

			/* Draw spin left arrow? */
			if(spin_ptr->allow_warp ? True :
			    (spin_ptr->cur_value > 0)
			)
			{
			    glRasterPos2i(
				x_start + label_width + (3 * DEF_XMARGIN),
				height - (y_start + (h / 2) -
				    (DEF_SCROLL_CURSOR_HEIGHT / 2) +
				    DEF_SCROLL_CURSOR_HEIGHT
			        )
			    );
			    glBitmap(
				DEF_SCROLL_CURSOR_WIDTH,
				DEF_SCROLL_CURSOR_HEIGHT,
				0.0, 0.0,
				DEF_SCROLL_CURSOR_WIDTH, 0.0,
				scroll_cursor_left_bm
			    );
			}
			/* Draw spin right arrow? */
			if(spin_ptr->allow_warp ? True :
                            (spin_ptr->cur_value < (spin_ptr->total_values - 1))
                        )
			{
			    glRasterPos2i(
				x_start + w - (3 * DEF_XMARGIN) -
				    DEF_SCROLL_CURSOR_WIDTH,
				height - (y_start + (h / 2) -
				    (DEF_SCROLL_CURSOR_HEIGHT / 2) +
				    DEF_SCROLL_CURSOR_HEIGHT
				)
			    );
			    glBitmap(
				DEF_SCROLL_CURSOR_WIDTH,
				DEF_SCROLL_CURSOR_HEIGHT,
				0.0, 0.0,
				DEF_SCROLL_CURSOR_WIDTH, 0.0,
				scroll_cursor_right_bm
			    );
			}

		    }
		}

                /* Draw outline if selected. */
                if(m->selected_object == n)
                {
                   int x_min, x_max, y_min, y_max;

                    x_min = x - (w / 2);
                    x_max = x_min + w;
                    y_min = y - (h / 2);
                    y_max = y_min + h;

                    glColor4d(0.0, 0.0, 0.0, 1.0);

                    GWDrawLine(display, x_min, y_max, x_min, y_min);
                    GWDrawLine(display, x_min, y_min, x_max, y_min);
                    GWDrawLine(display, x_max, y_min, x_max, y_max);
                    GWDrawLine(display, x_min, y_max, x_max, y_max);
                }
	    }
	    break;

          /* ******************************************************** */
	  case SAR_MENU_OBJECT_TYPE_MAP:
	    map_ptr = (sar_menu_map_struct *)ptr;

            /* Draw map object with its own drawing function first. */
            SARMenuMapDraw(display, m, map_ptr, n);

	    /* Calculate map object geometry for window frame drawing. */
            x = (int)(map_ptr->x * width);
            y = (int)(map_ptr->y * height);

	    /* Calculate size to be one pixel bigger (to ensure cover). */
            w = (int)(map_ptr->width * width) + 2;
	    h = (int)(map_ptr->height * height) + 2;

            xc = MAX(x - (w / 2) - 1, 0);
            yc = MAX(y - (h / 2) - 1, 0);

	    SARMenuDrawWindowBG(
		display, map_ptr->bg_image,
                xc, yc,
                w, h,
		False,		/* Do not draw base. */
                draw_shadows	/* Draw shadow. */
	    );
	    break;
	}

#undef DO_DRAW_IMAGE
#undef item_name_len
}


/*
 *      Draws the specified object n.
 */
void SARMenuDrawObject(
        gw_display_struct *display, sar_menu_struct *m,
        int n			/* Object number on m. */
)
{
	SARMenuDoDrawObject(
	    display, m, n, False
	);
}

/*
 *	Draws the specified menu m.
 *
 *	Does not swap the GW buffers.
 */
void SARMenuDrawAll(
	gw_display_struct *display,
	sar_menu_struct *m
)
{
	int i;

	if((display == NULL) || (m == NULL))
	    return;

	/* Draw background image. */
	if(m->bg_image != NULL)
	{
	    /* Draw to the size of the frame buffer, thus clearing it. */
	    SARImageDraw(
                display,
                m->bg_image,
	        0, 0,
	        display->width, display->height
	    );
	}
	else
	{
	    glClearColor(1.0, 1.0, 1.0, 0.0);
            glClearDepth(1.0);
            glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	}

	/* Draw each object. */
	for(i = 0; i < m->total_objects; i++)
	    SARMenuDoDrawObject(display, m, i, True);
}


/*
 *	Manages a pointer event assumed for the given menu.
 */
int SARMenuManagePointer(
	gw_display_struct *display, sar_menu_struct *m,
	int x, int y, int state, int btn_num
)
{
	int i, events_handled = 0;
	int xp, yp, w, h;
        int x_min, x_max, y_min, y_max, width, height;
	void *ptr;
        sar_menu_button_struct *button_ptr;
	sar_menu_message_box_struct *mesgbox_ptr;
        sar_menu_list_struct *list_ptr;
        sar_menu_switch_struct *switch_ptr;
	sar_menu_spin_struct *spin_ptr;
	sar_menu_map_struct *map_ptr;


	if((display == NULL) || (m == NULL))
	    return(events_handled);

        width = display->width;
        height = display->height;

	/* Itterate through all menu object/widgets. */
	for(i = 0; i < m->total_objects; i++)
	{
	    ptr = m->object[i];
            if(ptr == NULL)
                continue;

	    switch(*(int *)ptr)
	    {
	      /* ***************************************************** */
              case SAR_MENU_OBJECT_TYPE_BUTTON:
                button_ptr = ptr;
		xp = (int)(button_ptr->x * width) + button_ptr->offset_x;
                yp = (int)(button_ptr->y * height) + button_ptr->offset_y;
		w = button_ptr->width;
		h = button_ptr->height;

		/* Get coordinate bounds. */
                x_min = (int)MAX(xp - (w / 2), 0);
		x_max = (int)MAX(x_min + w, w);
		y_min = (int)MAX(yp - (h / 2), 0);
                y_max = (int)MAX(y_min + h, h);

		/* Pointer event in bounds? */
		if((x >= x_min) && (x < x_max) &&
                   (y >= y_min) && (y < y_max)
		)
		{
		    /* Handle by pointer event type. */
		    switch(state)
		    {
		      case GWEventTypePointerMotion:
			if(button_ptr->state == SAR_MENU_BUTTON_STATE_UNARMED)
			{
			    button_ptr->state = SAR_MENU_BUTTON_STATE_HIGHLIGHTED;
                            events_handled++;
                            DO_REDRAW_OBJECT(display, m, i);
			}
			break;

		      case GWEventTypeButtonPress:
                        events_handled++;
                        m->selected_object = i;     /* Select this object. */
			if(button_ptr->state != SAR_MENU_BUTTON_STATE_ARMED)
                        {
			    button_ptr->state = SAR_MENU_BUTTON_STATE_ARMED;
			    DO_REDRAW_OBJECT(display, m, i);
			}
			break;

		      case GWEventTypeButtonRelease:
			events_handled++;
                        m->selected_object = i;     /* Select this object. */
			if(button_ptr->state != SAR_MENU_BUTTON_STATE_HIGHLIGHTED)
                        {
			    button_ptr->state = SAR_MENU_BUTTON_STATE_HIGHLIGHTED;
                            DO_REDRAW_OBJECT(display, m, i);
			}
			if(button_ptr->func_cb != NULL)
			{
			    int n;
			    sar_menu_button_struct *button2_ptr;

			    /* Set all other buttons to unarmed. */
			    for(n = 0; n < m->total_objects; n++)
			    {
				ptr = m->object[n];
				if(ptr == NULL)
				    continue;

				if((*(int *)ptr) != SAR_MENU_OBJECT_TYPE_BUTTON)
				    continue;

				button2_ptr = ptr;

				if(button_ptr == button2_ptr)
				    continue;

				if(button2_ptr->state != SAR_MENU_BUTTON_STATE_UNARMED)
				{
				    button2_ptr->state = SAR_MENU_BUTTON_STATE_UNARMED;
                                    DO_REDRAW_OBJECT(display, m, n);
				}
			    }

			    /* Call callback function. */
			    button_ptr->func_cb(
				button_ptr,		/* Object. */
				button_ptr->client_data, /* Client data. */
				button_ptr->id_code	/* ID code. */
			    );
			}
			break;
		    }
		}
		else
		{
		    /* Pointer event out of bounds. */
                    switch(state)
                    {
                      case GWEventTypeButtonRelease:
		      case GWEventTypePointerMotion:
			if(button_ptr->state != SAR_MENU_BUTTON_STATE_UNARMED)
			{
			    button_ptr->state = SAR_MENU_BUTTON_STATE_UNARMED;
                            DO_REDRAW_OBJECT(display, m, i);
			}
			break;
		    }
		}
		break;

              /* ***************************************************** */
	      case SAR_MENU_OBJECT_TYPE_MESSAGE_BOX:
		mesgbox_ptr = (sar_menu_message_box_struct *)ptr;

                w = (int)(mesgbox_ptr->width * width);
                if(w <= 0)
                    w = (int)(width - (2 * DEF_MB_XMARGIN));
                if(w <= 0)
                    w = DEF_MB_XMARGIN;
                
                if(mesgbox_ptr->height > 0)
                    h = (int)(mesgbox_ptr->height * height);
                else
                    h = DEF_HEIGHT;
                if(h <= 0)
                    h = DEF_MB_YMARGIN;

                /* Get coordinate bounds. */
                x_min = (int)((mesgbox_ptr->x * width) - (w / 2));
                x_max = (int)(x_min + w);
                y_min = (int)((mesgbox_ptr->y * height) - (h / 2));
                y_max = (int)(y_min + h); 
              
                /* Pointer event in bounds? */
                if((x >= x_min) && (x < x_max) && 
                   (y >= y_min) && (y < y_max) &&
                   (mesgbox_ptr->message != NULL) &&
		   (state == GWEventTypeButtonPress)
                )
                {
                    int fh, fw;
                    int lines_visable, columns_visable,
			total_lines, scroll_width;


                    events_handled++;
                    m->selected_object = i;     /* Select this object. */
              
                    GWGetFontSize(
                        mesgbox_ptr->font,
                        NULL, NULL,
                        &fw, &fh
                    );

                    /* Calculate colums visable. */
                    if(fw > 0)
                        columns_visable = (w - DEF_SCROLL_CURSOR_WIDTH
			    - (2 * DEF_MB_XMARGIN)) / fw;
                    else
                        columns_visable = 1;
                    if(columns_visable < 1)
                        columns_visable = 1;

                    /* Calculate width (height) of scroll indicator. */
                    scroll_width = fh + (2 * DEF_MB_YMARGIN);  

                    /* Calculate lines visable. */
                    if(fh > 0)
                        lines_visable = (h - (2 * DEF_MB_YMARGIN)) / fh;
                    else
                        lines_visable = 0;
                    if(lines_visable < 0)
                        lines_visable = 0;

                    /* Calculate total number of lines. */
		    total_lines = SARMenuMessageBoxTotalLines(
			mesgbox_ptr->message, columns_visable
		    );

		    /* Let's make scrolling simple, check if y is above
		     * or below the midpoint.
		     */
		    if(y < (mesgbox_ptr->y * height))
		    {
			/* Scroll up. */
			mesgbox_ptr->scrolled_line--;
		    }
		    else
		    {
			/* Scroll down. */
			mesgbox_ptr->scrolled_line++;
		    }

		    if(mesgbox_ptr->scrolled_line >
			(total_lines - (lines_visable / 2))
		    )
			mesgbox_ptr->scrolled_line = total_lines -
			    (lines_visable / 2);
		    if(mesgbox_ptr->scrolled_line < 0)
			mesgbox_ptr->scrolled_line = 0;

                    DO_REDRAW_OBJECT(display, m, i);
		}
		break;

              /* ***************************************************** */
              case SAR_MENU_OBJECT_TYPE_LIST:
                list_ptr = ptr;

                w = (int)(list_ptr->width * width);
                if(w <= 0)
                    w = width - (2 * DEF_LIST_XMARGIN); 
                if(w <= 0)
                    w = DEF_LIST_XMARGIN;

                if(list_ptr->height > 0)
                    h = (int)(list_ptr->height * height);
                else
                    h = DEF_HEIGHT;
                if(h <= 0)
                    h = DEF_LIST_YMARGIN;

                /* Get coordinate bounds. */
                x_min = (int)((list_ptr->x * width) - (w / 2));  
                x_max = (int)(x_min + w);
                y_min = (int)((list_ptr->y * height) - (h / 2));
                y_max = (int)(y_min + h);

                /* Pointer event in bounds? */
                if((x >= x_min) && (x < x_max) &&
                   (y >= y_min) && (y < y_max)
                )
                {
		    int fh, fw, sel_num, prev_sel_num;
		    int items_visable;
		    int scroll = 0;

                    GWGetFontSize(
                        list_ptr->font, 
                        NULL, NULL,
                        &fw, &fh
                    );
                    /* Calculate items visible, remember to subtract two
		     * items for borders and heading.
                     */
                    if(fh > 0)
                        items_visable = MAX(
                            (h / fh) - 2,
                            0
                        );
                    else
                        items_visable = 0;

		    /* Update items visable on list. */
		    list_ptr->items_visable = items_visable;

		    /* Check if on scroll up or scroll down (scroll
		     * bar width assumed to be 10 pixels wide.
		     */
		    if(x >= (x_max - DEF_SCROLL_CURSOR_WIDTH - DEF_LIST_XMARGIN))
		    {
			if(y < (y_min + DEF_SCROLL_CURSOR_HEIGHT + fh + DEF_LIST_YMARGIN))
			    scroll = 1;
			else if(y >= (y_max - DEF_SCROLL_CURSOR_HEIGHT - DEF_LIST_YMARGIN))
			    scroll = 2;
		    }

                    /* Handle by pointer event type. */
                    switch(state)
                    {
                      case GWEventTypeButtonPress:
                        events_handled++;
                        m->selected_object = i;	/* Select this object. */

			if(scroll == 1)
			{
			    list_ptr->scrolled_item -= MAX(
				items_visable - 1,
				1
			    );
			    if(list_ptr->scrolled_item < 0)
				list_ptr->scrolled_item = 0;

			    DO_REDRAW_OBJECT(display, m, i);
			}
			else if(scroll == 2)
			{
                            list_ptr->scrolled_item += MAX(
                                items_visable - 1,
				1
                            );
			    if(list_ptr->scrolled_item >=
				(list_ptr->total_items - (items_visable / 2))
			    )
				list_ptr->scrolled_item =
				    list_ptr->total_items -
				    (items_visable / 2);
                            if(list_ptr->scrolled_item < 0)
                                list_ptr->scrolled_item = 0;

                            DO_REDRAW_OBJECT(display, m, i);
			}
			else
			{
		          /* Match selection. */

		          /* Calculate selection position, remember to -1
			   * for the heading which should not be selected.
			   */
			  if(fh > 0)
			    sel_num = list_ptr->scrolled_item +
                                ((y - DEF_LIST_YMARGIN - fh - y_min) / fh);
			  else
			    sel_num = -1;

			  /* Sanitize selection. */
			  if(sel_num < -1)
			    sel_num = -1;
			  if(sel_num >= list_ptr->total_items)
			    sel_num = list_ptr->total_items - 1;

			  prev_sel_num = list_ptr->selected_item;
			  list_ptr->selected_item = sel_num;

                          DO_REDRAW_OBJECT(display, m, i);

			  /* Call select notify function. */
			  if((list_ptr->select_cb != NULL) &&
			     (prev_sel_num != sel_num) &&
			     (sel_num >= 0) &&
			     (sel_num < list_ptr->total_items)
			  )
			  {
			    sar_menu_list_item_struct *item_ptr;

			    item_ptr = list_ptr->item[sel_num];
			    if(item_ptr != NULL)
                              list_ptr->select_cb(
                                list_ptr,		/* Object. */
                                item_ptr->client_data,	/* Client data. */
                                sel_num,		/* Selected item. */
				item_ptr		/* Pointer to item. */
                              );
			  }
		        }
			break;
		    }		    
		}
		break;

              /* ***************************************************** */
              case SAR_MENU_OBJECT_TYPE_SWITCH:
                switch_ptr = ptr;

		if(1)
		{
		    int fw, fh;

                    w = switch_ptr->width;
                    h = switch_ptr->height;

                    GWGetFontSize(
                        switch_ptr->font,
                        NULL, NULL,
                        &fw, &fh 
                    );

                    if(w <= 0)
                        w = MAX(
                            (2 * DEF_XMARGIN) + DEF_SWITCH_WIDTH,
                            (2 * DEF_XMARGIN) + ((switch_ptr->label == NULL) ?
                                0 : strlen(switch_ptr->label) * fw
                            )
                        );
                    if(h <= 0)
                        h = (3 * DEF_YMARGIN) + DEF_SWITCH_HEIGHT +
                            ((switch_ptr->label == NULL) ? 0 : fh);

                    /* Get coordinate bounds. */
                    x_min = (int)((switch_ptr->x * width) - (w / 2));
                    x_max = (int)(x_min + w);
                    y_min = (int)((switch_ptr->y * height) - (h / 2));
                    y_max = (int)(y_min + h);

                    /* Pointer event in bounds? */
                    if((x >= x_min) && (x < x_max) &&
                       (y >= y_min) && (y < y_max)
                    )
                    {
                        switch(state)
                        {
                          case GWEventTypeButtonPress:
                            events_handled++;
                            m->selected_object = i;	/* Select this object. */
			    switch_ptr->state = !switch_ptr->state;
                            DO_REDRAW_OBJECT(display, m, i);

                            if(switch_ptr->switch_cb != NULL)
			      switch_ptr->switch_cb(
			        switch_ptr,		/* Object. */
				switch_ptr->client_data,	/* Client data. */
				switch_ptr->id_code,	/* ID code. */
				switch_ptr->state	/* State. */
			      );
			    break;
		        }
		    }
		}
		break;

              /* ***************************************************** */
              case SAR_MENU_OBJECT_TYPE_SPIN:
                spin_ptr = ptr;

                if(1)
                {
                    int fw, fh, label_len, label_width;

                    GWGetFontSize(
                        spin_ptr->font,
                        NULL, NULL,
                        &fw, &fh
                    );

                    label_len = ((spin_ptr->label == NULL) ?
                        0 : strlen(spin_ptr->label)
                    );
                    label_width = (2 * DEF_XMARGIN) + (label_len * fw);

		    w = (int)(width * spin_ptr->width);
                    h = (int)MAX(
                        height * spin_ptr->height,
                        (4 * DEF_YMARGIN) + fh
                    );

                    /* Get coordinate bounds. */
                    x_min = (int)((spin_ptr->x * width) - (w / 2));
                    x_max = (int)(x_min + w);
		    x_min += label_width;

                    y_min = (int)((spin_ptr->y * height) - (h / 2));
                    y_max = (int)(y_min + h);

                    /* Pointer event in bounds? */
                    if((x >= x_min) && (x < x_max) &&
                       (y >= y_min) && (y < y_max)
                    )
                    {
			int x_mid = x_min + ((x_max - x_min) / 2);

                        switch(state)
                        {
                          case GWEventTypeButtonPress:
                            events_handled++;
                            m->selected_object = i;     /* Select this object. */

			    /* Pointer x coordinate at or above x
			     * midpoint?
			     */
			    if(x >= x_mid)
				SARMenuSpinDoInc(display, m, i);
			    else
				SARMenuSpinDoDec(display, m, i);
			    break;
			}
		    }
		}
		break;

              /* ***************************************************** */
              case SAR_MENU_OBJECT_TYPE_MAP:
                map_ptr = (sar_menu_map_struct *)ptr;
		events_handled += SARMenuMapManagePointer(
		    display, m, map_ptr, i,
		    x, y, state, btn_num
		);
		break;
	    }
	}


	return(events_handled);
}


/*
 *	Manages a key event assumed for the menu.
 */
int SARMenuManageKey(
	gw_display_struct *display,
        sar_menu_struct *m,
        int key, int state
)
{
	int i, prev_sel_num, events_handled = 0;
	int items_visable = 0;
	int fw, fh, w, h, width, height;
        void *ptr;
        sar_menu_button_struct *button_ptr;
	sar_menu_message_box_struct *mesgbox_ptr;
        sar_menu_list_struct *list_ptr;
	sar_menu_switch_struct *switch_ptr;
	sar_menu_spin_struct *spin_ptr;


	if((display == NULL) ||
           (m == NULL)
	)
	    return(events_handled);

	/* Always ignore key releases. */
	if(!state)
	    return(events_handled);

        width = display->width;  
        height = display->height;


	/* Menu global key functions. */
	switch(key)
	{
	  case '\t':
	    prev_sel_num = m->selected_object;

#define TYPE_SELECTABLE(t)	(\
	((t) != SAR_MENU_OBJECT_TYPE_LABEL) && \
	((t) != SAR_MENU_OBJECT_TYPE_PROGRESS) \
)

	    if(display->shift_key_state)
	    {
		/* Seek backwards untill we get a selectable object
                 * type or the list has cycled.
                 */
		int type, status = 0;
		m->selected_object--;
		while(m->selected_object >= 0)
		{
		    if(SARMenuIsObjectAllocated(m, m->selected_object))
		    {
			ptr = m->object[m->selected_object];
			type = (*(int *)ptr);
			if(TYPE_SELECTABLE(type))
			{
			    status = 1;
			    break;
			}
		    }
		    m->selected_object--;
		}
		/* Didn't get a match? */
		if(!status)
		{
		    /* From the end... */
		    m->selected_object = m->total_objects - 1;
		    while(m->selected_object > prev_sel_num)
		    {
                        if(SARMenuIsObjectAllocated(m, m->selected_object))
                        {
                            ptr = m->object[m->selected_object];
                            type = (*(int *)ptr);
                            if(TYPE_SELECTABLE(type))
                            {
                                status = 1;
                                break;
                            }
			}
			m->selected_object--;
		    }
		}
		/* Still no match? */
		if(!status)
		    m->selected_object = prev_sel_num;
	    }
	    else
	    {
                /* Seek forwards untill we get a selectable object
                 * type or the list has cycled.
                 */
                int type, status = 0;
                m->selected_object++;
                while(m->selected_object < m->total_objects)
                {
                    if(SARMenuIsObjectAllocated(m, m->selected_object))
                    {
                        ptr = m->object[m->selected_object];
                        type = (*(int *)ptr);
                        if(TYPE_SELECTABLE(type))
                        {
                            status = 1;
                            break;
                        }
                    }
                    m->selected_object++;
                }
                /* Didn't get a match? */
                if(!status)
                {
                    /* From the beginning... */
                    m->selected_object = 0;
                    while(m->selected_object < prev_sel_num)
                    {
                        if(SARMenuIsObjectAllocated(m, m->selected_object))
                        {
                            ptr = m->object[m->selected_object];
                            type = (*(int *)ptr);
                            if(TYPE_SELECTABLE(type))
                            {
                                status = 1;
                                break;
                            }
                        }
                        m->selected_object++;
                    }
                }
                /* Still no match? */
                if(!status)
                    m->selected_object = prev_sel_num;
	    }

	    /* Sanitize. */
	    if(m->selected_object >= m->total_objects)
		m->selected_object = 0;
	    else if(m->selected_object < 0)
		m->selected_object = m->total_objects - 1;

	    if(m->selected_object < 0)
		m->selected_object = 0;

            /* If a menu's selected object is this button and this
             * button is unarmed, it needs to be highlighted.
	     */
	    if(SARMenuIsObjectAllocated(m, prev_sel_num))
	    {
		ptr = m->object[prev_sel_num];
		if(*(int *)ptr == SAR_MENU_OBJECT_TYPE_BUTTON)
		{
		    button_ptr = ptr;
		    button_ptr->state = SAR_MENU_BUTTON_STATE_UNARMED;
		}
	    }
            if(SARMenuIsObjectAllocated(m, m->selected_object))
            {
                ptr = m->object[m->selected_object];
                if(*(int *)ptr == SAR_MENU_OBJECT_TYPE_BUTTON)
                {
                    button_ptr = ptr;
                    button_ptr->state = SAR_MENU_BUTTON_STATE_HIGHLIGHTED;
                }
            }

	    DO_REDRAW_MENU
	    events_handled++;
	    break;
	}
	if(events_handled > 0)
	    return(events_handled);


	/* Key on selected widget. */
	i = m->selected_object;
	if(SARMenuIsObjectAllocated(m, i))
	{
	    ptr = m->object[i];

	    /* Handle by widget type. */
	    switch(*(int *)ptr)
	    {
              /* **************************************************** */
	      case SAR_MENU_OBJECT_TYPE_BUTTON:
		button_ptr = ptr;
		switch(key)
		{
		  case '\n': case ' ':
                    if(button_ptr->func_cb != NULL)
                        button_ptr->func_cb(
                            button_ptr,			/* Object. */
			    button_ptr->client_data,	/* Client data. */
			    button_ptr->id_code		/* ID code. */
			);
		    events_handled++;
                    break;
		}
		break;

              /* **************************************************** */
              case SAR_MENU_OBJECT_TYPE_MESSAGE_BOX:
                mesgbox_ptr = (sar_menu_message_box_struct *)ptr;

                w = (int)(mesgbox_ptr->width * width);
                if(w <= 0)
                    w = (int)(width - (2 * DEF_MB_XMARGIN));
                if(w <= 0)
                    w = DEF_MB_XMARGIN;

                if(mesgbox_ptr->height > 0)
                    h = (int)(mesgbox_ptr->height * height);
                else
                    h = DEF_HEIGHT;
                if(h <= 0)
                    h = DEF_MB_YMARGIN;

                if(mesgbox_ptr->message != NULL)
                {
                    int	total_lines, columns_visable,
			lines_visable, scroll_width;


                    GWGetFontSize(
                        mesgbox_ptr->font,
                        NULL, NULL,
                        &fw, &fh
                    );

                    /* Calculate colums visable. */
                    if(fw > 0)
                        columns_visable = (w - DEF_SCROLL_CURSOR_WIDTH
			    - (2 * DEF_MB_XMARGIN)) / fw;
                    else
                        columns_visable = 1;
                    if(columns_visable < 1)
                        columns_visable = 1; 

                    /* Calculate width (height) of scroll indicator. */
                    scroll_width = fh + (2 * DEF_MB_YMARGIN);

                    /* Calculate lines visable. */
                    if(fh > 0)
			lines_visable = (h - (2 * DEF_MB_YMARGIN)) / fh;
                    else
                        lines_visable = 0;
                    if(lines_visable < 0)
                        lines_visable = 0;

                    /* Calculate total number of lines. */
                    total_lines = SARMenuMessageBoxTotalLines(
                        mesgbox_ptr->message, columns_visable
                    );

		    switch(key)
		    {
                      case GWKeyHome:
                        events_handled++;
                        mesgbox_ptr->scrolled_line = 0;
                        DO_REDRAW_OBJECT(display, m, i);
                        break;

                      case GWKeyEnd:
                        events_handled++;
                        mesgbox_ptr->scrolled_line =
			    total_lines - (lines_visable / 2);
                        if(mesgbox_ptr->scrolled_line < 0)
                            mesgbox_ptr->scrolled_line = 0;
                        DO_REDRAW_OBJECT(display, m, i);
                        break;

                      case GWKeyUp:
                        events_handled++;
                        mesgbox_ptr->scrolled_line--;
                        if(mesgbox_ptr->scrolled_line < 0)
                            mesgbox_ptr->scrolled_line = 0;
                        DO_REDRAW_OBJECT(display, m, i);
                        break;

                      case GWKeyDown:
                        events_handled++;
                        mesgbox_ptr->scrolled_line++;
                        if(mesgbox_ptr->scrolled_line >=
			    (total_lines - (lines_visable / 2))
			)
                            mesgbox_ptr->scrolled_line = total_lines - (lines_visable / 2);
                        if(mesgbox_ptr->scrolled_line < 0)
                            mesgbox_ptr->scrolled_line = 0;
                        DO_REDRAW_OBJECT(display, m, i);
                        break;

                      case GWKeyPageUp:
                        events_handled++;
                        mesgbox_ptr->scrolled_line -= lines_visable;
                        if(mesgbox_ptr->scrolled_line < 0)
                            mesgbox_ptr->scrolled_line = 0;
                        DO_REDRAW_OBJECT(display, m, i);
                        break;

                      case GWKeyPageDown:
                        events_handled++;
                        mesgbox_ptr->scrolled_line += lines_visable;
                        if(mesgbox_ptr->scrolled_line >=
                            (total_lines - (lines_visable / 2))
                        )
                            mesgbox_ptr->scrolled_line = total_lines - (lines_visable / 2);
                        if(mesgbox_ptr->scrolled_line < 0)
                            mesgbox_ptr->scrolled_line = 0;
                        DO_REDRAW_OBJECT(display, m, i);
                        break;
		    }
		}
		break;

	      /* **************************************************** */
              case SAR_MENU_OBJECT_TYPE_LIST:
                list_ptr = ptr;

                w = (int)(list_ptr->width * width);
                if(w <= 0)
                    w = width - (2 * DEF_LIST_XMARGIN);
                if(w <= 0)
                    w = DEF_LIST_XMARGIN;
                 
                if(list_ptr->height > 0)
                    h = (int)(list_ptr->height * height);
                else
                    h = DEF_HEIGHT;
                if(h <= 0)
                    h = DEF_LIST_YMARGIN;

                GWGetFontSize(
                    list_ptr->font,
                    NULL, NULL,
                    &fw, &fh
                );
		/* Calculate items visible, remember to subtract two items
		 * for borders and heading.
		 */
                if(fh > 0)
                    items_visable = MAX(
                        (h / fh) - 2,
                        0
                    );
                else
                    items_visable = 0;

		/* Update items visable on list. */
		list_ptr->items_visable = items_visable;

		/* Record previously selected item. */
		prev_sel_num = list_ptr->selected_item;

		switch(key)
		{
                  case GWKeyHome:
                    events_handled++;
                    list_ptr->scrolled_item = 0;
                    DO_REDRAW_OBJECT(display, m, i);
                    break;

                  case GWKeyEnd:
                    events_handled++;
                    list_ptr->scrolled_item = list_ptr->total_items -
			(items_visable / 2);
		    if(list_ptr->scrolled_item < 0)
			list_ptr->scrolled_item = 0;
                    DO_REDRAW_OBJECT(display, m, i);
                    break;

                  case GWKeyUp:
                    events_handled++;
		    list_ptr->selected_item--;
		    if(list_ptr->selected_item < 0)
			list_ptr->selected_item = 0;
		    if(list_ptr->selected_item < list_ptr->scrolled_item)
		    {
			list_ptr->scrolled_item -= (items_visable + 1);
			if(list_ptr->scrolled_item < 0)
                            list_ptr->scrolled_item = 0;
		    }
                    DO_REDRAW_OBJECT(display, m, i);
                    break;

                  case GWKeyDown:
                    events_handled++;
                    list_ptr->selected_item++;
                    if(list_ptr->selected_item >= list_ptr->total_items)
                        list_ptr->selected_item = list_ptr->total_items - 1;
		    if(list_ptr->selected_item < 0)
                       list_ptr->selected_item = 0;
		    if(list_ptr->selected_item > (list_ptr->scrolled_item +
			items_visable)
		    )
		    {
			list_ptr->scrolled_item = list_ptr->selected_item;
		    }
                    DO_REDRAW_OBJECT(display, m, i);
                    break;

		  case GWKeyPageUp:
		    events_handled++;
                    list_ptr->scrolled_item -= items_visable;
                    if(list_ptr->scrolled_item < 0)
                        list_ptr->scrolled_item = 0;

		    list_ptr->selected_item = list_ptr->scrolled_item;

                    DO_REDRAW_OBJECT(display, m, i);
		    break;

                  case GWKeyPageDown:
		    events_handled++;
                    list_ptr->scrolled_item += items_visable;
                    if(list_ptr->scrolled_item >=
                        (list_ptr->total_items - (items_visable / 2))
                    )
                        list_ptr->scrolled_item = list_ptr->total_items -
                            (items_visable / 2);
                    if(list_ptr->scrolled_item < 0)
                        list_ptr->scrolled_item = 0;
                    list_ptr->selected_item = list_ptr->scrolled_item;
                    DO_REDRAW_OBJECT(display, m, i);
                    break;
		}

                /* Call select notify function. */
                if((list_ptr->select_cb != NULL) &&
                   (prev_sel_num != list_ptr->selected_item) &&
                   (list_ptr->selected_item >= 0) &&
                   (list_ptr->selected_item < list_ptr->total_items)
                )
                {
                    sar_menu_list_item_struct *item_ptr;

                    item_ptr = list_ptr->item[list_ptr->selected_item];
                    if(item_ptr != NULL)
                        list_ptr->select_cb(
                            list_ptr,			/* Object. */
                            item_ptr->client_data,	/* Client data. */
                            list_ptr->selected_item, 	/* Selected item. */
                            item_ptr			/* Pointer to item. */
                        );
                }

		break; 

              /* **************************************************** */
              case SAR_MENU_OBJECT_TYPE_SWITCH:
                switch_ptr = ptr;

                switch(key)
                {
		  case ' ': case '\n':
		    events_handled++;
		    switch_ptr->state = !switch_ptr->state;

                    DO_REDRAW_OBJECT(display, m, i);

                    if(switch_ptr->switch_cb != NULL)
                        switch_ptr->switch_cb(
			    switch_ptr,			/* Object. */
			    switch_ptr->client_data,	/* Client data. */
			    switch_ptr->id_code,	/* ID code. */
			    switch_ptr->state		/* State. */
			);

		    break;
		}
		break;

              /* ***************************************************** */
              case SAR_MENU_OBJECT_TYPE_SPIN:
                spin_ptr = ptr;

                switch(key)
                {
                  case ' ': case '\n': case GWKeyRight: case GWKeyUp:
		    SARMenuSpinDoInc(display, m, i);
		    events_handled++;
		    break;

		  case GWKeyLeft: case GWKeyDown:
		    SARMenuSpinDoDec(display, m, i);
                    events_handled++;
                    break;
		}
		break;
	    }
	}

	return(events_handled);
}
