#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>

#include "../include/os.h"

#ifdef __MSW__
# include <windows.h>
#endif
  
#include <GL/gl.h>
         
#include "../include/string.h"  

#include "v3dtex.h"
                
#include "gw.h"
#include "stategl.h"

#include "image.h"
#include "menu.h"
#include "sar.h"


static sar_menu_map_marking_struct *SARMenuMapMatchMarkingPosition(
	gw_display_struct *display, sar_menu_map_struct *map_ptr,
        int x, int y, int start_marking_num, int *matched_marking_num
);

int SARMenuCreateMap(
	sar_menu_struct *m,
	double x, double y,
	double width, double height,
	GWFont *font,
	sar_menu_color_struct *color,
	const sar_image_struct **bg_image,	/* Total 9 images. */
	double scroll_x, double scroll_y,
	double m_to_pixels_coeff,
	int show_markings_policy
);
void SARMenuMapDeleteAllMarkings(sar_menu_map_struct *map_ptr);
void SARMenuMapDraw(
	gw_display_struct *display, sar_menu_struct *m,
	sar_menu_map_struct *map_ptr, int map_num
);
int SARMenuMapManagePointer(
        gw_display_struct *display, sar_menu_struct *m,
        sar_menu_map_struct *map_ptr, int menu_obj_num,
        int x, int y, int state, int btn_num
);


#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))

/*
 *      Default spacings (in pixels):
 *
 *	These values should match and coenciding values in menu.c.
 */
#define DEF_XMARGIN             5
#define DEF_YMARGIN             5

#define DEF_WIDTH               100
#define DEF_HEIGHT              100


#define DO_REDRAW_MENU		\
{ \
 SARMenuDrawAll(display, m); \
 GWSwapBuffer(display); \
}

#define DO_REDRAW_OBJECT(d,m,n)	\
{ \
 SARMenuDrawObject(display,m,n); \
 GWSwapBuffer(display); \
}



/*
 *	Returns the matched marking pointer at the given position
 *	in window coordinates relative to the upper left origin.
 *
 *	Returns NULL on failed match.
 *
 *	The search will start at the marking index specified by
 *	start_marking_num and returns the first match.
 */
static sar_menu_map_marking_struct *SARMenuMapMatchMarkingPosition(
	gw_display_struct *display, sar_menu_map_struct *map_ptr,
	int x, int y, int start_marking_num, int *matched_marking_num
)
{
	int i, width, height;
	int tolor_x, tolor_y;
	int marking_xp, marking_yp;
	int scroll_xp, scroll_yp;
	int	map_wp, map_hp,
		map_xp_cen, map_yp_cen,
		map_xp_min, map_yp_min,
		map_xp_max, map_yp_max;
	double mtop_coeff;
	sar_menu_map_marking_struct *marking_ptr;
	const sar_image_struct *icon;


	if(matched_marking_num != NULL)
	    (*matched_marking_num) = -1;

	if(map_ptr == NULL)
	    return(NULL);

        width = display->width;
        height = display->height;

        /* Bounds of map object on menu in pixels. */
        map_xp_cen = (int)(map_ptr->x * width);
        map_yp_cen = (int)(map_ptr->y * height);

        map_wp = (int)(map_ptr->width * width);
        map_hp = (int)(map_ptr->height * height);
        if((map_wp <= 0) ||
           (map_hp <= 0)
        )
            return(NULL);

        map_xp_min = MAX(map_xp_cen - (map_wp / 2), 0);
        map_yp_min = MAX(map_yp_cen - (map_hp / 2), 0);

        map_xp_max = map_xp_min + map_wp;
        map_yp_max = map_yp_min + map_hp;

	/* Flip given y coordinate. */
	y = (map_hp - (y - map_yp_min)) + map_yp_min;

        mtop_coeff = map_ptr->m_to_pixels_coeff;
 
	scroll_xp = map_ptr->scroll_x;
	scroll_yp = map_ptr->scroll_y;

        /* Itterate through markings on the map object. */
        for(i = start_marking_num; i < map_ptr->total_markings; i++)
        {
            marking_ptr = map_ptr->marking[i];
            if(marking_ptr == NULL)
                continue;

            /* Calculate marking position in pixels. */
            marking_xp = (int)((marking_ptr->x * mtop_coeff) +
                map_xp_cen + scroll_xp);
            marking_yp = (int)((marking_ptr->y * mtop_coeff) +
                map_yp_cen + scroll_yp);

	    /* Get icon of marking (if any). */
	    icon = marking_ptr->icon;
	    if(icon == NULL)
	    {
		tolor_x = 5;
		tolor_y = 5;
	    }
	    else
	    {
		tolor_x = icon->width / 2;
		tolor_y = icon->height / 2;
	    }

	    /* Given coordinates in bound with tolorance applied to
	     * marking position?
	     */
	    if(((marking_xp - tolor_x) < x) &&
               ((marking_xp + tolor_x) > x) &&
               ((marking_yp - tolor_y) < y) &&
               ((marking_yp + tolor_y) > y)
	    )
	    {
		if(matched_marking_num != NULL)
		    (*matched_marking_num) = i;
		return(marking_ptr);
	    }
	}
	return(NULL);
}


/*
 *	Creates a map object on menu m, returns the index number of
 *	the map object or -1 on failure.
 */
int SARMenuCreateMap(
        sar_menu_struct *m,
        double x, double y,
        double width, double height,
	GWFont *font,
	sar_menu_color_struct *color,
	const sar_image_struct **bg_image,	/* Total 9 images. */
	double scroll_x, double scroll_y,
        double m_to_pixels_coeff,
	int show_markings_policy
)
{
	int i, n;
        sar_menu_map_struct *map_ptr;


        if(m == NULL)
            return(-1);

        if(m->total_objects < 0)
            m->total_objects = 0;

        for(i = 0; i < m->total_objects; i++)
        {
            if(m->object[i] == NULL)
                break;
        }
        if(i < m->total_objects)
        {
            n = i;
        }
        else
        {
            n = m->total_objects;
            m->total_objects++;

            m->object = (void **)realloc(
                m->object,
                m->total_objects * sizeof(void *)
            );
            if(m->object == NULL)
            {
                m->total_objects = 0;
                return(-1);
            }

            m->object[n] = NULL;
        }   
            
        /* Allocate structure. */
        map_ptr = (sar_menu_map_struct *)calloc(
            1,
            sizeof(sar_menu_map_struct)
        );
        if(map_ptr == NULL)
            return(-1);
        else
            m->object[n] = (void *)map_ptr;
                
         
        /* Set map values. */
        map_ptr->type = SAR_MENU_OBJECT_TYPE_MAP;
        map_ptr->x = x;
        map_ptr->y = y;
        map_ptr->width = width;
        map_ptr->height = height;
	map_ptr->font = font;
	if(color != NULL)
	    memcpy(&map_ptr->color, color, sizeof(sar_menu_color_struct));
	map_ptr->show_markings_policy = show_markings_policy;
	map_ptr->bg_image = bg_image;
	map_ptr->scroll_x = (int)(-scroll_x * m_to_pixels_coeff);
        map_ptr->scroll_y = (int)(-scroll_y * m_to_pixels_coeff);
	map_ptr->m_to_pixels_coeff = m_to_pixels_coeff;
	map_ptr->marking = NULL;
	map_ptr->total_markings = 0;
	map_ptr->selected_marking = -1;

	return(n);
}

/*
 *	Delete all map markings on map object.
 */
void SARMenuMapDeleteAllMarkings(sar_menu_map_struct *map_ptr)
{
	if(map_ptr != NULL)
	{
	    int i;
	    sar_menu_map_marking_struct *marking_ptr;


	    for(i = 0; i < map_ptr->total_markings; i++)
	    {
		marking_ptr = map_ptr->marking[i];
		if(marking_ptr == NULL)
		    continue;

		/* Leave icon image data alone (shared). */
		marking_ptr->icon = NULL;

		/* Description. */
		free(marking_ptr->desc);
		marking_ptr->desc = NULL;

		/* Deallicate marking structure itself. */
		free(marking_ptr);
	    }

	    free(map_ptr->marking);
	    map_ptr->marking = NULL;

	    map_ptr->total_markings = 0;
	}
}

/*
 *	Draws the map object.
 */
void SARMenuMapDraw(
        gw_display_struct *display,
        sar_menu_struct *m,
        sar_menu_map_struct *map_ptr,
	int map_num
)
{
	int i, width, height;
	int scroll_xp, scroll_yp;
        int	map_wp, map_hp,
		map_xp_cen, map_yp_cen,
		map_xp_min, map_yp_min,
		map_xp_max, map_yp_max;

	int	marking_xp, marking_yp;
	double mtop_coeff;
	const sar_image_struct *image;
	v3d_texture_ref_struct *t;
	sar_menu_map_marking_struct *marking_ptr;


        if((display == NULL) ||
           (m == NULL) ||
	   (map_ptr == NULL)
        )
            return;

        width = display->width;
        height = display->height;

	mtop_coeff = map_ptr->m_to_pixels_coeff;

	scroll_xp = map_ptr->scroll_x;
	scroll_yp = map_ptr->scroll_y;

	/* Bounds of map object on menu in pixels. */
	map_xp_cen = (int)(map_ptr->x * width);
	map_yp_cen = (int)(map_ptr->y * height);

	map_wp = (int)(map_ptr->width * width);
	map_hp = (int)(map_ptr->height * height);
	if((map_wp < 1) || (map_hp < 1))
	    return;

        map_xp_min = MAX(map_xp_cen - (map_wp / 2), 0);
        map_yp_min = MAX(map_yp_cen - (map_hp / 2), 0);

        map_xp_max = map_xp_min + map_wp;
        map_yp_max = map_yp_min + map_hp;

	/* Convert all coordinates to lower-left origin for easy
	 * OpenGL calculations.
	 */
	i = map_yp_max;
	map_yp_cen = MAX(height - map_yp_cen, 0);
	map_yp_max = MAX(height - map_yp_min, 0);
	map_yp_min = MAX(height - i, 0);


	t = map_ptr->bg_tex;
	if(t != NULL)
	{
	    int tex_wp = (int)MAX(map_ptr->bg_tex_width * mtop_coeff, 10);
            int tex_hp = (int)MAX(map_ptr->bg_tex_height * mtop_coeff, 10);
	    double t_xc_min, t_yc_min, t_xc_max, t_yc_max;

	    /* Calculate texture coordinate bounds. */
	    t_xc_min = (double)(-scroll_xp + (tex_wp / 2) - (map_wp / 2))
		/ (double)tex_wp;
            t_xc_max = (double)(-scroll_xp + (tex_wp / 2) + (map_wp / 2))
                / (double)tex_wp;

            t_yc_max = (double)(scroll_yp + (tex_hp / 2) - (map_hp / 2))
                / (double)tex_hp;
            t_yc_min = (double)(scroll_yp + (tex_hp / 2) + (map_hp / 2))
                / (double)tex_hp;

	    StateGLTexEnvI(
		&display->state_gl, GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE,
		GL_MODULATE
	    );
	    StateGLEnable(&display->state_gl, GL_TEXTURE_2D);
	    V3DTextureSelect(t);
	    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
	    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);

	    glColor4d(0.5, 0.5, 0.5, 1.0);
            glBegin(GL_QUADS);
            {
		glTexCoord2d(t_xc_min, t_yc_min);
                glVertex2d(map_xp_min, map_yp_min);

		glTexCoord2d(t_xc_max, t_yc_min);
                glVertex2d(map_xp_max, map_yp_min);

                glTexCoord2d(t_xc_max, t_yc_max);
                glVertex2d(map_xp_max, map_yp_max);

                glTexCoord2d(t_xc_min, t_yc_max);
                glVertex2d(map_xp_min, map_yp_max);
            }
            glEnd();

	    StateGLDisable(&display->state_gl, GL_TEXTURE_2D);
	}
	else
	{
	    glColor4d(0.0, 0.0, 0.0, 1.0);
	    glBegin(GL_QUADS);
	    {
		glVertex2d((GLfloat)map_xp_min, (GLfloat)map_yp_min);
                glVertex2d((GLfloat)map_xp_max, (GLfloat)map_yp_min);
                glVertex2d((GLfloat)map_xp_max, (GLfloat)map_yp_max);
                glVertex2d((GLfloat)map_xp_min, (GLfloat)map_yp_max);
	    }
	    glEnd();
	}


	/* Begin drawing map markings. */
	for(i = 0; i < map_ptr->total_markings; i++)
	{
	    marking_ptr = map_ptr->marking[i];
	    if(marking_ptr == NULL)
		continue;

	    /* Calculate marking position in pixels. */
	    marking_xp = (int)((marking_ptr->x * mtop_coeff) +
		map_xp_cen + scroll_xp);
            marking_yp = (int)((marking_ptr->y * mtop_coeff) +
		map_yp_cen + scroll_yp);

	    /* Handle by type. */
	    switch(marking_ptr->type)
	    {
	      /* ***************************************************** */
	      case SAR_MENU_MAP_MARKING_TYPE_ICON:
		image = marking_ptr->icon;
		if(image != NULL)
		{
		    int	img_hw = image->width / 2,
			img_hh = image->height / 2;

		    /* Out of bounds? */
		    if((marking_xp <= (map_xp_min + img_hw)) ||
		       (marking_yp <= (map_yp_min + img_hh)) ||
		       (marking_xp >= (map_xp_max - img_hw)) ||
		       (marking_yp >= (map_yp_max - img_hh))
		    )
			break;

                    glPixelZoom(1.0, -1.0);
		    StateGLEnable(&display->state_gl, GL_ALPHA_TEST);
                    StateGLAlphaFunc(&display->state_gl, GL_GREATER, 0.5);
		    glRasterPos2i(
		        marking_xp - (image->width / 2),
		        marking_yp + (image->height / 2)
		    );
                    glDrawPixels(
                        image->width, image->height,
                        GL_RGBA, GL_UNSIGNED_BYTE,
                        image->data
                    );
		}
		break;

	      case SAR_MENU_MAP_MARKING_TYPE_INTERCEPT_LINE:
/* Need better bounds checking. */
		if(marking_xp <= map_xp_min)
		    marking_xp = map_xp_min + 1;
                else if(marking_xp >= map_xp_max)
                    marking_xp = map_xp_max - 1;

                if(marking_yp <= map_yp_min)
                    marking_yp = map_yp_min + 1;
                else if(marking_yp >= map_yp_max)
                    marking_yp = map_yp_max - 1;


		glColor4d(
		    (GLfloat)marking_ptr->fg_color.r,
		    (GLfloat)marking_ptr->fg_color.g,
		    (GLfloat)marking_ptr->fg_color.b,
		    (GLfloat)marking_ptr->fg_color.a
		);
		glBegin(GL_LINES);
		{
		    int marking_x2p = (int)((marking_ptr->x_end * mtop_coeff) +
			map_xp_cen + scroll_xp),
			marking_y2p = (int)((marking_ptr->y_end * mtop_coeff) +
			map_yp_cen + scroll_yp);

/* Need better bounds checking. */
                    if(marking_x2p <= map_xp_min)
                        marking_x2p = map_xp_min + 1;
                    else if(marking_x2p >= map_xp_max)
                        marking_x2p = map_xp_max - 1;

                    if(marking_y2p <= map_yp_min)
                        marking_y2p = map_yp_min + 1;
                    else if(marking_y2p >= map_yp_max)
                        marking_y2p = map_yp_max - 1;

		    glVertex2d((GLfloat)marking_xp, (GLfloat)marking_yp);
		    glVertex2d((GLfloat)marking_x2p, (GLfloat)marking_y2p);
		}
		glEnd();
		break;
	    }
	}

	/* Check if a marking is selected. */
	i = map_ptr->selected_marking;
	if((i >= 0) && (i < map_ptr->total_markings))
	{
	    marking_ptr = map_ptr->marking[i];
	    if((marking_ptr == NULL) ? 0 : (marking_ptr->desc != NULL))
	    {
		GWFont *font = map_ptr->font;
		sar_menu_color_struct *color = &map_ptr->color;
		const char *desc = (const char *)marking_ptr->desc;
		int fw, fh, len;

		len = strlen(desc);
		GWGetFontSize(font, NULL, NULL, &fw, &fh);
                GWSetFont(display, font);
                glColor4d(
                    (GLfloat)color->r,
                    (GLfloat)color->g,
                    (GLfloat)color->b,
                    (GLfloat)color->a
                );
		GWDrawString(
                    display,
                    map_xp_cen - (fw * len / 2),
		    height - (map_yp_min + fh + (5 + 16)),
		    desc
                );
	    }
	}
}


/*
 *	Manages the pointer event with respect to the given menu map
 *	object.
 */
int SARMenuMapManagePointer(
        gw_display_struct *display, sar_menu_struct *m,
	sar_menu_map_struct *map_ptr, int menu_obj_num,
        int x, int y, int state, int btn_num
)
{
        int events_handled = 0;
        int x_min, x_max, y_min, y_max, width, height, w, h;

        static Boolean map_translate_drag_state = False;
        static Boolean map_zoom_drag_state = False;
        static int prev_motion_x, prev_motion_y;
        static time_t last_motion_time = 0;


        if((display == NULL) || (m == NULL) || (map_ptr == NULL))
	    return(events_handled);

	width = display->width;
	height = display->height;

        /* If button states down and event is a release, reset all
         * states.
         */
        if((map_translate_drag_state || map_zoom_drag_state ) &&
           (state == GWEventTypeButtonRelease)
        )
        {
            map_translate_drag_state = False;
            map_zoom_drag_state = False;
        }

	w = (int)(map_ptr->width * width);
        if(w <= 0)
            w = (int)(width - (2 * DEF_XMARGIN));
        if(w <= 0)
            w = DEF_XMARGIN;

        if(map_ptr->height > 0)
            h = (int)(map_ptr->height * height);
        else
            h = DEF_HEIGHT;
        if(h <= 0)
            h = DEF_YMARGIN;

        /* Get coordinate bounds. */
        x_min = (int)((map_ptr->x * width) - (w / 2));
        x_max = (int)(x_min + w);
        y_min = (int)((map_ptr->y * height) - (h / 2));
        y_max = (int)(y_min + h);

        /* Pointer event in bounds? */
        if((x >= x_min) && (x < x_max) &&
           (y >= y_min) && (y < y_max)
        )
        {
            /* Calculate scroll bounds. */
            double mtop_coeff;
            int tex_w, tex_h;
            int scroll_x_min, scroll_y_min, scroll_x_max, scroll_y_max;

/* Calculates scroll bounds by first fetching latest mtop_coeff
 * value and updates variables tex_w, tex_h, scroll_x_min, scroll_y_min,
 * scroll_x_max, and scroll_y_max.
 */
#define GET_SCROLL_BOUNDS	\
{ \
 mtop_coeff = map_ptr->m_to_pixels_coeff; \
 \
 /* Get texture size in pixels. */ \
 tex_w = (int)(map_ptr->bg_tex_width * mtop_coeff); \
 tex_h = (int)(map_ptr->bg_tex_height * mtop_coeff); \
 \
 /* Calculate scroll bounds in pixels. */ \
 scroll_x_min = (w / 2) - (tex_w / 2); \
 scroll_x_max = (tex_w / 2) - (w / 2); \
 \
 scroll_y_min = (h / 2) - (tex_h / 2); \
 scroll_y_max = (tex_h / 2) - (h / 2); \
}
	    GET_SCROLL_BOUNDS

	    /* Handle by event type. */
	    switch(state)
	    {
	      case GWEventTypeButtonPress:
                events_handled++;
                m->selected_object = menu_obj_num;	/* Select this object. */

                switch(btn_num)
		{
                  case 1:
                    map_translate_drag_state = True;
                    last_motion_time = cur_millitime;
                    prev_motion_x = x;
                    prev_motion_y = y;

                    GWSetPointerCursor(
                        display, GWPointerCursorTranslate
                    );
                    DO_REDRAW_OBJECT(display, m, menu_obj_num);
                    break;

                  case 3:
                    map_zoom_drag_state = True;
                    last_motion_time = cur_millitime;
                    prev_motion_x = x;
                    prev_motion_y = y;

                    GWSetPointerCursor(
                        display, GWPointerCursorZoom
                    );
                    DO_REDRAW_OBJECT(display, m, menu_obj_num);
                    break;
                }
                break;

	      case GWEventTypeButtonRelease:
                GWSetPointerCursor(
                    display, GWPointerCursorStandard
                );

                /* Manipulate last motion time so next case
                 * checks okay.
                 */
                last_motion_time = cur_millitime - 1;

		/* Fall through. */
	      case GWEventTypePointerMotion:
                /* Too soon to handle a motion event? */
                if(last_motion_time == cur_millitime)
                    break;
                else
                    last_motion_time = cur_millitime;
                events_handled++;

                /* Translating? */
                if(map_translate_drag_state)
                {
/*
                    map_ptr->scroll_x = CLIP(
                        map_ptr->scroll_x + (x - prev_motion_x),
                        scroll_x_min, scroll_x_max
                    );
                    map_ptr->scroll_y = CLIP(
                        map_ptr->scroll_y - (y - prev_motion_y),
                        scroll_y_min, scroll_y_max
                    );
 */
                    map_ptr->scroll_x = map_ptr->scroll_x +
			(x - prev_motion_x);
                    map_ptr->scroll_y = map_ptr->scroll_y -
			(y - prev_motion_y);

		    DO_REDRAW_OBJECT(display, m, menu_obj_num);
                }
                /* Zooming? */
                else if(map_zoom_drag_state)
                {
                    double scroll_x_coeff, scroll_y_coeff;
                    double prev_m_to_pixels_coeff =
                        map_ptr->m_to_pixels_coeff;

                    /* Calculate new zoom. */
                    map_ptr->m_to_pixels_coeff = CLIP(
                        map_ptr->m_to_pixels_coeff +
                            ((y - prev_motion_y) *
                            map_ptr->m_to_pixels_coeff / 100),
                        0.002,		/* Zoomed out max. */
                        0.050		/* Zoomed in max. */
                    );

                    /* Need to calculate last scroll position coeff
                     * relative to the previous zoom dimension, then
                     * set the new scroll position by multiplying
                     * the scroll position coeff just calculated by
                     * the new zoom dimension.
                     */
                    if(map_ptr->bg_tex_width > 0)
                        scroll_x_coeff = (double)map_ptr->scroll_x /
                            (map_ptr->bg_tex_width *
                                prev_m_to_pixels_coeff);
                    else
                        scroll_x_coeff = 0.0;
                    map_ptr->scroll_x = (int)(map_ptr->bg_tex_width *
                        map_ptr->m_to_pixels_coeff * scroll_x_coeff);

                    if(map_ptr->bg_tex_height > 0)
                        scroll_y_coeff = -(double)map_ptr->scroll_y /
                            (map_ptr->bg_tex_height *
                                prev_m_to_pixels_coeff);
                    else
                        scroll_y_coeff = 0.0;

                    map_ptr->scroll_y = -(int)(map_ptr->bg_tex_height *
                        map_ptr->m_to_pixels_coeff * scroll_y_coeff);

                    /* Need to calculate scroll bounds again. */
                    GET_SCROLL_BOUNDS

                    /* Reclip scroll bounds. */
/* Its safe to scroll anywhere, plus we can see offmap markings.
                    map_ptr->scroll_x = CLIP(
                        map_ptr->scroll_x,
                        scroll_x_min, scroll_x_max
                    );
                    map_ptr->scroll_y = CLIP(
                        map_ptr->scroll_y,
                        scroll_y_min, scroll_y_max
                    );
 */
		    DO_REDRAW_OBJECT(display, m, menu_obj_num);
		}
		/* All else assume marking select, we can select a
		 * marking with the pointer only if we are showing
		 * all markings. Cannot select a marking if only showing
		 * the selected marking.
		 */
		else if(map_ptr->show_markings_policy == SAR_MENU_MAP_SHOW_MARKING_ALL)
		{
		    /* Do this even if button is in released state. */

		    int	prev_marking_num = map_ptr->selected_marking,
			matched_marking_num = -1,
			next_marking_num;
		    sar_menu_map_marking_struct *marking_ptr;


		    /* Itterate till we find a marking with a description. */
		    for(next_marking_num = 0; 1;)
		    {
			marking_ptr = SARMenuMapMatchMarkingPosition(
			    display, map_ptr,
			    x, y,
			    next_marking_num, &matched_marking_num
			);
			if((marking_ptr == NULL) || (matched_marking_num < 0))
			    break;

			/* This one has a description? */
			if(marking_ptr->desc != NULL)
			    break;

			next_marking_num = matched_marking_num + 1;
		    }

		    if(prev_marking_num != matched_marking_num)
		    {
			map_ptr->selected_marking = matched_marking_num;
/*
printf("Matched %i\n", matched_marking_num);
 */



			/* For this we gots to redraw the entire menu. */
			DO_REDRAW_MENU
		    }
		}

                /* Record previous pointer coordinates. */
                prev_motion_x = x;
                prev_motion_y = y;
                break;




            }	/* Handle by event type. */

	}	/* Pointer event in bounds? */


#undef GET_SCROLL_BOUNDS      

	return(events_handled);
}
