#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "../include/cfgfmt.h"
#include "../include/string.h"
#include "../include/strexp.h"
#include "../include/fio.h"
#include "../include/disk.h"

#include "sfm.h"
#include "obj.h"
#include "mission.h"
#include "sar.h"
#include "sarfio.h"

#include "config.h"


void *SARParmNew(int type);

void SARParmDelete(void *p);
void SARParmDeleteAll(void ***ptr, int *total);

static const char *NEXT_ARG(const char *s);
static double ATOF(const char *s);
static int ATOI(const char *s);
static long ATOL(const char *s);
static void CHOP_STR_BLANK(char *s);

static char *SARParmLoadFromFileGetParmString(FILE *fp);
static int SARParmLoadFromFileItterate(
        const char *filename,
        void ***parm, int *total_parms,
        const char *parm_str, const char *val_str,
        int *lines_read, int filter_parm_type
);
int SARParmLoadFromFile(
        const char *filename, int file_format,
        void ***parm, int *total_parms,
	int filter_parm_type,
        void *client_data,
        int (*progress_func)(void *, long, long)
);


#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
   
#define ISCOMMENT(c)    ((c) == SAR_COMMENT_CHAR)
#define ISCR(c)         (((c) == '\n') || ((c) == '\r'))

#define DEGTORAD(d)     ((d) * PI / 180)
#define RADTODEG(r)     ((r) * 180 / PI)


/*
 *	Allocates a new parameter structure of the given type or NULL
 *	on error.
 */
void *SARParmNew(int type)
{
	void *p = NULL;
	int len = 0;


	switch(type)
	{
	  case SAR_PARM_NAME:
	    len = sizeof(sar_parm_name_struct);
	    break;

          case SAR_PARM_DESCRIPTION:
            len = sizeof(sar_parm_description_struct);
            break;

          case SAR_PARM_PLAYER_MODEL_FILE:
            len = sizeof(sar_parm_player_model_file_struct);
            break;

	  case SAR_PARM_WEATHER:
	    len = sizeof(sar_parm_weather_struct);
	    break;

	  case SAR_PARM_TIME_OF_DAY:
	    len = sizeof(sar_parm_time_of_day_struct);
            break;

          case SAR_PARM_REGISTER_LOCATION:
            len = sizeof(sar_parm_register_location_struct);
            break;

          case SAR_PARM_SCENE_GPS:
            len = sizeof(sar_parm_scene_gps_struct);
            break;

          case SAR_PARM_SCENE_MAP:
            len = sizeof(sar_parm_scene_map_struct);
            break;

          case SAR_PARM_SCENE_ELEVATION:
            len = sizeof(sar_parm_scene_elevation_struct);
            break;

          case SAR_PARM_SCENE_CANT:
            len = sizeof(sar_parm_scene_cant_struct);
            break;

          case SAR_PARM_SCENE_GROUND_FLAGS:
            len = sizeof(sar_parm_ground_flags_struct);
            break;

          case SAR_PARM_SCENE_GROUND_TILE:
            len = sizeof(sar_parm_scene_ground_tile_struct);
            break;

          case SAR_PARM_TEXTURE_BASE_DIRECTORY:
            len = sizeof(sar_parm_texture_base_directory_struct);
            break;

          case SAR_PARM_TEXTURE_LOAD:
            len = sizeof(sar_parm_texture_load_struct);
            break;

          case SAR_PARM_MISSION_TYPE:
            len = sizeof(sar_parm_mission_type_struct);
            break;

          case SAR_PARM_MISSION_SCENE_FILE:
            len = sizeof(sar_parm_mission_scene_file_struct);
            break;

          case SAR_PARM_MISSION_TIME_LEFT:
            len = sizeof(sar_parm_mission_time_left_struct);
            break;

          case SAR_PARM_MISSION_BEGIN_AT:
            len = sizeof(sar_parm_mission_begin_at_struct);
            break;

          case SAR_PARM_MISSION_ARRIVE_AT:
            len = sizeof(sar_parm_mission_arrive_at_struct);
            break;

          case SAR_PARM_MISSION_MESSAGE_SUCCESS:
            len = sizeof(sar_parm_mission_message_success_struct);
            break;

          case SAR_PARM_MISSION_MESSAGE_FAIL:
            len = sizeof(sar_parm_mission_message_fail_struct);
            break;

          case SAR_PARM_MISSION_HUMANS_TALLY:
            len = sizeof(sar_parm_mission_humans_tally_struct);
            break;

          case SAR_PARM_MISSION_ADD_INTERCEPT:
            len = sizeof(sar_parm_mission_add_intercept_struct);
            break;

          case SAR_PARM_NEW_OBJECT:
            len = sizeof(sar_parm_new_object_struct);
            break;

          case SAR_PARM_NEW_HELIPAD:
            len = sizeof(sar_parm_new_helipad_struct);
            break;

          case SAR_PARM_NEW_RUNWAY:
            len = sizeof(sar_parm_new_runway_struct);
            break;

          case SAR_PARM_NEW_HUMAN:
            len = sizeof(sar_parm_new_human_struct);
            break;

	  case SAR_PARM_NEW_FIRE:
	    len = sizeof(sar_parm_new_fire_struct);
	    break;

	  case SAR_PARM_NEW_SMOKE:
	    len = sizeof(sar_parm_new_smoke_struct);
            break;

          case SAR_PARM_NEW_PREMODELED:
            len = sizeof(sar_parm_new_premodeled_struct);
            break;

	  case SAR_PARM_SELECT_OBJECT_BY_NAME:
	    len = sizeof(sar_parm_select_object_by_name);
	    break;

          case SAR_PARM_MODEL_FILE:
            len = sizeof(sar_parm_model_file_struct);
            break;

          case SAR_PARM_RANGE:
            len = sizeof(sar_parm_range_struct);
            break;

	  case SAR_PARM_RANGE_FAR:
	    len = sizeof(sar_parm_range_far_struct);
            break;

          case SAR_PARM_TRANSLATE:
            len = sizeof(sar_parm_translate_struct);
            break;

          case SAR_PARM_TRANSLATE_RANDOM:
            len = sizeof(sar_parm_translate_random_struct);
            break;

          case SAR_PARM_ROTATE:
            len = sizeof(sar_parm_rotate_struct);
            break;

          case SAR_PARM_NO_DEPTH_TEST:
            len = sizeof(sar_parm_no_depth_test_struct);
            break;

          case SAR_PARM_POLYGON_OFFSET:
            len = sizeof(sar_parm_polygon_offset_struct);
            break;

          case SAR_PARM_CONTACT_BOUNDS_SPHERICAL:
            len = sizeof(sar_parm_contact_bounds_spherical_struct);
            break;

          case SAR_PARM_CONTACT_BOUNDS_CYLENDRICAL:
            len = sizeof(sar_parm_contact_bounds_cylendrical_struct);
            break;

          case SAR_PARM_CONTACT_BOUNDS_RECTANGULAR:
            len = sizeof(sar_parm_contact_bounds_rectangular_struct);
            break;

          case SAR_PARM_GROUND_ELEVATION:
            len = sizeof(sar_parm_ground_elevation_struct);
            break;

          case SAR_PARM_OBJECT_NAME:
            len = sizeof(sar_parm_object_name_struct);
            break;

          case SAR_PARM_OBJECT_MAP_DESCRIPTION:
            len = sizeof(sar_parm_object_map_description_struct);
            break;

          case SAR_PARM_FUEL:
            len = sizeof(sar_parm_fuel_struct);
            break;

          case SAR_PARM_HITPOINTS:
            len = sizeof(sar_parm_hitpoints_struct);
            break;

          case SAR_PARM_ENGINE_STATE:
            len = sizeof(sar_parm_engine_state_struct);
            break;

          case SAR_PARM_PASSENGERS:
            len = sizeof(sar_parm_passengers_struct);
            break;

          case SAR_PARM_HUMAN_MESSAGE_ENTER:
            len = sizeof(sar_parm_human_message_enter_struct);
            break;

          case SAR_PARM_HUMAN_REFERANCE:
            len = sizeof(sar_parm_human_referance_struct);
            break;


	  default:
	    fprintf(
		stderr,
 "SARParmNew(): Unsupported type `%i'.\n",
		type
	    );
	    break;
	}

	/* If length is positive it means we got a valid structure. */
	if(len > 0)
	{
	    p = calloc(1, len);
	    if(p != NULL)
		(*(int *)p) = type;
	}

	return(p);
}

/*
 *	Deallocates the given parameter structure and all its allocated
 *	substructures.
 */
void SARParmDelete(void *p)
{
	int type;
	sar_parm_name_struct *p_name;
	sar_parm_description_struct *p_description;
	sar_parm_player_model_file_struct *p_player_model_file;
	sar_parm_weather_struct *p_weather;
	sar_parm_register_location_struct *p_register_location;
	sar_parm_scene_map_struct *p_scene_map;
	sar_parm_scene_ground_tile_struct *p_scene_ground_tile;
	sar_parm_texture_base_directory_struct *p_texture_base_directory;
	sar_parm_texture_load_struct *p_texture_load;
	sar_parm_mission_scene_file_struct *p_mission_scene_file;
	sar_parm_mission_begin_at_struct *p_mission_begin_at;
	sar_parm_mission_arrive_at_struct *p_mission_arrive_at;
	sar_parm_mission_message_success_struct *p_mission_message_success;
	sar_parm_mission_message_fail_struct *p_mission_message_fail;
	sar_parm_mission_add_intercept_struct *p_mission_add_intercept;
	sar_parm_new_helipad_struct *p_new_helipad;
	sar_parm_new_runway_struct *p_new_runway;
	sar_parm_new_human_struct *p_new_human;
	sar_parm_new_fire_struct *p_new_fire;
	sar_parm_new_smoke_struct *p_new_smoke;
	sar_parm_new_premodeled_struct *p_new_premodeled;
	sar_parm_select_object_by_name *p_select_object_by_name;
	sar_parm_model_file_struct *p_model_file;
	sar_parm_object_name_struct *p_object_name;
	sar_parm_object_map_description_struct *p_object_map_description;
	sar_parm_human_message_enter_struct *p_human_message_enter;
	sar_parm_human_referance_struct *p_human_referance;


	if(p == NULL)
	    return;

	type = (*(int *)p);
	switch(type)
	{
	  case SAR_PARM_NAME:
	    p_name = (sar_parm_name_struct *)p;
	    free(p_name->name);
	    break;

	  case SAR_PARM_DESCRIPTION:
	    p_description = (sar_parm_description_struct *)p;
	    free(p_description->description);
	    break;

	  case SAR_PARM_PLAYER_MODEL_FILE:
	    p_player_model_file = (sar_parm_player_model_file_struct *)p;
	    free(p_player_model_file->file);
	    break;

	  case SAR_PARM_WEATHER:
	    p_weather = (sar_parm_weather_struct *)p;
	    free(p_weather->weather_preset_name);
	    break;

	  case SAR_PARM_REGISTER_LOCATION:
	    p_register_location = (sar_parm_register_location_struct *)p;
	    free(p_register_location->name);
	    break;

	  case SAR_PARM_SCENE_MAP:
	    p_scene_map = (sar_parm_scene_map_struct *)p;
	    free(p_scene_map->file);
	    break;

	  case SAR_PARM_SCENE_GROUND_TILE:
	    p_scene_ground_tile = (sar_parm_scene_ground_tile_struct *)p;
	    free(p_scene_ground_tile->texture_name);
	    break;

	  case SAR_PARM_TEXTURE_BASE_DIRECTORY:
	    p_texture_base_directory = (sar_parm_texture_base_directory_struct *)p;
	    free(p_texture_base_directory->directory);
	    break;

	  case SAR_PARM_TEXTURE_LOAD:
	    p_texture_load = (sar_parm_texture_load_struct *)p;
	    free(p_texture_load->name);
	    free(p_texture_load->file);
	    break;

	  case SAR_PARM_MISSION_SCENE_FILE:
	    p_mission_scene_file = (sar_parm_mission_scene_file_struct *)p;
	    free(p_mission_scene_file->file);
	    break;

	  case SAR_PARM_MISSION_BEGIN_AT:
	    p_mission_begin_at = (sar_parm_mission_begin_at_struct *)p;
	    free(p_mission_begin_at->name);
	    break;

          case SAR_PARM_MISSION_ARRIVE_AT:
            p_mission_arrive_at = (sar_parm_mission_arrive_at_struct *)p;
            free(p_mission_arrive_at->name);
            break;

	  case SAR_PARM_MISSION_MESSAGE_SUCCESS:
	    p_mission_message_success = (sar_parm_mission_message_success_struct *)p;
	    free(p_mission_message_success->message);
	    break;

	  case SAR_PARM_MISSION_MESSAGE_FAIL:
	    p_mission_message_fail = (sar_parm_mission_message_fail_struct *)p;
	    free(p_mission_message_fail->message);
	    break;

	  case SAR_PARM_MISSION_ADD_INTERCEPT:
	    p_mission_add_intercept = (sar_parm_mission_add_intercept_struct *)p;
	    free(p_mission_add_intercept->name);
	    break;

	  case SAR_PARM_NEW_HELIPAD:
	    p_new_helipad = (sar_parm_new_helipad_struct *)p;
	    free(p_new_helipad->style);
	    free(p_new_helipad->label);
	    free(p_new_helipad->ref_obj_name);
	    break;

	  case SAR_PARM_NEW_RUNWAY:
	    p_new_runway = (sar_parm_new_runway_struct *)p;
	    free(p_new_runway->north_label);
	    free(p_new_runway->south_label);
	    break;

	  case SAR_PARM_NEW_HUMAN:
	    p_new_human = (sar_parm_new_human_struct *)p;
	    free(p_new_human->type_name);
	    break;

	  case SAR_PARM_NEW_FIRE:
	    p_new_fire = (sar_parm_new_fire_struct *)p;
	    break;

	  case SAR_PARM_NEW_SMOKE:
	    p_new_smoke = (sar_parm_new_smoke_struct *)p;
	    break;

	  case SAR_PARM_NEW_PREMODELED:
	    p_new_premodeled = (sar_parm_new_premodeled_struct *)p;
	    free(p_new_premodeled->model_type);
	    StringFreeArray(p_new_premodeled->argv, p_new_premodeled->argc);
	    break;

	  case SAR_PARM_SELECT_OBJECT_BY_NAME:
	    p_select_object_by_name = (sar_parm_select_object_by_name *)p;
	    free(p_select_object_by_name->name);
	    break;

	  case SAR_PARM_MODEL_FILE:
	    p_model_file = (sar_parm_model_file_struct *)p;
	    free(p_model_file->file);
	    break;

	  case SAR_PARM_OBJECT_NAME:
	    p_object_name = (sar_parm_object_name_struct *)p;
	    free(p_object_name->name);
	    break;

	  case SAR_PARM_OBJECT_MAP_DESCRIPTION:
	    p_object_map_description = (sar_parm_object_map_description_struct *)p;
	    free(p_object_map_description->description);
	    break;

	  case SAR_PARM_HUMAN_MESSAGE_ENTER:
	    p_human_message_enter = (sar_parm_human_message_enter_struct *)p;
	    free(p_human_message_enter->message);
	    break;

	  case SAR_PARM_HUMAN_REFERANCE:
	    p_human_referance = (sar_parm_human_referance_struct *)p;
	    free(p_human_referance->referance_name);
	    break;

	  default:
/* Don't warn, we're just deallocating if substructure had allocated
   resources.
	    fprintf(
		stderr,
 "SARParmDelete(): Unsupported parm type `%i'.\n",
		type
	    );
 */
	    break;
	}

	/* Deallocate structure itself. */
	free(p);

	return;
}

/*
 *	Deallocates the entire list of parameters.
 */
void SARParmDeleteAll(void ***ptr, int *total)
{
	int i;

	if((ptr == NULL) || (total == NULL))
	    return;

	for(i = 0; i < (*total); i++)
	{
	    SARParmDelete((*ptr)[i]);
	    (*ptr)[i] = NULL;
	}

	if((*ptr) != NULL)
	{
	    free(*ptr);
	    (*ptr) = NULL;
	}
	(*total) = 0;

	return;
}


/*
 *	Seeks s to next argument.
 *
 *	If s is "red green blue" then return will be "green blue".
 */
static const char *NEXT_ARG(const char *s)
{
	if(s == NULL)
	    return(s);

	/* Seek past current arg till next blank or end of string. */
	while(!ISBLANK(*s) && ((*s) != '\0'))
	    s++;

	/* Seek past spaces to next arg. */
	while(ISBLANK(*s))
	    s++;

	return(s);
}

/*
 *      Safe form of atof, returns a double.
 */
static double ATOF(const char *s)
{        
        if(s == NULL)  
            return(0.0);
        else
            return(atof(s));
}               

/*
 *	Safe form of atoi.
 */
static int ATOI(const char *s)
{
	if(s == NULL)
	    return(0);
	else
	    return(atoi(s));
}

/*
 *      Safe form of atol.
 */
static long ATOL(const char *s)
{
        if(s == NULL)
            return(0);
        else
            return(atol(s));
}

/*
 *	Terminates the string on the first black char encountered.
 */
static void CHOP_STR_BLANK(char *s)
{
	if(s == NULL)
	    return;

	while((*s) != '\0')
	{
	    if(ISBLANK(*s))
	    {
		(*s) = '\0';
		break;
	    }
	    else
	    {
		s++;
	    }
	}
}


/*
 *      Return a statically allocated string indicating the
 *      operation fetched from the pointed to fp.  Return can be
 *      an empty string if there was no op string to be found or
 *      error.
 *
 *      fp is positioned at the end of the op string which the next
 *      fetch can be used to get its argument. If a new line character
 *      is detected at the end of string on file, then the fp will be
 *      repositioned at that new line character. The next reading of fp
 *      will read that new line character.
 */
static char *SARParmLoadFromFileGetParmString(FILE *fp)
{
        int c, i;
#define len     80
        static char rtn_str[len];

        (*rtn_str) = '\0';

        if(fp == NULL)
            return(rtn_str);

        /* Seek past spaces. */
        FSeekPastSpaces(fp);

        /* Itterate to get parameter. */
        for(i = 0; i < len; i++)
        {
            c = fgetc(fp);
            if((c == EOF) ||
               ISBLANK(c)
            )                   
            {
                rtn_str[i] = '\0';
                break;
            } 
            /* Escape sequence? */
            else if(c == '\\')
            {
                c = fgetc(fp);
                if(c == EOF)
                {
                    rtn_str[i] = '\0';
                    break;
                }
        
                if(c != '\\')
                    c = fgetc(fp);

                if(c == EOF)
                {
                    rtn_str[i] = '\0';
                    break;
                }
            }   
            /* New line? */
            else if(ISCR(c))
            {
                /* New line right at the end of the op string, seek
                 * back one character.
                 */
                fseek(fp, -1, SEEK_CUR);
 
                rtn_str[i] = '\0';
                break;          
            }
                
            rtn_str[i] = (char)c;
        }
            
#undef len
        return(rtn_str);
}

/*
 *	Loads the value in parm_str and val_str into the given parm
 *	array.
 *
 *	The number of lines_read will be incremented.
 *
 *	If filter_parm_type is not -1 then only parms matching the
 * 	specified type will be added to the given parm list.
 *
 *	Returns 0 on failure or 1 on success.
 */
static int SARParmLoadFromFileItterate(
	const char *filename,
	void ***parm, int *total_parms,
	const char *parm_str, const char *val_str,
	int *lines_read, int filter_parm_type
)
{
	int i, parm_num;
	const char *cstrptr;
	void *p;


	if((parm_str == NULL) || (val_str == NULL))
	    return(0);

	/* Increment lines_read by itterating through value string. */
	i = (*lines_read) + 1;
	cstrptr = val_str;
	while((*cstrptr) != '\0')
	{
	    if(ISCR(*cstrptr))
		i++;

	    cstrptr++;
	}
	(*lines_read) = i;

	/* Seek val_str past initial spaces. */
	while(ISBLANK(*val_str))
	    val_str++;

/* Adds parm p to the given parm list. */
#define DO_ADD_PARM	\
{ \
 parm_num = (*total_parms); \
 (*total_parms) = parm_num + 1; \
 (*parm) = (void **)realloc(*parm, (*total_parms) * sizeof(void *)); \
 if((*parm) == NULL) \
 { \
  (*total_parms) = 0; \
  return(0); \
 } \
 else \
 { \
  (*parm)[parm_num] = p; \
 } \
}

/* Checks if filter parm type matches with the given parm type.
 * If matches or filter_parm_type is -1 then returns true.
 */
#define FILTER_CHECK(t)	((filter_parm_type == (t)) || \
                         (filter_parm_type < 0) \
                        )


	/* Begin matching the given parameter name string. */
	/* Name (of file). */
	if(!strcasecmp(parm_str, "name") &&
           FILTER_CHECK(SAR_PARM_NAME)
	)
	{
	    sar_parm_name_struct *p_name =
		(sar_parm_name_struct *)SARParmNew(
		    SAR_PARM_NAME
		);
	    p = p_name;

	    free(p_name->name);
	    p_name->name = strdup(val_str);

	    DO_ADD_PARM
	}
        /* Description (of file). */
        else if((!strcasecmp(parm_str, "description") ||
                 !strcasecmp(parm_str, "desc")
                ) && FILTER_CHECK(SAR_PARM_DESCRIPTION)
	)
        {
            sar_parm_description_struct *p_description =
                (sar_parm_description_struct *)SARParmNew(
                    SAR_PARM_DESCRIPTION
                );
            p = p_description;

            free(p_description->description);
            p_description->description = strdup(val_str);

            DO_ADD_PARM
        }
        /* Player model file. */
        else if(!strcasecmp(parm_str, "player_model_file") &&
                FILTER_CHECK(SAR_PARM_PLAYER_MODEL_FILE)
	)
        {
            sar_parm_player_model_file_struct *p_player_model_file =
                (sar_parm_player_model_file_struct *)SARParmNew(
                    SAR_PARM_PLAYER_MODEL_FILE
                );
            p = p_player_model_file;

	    cstrptr = val_str;
            free(p_player_model_file->file);
            p_player_model_file->file = StringCopyAlloc(cstrptr);
	    CHOP_STR_BLANK(p_player_model_file->file);

            DO_ADD_PARM
        }
	/* Weather. */
        else if(!strcasecmp(parm_str, "weather") &&
                FILTER_CHECK(SAR_PARM_WEATHER)
	)
        {
            sar_parm_weather_struct *p_weather =
                (sar_parm_weather_struct *)SARParmNew(
                    SAR_PARM_WEATHER
                ); 
            p = p_weather;

            free(p_weather->weather_preset_name);
            p_weather->weather_preset_name = StringCopyAlloc(val_str);
	    /* do not chop off name. */
/*	    CHOP_STR_BLANK(p_weather->weather_preset_name); */
            
            DO_ADD_PARM
        }
        /* Time of day. */
        else if(!strcasecmp(parm_str, "time_of_day") &&
                FILTER_CHECK(SAR_PARM_TIME_OF_DAY)
        )
        {
            sar_parm_time_of_day_struct *p_time_of_day =
                (sar_parm_time_of_day_struct *)SARParmNew(
                    SAR_PARM_TIME_OF_DAY
                );
            p = p_time_of_day;

/* Need to work on this. */
	    p_time_of_day->tod = 0.0;

            DO_ADD_PARM
        }

        /* Registered location. */
        else if((!strcasecmp(parm_str, "register_location") ||
                 !strcasecmp(parm_str, "reg_location") ||
                 !strcasecmp(parm_str, "reg_loc")
                ) && FILTER_CHECK(SAR_PARM_REGISTER_LOCATION)
	)
        {
            sar_parm_register_location_struct *p_register_location =
                (sar_parm_register_location_struct *)SARParmNew(
                    SAR_PARM_REGISTER_LOCATION
                );
            p = p_register_location;

	    /* Position (xyz). */
	    cstrptr = val_str;
	    p_register_location->pos.x = ATOF(cstrptr);
	    cstrptr = NEXT_ARG(cstrptr);
	    p_register_location->pos.y = ATOF(cstrptr);
            cstrptr = NEXT_ARG(cstrptr);
            p_register_location->pos.z = SFMFeetToMeters(ATOF(cstrptr));

	    /* Direction. */
            cstrptr = NEXT_ARG(cstrptr);
            p_register_location->dir.heading = DEGTORAD(ATOF(cstrptr));
            cstrptr = NEXT_ARG(cstrptr);
            p_register_location->dir.pitch = DEGTORAD(ATOF(cstrptr));
            cstrptr = NEXT_ARG(cstrptr);
            p_register_location->dir.bank = DEGTORAD(ATOF(cstrptr));

	    /* Name. */
            cstrptr = NEXT_ARG(cstrptr);
	    free(p_register_location->name);
	    p_register_location->name = StringCopyAlloc(cstrptr);
	    /* Do not chop name. */
/*	    CHOP_STR_BLANK(p_register_location->name); */

            DO_ADD_PARM
        }
        /* Scene global positioning. */
        else if((!strcasecmp(parm_str, "scene_gps") ||
                 !strcasecmp(parm_str, "scene_global_position")
                ) && FILTER_CHECK(SAR_PARM_SCENE_GPS)
        )
        {
            sar_parm_scene_gps_struct *p_scene_gps = 
                (sar_parm_scene_gps_struct *)SARParmNew(
                    SAR_PARM_SCENE_GPS
                );  
            p = p_scene_gps;

            /* Center offset in degrees. */
            cstrptr = val_str;
	    SARParseLongitudeDMS(cstrptr, &p_scene_gps->dms_x_offset);
            cstrptr = NEXT_ARG(cstrptr);
	    SARParseLatitudeDMS(cstrptr, &p_scene_gps->dms_y_offset);

	    /* Planet radius in meters. */
            cstrptr = NEXT_ARG(cstrptr);
            p_scene_gps->planet_radius = ATOF(cstrptr);

            DO_ADD_PARM
	}
        /* Scene map. */
        else if(!strcasecmp(parm_str, "scene_map") &&
                FILTER_CHECK(SAR_PARM_SCENE_MAP)
	)
        {
            sar_parm_scene_map_struct *p_scene_map =
                (sar_parm_scene_map_struct *)SARParmNew(
                    SAR_PARM_SCENE_MAP
                );
            p = p_scene_map;

            /* Size in meters. */
            cstrptr = val_str;
            p_scene_map->width = ATOF(cstrptr);
            cstrptr = NEXT_ARG(cstrptr);
            p_scene_map->height = ATOF(cstrptr);

	    /* Texture file. */
            cstrptr = NEXT_ARG(cstrptr);
            free(p_scene_map->file);
            p_scene_map->file = StringCopyAlloc(cstrptr);
	    CHOP_STR_BLANK(p_scene_map->file);

            DO_ADD_PARM
        }
	/* Scene globally applied elevation. */
        else if(!strcasecmp(parm_str, "scene_elevation") &&
                FILTER_CHECK(SAR_PARM_SCENE_ELEVATION)
	)
        {
	    sar_parm_scene_elevation_struct *p_scene_elevation =
                (sar_parm_scene_elevation_struct *)SARParmNew(
                    SAR_PARM_SCENE_ELEVATION
                );
            p = p_scene_elevation;

            /* Elevation in meters. */
            cstrptr = val_str;
            p_scene_elevation->elevation = SFMFeetToMeters(ATOF(cstrptr));

	    DO_ADD_PARM
	}
        /* Scene globally applied cant angle. */
        else if(!strcasecmp(parm_str, "scene_cant") &&
                FILTER_CHECK(SAR_PARM_SCENE_CANT)
	)
        {
            sar_parm_scene_cant_struct *p_scene_cant =
                (sar_parm_scene_cant_struct *)SARParmNew(
                    SAR_PARM_SCENE_CANT
                );
            p = p_scene_cant;

            /* Angle in radians. */
            cstrptr = val_str;
            p_scene_cant->cant = DEGTORAD(ATOF(cstrptr));
         
            DO_ADD_PARM
        }
        /* Scene ground base flags. */
        else if(!strcasecmp(parm_str, "scene_ground_flags") &&
                FILTER_CHECK(SAR_PARM_SCENE_GROUND_FLAGS)
	)
        {
            sar_parm_ground_flags_struct *p_scene_ground_flags =
                (sar_parm_ground_flags_struct *)SARParmNew(
                    SAR_PARM_SCENE_GROUND_FLAGS
                );
            p = p_scene_ground_flags;


	    p_scene_ground_flags->flags = 0;

	    /* Get flags. */
            cstrptr = val_str;
	    while((cstrptr == NULL) ? 0 : ((*cstrptr) != '\0'))
	    {
		if(strcasepfx(cstrptr, "is_water") ||
                   strcasepfx(cstrptr, "iswater")
		)
		    p_scene_ground_flags->flags |= SAR_SCENE_BASE_FLAG_IS_WATER;

		cstrptr = NEXT_ARG(cstrptr);
	    }

            DO_ADD_PARM
        }
	/* Scene ground tile. */
	else if(!strcasecmp(parm_str, "scene_ground_tile") &&
                FILTER_CHECK(SAR_PARM_SCENE_GROUND_TILE)
	)
        {
            sar_parm_scene_ground_tile_struct *p_scene_ground_tile =
                (sar_parm_scene_ground_tile_struct *)SARParmNew(
                    SAR_PARM_SCENE_GROUND_TILE
                );
            p = p_scene_ground_tile;

            /* Tile size in meters. */
            cstrptr = val_str;
            p_scene_ground_tile->tile_width = ATOI(cstrptr);
            cstrptr = NEXT_ARG(cstrptr);
            p_scene_ground_tile->tile_height = ATOI(cstrptr);

	    /* Close range in meters. */
            cstrptr = NEXT_ARG(cstrptr);
            p_scene_ground_tile->close_range = ATOF(cstrptr);

	    /* Far solid color. */
            cstrptr = NEXT_ARG(cstrptr);
            p_scene_ground_tile->color.r = ATOF(cstrptr);
            cstrptr = NEXT_ARG(cstrptr);
            p_scene_ground_tile->color.g = ATOF(cstrptr);
            cstrptr = NEXT_ARG(cstrptr);
            p_scene_ground_tile->color.b = ATOF(cstrptr);
            cstrptr = NEXT_ARG(cstrptr);
            p_scene_ground_tile->color.a = ATOF(cstrptr);

            /* Texture name. */
            cstrptr = NEXT_ARG(cstrptr);
            free(p_scene_ground_tile->texture_name);
            p_scene_ground_tile->texture_name = StringCopyAlloc(cstrptr);
	    CHOP_STR_BLANK(p_scene_ground_tile->texture_name);

            DO_ADD_PARM
	}
        /* Texture base directory. */
        else if(!strcasecmp(parm_str, "texture_base_directory") &&
                FILTER_CHECK(SAR_PARM_TEXTURE_BASE_DIRECTORY)
	)
        {
            sar_parm_texture_base_directory_struct *p_texture_base_directory =
                (sar_parm_texture_base_directory_struct *)SARParmNew(
                    SAR_PARM_TEXTURE_BASE_DIRECTORY
                );
            p = p_texture_base_directory;

            free(p_texture_base_directory->directory);
            p_texture_base_directory->directory = StringCopyAlloc(val_str);
	    CHOP_STR_BLANK(p_texture_base_directory->directory);

            DO_ADD_PARM
        }
        /* Texture load. */
        else if(!strcasecmp(parm_str, "texture_load") &&
                FILTER_CHECK(SAR_PARM_TEXTURE_LOAD)
	)
        {
            sar_parm_texture_load_struct *p_texture_load =
                (sar_parm_texture_load_struct *)SARParmNew(
                    SAR_PARM_TEXTURE_LOAD
                );
            p = p_texture_load;

	    /* Name. */
            cstrptr = val_str;
            free(p_texture_load->name);  
            p_texture_load->name = StringCopyAlloc(cstrptr);
	    CHOP_STR_BLANK(p_texture_load->name);

            /* File. */
            cstrptr = NEXT_ARG(cstrptr);
            free(p_texture_load->file);
            p_texture_load->file = StringCopyAlloc(cstrptr);
	    CHOP_STR_BLANK(p_texture_load->file);

            /* Priority. */
            cstrptr = NEXT_ARG(cstrptr);
            p_texture_load->priority = ATOF(cstrptr);

            DO_ADD_PARM
        }

        /* Mission type. */
        else if(!strcasecmp(parm_str, "mission_type") &&
                FILTER_CHECK(SAR_PARM_MISSION_TYPE)
	)
        {   
            sar_parm_mission_type_struct *p_mission_type =
                (sar_parm_mission_type_struct *)SARParmNew(
                    SAR_PARM_MISSION_TYPE
                );
            p = p_mission_type;

            /* Mission type. */
            cstrptr = val_str;
            p_mission_type->mission_type = ATOI(cstrptr);

            DO_ADD_PARM
        }
        /* Mission scene file. */
        else if(!strcasecmp(parm_str, "scene_file") &&
                FILTER_CHECK(SAR_PARM_MISSION_SCENE_FILE)
	)
        {
            sar_parm_mission_scene_file_struct *p_mission_scene_file =
                (sar_parm_mission_scene_file_struct *)SARParmNew(
                    SAR_PARM_MISSION_SCENE_FILE
                );
            p = p_mission_scene_file;

            free(p_mission_scene_file->file);
            p_mission_scene_file->file = strdup(val_str);

            DO_ADD_PARM
        }
        /* Mission time left. */
        else if(!strcasecmp(parm_str, "time_left") &&
                FILTER_CHECK(SAR_PARM_MISSION_TIME_LEFT)
	)
        {
            sar_parm_mission_time_left_struct *p_mission_time_left =
                (sar_parm_mission_time_left_struct *)SARParmNew(
                    SAR_PARM_MISSION_TIME_LEFT
                );
            p = p_mission_time_left;

            /* Time left code. */
            cstrptr = val_str;
            p_mission_time_left->time_left_type = ATOI(cstrptr);

	    /* Time left in seconds (type double). */
            cstrptr = NEXT_ARG(cstrptr);
            p_mission_time_left->time_left = ATOF(cstrptr);

            DO_ADD_PARM
        }
        /* Mission begin at. */
        else if(!strcasecmp(parm_str, "begin_at") &&
                FILTER_CHECK(SAR_PARM_MISSION_BEGIN_AT)
	)
        {
            sar_parm_mission_begin_at_struct *p_mission_begin_at =
                (sar_parm_mission_begin_at_struct *)SARParmNew(
                    SAR_PARM_MISSION_BEGIN_AT
                );
            p = p_mission_begin_at;

            /* Begin at object name. */
            cstrptr = val_str;
	    free(p_mission_begin_at->name);
	    p_mission_begin_at->name = StringCopyAlloc(cstrptr);
	    /* Do not chop off string. */
/*	    CHOP_STR_BLANK(p_mission_begin_at->name); */

            DO_ADD_PARM
        }
        /* Mission arrive at. */
        else if(!strcasecmp(parm_str, "arrive_at") &&
                FILTER_CHECK(SAR_PARM_MISSION_ARRIVE_AT)
	)
        {
            sar_parm_mission_arrive_at_struct *p_mission_arrive_at =
                (sar_parm_mission_arrive_at_struct *)SARParmNew(
                    SAR_PARM_MISSION_ARRIVE_AT
                );
            p = p_mission_arrive_at;

            /* Arrive at object name. */
            cstrptr = val_str;
            free(p_mission_arrive_at->name);
            p_mission_arrive_at->name = StringCopyAlloc(cstrptr);
	    /* Do not chop off string. */
/*	    CHOP_STR_BLANK(p_mission_arrive_at->name); */

            DO_ADD_PARM
        }
        /* Mission message success. */
        else if(!strcasecmp(parm_str, "message_success") &&
                FILTER_CHECK(SAR_PARM_MISSION_MESSAGE_SUCCESS)
	)
        {
            sar_parm_mission_message_success_struct *p_mission_message_success =
                (sar_parm_mission_message_success_struct *)SARParmNew(
                    SAR_PARM_MISSION_MESSAGE_SUCCESS
                );
            p = p_mission_message_success;

            /* Mission message success. */
            cstrptr = val_str;
            free(p_mission_message_success->message);
            p_mission_message_success->message = StringCopyAlloc(cstrptr);
	    /* Do not chop off string. */
/*	    CHOP_STR_BLANK(p_mission_message_success->message); */

            DO_ADD_PARM
        }
        /* Mission message fail. */
        else if(!strcasecmp(parm_str, "message_fail") &&
                FILTER_CHECK(SAR_PARM_MISSION_MESSAGE_FAIL)
	)
        {
            sar_parm_mission_message_fail_struct *p_mission_message_fail =
                (sar_parm_mission_message_fail_struct *)SARParmNew(
                    SAR_PARM_MISSION_MESSAGE_FAIL
                );
            p = p_mission_message_fail;

            /* Mission message fail. */
            cstrptr = val_str;
            free(p_mission_message_fail->message);
            p_mission_message_fail->message = StringCopyAlloc(cstrptr);
	    /* Do not chop off string. */
/*	    CHOP_STR_BLANK(p_mission_message_fail->message); */

            DO_ADD_PARM
        }
        /* Mission humans tally. */
        else if(!strcasecmp(parm_str, "humans_tally") &&
                FILTER_CHECK(SAR_PARM_MISSION_HUMANS_TALLY)
	)
        {
            sar_parm_mission_humans_tally_struct *p_mission_humans_tally =
                (sar_parm_mission_humans_tally_struct *)SARParmNew(
                    SAR_PARM_MISSION_HUMANS_TALLY
                );
            p = p_mission_humans_tally;

            /* Total. */
            cstrptr = val_str;
	    p_mission_humans_tally->total = ATOI(cstrptr);

	    /* Need rescue. */
	    cstrptr = NEXT_ARG(cstrptr);
            p_mission_humans_tally->need_rescue = ATOI(cstrptr);

            DO_ADD_PARM
        }
        /* Mission add intercept. */
        else if(!strcasecmp(parm_str, "add_intercept") &&
                FILTER_CHECK(SAR_PARM_MISSION_ADD_INTERCEPT)
	)
        {
            sar_parm_mission_add_intercept_struct *p_mission_add_intercept =
                (sar_parm_mission_add_intercept_struct *)SARParmNew(
                    SAR_PARM_MISSION_ADD_INTERCEPT
                );
            p = p_mission_add_intercept;

            /* Referance code. */
            cstrptr = val_str;
            p_mission_add_intercept->ref_code = ATOI(cstrptr);

	    /* Handle rest by referance code to determine how many
	     * arguments.
	     */
	    switch(p_mission_add_intercept->ref_code)
	    {
	      /* Arrive or begin at location:
	       *
	       *	<radius> <urgency>
	       */
	      case 3: case 2:
		/* Radius. */
		cstrptr = NEXT_ARG(cstrptr);
		p_mission_add_intercept->radius = ATOF(cstrptr);
                /* Urgency. */
                cstrptr = NEXT_ARG(cstrptr);
                p_mission_add_intercept->urgency = ATOF(cstrptr);
		break;

	      /* Standard intercept way point:
	       *
	       *	<x> <y> <z> <radius> <urgency>
	       */
	      default:
		/* Position. */
                cstrptr = NEXT_ARG(cstrptr);
                p_mission_add_intercept->pos.x = ATOF(cstrptr);
                cstrptr = NEXT_ARG(cstrptr);
                p_mission_add_intercept->pos.y = ATOF(cstrptr);
                cstrptr = NEXT_ARG(cstrptr);
                p_mission_add_intercept->pos.z = ATOF(cstrptr);
                /* Radius. */
                cstrptr = NEXT_ARG(cstrptr);
                p_mission_add_intercept->radius = ATOF(cstrptr);            
                /* Urgency. */
                cstrptr = NEXT_ARG(cstrptr);
                p_mission_add_intercept->urgency = ATOF(cstrptr);
		break;
	    }

            /* Last argument is intercept name. */
            cstrptr = NEXT_ARG(cstrptr);
            free(p_mission_add_intercept->name);
            p_mission_add_intercept->name = StringCopyAlloc(cstrptr);
	    /* Do not chop off string. */
/*	    CHOP_STR_BLANK(p_mission_add_intercept->name); */

            DO_ADD_PARM
        }

        /* New object. */
        else if((!strcasecmp(parm_str, "create_object") ||
                 !strcasecmp(parm_str, "new_object") ||
                 !strcasecmp(parm_str, "add_object")
	        ) && FILTER_CHECK(SAR_PARM_NEW_OBJECT)
	)
        {
            sar_parm_new_object_struct *p_new_object =
                (sar_parm_new_object_struct *)SARParmNew(
                    SAR_PARM_NEW_OBJECT
                );
            p = p_new_object;

            /* First int is object type. */
            cstrptr = val_str;
	    p_new_object->object_type = ATOI(cstrptr);

            DO_ADD_PARM
        }
        /* New helipad. */
        else if((!strcasecmp(parm_str, "create_helipad") ||
                 !strcasecmp(parm_str, "new_helipad") ||
                 !strcasecmp(parm_str, "add_helipad")
                ) && FILTER_CHECK(SAR_PARM_NEW_HELIPAD)
        )
        {
            sar_parm_new_helipad_struct *p_new_helipad =
                (sar_parm_new_helipad_struct *)SARParmNew(
                    SAR_PARM_NEW_HELIPAD
                );
            p = p_new_helipad;

            /* Begin parsing line, format:
             *
             *      <type_name> <length> <width> <recession> <label>
             *      <edge_lighting> <has_fuel> <has_repair> <has_drop_off>
             *      <restarting_point> <ref_obj_name> <follow_ref>
             *      <offset_x> <offset_y> <offset_z>
             *      <offset_h> <offset_p> <offset_b>
             *
             * Note, underscore characters ('_') in label will be 
             * substituted for spaces after parsing.
             */

            /* Style name. */
            cstrptr = val_str;
	    free(p_new_helipad->style);
            p_new_helipad->style = StringCopyAlloc(cstrptr);
	    CHOP_STR_BLANK(p_new_helipad->style);

	    /* Length and width in meters. */
	    cstrptr = NEXT_ARG(cstrptr);
	    p_new_helipad->length = ATOF(cstrptr);
            cstrptr = NEXT_ARG(cstrptr);
            p_new_helipad->width = ATOF(cstrptr);

	    /* Recession in meters. */
            cstrptr = NEXT_ARG(cstrptr);
            p_new_helipad->recession = SFMFeetToMeters(ATOF(cstrptr));

            /* Label. */
            cstrptr = NEXT_ARG(cstrptr);
            free(p_new_helipad->label);
            p_new_helipad->label = StringCopyAlloc(cstrptr);
	    CHOP_STR_BLANK(p_new_helipad->label);
	    /* Set has label flag? */
	    if(p_new_helipad->label != NULL)
		p_new_helipad->flags |= SAR_HELIPAD_FLAG_LABEL;

	    /* Edge lighting. */
	    cstrptr = NEXT_ARG(cstrptr);
	    if(StringIsYes(cstrptr))
		p_new_helipad->flags |= SAR_HELIPAD_FLAG_EDGE_LIGHTING;

            /* Has fuel. */
            cstrptr = NEXT_ARG(cstrptr);
            if(StringIsYes(cstrptr))
                p_new_helipad->flags |= SAR_HELIPAD_FLAG_FUEL;

            /* Has repair. */
            cstrptr = NEXT_ARG(cstrptr);
            if(StringIsYes(cstrptr))
                p_new_helipad->flags |= SAR_HELIPAD_FLAG_REPAIR;

            /* Has drop off. */
            cstrptr = NEXT_ARG(cstrptr);
            if(StringIsYes(cstrptr))
                p_new_helipad->flags |= SAR_HELIPAD_FLAG_DROPOFF;

            /* Is a restarting point? */
            cstrptr = NEXT_ARG(cstrptr);
            if(StringIsYes(cstrptr))
                p_new_helipad->flags |= SAR_HELIPAD_FLAG_RESTART_POINT;

	    /* Ref object name. */
	    cstrptr = NEXT_ARG(cstrptr);
            free(p_new_helipad->ref_obj_name);
            p_new_helipad->ref_obj_name = StringCopyAlloc(cstrptr);
	    CHOP_STR_BLANK(p_new_helipad->ref_obj_name);
            /* Set has ref object flag? */
            if(p_new_helipad->ref_obj_name != NULL)
                p_new_helipad->flags |= SAR_HELIPAD_FLAG_REF_OBJECT;

	    /* Referance offset. */
            cstrptr = NEXT_ARG(cstrptr);
            p_new_helipad->ref_offset.x = ATOF(cstrptr);
            cstrptr = NEXT_ARG(cstrptr);
            p_new_helipad->ref_offset.y = ATOF(cstrptr);
            cstrptr = NEXT_ARG(cstrptr);
            p_new_helipad->ref_offset.z = ATOF(cstrptr);

	    /* Referance direction. */
            cstrptr = NEXT_ARG(cstrptr);
            p_new_helipad->ref_dir.heading = DEGTORAD(ATOF(cstrptr));
            cstrptr = NEXT_ARG(cstrptr);
            p_new_helipad->ref_dir.pitch = DEGTORAD(ATOF(cstrptr));
            cstrptr = NEXT_ARG(cstrptr);
            p_new_helipad->ref_dir.bank = DEGTORAD(ATOF(cstrptr));

            DO_ADD_PARM
	}
        /* New runway. */
        else if((!strcasecmp(parm_str, "create_runway") ||
                 !strcasecmp(parm_str, "new_runway") ||
                 !strcasecmp(parm_str, "add_runway")
                ) && FILTER_CHECK(SAR_PARM_NEW_RUNWAY)
        )
        {
            sar_parm_new_runway_struct *p_new_runway =
                (sar_parm_new_runway_struct *)SARParmNew(
                    SAR_PARM_NEW_RUNWAY
                );
            p = p_new_runway;

	    /* Range in meters. */
	    cstrptr = val_str;
	    p_new_runway->range = ATOF(cstrptr);

	    /* Length and width in meters. */
	    cstrptr = NEXT_ARG(cstrptr);
            p_new_runway->length = ATOF(cstrptr);
            cstrptr = NEXT_ARG(cstrptr);
            p_new_runway->width = ATOF(cstrptr);

            DO_ADD_PARM
        }
        /* New human. */
        else if((!strcasecmp(parm_str, "create_human") ||
                 !strcasecmp(parm_str, "new_human") ||
                 !strcasecmp(parm_str, "add_human")
                ) && FILTER_CHECK(SAR_PARM_NEW_HUMAN)
        )
        {
            sar_parm_new_human_struct *p_new_human =
                (sar_parm_new_human_struct *)SARParmNew(
                    SAR_PARM_NEW_HUMAN
                );
            p = p_new_human;

	    /* Begin parsing line, format:
	     *
	     *	<type_name>
	     *	<need_rescue?> <sitting?> <lying?> <alert?>
	     *	<aware?> <in_water?> <on_streatcher?>
             */

	    /* First argument is type name. */
	    cstrptr = val_str;
	    free(p_new_human->type_name);
	    p_new_human->type_name = StringCopyAlloc(cstrptr);
	    CHOP_STR_BLANK(p_new_human->type_name);

            /* Get flags. */
            cstrptr = NEXT_ARG(cstrptr);
            while((cstrptr == NULL) ? 0 : ((*cstrptr) != '\0'))
            {
                /* Need rescue? */
                if(strcasepfx(cstrptr, "need_rescue") ||
                   strcasepfx(cstrptr, "needrescue")
                )
                    p_new_human->flags |= SAR_HUMAN_FLAG_NEED_RESCUE;
                /* Sitting? */
                else if(strcasepfx(cstrptr, "sitting") ||
                        strcasepfx(cstrptr, "sit")
                )
                    p_new_human->flags |= SAR_HUMAN_FLAG_SIT;
                /* Lying? */
                else if(strcasepfx(cstrptr, "lying") ||
                        strcasepfx(cstrptr, "lie")
                )
                    p_new_human->flags |= SAR_HUMAN_FLAG_LYING;
                /* Alert? */
                else if(strcasepfx(cstrptr, "alert"))
                    p_new_human->flags |= SAR_HUMAN_FLAG_ALERT;
                /* Aware? */
                else if(strcasepfx(cstrptr, "aware"))
                    p_new_human->flags |= SAR_HUMAN_FLAG_AWARE;
                /* In water? */
                else if(strcasepfx(cstrptr, "in_water") ||
                        strcasepfx(cstrptr, "inwater")
                )
                    p_new_human->flags |= SAR_HUMAN_FLAG_IN_WATER;
                /* On streatcher? */
                else if(strcasepfx(cstrptr, "on_streatcher") ||
                        strcasepfx(cstrptr, "onstreatcher")
                )
                    p_new_human->flags |= SAR_HUMAN_FLAG_ON_STREATCHER;

		cstrptr = NEXT_ARG(cstrptr);
	    }

	    DO_ADD_PARM
	}
        /* New fire object. */
        else if((!strcasecmp(parm_str, "create_fire") ||
                 !strcasecmp(parm_str, "new_fire") ||
                 !strcasecmp(parm_str, "add_fire")
                ) && FILTER_CHECK(SAR_PARM_NEW_FIRE)
        )
        {
            sar_parm_new_fire_struct *p_new_fire =
                (sar_parm_new_fire_struct *)SARParmNew(
                    SAR_PARM_NEW_FIRE
                );
            p = p_new_fire;

            /* Begin parsing line, format:
             *
             *      <radius> <height>
             */

            /* Radius in meters. */
            cstrptr = val_str;
            p_new_fire->radius = MAX(ATOF(cstrptr), 0.1);
            cstrptr = NEXT_ARG(cstrptr);
            p_new_fire->height = MAX(SFMFeetToMeters(ATOF(cstrptr)), 0.1);

	    DO_ADD_PARM
	}
        /* New smoke object. */
        else if((!strcasecmp(parm_str, "create_smoke") ||
                 !strcasecmp(parm_str, "new_smoke") ||
                 !strcasecmp(parm_str, "add_smoke")
                ) && FILTER_CHECK(SAR_PARM_NEW_SMOKE)
        )
        {
            sar_parm_new_smoke_struct *p_new_smoke =
                (sar_parm_new_smoke_struct *)SARParmNew(
                    SAR_PARM_NEW_SMOKE
                );
            p = p_new_smoke;

            /* Begin parsing line, format:
             *
             *      <offset_x> <offset_y> <offset_z> <radius_start>
	     *      <radius_max> <radius_rate> <hide_at_max> <respawn_int>
	     *      <total_units> <color_code>
             */

            /* Radius in meters. */
            cstrptr = val_str;
            p_new_smoke->offset.x = ATOF(cstrptr);
            cstrptr = NEXT_ARG(cstrptr);
            p_new_smoke->offset.y = ATOF(cstrptr);
            cstrptr = NEXT_ARG(cstrptr);
            p_new_smoke->offset.z = ATOF(cstrptr);
            cstrptr = NEXT_ARG(cstrptr);
            p_new_smoke->radius_start = ATOF(cstrptr);
            cstrptr = NEXT_ARG(cstrptr);
            p_new_smoke->radius_max = ATOF(cstrptr);
            cstrptr = NEXT_ARG(cstrptr);
            p_new_smoke->radius_rate = ATOF(cstrptr);
            cstrptr = NEXT_ARG(cstrptr);
            p_new_smoke->hide_at_max = ATOI(cstrptr);
            cstrptr = NEXT_ARG(cstrptr);
            p_new_smoke->respawn_int = ATOL(cstrptr);
            cstrptr = NEXT_ARG(cstrptr);
            p_new_smoke->total_units = ATOI(cstrptr);
            cstrptr = NEXT_ARG(cstrptr);
	    p_new_smoke->color_code = ATOI(cstrptr);

            DO_ADD_PARM
        }
        /* New premodeled object. */
        else if((!strcasecmp(parm_str, "create_premodeled") ||
                 !strcasecmp(parm_str, "new_premodeled") ||
                 !strcasecmp(parm_str, "add_premodeled")
                ) && FILTER_CHECK(SAR_PARM_NEW_PREMODELED)
        )
        {
            sar_parm_new_premodeled_struct *p_new_premodeled =
                (sar_parm_new_premodeled_struct *)SARParmNew(
                    SAR_PARM_NEW_PREMODELED
                );
            p = p_new_premodeled;

	    /* First argument is model type. */
	    cstrptr = val_str;
	    free(p_new_premodeled->model_type);
	    p_new_premodeled->model_type = StringCopyAlloc(cstrptr);
	    CHOP_STR_BLANK(p_new_premodeled->model_type);

	    /* Get remaining arguments as an exploded string array. */
	    StringFreeArray(p_new_premodeled->argv, p_new_premodeled->argc);
	    cstrptr = NEXT_ARG(cstrptr);
	    p_new_premodeled->argv = strexp(cstrptr, &p_new_premodeled->argc);

            DO_ADD_PARM
        }

        /* Select object by name. */
        else if(!strcasecmp(parm_str, "select_object_by_name") &&
                FILTER_CHECK(SAR_PARM_SELECT_OBJECT_BY_NAME)
        )
        {
            sar_parm_select_object_by_name *p_select_object_by_name =
                (sar_parm_select_object_by_name *)SARParmNew(
                    SAR_PARM_SELECT_OBJECT_BY_NAME
                );
            p = p_select_object_by_name;

            /* Name. */
            cstrptr = val_str;
            free(p_select_object_by_name->name);
            p_select_object_by_name->name = StringCopyAlloc(cstrptr);
	    /* Do not chop off string. */
/*	    CHOP_STR_BLANK(p_select_object_by_name->name); */

            DO_ADD_PARM
	}

	/* Model file. */
        else if(!strcasecmp(parm_str, "model_file") &&
                FILTER_CHECK(SAR_PARM_MODEL_FILE)
	)
        {
            sar_parm_model_file_struct *p_model_file =
                (sar_parm_model_file_struct *)SARParmNew(
                    SAR_PARM_MODEL_FILE
                );
            p = p_model_file;

	    /* File name. */
	    cstrptr = val_str;
	    free(p_model_file->file);
            p_model_file->file = StringCopyAlloc(cstrptr);
	    CHOP_STR_BLANK(p_model_file->file);

	    DO_ADD_PARM
	}
	/* Visual range. */
        else if(!strcasecmp(parm_str, "range") &&
                FILTER_CHECK(SAR_PARM_RANGE)
        )
        {
            sar_parm_range_struct *p_range =
                (sar_parm_range_struct *)SARParmNew(
                    SAR_PARM_RANGE
                );
            p = p_range;

	    /* Range in meters. */
	    cstrptr = val_str;
	    p_range->range = ATOF(cstrptr);

            DO_ADD_PARM
        }
	/* Visual range far model display. */
	else if(!strcasecmp(parm_str, "range_far") &&
                FILTER_CHECK(SAR_PARM_RANGE_FAR)
        )
        {
            sar_parm_range_far_struct *p_range_far =
                (sar_parm_range_far_struct *)SARParmNew(
                    SAR_PARM_RANGE_FAR
                );
            p = p_range_far;

            /* Far range in meters. */
            cstrptr = val_str;
            p_range_far->range_far = ATOF(cstrptr);

            DO_ADD_PARM
        }
	/* Translate. */
        else if((!strcasecmp(parm_str, "translate") ||
                 !strcasecmp(parm_str, "translation")
                ) && FILTER_CHECK(SAR_PARM_TRANSLATE)
	)
        {
            sar_parm_translate_struct *p_translate =
                (sar_parm_translate_struct *)SARParmNew(
                    SAR_PARM_TRANSLATE
                );
            p = p_translate;

            /* Translate in meters. */
            cstrptr = val_str;
            p_translate->translate.x = ATOF(cstrptr);
	    cstrptr = NEXT_ARG(cstrptr);
            p_translate->translate.y = ATOF(cstrptr);
            cstrptr = NEXT_ARG(cstrptr);
            p_translate->translate.z = SFMFeetToMeters(ATOF(cstrptr));

            DO_ADD_PARM
        }
        /* Translate random. */
        else if((!strcasecmp(parm_str, "translate_random") ||
                 !strcasecmp(parm_str, "translation_random")
                ) && FILTER_CHECK(SAR_PARM_TRANSLATE_RANDOM)
        )
        {
            sar_parm_translate_random_struct *p_translate_random =
                (sar_parm_translate_random_struct *)SARParmNew(
                    SAR_PARM_TRANSLATE_RANDOM
                );
            p = p_translate_random;

	    /* Bounds in meters. */
	    cstrptr = val_str;
	    p_translate_random->radius_bound = ATOF(cstrptr);

	    cstrptr = NEXT_ARG(cstrptr);
	    p_translate_random->z_bound = ATOF(cstrptr);

            DO_ADD_PARM
        }
        /* Rotate. */
        else if(!strcasecmp(parm_str, "rotate") &&
                FILTER_CHECK(SAR_PARM_ROTATE)
	)
        {
            sar_parm_rotate_struct *p_rotate =
                (sar_parm_rotate_struct *)SARParmNew(
                    SAR_PARM_ROTATE
                );
            p = p_rotate;

            /* Rotate in radians. */
            cstrptr = val_str;
            p_rotate->rotate.heading = DEGTORAD(ATOF(cstrptr));
            cstrptr = NEXT_ARG(cstrptr);
            p_rotate->rotate.pitch = DEGTORAD(ATOF(cstrptr));
            cstrptr = NEXT_ARG(cstrptr);
            p_rotate->rotate.bank = DEGTORAD(ATOF(cstrptr));

            DO_ADD_PARM
        }   
        /* No depth test. */
        else if(!strcasecmp(parm_str, "no_depth_test") &&
                FILTER_CHECK(SAR_PARM_NO_DEPTH_TEST)
	)
        {
            sar_parm_no_depth_test_struct *p_no_depth_test =
                (sar_parm_no_depth_test_struct *)SARParmNew(
                    SAR_PARM_NO_DEPTH_TEST
                );
            p = p_no_depth_test;

            /* No arguments. */

            DO_ADD_PARM
        }
        /* Polygon offset. */
        else if(!strcasecmp(parm_str, "polygon_offset") &&
                FILTER_CHECK(SAR_PARM_POLYGON_OFFSET)
        )
        {
            sar_parm_polygon_offset_struct *p_polygon_offset =
                (sar_parm_polygon_offset_struct *)SARParmNew(
                    SAR_PARM_POLYGON_OFFSET
                );
            p = p_polygon_offset;

	    /* Reset flags. */
	    p_polygon_offset->flags = SAR_OBJ_FLAG_POLYGON_OFFSET;

            /* Get flags. */
            cstrptr = val_str;
            while((cstrptr == NULL) ? 0 : ((*cstrptr) != '\0'))
            {
                /* Reverse offset? */
                if(strcasepfx(cstrptr, "reverse") ||
                   strcasepfx(cstrptr, "rev")
                )
		{
		    p_polygon_offset->flags &= ~SAR_OBJ_FLAG_POLYGON_OFFSET;
		    p_polygon_offset->flags |= SAR_OBJ_FLAG_POLYGON_OFFSET_REVERSE;
		}
                /* Write depth after polygon offsetting? */
                else if(strcasepfx(cstrptr, "write_depth") ||
                        strcasepfx(cstrptr, "writedepth")
                )
		{
                    p_polygon_offset->flags |= SAR_OBJ_FLAG_POLYGON_OFFSET_WRITE_DEPTH;
		}

		cstrptr = NEXT_ARG(cstrptr);
	    }

            DO_ADD_PARM
        }
        /* Countact bounds spherical. */
        else if(!strcasecmp(parm_str, "contact_spherical") &&
                FILTER_CHECK(SAR_PARM_CONTACT_BOUNDS_SPHERICAL)
	)
        {
            sar_parm_contact_bounds_spherical_struct *p_contact_bounds_spherical =
                (sar_parm_contact_bounds_spherical_struct *)SARParmNew(
                    SAR_PARM_CONTACT_BOUNDS_SPHERICAL
                );
            p = p_contact_bounds_spherical;

            /* Radius in meters. */
	    cstrptr = val_str;
            p_contact_bounds_spherical->radius = ATOF(cstrptr);

            DO_ADD_PARM
        }
        /* Countact bounds cylendrical. */
        else if(!strcasecmp(parm_str, "contact_cylendrical") &&
                FILTER_CHECK(SAR_PARM_CONTACT_BOUNDS_CYLENDRICAL)
	)
        {
            sar_parm_contact_bounds_cylendrical_struct *p_contact_bounds_cylendrical =
                (sar_parm_contact_bounds_cylendrical_struct *)SARParmNew(
                    SAR_PARM_CONTACT_BOUNDS_CYLENDRICAL
                );
            p = p_contact_bounds_cylendrical;

            /* Radius in meters. */
            cstrptr = val_str;
            p_contact_bounds_cylendrical->radius = ATOF(cstrptr);

	    /* Height min and max in meters. */
	    cstrptr = NEXT_ARG(cstrptr);
	    p_contact_bounds_cylendrical->height_min = ATOF(cstrptr);
            cstrptr = NEXT_ARG(cstrptr);
            p_contact_bounds_cylendrical->height_max = ATOF(cstrptr);

            DO_ADD_PARM
        }
        /* Countact bounds rectangular. */
        else if(!strcasecmp(parm_str, "contact_rectangular") &&
                FILTER_CHECK(SAR_PARM_CONTACT_BOUNDS_RECTANGULAR)
        )
        {
            sar_parm_contact_bounds_rectangular_struct *p_contact_bounds_rectangular =
                (sar_parm_contact_bounds_rectangular_struct *)SARParmNew(
                    SAR_PARM_CONTACT_BOUNDS_RECTANGULAR
                );
            p = p_contact_bounds_rectangular;

            /* X min and max. */
            cstrptr = val_str;
            p_contact_bounds_rectangular->x_min = ATOF(cstrptr);
	    cstrptr = NEXT_ARG(cstrptr);
            p_contact_bounds_rectangular->x_max = ATOF(cstrptr);

            /* Y min and max. */
            cstrptr = NEXT_ARG(cstrptr);
            p_contact_bounds_rectangular->y_min = ATOF(cstrptr);
            cstrptr = NEXT_ARG(cstrptr);
            p_contact_bounds_rectangular->y_max = ATOF(cstrptr);

            /* Z min and max. */
            cstrptr = NEXT_ARG(cstrptr);
            p_contact_bounds_rectangular->z_min = ATOF(cstrptr);
            cstrptr = NEXT_ARG(cstrptr);
            p_contact_bounds_rectangular->z_max = ATOF(cstrptr);

            DO_ADD_PARM
        }
        /* Ground elevation. */
        else if(!strcasecmp(parm_str, "ground_elevation") &&
                FILTER_CHECK(SAR_PARM_GROUND_ELEVATION)
	)
        {
            sar_parm_ground_elevation_struct *p_ground_elevation =
                (sar_parm_ground_elevation_struct *)SARParmNew(
                    SAR_PARM_GROUND_ELEVATION
                );
            p = p_ground_elevation;

	    /* Ground elevation in meters. */
	    cstrptr = val_str;
	    p_ground_elevation->elevation = SFMFeetToMeters(ATOF(cstrptr));

            DO_ADD_PARM
        }
        /* Object name. */
        else if((!strcasecmp(parm_str, "object_name") ||
                 !strcasecmp(parm_str, "obj_name")
                ) && FILTER_CHECK(SAR_PARM_OBJECT_NAME)
	)
        {
            sar_parm_object_name_struct *p_object_name =
                (sar_parm_object_name_struct *)SARParmNew(
                    SAR_PARM_OBJECT_NAME
                );
            p = p_object_name;

            cstrptr = val_str;
	    free(p_object_name->name);
  	    p_object_name->name = StringCopyAlloc(cstrptr);
	    /* Do not chop off string. */
/*	    CHOP_STR_BLANK(p_object_name->name); */

            DO_ADD_PARM
        }
        /* Object map description. */
        else if((!strcasecmp(parm_str, "object_map_description") ||
                 !strcasecmp(parm_str, "object_map_desc") ||
                 !strcasecmp(parm_str, "obj_map_description") ||
                 !strcasecmp(parm_str, "obj_map_desc")
                ) && FILTER_CHECK(SAR_PARM_OBJECT_MAP_DESCRIPTION)
        )
        {
            sar_parm_object_map_description_struct *p_object_map_description =
                (sar_parm_object_map_description_struct *)SARParmNew(
                    SAR_PARM_OBJECT_MAP_DESCRIPTION
                );
            p = p_object_map_description;

            cstrptr = val_str;
            free(p_object_map_description->description);
            p_object_map_description->description = StringCopyAlloc(cstrptr);
            /* Do not chop off string. */
/*	    CHOP_STR_BLANK(p_object_map_description->description); */

            DO_ADD_PARM
        }
        /* Object fuel. */
        else if(!strcasecmp(parm_str, "fuel") &&
                FILTER_CHECK(SAR_PARM_FUEL)
	)
        {
            sar_parm_fuel_struct *p_fuel =
                (sar_parm_fuel_struct *)SARParmNew(
                    SAR_PARM_FUEL
                );
            p = p_fuel;

	    /* Get fuel in kg. */
            cstrptr = val_str;
	    p_fuel->fuel = ATOF(cstrptr);

            cstrptr = NEXT_ARG(cstrptr);
            p_fuel->fuel_max = ATOF(cstrptr);

            DO_ADD_PARM
        }
        /* Object hit points. */
        else if(!strcasecmp(parm_str, "hitpoints") &&
                FILTER_CHECK(SAR_PARM_HITPOINTS)
	)
        {
            sar_parm_hitpoints_struct *p_hitpoints =
                (sar_parm_hitpoints_struct *)SARParmNew(
                    SAR_PARM_HITPOINTS
                );
            p = p_hitpoints;

            cstrptr = val_str;
            p_hitpoints->hitpoints = ATOF(cstrptr);

            cstrptr = NEXT_ARG(cstrptr);
            p_hitpoints->hitpoints_max = ATOF(cstrptr);

            DO_ADD_PARM
        }
        /* Engine state. */
        else if(!strcasecmp(parm_str, "engine_state") &&
                FILTER_CHECK(SAR_PARM_ENGINE_STATE)
	)
        {
            sar_parm_engine_state_struct *p_engine_state =
                (sar_parm_engine_state_struct *)SARParmNew(
                    SAR_PARM_ENGINE_STATE
                );
            p = p_engine_state;

            cstrptr = val_str;
            p_engine_state->state = ATOI(cstrptr);

            DO_ADD_PARM
	}
        /* Passengers. */
        else if(!strcasecmp(parm_str, "passengers") &&
                FILTER_CHECK(SAR_PARM_PASSENGERS)
	)
        {
            sar_parm_passengers_struct *p_passengers =
                (sar_parm_passengers_struct *)SARParmNew(
                    SAR_PARM_PASSENGERS
                );
            p = p_passengers;

            cstrptr = val_str;
            p_passengers->passengers = ATOI(cstrptr);

            cstrptr = NEXT_ARG(cstrptr);
            p_passengers->passengers_max = ATOI(cstrptr);

            DO_ADD_PARM
        }
        /* Human message enter. */
        else if((!strcasecmp(parm_str, "set_human_message_enter") ||
                 !strcasecmp(parm_str, "set_human_mesg_enter")
                ) && FILTER_CHECK(SAR_PARM_HUMAN_MESSAGE_ENTER)
	)
        {
            sar_parm_human_message_enter_struct *p_human_message_enter =
                (sar_parm_human_message_enter_struct *)SARParmNew(
                    SAR_PARM_HUMAN_MESSAGE_ENTER
                );
            p = p_human_message_enter;

            cstrptr = val_str;
	    free(p_human_message_enter->message);
	    p_human_message_enter->message = StringCopyAlloc(cstrptr);
            /* Do not chop off string. */
/*	    CHOP_STR_BLANK(p_human_message_enter->message); */

            DO_ADD_PARM
        }
        /* Human referance. */
        else if((!strcasecmp(parm_str, "human_referance") ||
                 !strcasecmp(parm_str, "human_ref")
                ) && FILTER_CHECK(SAR_PARM_HUMAN_REFERANCE)
        )
        {
            sar_parm_human_referance_struct *p_human_referance =
                (sar_parm_human_referance_struct *)SARParmNew(
                    SAR_PARM_HUMAN_REFERANCE
                );
            p = p_human_referance;

	    /* Begin parsing line, format:
	     *
	     *	<ref_name>
	     *  <run_to?> <run_away?>
	     */

            /* First argument is referance object name. */
            cstrptr = val_str;
            free(p_human_referance->referance_name);
            p_human_referance->referance_name = StringCopyAlloc(cstrptr);
            CHOP_STR_BLANK(p_human_referance->referance_name);

            /* Get flags. */
            cstrptr = NEXT_ARG(cstrptr);
            while((cstrptr == NULL) ? 0 : ((*cstrptr) != '\0'))
            {
                /* Run towards? */
                if(strcasepfx(cstrptr, "run_towards") ||
                   strcasepfx(cstrptr, "run_to")
                )
                    p_human_referance->flags |= SAR_HUMAN_FLAG_RUN_TOWARDS;
                /* Run away? */
                else if(strcasepfx(cstrptr, "run_away"))
                    p_human_referance->flags |= SAR_HUMAN_FLAG_RUN_AWAY;

                cstrptr = NEXT_ARG(cstrptr);
            }

            DO_ADD_PARM
	}

	/* Unsupported parameter. */
	else
	{
	    /* Print warning only if we are not filtering any
	     * parm types.
	     */
	    if(filter_parm_type < 0)
		fprintf(
		    stderr,
"%s: Line %i: Unsupported parameter `%s'.\n",
		    filename, (*lines_read), parm_str
		);
	}

#undef FILTER_CHECK
#undef DO_ADD_PARM

	return(1);
}

/*
 *	Loads a set of parameters from file.
 *
 *	The given file_format is a hint to the file's format.
 *
 *	Returns non-zero on error.
 *
 *	The given parm and total_parms are assumed valid but undefined.
 */
int SARParmLoadFromFile(
        const char *filename, int file_format,
	void ***parm, int *total_parms,
	int filter_parm_type,
        void *client_data,
        int (*progress_func)(void *, long, long)
)
{
        int c, lines_read = 0;
        FILE *fp;
        long file_size;
	const char *parm_str;
	char *val_str;
	struct stat stat_buf;


	/* Reset loaded parms array. */
	(*parm) = NULL;
	(*total_parms) = 0;

	if(filename == NULL)
	    return(-1);

	/* Check if file exists. */
        if(stat(filename, &stat_buf))
            return(-1);

	/* Get file size. */
        file_size = stat_buf.st_size;

        /* Open file. */
        fp = FOpen(filename, "rb");
        if(fp == NULL)
            return(-1);

        /* Begin reading file. */
        while(1)
        {
            c = fgetc(fp);
            if(c == EOF)
                break;

	    /* Call progress callback. */
	    if(progress_func != NULL)
	    {
		if(progress_func(client_data, ftell(fp), file_size))
		    break;
	    }

            /* Read past leading spaces. */
            if(ISBLANK(c))
            {
                FSeekPastSpaces(fp);

                /* Get first non space character. */
                c = fgetc(fp);
                if(c == EOF)
                    break;
            }

            /* Newline character? */
            if(ISCR(c))
	    {
		lines_read++;
                continue;
	    }

            /* Comment? */
            if(ISCOMMENT(c))
            {
                FSeekNextLine(fp);      /* Seek to next line. */
		lines_read++;
                continue;
            }

            /* Seek back one character for fetching this parm. */
            fseek(fp, -1, SEEK_CUR);

            /* Get pointer to parameter string. */
            parm_str = (const char *)SARParmLoadFromFileGetParmString(fp);

	    /* Get value string, escape sequences parsed for escaped
	     * newline characters (escaped newlines not saved).
	     */
	    val_str = FGetString(fp);

	    /* Handle this parameter. */
	    SARParmLoadFromFileItterate(
		filename, parm, total_parms,
		parm_str, (const char *)val_str,
		&lines_read, filter_parm_type
	    );

	    /* Deallocate value string. */
	    if(val_str != NULL)
	    {
		free(val_str);
		val_str = NULL;
	    }
	}

	/* Close file. */
	FClose(fp);

	return(0);
}
