#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>

#include "sfm.h"

#include "horizon.h"
#include "sarreality.h"

#include "obj.h"

#include "simcb.h"
#include "simutils.h"
#include "simsurface.h"
#include "simcontact.h"
#include "simop.h"
#include "simmanage.h"

#include "sar.h"


int SARSimUpdateScene(
        sar_core_struct *core_ptr,
        sar_scene_struct *scene
);
int SARSimUpdateSceneObjects(
        sar_core_struct *core_ptr,
        sar_scene_struct *scene
);


#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))

#define DEGTORAD(d)     ((d) * PI / 180)
#define RADTODEG(r)     ((r) * 180 / PI)


/*
 *	Updates the given scene structure's members (but does not update
 *	any objects).
 *
 *	Time of day will be updated on the scene, then along with the
 *	primary light position light_pos and color light_color.
 *
 *	Horizon texture will be (re)generated as need per a given interval
 *	(every 5 minutes).
 *
 *	Returns 0 on success and non-zero on error or memory pointer
 *	change.
 */
int SARSimUpdateScene(sar_core_struct *core_ptr, sar_scene_struct *scene)
{
	double scene_lumination_coeff, horizon_sat_max_coeff;
	int prev_tod_code, new_tod_code;
	sar_position_struct *pos;
	sar_color_struct *c;
	sar_scene_horizon_struct *horizon_ptr;

/* Macro to convert hours into seconds. */
#define HTOS(h)	((h) * 3600.0)


	if((core_ptr == NULL) || (scene == NULL))
	    return(-1);


	/* Get pointer to horizon structure. */
	horizon_ptr = &scene->horizon;


	/* Update time of day (in seconds) since midnight. */
	scene->tod += (double)(
	    (double)lapsed_millitime * time_compression / 1000.0
	);

	/* Sanitize time of day. */
	while(scene->tod >= HTOS(24.0))
	    scene->tod -= HTOS(24.0);
	while(scene->tod < HTOS(0.0))
	    scene->tod += HTOS(24.0);

	/* Record previous time of day code. */
	prev_tod_code = scene->tod_code;

	/* Set time of day code depending on time of day in seconds
	 * updated above.
	 */
	if(scene->tod < HTOS(6.0))		/* Before 6:00 am. */
	    new_tod_code = SAR_SCENE_TOD_NIGHT;
	else if(scene->tod < HTOS(7.0))		/* Before 7:00 am. */
	    new_tod_code = SAR_SCENE_TOD_DAWN;
        else if(scene->tod < HTOS(17.0))	/* Before 5:00 pm. */
            new_tod_code = SAR_SCENE_TOD_DAY;
	else if(scene->tod < HTOS(18.0))	/* Before 6:00 pm. */
	    new_tod_code = SAR_SCENE_TOD_DUSK;
	else					/* After 6:00 pm. */
	    new_tod_code = SAR_SCENE_TOD_NIGHT;

	scene->tod_code = new_tod_code;


	/* Calculate global scene lumination coeff for time 4:00 to
	 * 20:00, where 0.0 is darkest and 1.0 is brightest. Darkest
	 * occures at 4:00 and 20:00, brightest is at 12:00.
	 *
	 * Also calculate horizon saturation extreme coefficient, the
	 * coefficient to determine the amount of dawn/dusk saturation
	 * on the horizon.
	 */
	/* First calculate from -1.0 to 1.0 linearly. */
	scene_lumination_coeff = (scene->tod - HTOS(12.0)) / HTOS(8.0);
	if(scene_lumination_coeff < 0.0)
	{
	    /* Before 12:00. */

            /* Horizon saturation coefficient, calculate from
             * scene lumination so far (with power of 2).   
             */
	    horizon_sat_max_coeff = CLIP(
                1.0 - pow(-scene_lumination_coeff, 2), 0.0, 1.0
            );

	    /* Scene lumination; flip, change sign, and apply non-linear
	     * curvature (power of 5).
	     */
	    scene_lumination_coeff = CLIP(
		1.0 - pow(-scene_lumination_coeff, 5), 0.0, 1.0
	    );
	}
	else
	{
	    /* 12:00 of after. */

            /* Horizon saturation coefficient, calculate from
	     * scene lumination so far (with power of 2).
	     */
            horizon_sat_max_coeff = CLIP(
                1.0 - pow(scene_lumination_coeff, 2), 0.0, 1.0
            );

	    /* Scene lumination; flip and apply non-linear
	     * curvature (power of 5).
	     */
	    scene_lumination_coeff = CLIP(
		1.0 - pow(scene_lumination_coeff, 5), 0.0, 1.0
	    );
	}

	/* Move scene's global light position relative to camera based
	 * on time of day.
	 *
	 * The scene's global light is in a orbital pattern depending on
	 * the time of day, to simulate it as the nearest star (the sun).
	 */
	pos = &scene->light_pos;	/* Scene global light position. */
	if(scene->tod > HTOS(12.0))
	{
	    double orbital_coeff;

	    /* Afternoon or later, calculate orbital position coefficient
	     * linearly from 12:00 (coeff 1.0) to 21:00 (coeff 0.0).
	     */
	    orbital_coeff = 1.0 - MIN(
		(scene->tod - HTOS(12.0)) / HTOS(9.0),
		1.0
	    );

	    /* Calculate longitude aligned orbital position of light
	     * relative to camera's xy plane.
	     */
	    pos->x = -(cos(orbital_coeff * (0.5 * PI)) *
		SAR_PRILIGHT_TO_CAMERA_RAD
	    );
	    pos->y = 0.0;
	    pos->z = sin(orbital_coeff * (0.5 * PI)) *
		SAR_PRILIGHT_TO_CAMERA_RAD;
	}
	else
	{
	    double orbital_coeff;

	    /* Morning, calculate orbital position coefficient
             * linearly from 6:00 (coeff 0.0) to 12:00 (coeff 1.0).
             */
            orbital_coeff = MAX(
                (scene->tod - HTOS(6.0)) / HTOS(6.0),
                0.0
            );

            /* Calculate longitude aligned orbital position of light    
             * relative to camera's xy plane.
             */
            pos->x = cos(orbital_coeff * (0.5 * PI)) *
                SAR_PRILIGHT_TO_CAMERA_RAD;
            pos->y = 0.0;
            pos->z = sin(orbital_coeff * (0.5 * PI)) *
                SAR_PRILIGHT_TO_CAMERA_RAD;
	}
	/* Apply latitude offset to scene's global light position. */
	if(scene->planet_radius > 0.0)
	{
	    double lat_offset_deg = RADTODEG(pos->y / scene->planet_radius) +
		scene->dms_y_offset;
	    double lat_offset_rad;

	    lat_offset_rad = CLIP(
		DEGTORAD(lat_offset_deg),
		-0.5 * PI, 0.5 * PI
	    );
	    pos->y = pos->y + (SAR_PRILIGHT_TO_CAMERA_RAD *
		sin(-lat_offset_rad));
	    pos->z = MAX(pos->z - (SAR_PRILIGHT_TO_CAMERA_RAD *
		sin(-lat_offset_rad)), 0.0);
	}

	/* Adjust scene's global light color. */
        c = &scene->light_color;	/* Scene global light color. */
	c->a = 1.0;
	c->r = scene_lumination_coeff;
	c->g = scene_lumination_coeff;
	c->b = scene_lumination_coeff;


	/* Regenerate horizon as needed every 5 minutes. */
	if(horizon_ptr->last_tod != (int)(scene->tod / (5 * 60)))
	{
	    int i, total, tex_num;
	    double rc, gc, bc;		/* Coefficient color change. */
	    double darken_coeff, midpoint = 0.88;
	    sar_color_struct start_color, end_color;


	    /* Update last tod on horizon structure to the current
	     * to current time in 5 minute units so we know when to
	     * update the horizon again.
	     */
	    horizon_ptr->last_tod = (int)(scene->tod / (5 * 60));

/* Creates a new horizon texture on the horizon_ptr. */
#define DO_CREATE_TEXTURE	\
{ \
 tex_num = horizon_ptr->total_textures; \
 horizon_ptr->total_textures++; \
 horizon_ptr->texture = (v3d_texture_ref_struct **)realloc( \
  horizon_ptr->texture, \
  horizon_ptr->total_textures * sizeof(v3d_texture_ref_struct *) \
 ); \
 if(horizon_ptr->texture == NULL) \
 { \
  horizon_ptr->total_textures = 0; \
 } \
 else \
 { \
  horizon_ptr->texture[tex_num] = SARCreateHorizonTexture( \
   NULL, /* No name. */ \
   &start_color, &end_color, \
   32, midpoint \
  ); \
 } \
}

	    /* Adjust gradient horizon textures by recreating them.
	     * Do not add lumination/gamma to them, the light color
	     * will be multiplied during drawing.
	     */

	    for(i = 0; i < horizon_ptr->total_textures; i++)
		V3DTextureDestroy(horizon_ptr->texture[i]);
	    free(horizon_ptr->texture);
	    horizon_ptr->texture = NULL;
	    horizon_ptr->total_textures = 0;


	    /* Create textures, starting with most brightest. */
	    memcpy(
		&start_color, &scene->sky_nominal_color,
		sizeof(sar_color_struct)
	    );

	    /* Create 7 horizon textures from brightest to darkest. */
	    total = 7;
	    for(i = 0; i < total; i++)
	    {
		darken_coeff = (double)i / (double)(total - 1);

		rc = MIN(
		    (scene->sky_brighten_color.r * (1.0 - darken_coeff)) +
		    (scene->sky_darken_color.r * darken_coeff),
		    1.0
		);
                gc = MIN(
                    (scene->sky_brighten_color.g * (1.0 - darken_coeff)) +
                    (scene->sky_darken_color.g * darken_coeff),
                    1.0
                );
                bc = MIN(
                    (scene->sky_brighten_color.b * (1.0 - darken_coeff)) +
                    (scene->sky_darken_color.b * darken_coeff),
                    1.0
                );

                end_color.r = MIN(
		    (1.0 * horizon_sat_max_coeff) +
			(rc * (1.0 - horizon_sat_max_coeff)),
		    1.0
		);
                end_color.g = MIN(
                    (1.0 * horizon_sat_max_coeff) +
			(gc * (1.0 - horizon_sat_max_coeff)), 
                    1.0
                );
                end_color.b = MIN(
                    (1.0 * horizon_sat_max_coeff) +
			(bc * (1.0 - horizon_sat_max_coeff)),
                    1.0
                );

		DO_CREATE_TEXTURE
	    }
	}
#undef DO_CREATE_TEXTURE
#undef HTOS

	return(0);
}

/*
 *	First updates the timmings on the scene structure's SFMRealmStruct
 *	and then updates all objects with respect to the given scene and
 *	core structure.
 *
 *	Returns 0 on success and non-zero on error or memory pointer
 *	change.
 */
int SARSimUpdateSceneObjects(
	sar_core_struct *core_ptr, sar_scene_struct *scene
)
{
	int i, status;
	int *obj_total;
        sar_object_struct *obj_ptr, ***obj_pa;
	sar_object_aircraft_struct *obj_aircraft_ptr;
	sar_contact_bounds_struct *cb;
	SFMModelStruct *fdm;


	if((core_ptr == NULL) || (scene == NULL))
	    return(-1);

	/* Update timming on realm. */
	if(scene->realm != NULL)
	{
	    SFMSetTimming(scene->realm, lapsed_millitime);
	    SFMSetTimeCompression(scene->realm, time_compression);
	}


	/* Begin handling each object on core structure. */
        obj_pa = &core_ptr->object;
        obj_total = &core_ptr->total_objects;
        for(i = 0; i < (*obj_total); i++)
        {
            obj_ptr = (*obj_pa)[i];
            if(obj_ptr == NULL)
                continue;

	    status = 0;		/* Reset `SFM handled' value. */

	    /* SFM realm structure allocated? */
	    if(scene->realm != NULL)
	    {
		/* Handle by object type. */
		switch(obj_ptr->type)
		{
		  case SAR_OBJ_TYPE_AIRCRAFT:
		    obj_aircraft_ptr = (sar_object_aircraft_struct *)obj_ptr->data;
		    if(obj_aircraft_ptr == NULL)
			break;

                    fdm = obj_aircraft_ptr->fdm;
                    if(fdm == NULL)
			break;

		    /* Set object values to SFM structure. */
                    SARSimSetSFMValues(scene, obj_ptr);

                    /* Apply control positions if this is the
                     * player object.
                     */
                    if(obj_ptr == scene->player_obj_ptr)
		    {
                        SARSimApplyGCTL(core_ptr, obj_ptr);
		    }
		    else
		    {

 /* Otherwise apply AI? */

		    }

		    /* Set status to 1, noting that the SFM structure
		     * was handled (which we will actually do so just
		     * below).
		     */
		    status = 1;

                    /* Begin SFM updating. */
                    if(SFMForceApplyArtificial(scene->realm, fdm))
                        break;
/* Swapped, calculating natural forces after artifical gives us
 * more accurate center to ground height.
 */
                    if(SFMForceApplyNatural(scene->realm, fdm))
			break;
		    if(SFMForceApplyControl(scene->realm, fdm))
			break;

                    /* Get new object values from the just processed
		     * SFM.
		     */
                    SARSimGetSFMValues(scene, obj_ptr);
		    break;

 /* Add SFM handling support for other types of objects as needed here. */

		}
	    }
	    /* Check if object survived SFM handling, in other words is
	     * it still allocated?
	     */
	    if(status)
	    {
		/* SFM was handled on the object, so check if object
		 * is still allocated.
		 */
		if(SARObjIsAllocated(*obj_pa, *obj_total, i))
		    obj_ptr = (*obj_pa)[i];
		else
		    continue;
	    }

            /* Apply natural forces to object. */
            if(SARSimApplyNaturalForce(core_ptr, obj_ptr))
		continue;

            /* Apply artificial forces to object. */
            if(SARSimApplyArtificialForce(core_ptr, obj_ptr))
		continue;


            /* Get pointer to object's contact bounds structure (which
	     * may be NULL).
	     */
	    cb = obj_ptr->contact_bounds;

	    /* Can this object crash into other objects? */
            if(((cb == NULL) ? 0 : cb->crash_flags) &
		SAR_CRASH_FLAG_CRASH_OTHER
	    )
            {
		/* Perform object to object crash contact check. */
                if(SARSimCrashContactCheck(core_ptr, obj_ptr) > -1)
                {
                    /* Crash contact into another object has occured,
		     * so we need to check if this object is still
                     * allocated.
                     */
                    if(SARObjIsAllocated(*obj_pa, *obj_total, i))
                        obj_ptr = (*obj_pa)[i];
                    else
                        continue;
                }
            }

            /* Rescue basket contact check (only if this object is a
	     * type that has a rescue basket). If a contact is made
	     * then a valid index is returned and the proper procedure
	     * will have been taken.
	     */
	    if(SARSimRescueBasketContactCheck(core_ptr, obj_ptr) > -1)
	    {
		/* Rescue basket contact has occured, need to check if
		 * this object is still allocated.
		 */
		if(SARObjIsAllocated(*obj_pa, *obj_total, i))
		    obj_ptr = (*obj_pa)[i];
		else
		    continue;
            }

/* Spawn smoke? */
/* SmokeSpawnCheck */

            /* Check if the object is `too old', if it is then it
	     * will be deleted.
	     */
            if(SARSimDoMortality(core_ptr, obj_ptr))
	    {
                /* Object has been deleted, check if it is still
		 * allocated (which it probably won't be).
                 */
                if(SARObjIsAllocated(*obj_pa, *obj_total, i))
                    obj_ptr = (*obj_pa)[i];
		else
		    continue;
	    }

/* Add more object updates here. */

        }

	return(0);
}
