/* list_sids.c
 *
 * prints a list of all the SIDs and security contexts currently in use
 */

#include <ss.h>
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>

static void nomem();
static int list_sids();

#define DEFAULT_SIZE 255

char *malloc();


int compar(int *x, int *y)
{
	if (*x < *y)
		return (-1);
	else if (*x > *y)
		return (1);
	else
		return (0);
}


int main(int argc, char **argv)
{
	int ret_val, i, nel;
	security_id_t *sids;
	char *scontext;
	unsigned int ctxt_len;

	if (argc > 1) {
		ret_val = list_sids(argc, argv);
		exit(ret_val);
	}

	nel = DEFAULT_SIZE;
	sids = (security_id_t *) malloc(sizeof(security_id_t) * nel);
	if (!sids)
		nomem(&ret_val);

	ret_val = security_get_sids(sids, &nel);

	if (ret_val && (nel > DEFAULT_SIZE)) {
		/* A larger sids array was needed */
		free(sids);
		sids =
		    (security_id_t *) malloc(sizeof(security_id_t) * nel);
		if (!sids)
			nomem(&ret_val);
		ret_val = security_get_sids(sids, &nel);
	}

	if (ret_val) {
		perror("security_get_sids");
		free(scontext);
		exit(2);
	}

	qsort(sids, (size_t) nel, sizeof(security_id_t),
	      (int (*)()) compar);

	for (i = 0; i < nel; i++) {
		scontext = malloc(sizeof(char) * DEFAULT_SIZE);
		ctxt_len = DEFAULT_SIZE;
		memset((void *) scontext, '\0', DEFAULT_SIZE);

		if (!scontext)
			nomem(&ret_val);

		ret_val =
		    security_sid_to_context(sids[i], scontext, &ctxt_len);
		if (ret_val && (ctxt_len > DEFAULT_SIZE)) {
			/* A larger string for scontext was needed */
			scontext =
			    realloc(scontext, sizeof(char) * ctxt_len);
			if (!scontext)
				nomem(&ret_val);

			ret_val =
			    security_sid_to_context(sids[i], scontext,
						    &ctxt_len);
		}

		if (!ret_val) {
			(void) printf("\nSID %3ld -> Scontext  %s",
				      sids[i], scontext);
			(void) fflush(stdout);
		}

		free(scontext);
	}

	(void) printf("\n\n");
	(void) fflush(stdout);

	free(sids);

	exit(0);
}


void nomem(int *ret_val)
{
	*ret_val = -ENOMEM;

	(void) fprintf(stderr, "\nget_sids: out of memory\n\n");
	(void) fflush(stderr);

	exit(2);
}

static int list_sids(int argc, char **argv)
{
	char *ctxt, *ep, *s_sid;
	int len = 255, rv, k, ret_val;
	security_id_t sid;
	long tmp;

	ret_val = 0;

	for (k = 1; k < argc; k++) {
		s_sid = argv[k];

		tmp = strtol(s_sid, &ep, 10);
		if (*ep) {
			(void) fprintf(stderr, "SID '%s' is not numeric\n",
				       s_sid);
			(void) fflush(stderr);
			ret_val = 1;
			continue;
		}
		sid = (security_id_t) tmp;

		ctxt = malloc(len + 1);
		if (ctxt) {
			rv = security_sid_to_context(sid, ctxt, &len);
			if (rv) {
				if (errno == ENOSPC) {
					ctxt = realloc(ctxt, len + 1);
					if (ctxt) {
						rv = security_sid_to_context(sid, ctxt, &len);
						if (rv) {
							(void)
							    fprintf(stderr,
								    "security_sid_to_context(%d): %s\n",
								    sid,
								    strerror
								    (errno));
							(void)
							    fflush(stderr);
							free(ctxt);
							ret_val = 1;
						} else {
							(void)
							    printf
							    ("SID %3ld -> Scontext  %s\n",
							     sid, ctxt);
							(void)
							    fflush(stdout);
							free(ctxt);
						}
					} else {
						(void) fprintf(stderr,
							       "calloc(): %s\n",
							       strerror
							       (errno));
						(void) fflush(stderr);
						exit(1);
					}
				} else {
					(void) fprintf(stderr,
						       "security_sid_to_context(%d): %s\n",
						       sid,
						       strerror(errno));
					(void) fflush(stderr);
					free(ctxt);
					ret_val = 1;
				}
			} else {
				(void) printf("SID %3ld -> Scontext  %s\n",
					      sid, ctxt);
				(void) fflush(stdout);
				free(ctxt);
			}
		} else {
			(void) fprintf(stderr, "calloc(): %s\n",
				       strerror(errno));
			(void) fflush(stderr);
			exit(1);
		}
	}

	return (ret_val);
}
