/*
 * tcp_server
 * 
 * Create a listening socket on a specified port, 
 * accept connections and echo a message.
 *
 * Permit testing of socket_secure, listen_secure, 
 * accept_secure, getsockname_secure and getpeername_secure.
 * 
 * Calls socket_secure if '-s socket_context' is specified.
 * Calls listen_secure if '-u' or '-n newconn_context' is specified.
 *
 * Usage:  tcp_server [-s socket_context] [-u] [-n newconn_context] port
 *
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <getopt.h>
#include <stdio.h>

#include <socket_secure.h>
#include <ss.h>

void usage(char *progname) 
{
	printf("usage:  %s [-s socket_context] [-u] [-n newconn_context] port\n", progname);
	exit(1);
}


int main(int argc, char **argv)
{
	struct sockaddr_in sin;
	char buf[1024], context[255];
	int s, sinlen, ret, buflen, ctrl, c, useclient = 0, contextlen;
	security_id_t so_sid = 0, newconn_sid = 0, peer_sid = 0, tmp_sid = 0; 


	while ( (c = getopt(argc, argv, "s:n:u")) != EOF) {
		switch (c) {
		case 's':
		case 'n':
			ret = security_context_to_sid(optarg,strlen(optarg)+1,&tmp_sid);
			if (ret) {
				perror(optarg);
				exit(ret);
			}
			if (c == 's')
				so_sid = tmp_sid;
			else if (c == 'n')
				newconn_sid = tmp_sid;
			break;
		case 'u':
			useclient = 1;
			break;
		default:
			usage(argv[0]);
		}
	}

	if (optind != (argc - 1)) {
		usage(argv[0]);
	}

	/* Create the listening socket */
	if (so_sid) 
		s = socket_secure(AF_INET, SOCK_STREAM, 0, so_sid);
	else 
		s = socket(AF_INET, SOCK_STREAM, 0);
	if (s == -1) {
		perror("socket");
		exit(1);
	}

	/* Obtain the security context of the listening socket */
	sin.sin_family = AF_INET;
	sinlen = sizeof(struct sockaddr_in);
	ret = getsockname_secure(s, &sin, &sinlen, &tmp_sid);
	if (ret) {
		perror("getsockname_secure");
		exit(1);
	}

	contextlen = sizeof(context);
	ret = security_sid_to_context(tmp_sid, context, &contextlen);
	if (ret) {
		perror("security_sid_to_context");
		exit(1);
	}
	
	printf("Created socket with context %s\n", context);

	/* Bind to the specified port */
	sin.sin_port = atoi(argv[optind++]);
	sin.sin_port = htons(sin.sin_port);
	sin.sin_addr.s_addr = INADDR_ANY;
	sinlen = sizeof(struct sockaddr_in);
	ret = bind(s, (struct sockaddr *)&sin, sinlen);
	if (ret == -1) {
		perror("bind");
		exit(1);
	}

	/* Set state of socket to accept connections */
	if (newconn_sid || useclient) 
		ret = listen_secure(s, 5, newconn_sid, useclient);
	else
		ret = listen(s, 5);
	if (ret == -1) {
		perror("listen");
		exit(1);
	}
	
	while (1) {
		/* Accept a new connection and obtain the peer socket context */
		ctrl = accept_secure(s, (struct sockaddr*)&sin, &sinlen, &peer_sid);
		if (ctrl == -1) {
			perror("accept_secure");
			exit(1);
		}

		contextlen = sizeof(context);
		ret = security_sid_to_context(peer_sid, context, &contextlen);
		if (ret) {
			perror("security_sid_to_context");
			exit(1);
		}
	
		printf("Accepted connection from peer (%s,%d) with context %s\n", 
		       inet_ntoa(sin.sin_addr), ntohs(sin.sin_port), context);

		/* Obtain the new connection socket security context */
		ret = getsockname_secure(ctrl, &sin, &sinlen, &tmp_sid);
		if (ret) {
			perror("getsockname_secure");
			exit(1);
		}

		contextlen = sizeof(context);
		ret = security_sid_to_context(tmp_sid, context, &contextlen);
		if (ret) {
			perror("security_sid_to_context");
			exit(1);
		}

		printf("New socket created by connection has context %s\n", 
		       context);

		/* Verify that getpeername_secure is consistent with accept_secure */
		ret = getpeername_secure(ctrl, (struct sockaddr*)&sin, &sinlen, &tmp_sid);
		if (ret) {
			perror("getpeername_secure");
			exit(1);
		}

		if (tmp_sid != peer_sid) {
			printf("Getpeername_secure returned a different SID than accept_secure!\n");
			exit(1);
		}

		/* Receive and echo a message */
		buf[0] = 0;
		ret = recv(ctrl, buf, sizeof(buf), 0);
		if (ret == -1) {
			perror("recv");
			exit(1);
		}		    

		printf("Received message ##%s##\n", buf);
	    
		ret = send(ctrl, buf, sizeof(buf), 0);
		if (ret == -1) {
			perror("send");
			exit(1);
		}

		/* Close the connection */
		close(ctrl);
	}
}
