/*
 * Flask Netlink Userspace Library
 *
 * Copyright (c) 2001-2002 James Morris <jmorris@intercode.com.au>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */
#ifndef _LIBFLNETLINK_H_
#define _LIBFLNETLINK_H_

#include <errno.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/uio.h>
#include <asm/types.h>
#include <linux/netlink.h>
#include <linux/flask/flask_types.h>
#include <linux/flask/flnetlink.h>

#include "libflutil.h"

/* glibc damage */
#ifndef MSG_TRUNC
#define MSG_TRUNC 0x20
#endif

#define FLN_F_ACKS	0x00000001
#define FLN_F_BCAST	0x00000002

struct fln_handle
{
	int fd;
	u_int32_t flags;
	u_int32_t groups;
	u_int32_t sequence;
	struct sockaddr_nl local;
	struct sockaddr_nl peer;
};

struct fln_handle *fln_create_handle(u_int32_t flags, u_int32_t groups);
int fln_destroy_handle(struct fln_handle *h);

/* Timeout is in microseconds */
ssize_t fln_read(const struct fln_handle *h, unsigned char *buf, size_t len, int timeout);

int fln_message_type(const unsigned char *buf);
int fln_get_msgerr(const unsigned char *buf);
void *fln_get_msg(const unsigned char *buf);
char *fln_errstr(void);
void fln_perror(const char *s);
int fln_errno_set(void);
u_int32_t fln_sequence(struct fln_handle *h);
u_int32_t fln_current_seq(struct fln_handle *h);
int fln_fd(struct fln_handle *h);
u_int32_t fln_message_seq(const unsigned char *buf);
pid_t fln_message_pid(const unsigned char *buf);

/*
 * Perimeter table.
 */
int fln_perimeter_add(struct fln_handle *h, u_int32_t addr, u_int32_t mask);
int fln_perimeter_del(struct fln_handle *h, u_int32_t addr, u_int32_t mask);
int fln_perimeter_dump(struct fln_handle *h);
int fln_perimeter_flush(struct fln_handle *h);
void fln_parse_perim_entry(struct fln_handle *h, unsigned char *buf,
                           int status, void (*userfn)(struct flmsg_perim_entry *entry));

                                                  

/*
 * Mapping cache.
 */
int fln_cache_map_res(struct fln_handle *h, struct flmsg_map_res *msg, size_t len);
int fln_cache_get(struct fln_handle *h, u_int32_t addr, security_id_t rsid);
int fln_cache_dump(struct fln_handle *h);
int fln_cache_flush(struct fln_handle *h);
void fln_parse_cache_map(struct fln_handle *h, unsigned char *buf, int status,
                         void (*userfn)(struct flmsg_map_res *res));
                                                      
/*
 * Packet queue
 */
int fln_queue_dump(struct fln_handle *h);
int fln_queue_flush(struct fln_handle *h);
void fln_parse_queue_entry(struct fln_handle *h, unsigned char *buf, int status,
                           void (*userfn)(struct flmsg_queue_entry *entry));

/* Helpers */
void print_perimtab_entry(struct flmsg_perim_entry *entry);
void print_cache_entry(struct flmsg_base *base);
void print_queue_entry(struct flmsg_queue_entry *entry);
void print_mapreq(struct flmsg_map_req *req);
void print_mapres(struct flmsg_map_res *res);

#endif	/* _LIBFLNETLINK_H_ */


