/*
 * Security Context Mapping Protocol Daemon
 *
 * Peer database, synchronised with the kernel perimeter table.
 *
 * Copyright (c) 2001-2002 James Morris <jmorris@intercode.com.au>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <stdarg.h>
#include <syslog.h>
#include <linux/flask/flask_types.h>
#include <linux/flask/flnetlink.h>

#include "libflnetlink.h"
#include "libflutil.h"

#include "list.h"
#include "perimtab.h"
#include "server.h"
#include "flnetlink.h"
#include "debug.h"

struct perimtab_entry {
	struct list_head list;
	u_int32_t addr;
	u_int32_t mask;
};

static LIST_HEAD(perimtab_list);

static struct perimtab_entry *perimtab_find(u_int32_t addr, u_int32_t mask)
{
	struct list_head *i;
	
	for (i = perimtab_list.next; i != &perimtab_list; i = i->next) {
		struct perimtab_entry *entry = (struct perimtab_entry *)i;
		
		if (addr == entry->addr && mask == entry->mask)
			return entry;
	}
	
	return NULL;
}

static int perimtab_append(u_int32_t addr, u_int32_t mask)
{
	struct perimtab_entry *entry;
	
	if (perimtab_find(addr, mask))
		return 1;
	
	entry = malloc(sizeof(struct perimtab_entry));
	if (entry == NULL)
		return 0;

	memset(entry, 0, sizeof(struct perimtab_entry));
	
	entry->addr = addr;
	entry->mask = mask;
	
	list_add_tail(&entry->list, &perimtab_list);
	
	debug("perimtab add %u.%u.%u.%u/%d", NIPQUAD(addr), ip_masklen(mask));
	
	return 1;
}

static int perimtab_delete(u_int32_t addr, u_int32_t mask)
{
	struct perimtab_entry *entry;
	
	entry = perimtab_find(addr, mask);
	if (entry == NULL)
		return 0;
	
	list_del(&entry->list);
	free(entry);
	
	debug("perimtab del %u.%u.%u.%u/%d", NIPQUAD(addr), ip_masklen(mask));

	return 1;
}

int perimtab_match(u_int32_t addr)
{
	struct list_head *i;
	
	for (i = perimtab_list.next; i != &perimtab_list; i = i->next) {
		struct perimtab_entry *entry = (struct perimtab_entry *)i;
		
		if (entry->addr == (addr & entry->mask))
			return 1;
	}
	
	return 0;
}

void perimtab_append_entry(struct flmsg_perim_entry *entry)
{
	int rc;
	
	rc = perimtab_append(entry->entry.addr, entry->entry.mask);
	if (!rc)
		syslog(LOG_ERR, "%s: failed to add %u.%u.%u.%u/%d: %m",
		       __FUNCTION__, NIPQUAD(entry->entry.addr),
		       ip_masklen(entry->entry.mask));
	
	return;
}
                
void perimtab_delete_entry(struct flmsg_perim_entry *entry)
{
	int rc;
	
	rc = perimtab_delete(entry->entry.addr, entry->entry.mask);
	if (!rc)
		syslog(LOG_ERR, "%s: failed to delete %u.%u.%u.%u/%d",
		       __FUNCTION__, NIPQUAD(entry->entry.addr),
		       ip_masklen(entry->entry.mask));
	
	return;
}

void perimtab_flush(void)
{
	while (perimtab_list.next != &perimtab_list) {
		struct list_head *i = perimtab_list.next;
		struct perimtab_entry *entry = (struct perimtab_entry *)i;
		
        	list_del(i);
        	
        	debug("perimtab flush %u.%u.%u.%u/%d",
        	      NIPQUAD(entry->addr), ip_masklen(entry->mask));
        	    
        	free(entry);
        }

	return;
}

int perimtab_init(void)
{
	int rc;
	
	rc = flnetlink_perimeter_dump();
	if (rc < 0) {
		syslog(LOG_ERR, "flnetlink_perimeter_dump: %m: %s",
		       fln_errstr());
		return -1;
	}
	
	return 0;
}

void perimtab_exit(void)
{
	perimtab_flush();
}
